/*************************************************************************
 *
 *  $RCSfile: socket.c,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: vg $ $Date: 2003/04/15 17:42:01 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <premac.h>
#include <OpenTptInternet.h>
#include <OpenTptLinks.h>
#include <OsUtils.h>
#include <postmac.h>

#include <string.h>

#include <osl/socket.h>
#include <osl/diagnose.h>

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif


#include <sockimpl.h>
#include <timerimp.h>

#define SOCKET_ERROR	-1

#define MAX_TIMEOUT 2000	// 2000 ms maximaler Timeout
#define SETERROR(retval,err) (((socket_error = (err)) == (err)) ? (retval) : (retval))

int __OT_installed = 0;

#define OT_CHECK(ret)	if (!__OT_installed) return ret;
oslSocketError	socket_error = osl_Socket_E_None;
#define FATAL(value)	OSL_TRACE("Fatal error in %s line %d: %d", __FILE__, __LINE__, value)
#define ABORT()	FATAL(0)


/*****************************************************************************/
/* osl_createEmptySocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createEmptySocketAddr(oslAddrFamily Family)
{
	TSocketAddress *addr = OTAllocMem(sizeof(TSocketAddress));
	if (addr != NULL)
	{
		switch (Family)
		{
			case osl_Socket_FamilyInet:
				OTInitInetAddress(&addr->fInet, 0, kOTAnyInetAddress);
				break;
			default:
				OTFreeMem(addr);
				addr = NULL;
				break;
		}
	}

	return addr;
}

/*****************************************************************************/
/* osl_copySocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_copySocketAddr(oslSocketAddr Addr)
{
	if (Addr != NULL)
	{
		TSocketAddress	*addr = OTAllocMem(sizeof(TSocketAddress));

		if (addr != NULL)
			*addr = *(TSocketAddress *)Addr;

		return (oslSocketAddr)addr;
	}
	else
		return NULL;
}

/*****************************************************************************/
/* osl_resolveHostname */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_resolveHostname(rtl_uString *strHostname)
{
	InetHostInfo	info;
	OSStatus		status;
	oslSocketAddr	retAddr = NULL;

	status = OTResolveInetHostName(strHostname->buffer, &info);

	if (status == kOTNoError)
	{
		TSocketAddress	addr;
		OTInitInetAddress(&addr.fInet, 0, info.addrs[0]);
		retAddr = osl_copySocketAddr((oslSocketAddr)&addr);
	}

	if (status != kOTNoError)
		socket_error = osl_Socket_E_InvalidError;

	return retAddr;
}

/*****************************************************************************/
/* osl_getlocalHostname */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getLocalHostname(rtl_uString **strLocalHostname)
{
	InetInterfaceInfo	info;
	OSStatus			status;
	rtl_uString			*strBuffer = *strLocalHostname;

	OT_CHECK(osl_Socket_Error);
	status = OTInetGetInterfaceInfo(&info, kDefaultInetInterface);
	if (status == kOTNoError)
	{
		TSocketAddress	addr;

		OTInitInetAddress(&addr.fInet, 0, info.fAddress);
		return osl_getHostnameOfSocketAddr((oslSocketAddr)&addr, strBuffer->buffer);
		/* sock_error is implicitly set */
	}

	socket_error = osl_Socket_E_InvalidError;
	return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_createInetSocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createInetSocketAddr(rtl_uString *strDottedAddr, sal_Int32 Port)
{
	OSStatus	status;
	InetHost	host;

	status = OTInetStringToHost((sal_Char*)(strDottedAddr->buffer), &host);
	if (status == kOTNoError)
	{
		TSocketAddress	*addr = OTAllocMem(sizeof(TSocketAddress));

		if (addr != NULL)
			OTInitInetAddress(&addr->fInet, Port, host);

		return (oslSocketAddr)addr;
	}
	else
		return NULL;
}

/*****************************************************************************/
/* osl_createIpxSocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createIpxSocketAddr(rtl_uString * strNetNumber,
                                               rtl_uString * strNodeNumber,
                                               sal_uInt32 SocketNumber)
{
#pragma unused(strNetNumber, strNodeNumber, SocketNumber)
	return NULL;
}

/*****************************************************************************/
/* osl_destroySocketAddr */
/*****************************************************************************/
void SAL_CALL osl_destroySocketAddr(oslSocketAddr Addr)
{
	if (Addr)
		OTFreeMem(Addr);
}

/*****************************************************************************/
/* osl_getServicePort */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getServicePort(rtl_uString *strServicename, rtl_uString *strProtocol)
{
#pragma unused(strServicename, strProtocol)
	return OSL_INVALID_PORT; /* not supported by Open Transport */
}

/*****************************************************************************/
/* osl_getFamilyOfSocketAddr */
/*****************************************************************************/
oslAddrFamily SAL_CALL osl_getFamilyOfSocketAddr(oslSocketAddr Addr)
{
	if (Addr)
	{
		switch(((TSocketAddress *)Addr)->fAddressType)
		{
			case AF_INET:
				return osl_Socket_FamilyInet;
			case AF_8022:
				return osl_Socket_FamilyIpx;
			default:
				return osl_Socket_FamilyInvalid;
		}
	}
	else
		return osl_Socket_FamilyInvalid;
}

/*****************************************************************************/
/* osl_getInetPortOfSocketAddr */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getInetPortOfSocketAddr(oslSocketAddr Addr)
{
	if (Addr)
	{
		switch(((TSocketAddress *)Addr)->fAddressType)
		{
			case AF_INET:
				return ((TSocketAddress *)Addr)->fInet.fPort;
			default:
				return OSL_INVALID_PORT;
		}
	}
	else
		return OSL_INVALID_PORT;
}

/*****************************************************************************/
/* osl_setInetPortOfSocketAddr */
/*****************************************************************************/
sal_Bool SAL_CALL osl_setInetPortOfSocketAddr(oslSocketAddr Addr, sal_Int32 Port)
{
	if (Addr)
	{
		switch(((TSocketAddress *)Addr)->fAddressType)
		{
			case AF_INET:
				((TSocketAddress *)Addr)->fInet.fPort = Port;
				return sal_True;
			default:
				return sal_False;
		}
	}
	else
		return sal_False;
}

/*****************************************************************************/
/* osl_getHostnameOfSocketAddr */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getHostnameOfSocketAddr(oslSocketAddr Addr, rtl_uString **strHostname)
{
	if (Addr)
	{
		TSocketAddress	*addr = (TSocketAddress *)Addr;
		InetDomainName	name;
		rtl_uString		*strBuffer = *strHostname;

		switch(addr->fAddressType)
		{
			case AF_INET:
				if (OTResolveInetHost(addr->fInet.fHost, name) == kOTNoError)
					break;
			default:
				return osl_Socket_Error;
		}

		if (strBuffer && strBuffer->length > 0)
		{
			strncpy(strBuffer->buffer, name, strBuffer->length);
			strBuffer->buffer[strBuffer->length-1] = '\0';
		}
		return	osl_Socket_Ok;
	}
	else
		return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_getDottedInetAddrOfSocketAddr */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getDottedInetAddrOfSocketAddr(oslSocketAddr Addr, rtl_uString **strDottedInetAddr)
{
	if (Addr)
	{
		TSocketAddress	*addr = (TSocketAddress *)Addr;
		sal_Char		szDotted[32];
		rtl_uString		*strBuffer = *strDottedInetAddr;

		switch(addr->fAddressType)
		{
			case AF_INET:
				OTInetHostToString(addr->fInet.fHost, szDotted);
				break;
			default:
				return osl_Socket_Error;
		}

		if (strBuffer && strBuffer->length > 0)
		{
			strncpy(strBuffer->buffer, szDotted, strBuffer->length);
			strBuffer->buffer[strBuffer->length-1] = '\0';
		}
		return osl_Socket_Ok;
	}
	else
		return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_getIpxNetNumber */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getIpxNetNumber(oslSocketAddr Addr, oslSocketIpxNetNumber NetNumber)
{
#pragma unused(Addr, NetNumber)
	return osl_Socket_Error;	/* not supported by Open Transport */
}

/*****************************************************************************/
/* osl_getIpxNodeNumber */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getIpxNodeNumber(oslSocketAddr Addr, oslSocketIpxNodeNumber NodeNumber)
{
#pragma unused(Addr, NodeNumber)
	return osl_Socket_Error;	/* not supported by Open Transport */
}

/*****************************************************************************/
/* osl_getIpxSocketNumber */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getIpxSocketNumber(oslSocketAddr Addr)
{
#pragma unused(Addr)
	return OSL_INVALID_IPX_SOCKET_NO;	/* not supported by Open Transport */
}

/*****************************************************************************/
/**																			**/
/**          Section          HOSTADDR                                      **/
/**																			**/
/*****************************************************************************/

/*****************************************************************************/
/* osl_createHostAddr */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_createHostAddr(rtl_uString *strHostname, const oslSocketAddr Addr)
{
	if (Addr && strHostname)
	{
		THostAddress*	host = OTAllocMem(sizeof(THostAddress));

		if (host != NULL)
		{
			OTStrCopy(host->name, strHostname->buffer);
			host->addr = *(TSocketAddress *)Addr;
		}
		return (oslHostAddr)host;
	}
	else
		return NULL;
}

/*****************************************************************************/
/* osl_createHostAddrByName */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_createHostAddrByName(rtl_uString *strHostname)
{
	oslHostAddr		hostAddr = NULL;
	OSStatus		status;
	InetHostInfo	info;

	status = OTResolveInetHostName(strHostname->buffer, &info);
	if (status == kOTNoError)
	{
		TSocketAddress	addr;

		OTInitInetAddress(&addr.fInet, 0, info.addrs[0]);
		hostAddr = osl_createHostAddr(info.name, (oslSocketAddr)&addr);
	}

	return (oslHostAddr)hostAddr;
}

/*****************************************************************************/
/* osl_createHostAddrByAddr */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_createHostAddrByAddr (const oslSocketAddr Addr)
{
	InetDomainName	szDomain;
	if (osl_getHostnameOfSocketAddr(Addr, szDomain) == osl_Socket_Ok)
		return osl_createHostAddr(szDomain, Addr);
	else
		return NULL;
}

/*****************************************************************************/
/* osl_copyHostAddr */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_copyHostAddr (const oslHostAddr Addr)
{
    THostAddress *host = (THostAddress *)Addr;
    if (host)
        return osl_createHostAddr(host->name, (oslSocketAddr)&host->addr);
    else
        return NULL;
}

/*****************************************************************************/
/* osl_destroyHostAddr */
/*****************************************************************************/
void SAL_CALL osl_destroyHostAddr (oslHostAddr Addr)
{
	if (Addr)
		OTFreeMem(Addr);
}

/*****************************************************************************/
/* osl_getHostnameOfHostAddr */
/*****************************************************************************/
void SAL_CALL osl_getHostnameOfHostAddr(const oslHostAddr Addr, rtl_uString **strHostname)
{
	rtl_uString *strBuffer = *strHostname;

	if (Addr && strBuffer) {
		strncpy(strBuffer->buffer, ((THostAddress *)Addr)->name, strBuffer->length);
		strBuffer->buffer[strBuffer->length-1] = '\0';
	}
}

/*****************************************************************************/
/* osl_getSocketAddrOfHostAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_getSocketAddrOfHostAddr(const oslHostAddr Addr)
{
	if (Addr)
		return (const oslSocketAddr)&((THostAddress *)Addr)->addr;
	else
		return NULL;
}

/*****************************************************************************/
/* osl_createSocket */
/*****************************************************************************/
oslSocket SAL_CALL osl_createSocket(oslAddrFamily	Family,
						   oslSocketType	Type,
						   oslProtocol      Protocol)
{
	OT_CHECK(NULL);

	if (Family == osl_Socket_FamilyInet && Protocol == osl_Socket_ProtocolIp)
	{
		const sal_Char	*pszCfgPath;
		TSocket		*socket;
		OSStatus	status;

		switch (Type) {
			case osl_Socket_TypeStream:
				pszCfgPath = kTCPName;
				break;
			case osl_Socket_TypeDgram:
				pszCfgPath = kUDPName;
				break;
			case osl_Socket_TypeRaw:
				pszCfgPath = kRawIPName;
				break;
			default:
				return NULL;
		}

		socket = OTAllocMem(sizeof(TSocket));
		if (socket) {
			socket->pszCfgPath = pszCfgPath;
			socket->listening = false;
			socket->result = kOTNoError;
			socket->inconList.fHead = NULL;
			OTResetEvent(&socket->fSyncComplete);
			OTInitInetAddress(&socket->fDatagram.defaultPeer.fInet, 0, kOTAnyInetAddress);

			status = OTAsyncOpenEndpoint(OTCreateConfiguration(pszCfgPath), 0, &socket->epInfo, SocketNotifier, socket);
			socket->servtype = socket->epInfo.servtype;
			if (status == kOTNoError)
				status = OTGetEvent(&socket->fSyncComplete);
			if (status != kOTNoError)
			{
				OTFreeMem(socket);
				socket = NULL;
			}
		}
		return (oslSocket)socket;
	} else
		return NULL;
}

/*****************************************************************************/
/* osl_copySocket */
/*****************************************************************************/
oslSocket SAL_CALL osl_copySocket(oslSocket Socket)
{
	TSocket	*newSocket;

	if (Socket == NULL)
		return NULL;

	newSocket = OTAllocMem(sizeof(TSocket));

	if (newSocket)
		*newSocket = *(TSocket *)Socket;

	return newSocket;
}

/*****************************************************************************/
/* osl_destroySocket */
/*****************************************************************************/
void SAL_CALL osl_destroySocket(oslSocket Socket)
{
	TSocket *s = (TSocket *)Socket;

	if (s)
	{
		int i;

		OTUnbind(s->epRef);
		for (i = 0; i < 100; i++)
			osl_yieldThread();
		OTCloseProvider(s->epRef);
		for (i = 0; i < 100; i++)
			osl_yieldThread();
		OTFreeMem(s);
	}
}

/*****************************************************************************/
/* osl_getLocalAddrOfSocket */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_getLocalAddrOfSocket(oslSocket Socket)
{
	TSocket		*s = (TSocket *)Socket;
	TBind		bind;
	OSStatus	status;
	InetAddress	addr;

	if (Socket == NULL)
		return NULL;

	bind.addr.buf = (UInt8 *)&addr;
	bind.addr.maxlen = sizeof(InetAddress);

	status = OTGetProtAddress(s->epRef, &bind, nil);
	if (status == kOTNoError)
		status = OTGetEvent(&s->fSyncComplete);

	if (status != kOTNoError)
		return NULL;

	return osl_copySocketAddr((oslSocketAddr)&addr);
}

/*****************************************************************************/
/* osl_getLocalAddrOfSocket */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_getPeerAddrOfSocket(oslSocket Socket)
{
	TSocket		*s = (TSocket *)Socket;
	TBind		bind;
	OSStatus	status;
	InetAddress	addr;

	if (Socket == NULL)
		return NULL;

	bind.addr.buf = (UInt8 *)&addr;
	bind.addr.maxlen = sizeof(InetAddress);

	status = OTGetProtAddress(s->epRef, nil, &bind);
	if (status == kOTNoError)
		status = OTGetEvent(&s->fSyncComplete);

	if (status != kOTNoError)
		return NULL;

	return osl_copySocketAddr((oslSocketAddr)&addr);
}

/*****************************************************************************/
/* osl_bindAddrToSocket */
/*****************************************************************************/
sal_Bool SAL_CALL osl_bindAddrToSocket(oslSocket Socket, oslSocketAddr Addr)
{
	TSocket		*s = (TSocket *)Socket;
	OSStatus	status;
	TBind		tbReturn;
	TSocketAddress	retAddr;
	if (Socket == NULL)
		return sal_False;

	tbReturn.addr.buf = (UInt8 *)&retAddr;
	tbReturn.addr.maxlen = sizeof(TSocketAddress);

	if (Addr != NULL) {
		TBind		tbRequest;

		tbRequest.addr.buf = (UInt8 *)&((TSocketAddress *)Addr)->fInet;
		tbRequest.addr.len = sizeof(TSocketAddress);
		tbRequest.qlen = 0;

		status = OTBind(s->epRef, &tbRequest, &tbReturn);
	} else
		status = OTBind(s->epRef, nil, &tbReturn);

	if (status == kOTNoError)
		status = OTGetEvent(&s->fSyncComplete);
#if OSL_DEBUG_LEVEL > 1
	OTInetHostToString(retAddr.fInet.fHost, s->debug_localAddr);
#endif

	s->listening = false;

	return (sal_Bool)(status == kOTNoError);
}

/*****************************************************************************/
/* osl_connectSocketTo */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_connectSocketTo(oslSocket Socket, oslSocketAddr Addr, const TimeValue* pTimeout)
{
	TSocket				*pSocket = (TSocket *)Socket;
	TSocketAddress		*pAddr = (TSocketAddress *)Addr;
	OSStatus			status;
	OTResult			epState, result;
	TCall				call;
	long				timeout;

//	SysBeep(100);
/* Idiotenfehler abfangen */
	if (Socket == NULL)
		return SETERROR(osl_Socket_Error, osl_Socket_E_NotSocket);

	if (Addr == NULL)
		return SETERROR(osl_Socket_Error, osl_Socket_E_AddrNotAvail);

/* Jetzt ermitteln, ob der Socket schon gebunden ist */
	epState = OTGetEndpointState(pSocket->epRef);

/* Laut BSD Spezifikation muss der Socket implizit gebunden werden  */
	if (epState == T_UNBND)
	{
		if (!osl_bindAddrToSocket(Socket, NULL))
			return osl_Socket_Error;
	}

/* Wenn es kein Stream-Socket ist wird nur ein Default Peer angegeben */
	switch (osl_getSocketType(Socket))
	{
		case osl_Socket_TypeStream:
			/* Weiter geht's */
			break;
		case osl_Socket_TypeDgram:
			pSocket->fDatagram.defaultPeer = *pAddr;
			return SETERROR(osl_Socket_Ok, osl_Socket_E_None);
		case osl_Socket_TypeInvalid:
		default:
			return SETERROR(osl_Socket_Error, osl_Socket_E_AfNoSupport);
	}

/* Wichtig!! Der Reset des Sync-Objekt muss vor dem Connect erfolgen */
/* sonst bekommen wir ein Timing-Problem */

	OTResetEvent(&pSocket->fSyncComplete);

/* Jetzt den Connect initiieren */
	call.addr.buf = (UInt8 *)pAddr;
	call.addr.len = sizeof(TSocketAddress);
	call.opt.len = 0;
	call.udata.len = 0;
/* Beim asynchronen Connect keine Sequence */
	call.sequence = 0;
	status = OTConnect(pSocket->epRef, &call, nil);
	switch (status)		/* Fehlerbedingungen */
	{
		case kOTNoError:
			return SETERROR(osl_Socket_Ok, osl_Socket_E_None);
		case kOTNoDataErr:
			if (OTIsNonBlocking(pSocket->epRef))
				return SETERROR(osl_Socket_Error, osl_Socket_E_WouldBlock);
			/* Weiter geht's */
			break;
		case kOTOutStateErr:
			epState = OTGetEndpointState(pSocket->epRef);
			switch (epState)
			{
				case T_DATAXFER:
					return SETERROR(osl_Socket_Error, osl_Socket_E_IsConnected);
				default:
					FATAL(epState);
					return SETERROR(osl_Socket_Error, osl_Socket_E_InvalidError);
			}
			break;
		default:
			FATAL(status);
			return SETERROR(osl_Socket_Error, osl_Socket_E_InvalidError);
	}

	/* Wenn wir hier ankommen sind wir Blocking und der Connect ist noch nicht fertig */
	/* Also warten wir mit einem Timeout darauf */

	if (pTimeout)
		timeout = pTimeout->Seconds * 1000L + pTimeout->Nanosec / 1000000L;
	else
		timeout = INFINITE;

#if defined(__POWERPC__)

	/* Wenn ein Connect mit dem Localhost versu versucht wird, dann hoechstens */
	/* 2 Sekunden Timeout */

	if (pAddr->fInet.fHost == (InetHost)0x7F000001 && (timeout > MAX_TIMEOUT || timeout == INFINITE))
		timeout = MAX_TIMEOUT;
#endif /* __POWERPC__ */
	result = OTWaitEvent(&pSocket->fSyncComplete, timeout);

	switch (result)
	{
		case T_CONNECT:
			epState = OTGetEndpointState(pSocket->epRef);
			switch (epState)
			{
				case T_DATAXFER:
					return SETERROR(osl_Socket_Ok, osl_Socket_E_None);
				case T_IDLE:
				case T_INREL:
					return SETERROR(osl_Socket_Error, osl_Socket_E_ConnRefused);
				default:
					FATAL(epState);
					return SETERROR(osl_Socket_Error, osl_Socket_E_InvalidError);
			}
		case T_TIMEOUT:
			return SETERROR(osl_Socket_Error, osl_Socket_E_TimedOut);
		default:
			FATAL(result);
			return SETERROR(osl_Socket_Error, osl_Socket_E_InvalidError);
	}
}

/*****************************************************************************/
/* osl_listenOnSocket */
/*****************************************************************************/
sal_Bool SAL_CALL osl_listenOnSocket(oslSocket Socket, sal_Int32 MaxPendingConnections)
{
	TSocket		*s = (TSocket *)Socket;
	OSStatus	status = kOTNoError;
	TBind		bind;
	InetAddress	addr;
	if (Socket == NULL)
		return sal_False;

	if (MaxPendingConnections == -1)
		MaxPendingConnections = 5;

	bind.addr.buf = (UInt8 *)&addr;
	bind.addr.maxlen = sizeof(InetAddress);

	status = OTGetProtAddress(s->epRef, &bind, nil);
	if (status == kOTNoError)
		status = OTGetEvent(&s->fSyncComplete);
	if (status == kOTNoError)
		status = OTUnbind(s->epRef);
	if (status == kOTNoError)
		status = OTGetEvent(&s->fSyncComplete);

	bind.qlen = MaxPendingConnections;
	if (status == kOTNoError)
		status = OTBind(s->epRef, &bind, nil);
	if (status == kOTNoError)
		status = OTGetEvent(&s->fSyncComplete);

	if (status == kOTNoError)
		s->listening = true;

	return (sal_Bool)(status == kOTNoError);
}

/*****************************************************************************/
/* osl_acceptConnectionOnSocket */
/*****************************************************************************/
static OSStatus OTListenConnect(TSocket *s)
{
	TCall		call;
	InetAddress	addr;
	OSStatus	status;
	OTResult	result;

	call.addr.buf = (UInt8 *)&addr;
	call.addr.maxlen = sizeof(addr);
	call.opt.maxlen = 0;
	call.udata.maxlen = 0;
	status = OTListen(s->epRef, &call);
	if (status == kOTNoError)
		OTEnqueueInConnect(&s->inconList, &call);
	else if (status == kOTLookErr)
	{
		result = OTLook(s->epRef);
		if (result == T_DISCONNECT)
		{
			TDiscon	discon;

			discon.udata.maxlen = 0;
			if (OTRcvDisconnect(s->epRef, &discon) == kOTNoError)
				OTDequeueInConnect(&s->inconList, discon.sequence);
			s->result = result;
		}
	}
	return status;
}

static OSStatus OTAcceptConnect(TSocket *s, TSocket *acceptor, TCall *call)
{
	OSStatus	status;

	status = OTGetInConnect(&s->inconList, call);

	if (status == kOTNoError)
	{
		status = OTAccept(s->epRef, acceptor->epRef, call);

		/* if accept was successfull - remove it from list */
		if (status == kOTNoError) /* Accept success */
		{
			status = OTGetEvent(&s->fSyncComplete);
			if (status == kOTNoError)
				OTDequeueInConnect( &s->inconList, call->sequence);
			else
				ABORT();
		}
	}
	return status;
}

oslSocket SAL_CALL osl_acceptConnectionOnSocket(oslSocket Socket, oslSocketAddr* pAddr)
{
	TSocket	*s = (TSocket *)Socket;
	TSocket	*acceptor;
	TCall	call;
	TSocketAddress	addr;

	if (Socket == NULL)
		return NULL;

	acceptor = (TSocket *)osl_createSocket(osl_Socket_FamilyInet, osl_getSocketType(Socket), osl_Socket_ProtocolIp);
	call.addr.buf = (UInt8 *)&addr;
	call.addr.maxlen = sizeof(addr);
	call.opt.maxlen = 0;
	call.udata.maxlen = 0;

	if (acceptor) {
		OSStatus		status;
		do
		{
			do
			{
				status = OTListenConnect(s);
				osl_yieldThread();
			} while ((status == kOTNoDataErr || status == kOTLookErr) && !OTIsNonBlocking(s->epRef));

			if (status == kOTNoError)
				status = OTAcceptConnect(s, acceptor, &call);

			osl_yieldThread();

		} while (!OTIsNonBlocking(s->epRef) && (status == kOTNoDataErr || status == kOTLookErr));


		if (status == kOTNoError)
			*pAddr = osl_copySocketAddr(osl_getPeerAddrOfSocket(acceptor));
		else if (status != kOTNoError) {
			osl_destroySocket((oslSocket)acceptor);
			acceptor = NULL;
		}
	}

	return acceptor;
}

/*****************************************************************************/
/* osl_receiveSocket */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_receiveSocket(oslSocket Socket, void* pBuffer, sal_uInt32 BytesToRead, oslSocketMsgFlag MsgFlag)
{
	TSocket			*s = (TSocket *)Socket;
	OTFlags			flags;
	OTResult		status;
	oslSocketType	type;

	if (Socket == NULL)
		return kOTBadReferenceErr;

	type = osl_getSocketType(Socket);

	if (type == osl_Socket_TypeDgram || type == osl_Socket_TypeRaw) {
		TSocketAddress	addr;
		return osl_receiveFromSocket(Socket, (oslSocketAddr)&addr, pBuffer, BytesToRead, MsgFlag);
	}

	status = OTRcv(s->epRef, pBuffer, BytesToRead, &flags);

	if (status == kOTNoDataErr) {
		OTResult	result;
		if (OTIsNonBlocking(s->epRef))
			return SETERROR(SOCKET_ERROR, osl_Socket_E_WouldBlock);

		result = OTWaitForPending(s->epRef, INFINITE);
		if (result == T_DATA || result == T_EXDATA)
			status = OTRcv(s->epRef, pBuffer, BytesToRead, &flags);
		else
			status = kOTOutStateErr;
	}

	if (status == kOTOutStateErr) {
		status = OTGetEndpointState(s->epRef);
		switch (status) {
			case T_UNINIT:
				return -1;
			default:
				return SETERROR(0, osl_Socket_E_None);
		}
	}
	else if (status < 0)
	{
		return SETERROR(0, osl_Socket_E_None);
	}
	else
		return SETERROR(status, osl_Socket_E_None);
}

/*****************************************************************************/
/* osl_receiveFromSocket */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_receiveFromSocket(oslSocket Socket, oslSocketAddr SenderAddr, void* pBuffer, sal_uInt32 BufferSize,
						  oslSocketMsgFlag MsgFlag)
{
	TSocket			*s = (TSocket *)Socket;
	OTFlags			flags;
	TUnitData		data;
	OSStatus		status;
	oslSocketType	type;

	if (Socket == NULL)
		return kOTBadReferenceErr;

	type = osl_getSocketType(Socket);

	if (type == osl_Socket_TypeStream) {
		int 			Result;
		oslSocketAddr	PeerAddr;

		Result = osl_receiveSocket(Socket, pBuffer, BufferSize, MsgFlag);
		if (Result > 0) {
			PeerAddr = osl_getPeerAddrOfSocket(Socket);
			*(TSocketAddress *)SenderAddr = *(TSocketAddress *)PeerAddr;
			osl_destroySocketAddr(PeerAddr);
		}
		return Result;
	}

	data.addr.buf = (UInt8 *)(TSocketAddress *)SenderAddr;
	data.addr.maxlen = sizeof(TSocketAddress);
	data.opt.maxlen = 0;
	data.udata.buf = (UInt8 *)pBuffer;
	data.udata.maxlen = BufferSize;
	data.udata.len = 0;
	status = OTRcvUData(s->epRef, &data, &flags);

	if (status == kOTNoDataErr) {
		OTResult	result;
		if (OTIsNonBlocking(s->epRef))
			return SETERROR(SOCKET_ERROR, osl_Socket_E_WouldBlock);

		result = OTWaitForPending(s->epRef, INFINITE);
		if (result == T_DATA)
			status = OTRcvUData(s->epRef, &data, &flags);
		else
			status = kOTOutStateErr;
	}

	if (status == kOTNoError)
		return data.udata.len;
	else
		return status;
}

/*****************************************************************************/
/* osl_sendSocket */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_sendSocket(oslSocket Socket, const void* pBuffer, sal_uInt32 BytesToSend, oslSocketMsgFlag MsgFlag)
{
	TSocket		*s = (TSocket *)Socket;
	OTResult	result;

	if (Socket == NULL)
		return kOTBadReferenceErr;

	if ((s->servtype & T_CLTS) == T_CLTS)
		return osl_sendToSocket(Socket, (oslSocketAddr)&s->fDatagram.defaultPeer, pBuffer, BytesToSend, MsgFlag);

	result = OTSnd(s->epRef, (void *)pBuffer, BytesToSend, 0);

	if (result == 0)
		result = kOTFlowErr;
	while (result == kOTFlowErr)
			result = OTSnd(s->epRef, (void *)pBuffer, BytesToSend, 0);
	return result;
}

/*****************************************************************************/
/* osl_sendToSocket */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_sendToSocket(oslSocket Socket,
		oslSocketAddr ReceiverAddr, const void* pBuffer,
		sal_uInt32 BytesToSend, oslSocketMsgFlag MsgFlag)
{
	TSocket		*s = (TSocket *)Socket;
	OSStatus	status;
	TUnitData	data;
	oslSocketType	type;
	OTResult	epState;

	if (Socket == NULL)
		return kOTBadReferenceErr;

	type = osl_getSocketType(Socket);

/* Jetzt ermitteln, ob der Socket schon gebunden ist */
	epState = OTGetEndpointState(s->epRef);

	if (epState == T_UNBND)
	{
		if (!osl_bindAddrToSocket(Socket, NULL))
			return osl_Socket_Error;
	}

	if ((s->servtype & T_CLTS) != T_CLTS)
		return osl_sendSocket(Socket, pBuffer, BytesToSend, MsgFlag);

	data.addr.buf = (UInt8 *)(TSocketAddress *)ReceiverAddr;
	data.addr.len = sizeof(TSocketAddress);
	data.opt.len = 0;
	data.udata.buf = (UInt8 *)pBuffer;
	data.udata.len = BytesToSend;

	status = OTSndUData(s->epRef, &data);
	while (status == kOTFlowErr && !OTIsBlocking(s->epRef))
			status = OTSndUData(s->epRef, &data);

	if (status == kOTNoError)
		return data.udata.len;
	else
		return status;
}

/*****************************************************************************/
/* osl_isReceiveReady */
/*****************************************************************************/
#define IS_SOCK_STREAM(psocket)	(((psocket)->servtype & T_CLTS) != T_CLTS)
sal_Bool SAL_CALL osl_isReceiveReady(oslSocket Socket, const TimeValue* pTimeout)
{
	TSocket		*s = (TSocket *)Socket;
	long		timeout;
	OTResult	result;

	if (Socket == NULL)
		return sal_False;

	if (pTimeout)
		timeout = pTimeout->Seconds * 1000L + pTimeout->Nanosec / 1000000L;
	else
		timeout = INFINITE;
// Wenn der Socket gerade seinen Status ndert geht nichts!
	if (OTLook(s->epRef) == kOTStateChangeErr)
		return sal_False;

// Jetzt erstmal den Typ des Sockets bestimmen
	if (IS_SOCK_STREAM(s))
	{
		// Fr einen socket vom Typ SOCK_STREAM heit readable, da generell
		// ein Ereignis anliegt (Daten, Connection Request aber auch Disconnect)

		result = OTWaitForPending(s->epRef, timeout);
		return (sal_Bool)(result > 0);
	}
	else
	{
		// Fr den Typ SOCK_DGRAM mssen Daten anliegen
		result = OTWaitForPending(s->epRef, timeout);
		return (sal_Bool)(result == T_DATA);
	}
}

/*****************************************************************************/
/* osl_isSendReady */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isSendReady(oslSocket Socket, const TimeValue* pTimeout)
{
	TSocket		*s = (TSocket *)Socket;
	long		timeout;
	OTResult	epState;

	if (Socket == NULL)
		return sal_False;

	if (pTimeout)
		timeout = pTimeout->Seconds * 1000L + pTimeout->Nanosec / 1000000L;
	else
		timeout = INFINITE;
/* If the endpoint state is changing, wait until change is complete */
	if (OTLook(s->epRef) == kOTStateChangeErr)
		return sal_False;
/* Now determine the endpoint state to interprete the request properly */

	epState = OTGetEndpointState(s->epRef);

	if (epState == T_DATAXFER || epState == T_INREL)
		return sal_True;
	else if (epState == T_IDLE && osl_getSocketType(Socket) == osl_Socket_TypeDgram)
		return sal_True;
	else
		return sal_False;
}

/*****************************************************************************/
/* osl_isExceptionPending */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isExceptionPending(oslSocket Socket, const TimeValue* pTimeout)
{
	TSocket		*s = (TSocket *)Socket;
	long		timeout;
	OTResult	result;

	if (Socket == NULL)
		return sal_False;

	if (pTimeout)
		timeout = pTimeout->Seconds * 1000L + pTimeout->Nanosec / 1000000L;
	else
		timeout = INFINITE;

/* If the endpoint state is changing, wait until change is complete */
	if (OTLook(s->epRef) == kOTStateChangeErr)
		return sal_False;
	result = OTWaitForPending(s->epRef, timeout);

	return (sal_Bool)(result == T_EXDATA || result == T_DISCONNECT);
}

/*****************************************************************************/
/* osl_shutDownSocket */
/*****************************************************************************/
sal_Bool SAL_CALL osl_shutdownSocket(oslSocket Socket,
						   oslSocketDirection Direction)
{
	TSocket	*s = (TSocket *)Socket;
	OSStatus	status;

	if (Socket == NULL)
		return sal_False;

	switch (Direction) {
		case osl_Socket_DirRead:
			status = OTRcvOrderlyDisconnect(s->epRef);
			break;
		case osl_Socket_DirWrite:
			status = OTSndOrderlyDisconnect(s->epRef);
			break;
		case osl_Socket_DirReadWrite:
			status = OTRcvOrderlyDisconnect(s->epRef);
			status = OTSndOrderlyDisconnect(s->epRef);
			break;
		default:
			return sal_False;
	}

	return (sal_Bool)(status == kOTNoError);
}

/** Retrieves attributes associated with the socket.
	@param Socket is the socket to query.
	@param Level selects the level for which an option should be queried.
	Valid values are:
	<ul>
	<li> osl_sol_socket:	Socket Level
	<li> osl_sol_tcp:		Level of Transmission Control Protocol
	</ul>

	@param Option denotes the option to query.
	Valid values (depending on the Level) are:
	<ul>
	<li> osl_so_debug,
	<li> osl_so_acceptconn,
	<li> osl_so_reuseaddr,
	<li> osl_so_keepalive,
	<li> osl_so_dontroute,
	<li> osl_so_broadcast,
	<li> osl_so_useloopback,
	<li> osl_so_linger,
	<li> osl_so_oobinline,
	<li> osl_so_sndbuf,
	<li> osl_so_rcvbuf,
	<li> osl_so_sndlowat,
	<li> osl_so_rcvlowat,
	<li> osl_so_sndtimeo,
	<li> osl_so_rcvtimeo,
	<li> osl_so_error,
	<li> osl_so_type,
	<li> osl_so_tcp_nodelay,	(sol_tcp)
	</ul>
	If not above mentioned otherwise, the options are only valid for
	level sol_socket.

	@param pBuffer Pointer to a Buffer with enough room to take the desired
	attribute-value.
	@param BufferSize contains the length of the Buffer.
	@return -1 if an error occured or else the size of the data copied into
	pBuffer.
*/
long SAL_CALL osl_getSocketOption(oslSocket Socket,
						oslSocketOptionLevel	Level,
						oslSocketOption			Option,
						void*					pBuffer,
						sal_uInt32						BufferLen)
 {
 #pragma unused(Socket, Level, Option, pBuffer, BufferLen)
 	return 0;
 }
/** Sets the sockets attributes.
	@param Socket is the socket to modify.
	@param Level selects the level for which an option should be changed.
	Valid values are:
	<ul>
	<li> osl_sol_socket:	Socket Level
	<li> osl_sol_tcp:		Level of Transmission Control Protocol
	</ul>

	@param Option denotes the option to modify.
	Valid values (depending on the Level) are:
	<ul>
	<li> osl_so_debug,
	<li> osl_so_acceptconn,
	<li> osl_so_reuseaddr,
	<li> osl_so_keepalive,
	<li> osl_so_dontroute,
	<li> osl_so_broadcast,
	<li> osl_so_useloopback,
	<li> osl_so_linger,
	<li> osl_so_oobinline,
	<li> osl_so_sndbuf,
	<li> osl_so_rcvbuf,
	<li> osl_so_sndlowat,
	<li> osl_so_rcvlowat,
	<li> osl_so_sndtimeo,
	<li> osl_so_rcvtimeo,
	<li> osl_so_error,
	<li> osl_so_type,
	<li> osl_so_tcp_nodelay,	(sol_tcp)
	</ul>
	If not above mentioned otherwise, the options are only valid for
	level sol_socket.

	@param pBuffer Pointer to a Buffer which contains the attribute-value.
	@param BufferSize contains the length of the Buffer.
	@return sal_True if the option could be changed.
*/
sal_Bool SAL_CALL osl_setSocketOption(oslSocket Socket,
							oslSocketOptionLevel	Level,
							oslSocketOption			Option,
							void*					pBuffer,
							sal_uInt32						BufferLen)
{
 #pragma unused(Socket, Level, Option, pBuffer, BufferLen)
	return sal_True;
}

/*****************************************************************************/
/* osl_enableBlockingMode */
/*****************************************************************************/
sal_Bool SAL_CALL osl_enableNonBlockingMode(oslSocket Socket, sal_Bool On)
{
	TSocket *s = (TSocket *)Socket;

	if (Socket == NULL)
		return sal_False;

	if (On)
		return (sal_Bool)(OTSetNonBlocking(s->epRef) == kOTNoError);
	else
		return (sal_Bool)(OTSetBlocking(s->epRef) == kOTNoError);
}

/*****************************************************************************/
/* osl_isNonBlockingMode */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isNonBlockingMode(oslSocket Socket)
{
	TSocket *s = (TSocket *)Socket;

	if (Socket == NULL)
		return sal_False;

	return OTIsNonBlocking(s->epRef);
}

/*****************************************************************************/
/* osl_isNonBlockingMode */
/*****************************************************************************/
oslSocketType SAL_CALL osl_getSocketType(oslSocket Socket)
{
	TSocket *s = (TSocket *)Socket;

	if (Socket == NULL)
		return osl_Socket_TypeInvalid;

	if (strcmp(s->pszCfgPath, kTCPName) == 0)
		return osl_Socket_TypeStream;
	else if (strcmp(s->pszCfgPath, kUDPName) == 0)
		return osl_Socket_TypeDgram;
	else if (strcmp(s->pszCfgPath, kRawIPName) == 0)
		return osl_Socket_TypeRaw;
	else
		return osl_Socket_TypeInvalid;
}

/*****************************************************************************/
/* osl_getLastSocketErrorDescription */
/*****************************************************************************/
void SAL_CALL osl_getLastSocketErrorDescription(oslSocket Socket, rtl_uString **strError)
{
	rtl_uString	*strBuffer = *strError;

	if (strBuffer) {
		strncpy(strBuffer->buffer, "An error occured - are you interested in reason?!", strBuffer->length);
		strBuffer->buffer[strBuffer->length-1] = '\0';
	}
}

/*****************************************************************************/
/* osl_getLastSocketError */
/*****************************************************************************/
oslSocketError SAL_CALL osl_getLastSocketError(oslSocket Socket)
{
	oslSocketError	error;

	error = socket_error;

	return error;
}

/** Creates a set of sockets to be used with osl_demultiplexSocketEvents().
	@return A oslSocketSet or 0 if creation failed.
*/
oslSocketSet SAL_CALL osl_createSocketSet()
{
	oslSocket *set = OTAllocMem(sizeof(oslSocket) * kMaxSocketSetSize);

	if (set) {
		int n;

		for (n = 0; n < kMaxSocketSetSize; n++)
			set[n] = NULL;
	}

	return (oslSocketSet)set;
}

/** Destroys a oslSocketSet.
*/
void SAL_CALL osl_destroySocketSet(oslSocketSet Set)
{
	if (Set)
		OTFreeMem((oslSocket *)Set);
}

/** Clears the set from all previously added sockets.
	@param Set the set to be cleared.
*/
void SAL_CALL osl_clearSocketSet(oslSocketSet Set)
{
	oslSocket *set = (oslSocket *)Set;

	if (set) {
		int n;

		for (n = 0; n < kMaxSocketSetSize; n++)
			set[n] = NULL;
	}
}

/** Adds a socket to the set.
	@param Set the set were the socket is added.
	@param Socket the socket to be added.
*/
void SAL_CALL osl_addToSocketSet(oslSocketSet Set, oslSocket Socket)
{
	oslSocket *set = (oslSocket *)Set;

	if (Set && Socket) {
		int			n = 0;

		while (n < kMaxSocketSetSize && set[n] != NULL)
			n++;
		if (n < kMaxSocketSetSize && set[n] == NULL)
			set[n] = Socket;
	}
}

/** Removes a socket from the set.
	@param Set the set were the socket is removed from.
	@param Socket the socket to be removed.
*/
void SAL_CALL osl_removeFromSocketSet(oslSocketSet Set, oslSocket Socket)
{
	oslSocket *set = (oslSocket *)Set;

	if (Set && Socket) {
		int			n = 0;

		while (n < kMaxSocketSetSize && set[n] != Socket)
			n++;
		if (n < kMaxSocketSetSize && set[n] == Socket)
			set[n] = NULL;
	}
}

/** Checks if socket is in the set.
	@param Set the set to be checked.
	@param Socket check if this socket is in the set.
	@return sal_True if socket is in the set.
*/
sal_Bool SAL_CALL osl_isInSocketSet(oslSocketSet Set, oslSocket Socket)
{
	oslSocket *set = (oslSocket *)Set;

	if (Set && Socket) {
		int			n = 0;

		while (n < kMaxSocketSetSize && set[n] != Socket)
			n++;
		return (sal_Bool)(n < kMaxSocketSetSize && set[n] == Socket);
	}
	return sal_False;
}

static pascal void _SelectTimerTask(void *arg)
{
	*(sal_Bool *)arg = true;
}

long SAL_CALL osl_demultiplexSocketEvents(oslSocketSet IncomingSet,
								oslSocketSet OutgoingSet,
								oslSocketSet OutOfBandSet,
								const TimeValue* pTimeout)
{
	oslSocket 	rcvList[kMaxSocketSetSize] = { NULL };
	oslSocket 	sndList[kMaxSocketSetSize] = { NULL };
	oslSocket	oobList[kMaxSocketSetSize] = { NULL };
	oslSocket	*testList;
	TimeValue	zero = {0, 0};
	long		timeout, task;
	int			n, count = 0;
	sal_Bool		timeElapsed = false;
	if (pTimeout)
		timeout = pTimeout->Seconds * 1000L + pTimeout->Nanosec / 1000000L;
	else
		timeout = INFINITE;
	if (timeout >= 0) {
		task = CreateTimerTask(_SelectTimerTask, &timeElapsed);
		ScheduleTimerTask(task, timeout);
	}
	do {
		if (IncomingSet != NULL) {
			testList = (oslSocket *)IncomingSet;
			for (n = 0; n < kMaxSocketSetSize; n++) {
				if (testList[n] != NULL && osl_isReceiveReady(testList[n], &zero)) {
					count++;
					rcvList[n] = testList[n];
				}
			}
		}

		if (OutgoingSet != NULL) {
			testList = (oslSocket *)OutgoingSet;
			for (n = 0; n < kMaxSocketSetSize; n++) {
				if (testList[n] != NULL && osl_isSendReady(testList[n], &zero)) {
					count++;
					sndList[n] = testList[n];
				}
			}
		}
		if (OutOfBandSet != NULL) {
			testList = (oslSocket *)OutOfBandSet;
			for (n = 0; n < kMaxSocketSetSize; n++) {
				if (testList[n] != NULL && osl_isExceptionPending(testList[n], &zero)) {
					count++;
					oobList[n] = testList[n];
				}
			}
		}
	} while (!count && !timeElapsed);

	if (timeout >= 0) {
		CancelTimerTask(task);
		DestroyTimerTask(task);
	}
	if (IncomingSet)
		memcpy(IncomingSet, rcvList, sizeof(oslSocket) * kMaxSocketSetSize);
	if (OutgoingSet)
		memcpy(OutgoingSet, sndList, sizeof(oslSocket) * kMaxSocketSetSize);
	if (OutOfBandSet)
		memcpy(OutOfBandSet, oobList, sizeof(oslSocket) * kMaxSocketSetSize);

	return count;
}

sal_Bool SAL_CALL osl_isEqualSocketAddr( oslSocketAddr Addr1, oslSocketAddr Addr2)
{
	return sal_False;
}

/*****************************************************************************/
/* osl_closeSocket  */
/*****************************************************************************/
void SAL_CALL osl_closeSocket(oslSocket Socket)
{
}

/*****************************************************************************/
/* osl_createInetBroadcastAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createInetBroadcastAddr (
	rtl_uString *strDottedAddr,
	sal_Int32    Port)
{
	return NULL;
}


