#! /usr/bin/env python
# -*- coding: utf-8 -*-

#   OpenTeacher
#   depends on: python-qt4
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

from PyQt4 import QtCore

import random
import errors

class Word:
	"""This class is the representation of a word inside OpenTeacher, it's used
	   everywhere. Mostly in combination with a WordList-object."""
	def __init__(self):
		#Set default values for properties
		self.question = u""
		self.answer = u""
		self.secondAnswer = u""
		self.id = 0
		self.worth = 1.0
		self.errorCount = 0
		self.testCount = 0
		
		# Counter in lessons
		self.lessonCount = 0
		self.errorLessonCount = 0

	#Setter
	def __setattr__(self, name, value):
		#Numbers are forced to ints
		if name in ("id", "errorCount", "testCount"):
			self.__dict__[name] = int(value)
		#Worth should be a float
		elif name in ("worth",):
			self.__dict__[name] = float(value)
		#Strings are forced to unicode and are stripped from whitespace at the ends.
		elif name in ("question", "answer", "secondAnswer"):
			self.__dict__[name] = unicode(value).strip()
		#Adding custom attributes is allowed, like normal in python
		else:
			self.__dict__[name] = value

	@property
	def results(self):
		return u"%s/%s" % (self.errorCount, self.testCount)

	#Tells if second answer is set, returns bool
	def secondAnswerSet(self):
		return self.secondAnswer != u""

	#Reverses a word. Creates a new word, because otherwise the old word is
	#damaged. (A word is mutable, and mutable objects are passed by reference
	#in python.)
	def reverse(self):
		newWord = Word()
		newWord.question = self.answer
		newWord.answer = self.question
		newWord.testCount = self.testCount
		newWord.errorCount = self.errorCount
		newWord.worth = self.worth
		newWord.id = self.id

		return newWord

class WordList:
	def __init__(self):
		#Sets default values
		self.words = []
		self.title = u""
		self.questionLanguage = u""
		self.answerLanguage = u""

	#Functions which allow WordList to be threated pretty much like a python-list
	def __len__(self):
		return len(self.words)

	def __iter__(self):
		return iter(self.words)

	def __getitem__(self, key):
		return self.words[key]

	def __delitem__(self, key):
		del self.words[key]

	def index(self, word):
		return self.words.index(word)

	#Setter
	def __setattr__(self, name, value):
		#Forces to unicode, strips from spaces
		if name in ["title", "questionLanguage", "answerLanguage"]:
			self.__dict__[name] = unicode(value).strip()
		#Normal access
		else:
			self.__dict__[name] = value

	#Adds a Word-instance to the wordList
	def addWord(self, word):
		self.words.append(word)

	#Methods telling if a property is set (not the default)
	def titleSet(self):
		return self.title != u""

	def questionLanguageSet(self):
		return self.questionLanguage != u""

	def answerLanguageSet(self):
		return self.answerLanguage != u""

#Define some 'enums' (or something looking like it).
class QuestionOrder:
	QUESTION_ANSWER, ANSWER_QUESTION = xrange(2)

class LessonType:
	SMART, INTERVAL, ALL_RIGHT, ALL_ONCE = xrange(4)

class LessonOrder:
	RANDOM, NORMAL, REVERSE, SORTED_ASC, SORTED_DESC = xrange(5)

class WordsToAsk:
	ALL_WORDS, NEVER_CORRECTLY_ANSWERED_WORDS, HARD_WORDS = xrange(3)

class LessonWordList:
	"""This class manages the wordList during lessons. It can filter the wordList
	   based on some properties. (See above 'enums'.) It also handles setting the
	   results to the words, and the word sequence during a test.

	   Methods:
		   filter
		   getNext
		   answerRight
		   answerWrong
		   correctPreviousWord

	   Properties:
		   wordList
		   firstRoundDone

		   questionOrder - should have one of the pre-defined values of QuestionOrder
		   lessonType - should have one of the pre-defined values of LessonType
		   lessonOrder - should have one of the pre-defined values of LessonOrder
		   wordsToAsk - should have one of the pre-defined values of WordsToAsk

		   numberOfAnswers
		   numberOfRightAnswers
		   numberOfQuestions

		   dutchNote
		   percentsNumber
		   percentsNote
		   americanNote
		   germanNote
		   frenchNote
		   ectsNote
		   """

	class LessonDoneException(Exception):
		"""This exception is raised when all the words are asked, so the user
		   of LessonWordList is notified. This is done by an exception instead
		   of a return value because it was needed to raise this exception also
		   from private methods, called by the public methods the user uses."""

	def __init__(self, wordList):
		#Set the wordList
		self.wordList = wordList
		#Check if the wordList isn't empty, raise a NoWordsEnteredError if
		#it is.
		if len(self.wordList) == 0:
			raise errors.NoWordsEnteredError()

		#This lists keep the indexes of this lesson round
		self.indexes = []

		#Set the defaults for the properties
		self.questionOrder = QuestionOrder.QUESTION_ANSWER
		self.lessonType = LessonType.SMART
		self.lessonOrder = LessonOrder.RANDOM
		self.wordsToAsk = WordsToAsk.ALL_WORDS
		
		#Set the lesson counters
		for word in self.wordList.words:
			word.lessonCount = 0
			word.errorLessonCount = 0

	def filter(self):
		"""Filter the current wordList with the settings set earlier, the result is
		   a list of indexes, fitting on the 'self.wordList'-list."""
		#If the lessonOrder is random, create a shuffled indexes-list
		if self.lessonOrder == LessonOrder.RANDOM:
			self.indexes = range(len(self.wordList))
			random.shuffle(self.indexes)

		#If the lessonOrder is reverse, reverse the indexes-list.
		elif self.lessonOrder == LessonOrder.REVERSE:
			self.indexes = range(len(self.wordList))
			self.indexes.reverse()

		#If the lessonOrder is sorted ascending, create a indexes-list sorted ascending by question
		elif self.lessonOrder == LessonOrder.SORTED_ASC:
			self.indexes = []
			for word in sorted(self.wordList, key=lambda word:word.question):
				self.indexes.append(self.wordList.index(word))
		#If the lessonOrder is sorted descending, do the same as asc, only reverse the list.
		elif self.lessonOrder == LessonOrder.SORTED_DESC:
			self.indexes = []
			sortedWordList = sorted(self.wordList, key=lambda word:word.question)
			sortedWordList.reverse()
			for word in sortedWordList:
				self.indexes.append(self.wordList.index(word))
		#If the lessonOrder is NORMAL, just get all the indexes.
		else:
			self.indexes = range(len(self.wordList))

		#Filter on the 'WordsToAsk'-column
		if self.wordsToAsk == WordsToAsk.NEVER_CORRECTLY_ANSWERED_WORDS:
			#Only never correctly answered words
			for index in self.indexes[:]:
				word = self.wordList[index]
				if word.errorCount != word.testCount:
					self.indexes.remove(index)
		elif self.wordsToAsk == WordsToAsk.HARD_WORDS:
			#Only hard words (>50% wrong)
			for index in self.indexes[:]:
				word = self.wordList[index]
				#Only delete words if they are asked earlier
				if word.testCount != 0:
					#Delete word if it isn't a hard word
					if float(word.errorCount) <= (float(word.testCount) /2):
						self.indexes.remove(index)
		#Else (ALL_WORDS): don't filter. (I made it pass, for clearness)
		else:
			pass

		#Check if there were words left after the filtering.
		if len(self.indexes) == 0:
			raise errors.NoWordsLeftError()

		#Reset the lesson
		self.numberOfQuestions = len(self.indexes)
		self.numberOfAnswers = 0
		self.numberOfRightAnswers = 0
		self._score = 0
		self._wordsToAskAgainCounter = 0
		
		#Reset word worth
		for index in self.indexes:
			self.wordList[index].worth = 1.0

	def getNext(self):
		"""Gets the next word in the test. If the lesson is done,
		   LessonDoneException is raised."""
		
		#circulate the currentIndex to the previousIndex
		try:
			self.previousIndex = self.currentIndex
		except AttributeError:
			pass

		if self.indexes == []:
			raise self.LessonDoneException()
		else:
			#Get a new currentIndex
			self.currentIndex = self.indexes.pop(0)

		#Get the current word with that index
		currentWord = self.wordList[self.currentIndex]
		#If the user wants that the answers are asked instead of the questions:
		if self.questionOrder == QuestionOrder.ANSWER_QUESTION:
			#turn question and answer
			currentWord = currentWord.reverse()
		#Return the current word
		return currentWord

	def _wordAnswered(self):
		"""Just increments some counters"""
		self.wordList[self.currentIndex].testCount += 1
		self.wordList[self.currentIndex].lessonCount += 1
		self.numberOfAnswers += 1

	def answerRight(self):
		word = self.wordList[self.currentIndex]

		#Mark word answered
		self._wordAnswered()

		#Update score
		self.numberOfRightAnswers += 1
		self._score += word.worth
		if word.worth != 1.0:
			self._wordsToAskAgainCounter -= 1
		
		# Interval lesson type
		if self.lessonType == LessonType.INTERVAL:
			try:
				# Check if this item has been asked and answered right enough
				if (float(word.lessonCount - word.errorLessonCount) / float(word.lessonCount) >= 0.8 and \
				   word.lessonCount > 3) or \
				   len(self.indexes) == 0:
					# Remove item
					self.indexes.pop(self.currentIndex)
				else:
					# Add item 3 places further
					self.indexes.insert(self.currentIndex + 3, self.currentIndex)
					self.numberOfQuestions += 1
			except IndexError:
				pass
				

	def answerWrong(self):
		#Mark word answered
		self._wordAnswered()
		
		word = self.wordList[self.currentIndex]
		#Update errorCount
		word.errorCount += 1
		#Update errorLessonCount
		word.errorLessonCount += 1
		#Update the worth of the word
		word.worth /= 2.0

		if self.lessonType in (LessonType.ALL_RIGHT,
							   LessonType.SMART):
			#Ask word again at the end, if it isn't already the last
			#asked one
			try:
				if self.indexes[-1] != self.currentIndex:
					self.indexes.append(self.currentIndex)
					self.numberOfQuestions += 1
					self._wordsToAskAgainCounter += 1
			except IndexError:
				pass
		if self.lessonType == LessonType.SMART:
			#Ask word also again quicker, if it isn't already asked
			#around that position
			try:
				if self.indexes[1] != self.currentIndex and \
				   self.indexes[2] != self.currentIndex:
					self.indexes.insert(2, self.currentIndex)
					self.numberOfQuestions += 1
					self._wordsToAskAgainCounter += 1
			except IndexError:
				pass
		if self.lessonType == LessonType.INTERVAL:
			#Ask word also again quicker, if it isn't already asked
			#around that position
			try:
				self.indexes.insert(self.currentIndex + 3, self.currentIndex)
				self.numberOfQuestions += 1
				self._wordsToAskAgainCounter += 1
			except IndexError:
				pass

	#Corrects the previous word, doesn't check if that word was wrong!
	def correctPreviousWord(self):
		"""Marks the previous word right. If that was the only word still in the lesson,
		   LessonDoneException is raised."""
		word = self.wordList[self.previousIndex]

		#The errorcount of the previous word is decremented.
		word.errorCount -= 1
		#The worth is corrected
		word.worth *= 2
		#The lessonCounter is corrected.
		self.numberOfRightAnswers += 1
		#The score is corrected	
		self._score += word.worth

		try:
			if self.indexes[-1] == self.previousIndex:
				#remove the last item from the list, if it is the
				#previousIndex
				del(self.indexes[-1])
				self.numberOfQuestions -= 1
				self._wordsToAskAgainCounter -= 1
		except IndexError:
			pass

		try:
			if self.indexes[1] == self.previousIndex: #smartIndex shifted if one
				del self.indexes[1]
				self.numberOfQuestions -= 1
				self._wordsToAskAgainCounter -= 1
		except IndexError:
			pass

	@property
	def dutchNote(self):
		"""Calculates note: rightAnswers/numberOfAnswers * 9 + 1"""
		try:
			return "%0.1f" % (float(self._score) / float(self.numberOfAnswers + self._wordsToAskAgainCounter) * 9.0 + 1.0)
		except ZeroDivisionError:
			return "1.0"

	@property
	def percentsNumber(self):
		"""Calculates percents of right answers: rightAnswers/numberOfAnsers * 100"""
		return int(float(self._score)/float(self.numberOfAnswers + self._wordsToAskAgainCounter) * 100)

	@property
	def percentsNote(self):
		return unicode(self.percentsNumber) +  u"%"

	@property
	def americanNote(self):
		"""Converts percents to USA-notes, uses this (dutch) table:
		   http://nl.wikipedia.org/wiki/Onderwijs_in_de_Verenigde_Staten#Schoolcijfers"""
		percents = self.percentsNumber
		if percents >= 97:
			return "A+"
		elif percents >= 93:
			return "A"
		elif percents >= 90:
			return "A-"
		elif percents >= 87:
			return "B+"
		elif percents >= 83:
			return "B"
		elif percents >= 80:
			return "B-"
		elif percents >= 77:
			return "C+"
		elif percents >= 73:
			return "C"
		elif percents >= 70:
			return "C-"
		elif percents >= 67:
			return "D+"
		elif percents >= 63:
			return "D"
		elif percents >= 60:
			return "D-"
		else:
			return "F"

	@property
	def germanNote(self):
		"""uses this (german) table: http://de.wikipedia.org/wiki/Schulnote#Berufskolleg"""
		percents = self.percentsNumber
		if percents >= 92:
			return "1"
		elif percents >= 81:
			return "2"
		elif percents >= 67:
			return "3"
		elif percents >= 50:
			return "4"
		elif percents >= 30:
			return "5"
		else:
			return "6"

	@property
	def frenchNote(self):
		return str(int(float(self._score)/float(self.numberOfAnswers + self._wordsToAskAgainCounter) * 20))

	@property
	def ectsNote(self):
		percents = self.percentsNumber
		if percents >= 70:
			return "A"
		elif percents >= 60:
			return "B"
		elif percents >= 55:
			return "C"
		elif percents >= 50:
			return "D"
		elif percents >= 40:
			return "E"
		elif percents >= 30:
			return "FX"
		else:
			return "F"
