//                                               -*- C++ -*-
/**
 *  @file  Log.hxx
 *  @brief Log records all user information to a file or tty
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-29 08:12:01 +0200 (Fri, 29 Jul 2011) $
 *  Id:      $Id: Log.hxx 2051 2011-07-29 06:12:01Z schueller $
 */
#ifndef OPENTURNS_LOG_HXX
#define OPENTURNS_LOG_HXX

#include <fstream>
#include <map>
#include "OTprivate.hxx"
#include "AtomicFunctions.hxx"

#ifdef WIN32
#undef ERROR
#endif

#define LOGDEBUG(st)   do { if (OT::Base::Common::Log::HasDebug()  ) OT::Base::Common::Log::Debug(st);   } while(0)
#define LOGWRAPPER(st) do { if (OT::Base::Common::Log::HasWrapper()) OT::Base::Common::Log::Wrapper(st); } while(0)
#define LOGINFO(st)    do { if (OT::Base::Common::Log::HasInfo()   ) OT::Base::Common::Log::Info(st);    } while(0)
#define LOGUSER(st)    do { if (OT::Base::Common::Log::HasUser()   ) OT::Base::Common::Log::User(st);    } while(0)
#define LOGWARN(st)    do { if (OT::Base::Common::Log::HasWarn()   ) OT::Base::Common::Log::Warn(st);    } while(0)
#define LOGERROR(st)   do { if (OT::Base::Common::Log::HasError()  ) OT::Base::Common::Log::Error(st);   } while(0)
#define LOGTRACE(st)   do { if (OT::Base::Common::Log::HasTrace()  ) OT::Base::Common::Log::Trace(st);   } while(0)


namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

#ifndef SWIG
      struct _Prefix
      {
        typedef String Value;
        Value color_, nocolor_, prefix_;
        _Prefix() : color_(), nocolor_(), prefix_() {}
        _Prefix( const Value & color, const Value & nocolor, const Value & prefix) : color_(color), nocolor_(nocolor), prefix_(prefix) {}
      };
#endif


      /**
       * @class log
       * @brief Records all user information to a file or tty
       * @internal
       */

      class Log
      {
      public:

        typedef unsigned long Severity;

      private:
        static Log & GetInstance();
        static void Initialization();
        static void Release();

      public:

        /** Those flags should be ORed */
        static const Severity DBG;
        static const Severity WRAPPER;
        static const Severity INFO;
        static const Severity USER;
        static const Severity WARN;
        static const Severity ERROR;
        static const Severity TRACE;

        static const Severity DEFAULT;
        static const Severity NONE;
        static const Severity ALL;

#ifndef SWIG
        /** Change the Log */
        static void Reset();
#endif

        /** Log messages according to the DBG level
         * @param msg The message to be logged
         */
        static void Debug(const String & msg);
        static inline Bool HasDebug() { return Flags() & Log::DBG; }

        /** Log messages according to the WRAPPER level
         * @param msg The message to be logged
         */
        static void Wrapper(const String & msg);
        static inline Bool HasWrapper() { return Flags() & Log::WRAPPER; }

        /** Log messages according to the INFO level
         * @param msg The message to be logged
         */
        static void Info(const String & msg);
        static inline Bool HasInfo() { return Flags() & Log::INFO; }

        /** Log messages according to the USER level
         * @param msg The message to be logged
         */
        static void User(const String & msg);
        static inline Bool HasUser() { return Flags() & Log::USER; }

        /** Log messages according to the WARN level
         * @param msg The message to be logged
         */
        static void Warn(const String & msg);
        static inline Bool HasWarn() { return Flags() & Log::WARN; }

        /** Log messages according to the ERROR level
         * @param msg The message to be logged
         */
        static void Error(const String & msg);
        static inline Bool HasError() { return Flags() & Log::ERROR; }

        /** Log messages according to the TRACE level
         * @param msg The message to be logged
         */
        static void Trace(const String & msg);
        static inline Bool HasTrace() { return Flags() & Log::TRACE; }

        /** Divert the output to a file
         * @param file The filename where the log will be written
         */
        static void SetFile(const FileName & file);

        /** Set the level flags for the messages logged to the file
         * @param flags An integer built from ORed level flags
         */
        static void Show(Severity flags);

        /** Get the current level flags
         * @return An integer built from ORed level flags
         */
        static Severity Flags();

        /** Flush pending messages
         * @internal
         */
        static void Flush();

        /** Does Log show repeated messages or not
         *  If repeat is false then Log shows every messages it receives
         *  even if they are identical to the previous ones.
         *  If repeat is true then Log only shows the first message
         *  and a message couting how much identical messages were
         *  received after that.
         */
        static void Repeat( Bool repeat );


#ifndef SWIG
        struct Entry
        {
          Severity sev_;
          String msg_;
          Entry(Severity sev, String msg) : sev_(sev), msg_(msg) {}
          Entry() : sev_(0), msg_() {}
          Bool operator== (const Entry & other) const { return (this->sev_ == other.sev_) && (this->msg_ == other.msg_); }
        }; /* end struct Entry */
#endif
        ~Log();

      private:
        Log();
        void push(const Entry & entry);
        void printRepeatedMessage(const Entry & entry);
        void flush();
        void repeat( Bool r );

        /** Set the name of the log file */
        void setFile(const FileName & file);

        /** Human readable log */
        std::map<Severity, _Prefix > logName_;

        /** The environment variable name */
        const char * openturnsLogSeverityVariableName_;

        /** Set Severity according to Openturns LogSeverity Variable */
        void initSeverityFromEnvironment();

        /** The file where to write messages */
        std::ostream * p_file_;

        /** Remember the previous message */
        mutable Entry previousMessage_;
        mutable UnsignedLong count_;
        mutable AtomicInt repeat_;

        friend class Log_init; /* friendship for static member initialization */
      }; /* end class Log */



      /** This struct initializes all static members of Log */
      struct Log_init {
        Log_init();
      }; /* end class Log_init */

      static Log_init __Log_initializer;


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LOG_HXX */
