//                                               -*- C++ -*-
/**
 *  @file  LinearCombinationEvaluationImplementation.cxx
 *  @brief The evaluation part of linear combination of polynomials
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "LinearCombinationEvaluationImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Description.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      using Func::NumericalMathFunction;
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<NumericalMathFunction>);

      static Common::Factory<PersistentCollection<NumericalMathFunction> > RegisteredFactory("PersistentCollection<NumericalMathFunction>");
    } /* namespace Type */

    namespace Func
    {

      CLASSNAMEINIT(LinearCombinationEvaluationImplementation);

      static Base::Common::Factory<LinearCombinationEvaluationImplementation> RegisteredFactory("LinearCombinationEvaluationImplementation");

      typedef Type::Description                Description;

      /* Default constructor */
      LinearCombinationEvaluationImplementation::LinearCombinationEvaluationImplementation()
        : NumericalMathEvaluationImplementation(),
          functionsCollection_(0),
          coefficients_(0)
      {
        // Nothing to do
      }


      /* Parameters constructor */
      LinearCombinationEvaluationImplementation::LinearCombinationEvaluationImplementation(const NumericalMathFunctionCollection & functionsCollection,
                                                                                           const NumericalPoint & coefficients)
        : NumericalMathEvaluationImplementation(),
          functionsCollection_(0),
          coefficients_(0)
      {
        setFunctionsCollectionAndCoefficients(functionsCollection, coefficients);
      }


      /* Virtual constructor */
      LinearCombinationEvaluationImplementation * LinearCombinationEvaluationImplementation::clone() const
      {
        return new LinearCombinationEvaluationImplementation(*this);
      }


      /* String converter */
      String LinearCombinationEvaluationImplementation::__repr__() const
      {
        return OSS() << "class=" << GetClassName()
                     << " functions=" << functionsCollection_
                     << " coefficients=" << coefficients_.__repr__();
      }

      String LinearCombinationEvaluationImplementation::__str__(const String & offset) const
      {
        OSS oss(false);
        oss << offset;
        const UnsignedLong size(functionsCollection_.getSize());
        for (UnsignedLong i = 0; i < size; ++i) oss << (i > 0 ? "+" : "") << "(" << coefficients_[i] << ")*" << functionsCollection_[i].getEvaluationImplementation()->__str__();
        return oss;
      }


      /* Evaluation operator */
      LinearCombinationEvaluationImplementation::NumericalPoint LinearCombinationEvaluationImplementation::operator () (const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong inputDimension(getInputDimension());
        if (inP.getDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given point has an invalid dimension. Expect a dimension " << inputDimension << ", got " << inP.getDimension();
        const UnsignedLong size(functionsCollection_.getSize());
        NumericalPoint result(getOutputDimension());
        for (UnsignedLong i = 0; i < size; ++i) result += coefficients_[i] * functionsCollection_[i](inP);
        return result;
      }

      /* Coefficients accessor */
      LinearCombinationEvaluationImplementation::NumericalPoint LinearCombinationEvaluationImplementation::getCoefficients() const
      {
        return coefficients_;
      }

      /* Functions accessor */
      LinearCombinationEvaluationImplementation::NumericalMathFunctionCollection LinearCombinationEvaluationImplementation::getFunctionsCollection() const
      {
        return functionsCollection_;
      }

      void LinearCombinationEvaluationImplementation::setFunctionsCollectionAndCoefficients(const NumericalMathFunctionCollection & functionsCollection,
                                                                                            const NumericalPoint & coefficients)
      {
        const UnsignedLong size(functionsCollection.getSize());
        // Check for empty functions collection
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a linear combination from an empty collection of functions.";
        // Check for incompatible number of functions and coefficients
        if (size != coefficients.getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot build a linear combination with a different number of functions and coefficients.";
        // Check for coherent input and output dimensions of the functions
        UnsignedLong inputDimension(functionsCollection[0].getInputDimension());
        UnsignedLong outputDimension(functionsCollection[0].getOutputDimension());
        for (UnsignedLong i = 1; i < size; ++i)
          {
            if (functionsCollection[i].getInputDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given functions have incompatible input dimension.";
            if (functionsCollection[i].getOutputDimension() != outputDimension) throw InvalidArgumentException(HERE) << "Error: the given functions have incompatible output dimension.";
          }
        functionsCollection_ = functionsCollection;
        coefficients_ = coefficients;
        setDescription(functionsCollection[0].getDescription());
      }


      /* Input dimension accessor */
      UnsignedLong LinearCombinationEvaluationImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        return functionsCollection_[0].getInputDimension();
      }

      /* Output dimension accessor */
      UnsignedLong LinearCombinationEvaluationImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return functionsCollection_[0].getOutputDimension();
      }


      /* Gradient according to the marginal parameters */
      LinearCombinationEvaluationImplementation::Matrix LinearCombinationEvaluationImplementation::parametersGradient(const NumericalPoint & inP) const
      {
        Matrix result(getParameters().getDimension(), getOutputDimension());
        const UnsignedLong size(functionsCollection_.getSize());
        // Get the parameters gradients for each atom and stack them into the result
        UnsignedLong rowIndex(0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            // Extract the atom gradient
            const Matrix atomParametersGradient(functionsCollection_[i].parametersGradient(inP));
            const UnsignedLong rowDimension(atomParametersGradient.getNbRows());
            const UnsignedLong columnDimension(atomParametersGradient.getNbColumns());
            // Scale the atom gradient and copy it into the result
            const NumericalScalar coefficient(coefficients_[i]);
            for (UnsignedLong j = 0; j < rowDimension; ++j)
              {
                for (UnsignedLong k = 0; k < columnDimension; ++k)
                  {
                    result(rowIndex, k) = coefficient * atomParametersGradient(j, k);
                  }
                ++rowIndex;
              }
          }
        return result;
      }

      /* Parameters value and description accessor */
      LinearCombinationEvaluationImplementation::NumericalPointWithDescription LinearCombinationEvaluationImplementation::getParameters() const
      {
        NumericalPointWithDescription parameters(0);
        Description description(0);
        const UnsignedLong size(functionsCollection_.getSize());
        for (UnsignedLong i = 0; i < size; ++i)
          {
            // Extract the parameters of the current atom
            NumericalPointWithDescription atomParameters(functionsCollection_[i].getParameters());
            Description atomDescription(atomParameters.getDescription());
            const UnsignedLong atomSize(atomParameters.getDimension());
            // Copy the parameters value and description
            for (UnsignedLong j = 0; j < atomSize; ++j)
              {
                parameters.add(atomParameters[i]);
                description.add(atomDescription[i]);
              }
          }
        parameters.setDescription(description);
        return parameters;
      }

      /* Method save() stores the object through the StorageManager */
      void LinearCombinationEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
        PersistentObject::save(adv);
        adv.saveAttribute( "functionsCollection_", functionsCollection_ );
        adv.saveAttribute( "coefficients_", coefficients_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void LinearCombinationEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
        PersistentObject::load(adv);
        adv.loadAttribute( "functionsCollection_", functionsCollection_ );
        adv.loadAttribute( "coefficients_", coefficients_ );
      }



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
