//                                               -*- C++ -*-
/**
 *  @file  NumericalMathGradientImplementation.hxx
 *  @brief Abstract top-level class for all numerical math gradient implementations
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: NumericalMathGradientImplementation.hxx 1866 2011-04-11 10:32:27Z schueller $
 */
#ifndef OPENTURNS_NUMERICALMATHGRADIENTIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALMATHGRADIENTIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "Exception.hxx"
#include "Description.hxx"
#include "Indices.hxx"
#include "Matrix.hxx"
#include "Tensor.hxx"
#include "Pointer.hxx"
#include "StorageManager.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class NumericalMathGradientImplementation
       *
       * This class offers an abstract interface for the implementation
       * of an real numerical mathematical Gradient into the platform.
       */
      class NumericalMathGradientImplementation
        : public Common::PersistentObject
      {
        CLASSNAME;
      public:

        typedef Pointer<NumericalMathGradientImplementation> Implementation;
        typedef Type::Description                                    Description;
        typedef Type::Indices                                        Indices;
        typedef Type::NumericalPoint                                 NumericalPoint;
        typedef Type::Matrix                                         Matrix;
        typedef Type::Tensor                                         Tensor;
        typedef Common::InvalidArgumentException                     InvalidArgumentException;
        typedef Common::InternalException                            InternalException;
        typedef Common::StorageManager                               StorageManager;


        /** Default constructor */
        NumericalMathGradientImplementation();

        /** The Gradient that actually realizes assignment between objects */

        /** Virtual constructor */
        virtual NumericalMathGradientImplementation * clone() const;

        /** Comparison operator */
        Bool operator ==(const NumericalMathGradientImplementation & other) const;

        /** String converter */
        virtual String __repr__() const;
        virtual String __str__(const String & offset = "") const;


        /** Description Accessor */
        void setDescription(const Description & description);
        Description getDescription() const;

        /** Get the i-th marginal evaluation */
        virtual NumericalMathGradientImplementation * getMarginal(const UnsignedLong i) const;

        /** Get the evaluation corresponding to indices components */
        virtual NumericalMathGradientImplementation * getMarginal(const Indices & indices) const;

        /** Test for actual implementation */
        virtual Bool isActualImplementation() const;



        /* Here is the interface that all derived class must implement */

        /** Gradient method */
        virtual Matrix gradient(const NumericalPoint & inP) const
          /* throw(InvalidArgumentException, InternalException) */;

        /** Accessor for input point dimension */
        virtual UnsignedLong getInputDimension() const
          /* throw(InternalException) */;

        /** Accessor for output point dimension */
        virtual UnsignedLong getOutputDimension() const
          /* throw(InternalException) */;

        /** Get the number of calls to operator() */
        UnsignedLong getCallsNumber() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

        /** Number of calls since the construction */
        mutable UnsignedLong callsNumber_;

      private:

        /** The description of all components */
        Description description_;


      }; /* class NumericalMathGradientImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALMATHGRADIENTIMPLEMENTATION_HXX */
