//                                               -*- C++ -*-
/**
 * @file  CorrelationAnalysis.hxx
 * @brief CorrelationAnalysis implements computation of correlation coefficients
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-06-30 11:29:46 +0200 (Thu, 30 Jun 2011) $
 *  Id:      $Id: CorrelationAnalysis.hxx 1975 2011-06-30 09:29:46Z schueller $
 */

#ifndef OPENTURNS_CORRELATIONANALYSIS_HXX
#define OPENTURNS_CORRELATIONANALYSIS_HXX

#include "OTprivate.hxx"
#include "NumericalSample.hxx"
#include "NumericalPoint.hxx"
#include "NumericalMathFunction.hxx"
#include "SymmetricTensor.hxx"
#include "SobolIndicesParameters.hxx"
#include "SobolIndicesResult.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      /**
       * @class CorrelationAnalysis
       *
       */

      class CorrelationAnalysis
      {
      public:

        typedef Type::NumericalPoint        NumericalPoint;
        typedef Func::NumericalMathFunction NumericalMathFunction;

        /** Default constructor */
        CorrelationAnalysis();

        /** Compute the Pearson correlation coefficient between the component number index of the input sample and the 1D outputSample */
        static NumericalScalar PearsonCorrelation(const NumericalSample & inputSample,
                                                  const NumericalSample & outputSample,
                                                  const UnsignedLong index = 0);

        /** Compute the Spearman correlation coefficient between the component number index of the input sample and the 1D outputSample */
        static NumericalScalar SpearmanCorrelation(const NumericalSample & inputSample,
                                                   const NumericalSample & outputSample,
                                                   const UnsignedLong index = 0);

        /** Compute the Standard Regression Coefficients (SRC) between the input sample and the output sample */
        static NumericalPoint SRC(const Base::Stat::NumericalSample & inputSample,
                                  const Base::Stat::NumericalSample & outputSample);

        /** Compute the Partial Correlation Coefficients (PCC) between the input sample and the output sample */
        static NumericalPoint PCC(const NumericalSample & inputSample,
                                  const NumericalSample & outputSample);

        /** Compute the Standard Rank Regression Coefficients (SRRC) between the input sample and the output sample */
        static NumericalPoint SRRC(const NumericalSample & inputSample,
                                   const NumericalSample & outputSample);

        /** Compute the Partial Rank Correlation Coefficients (PRCC) between the input sample and the output sample */
        static NumericalPoint PRCC(const NumericalSample & inputSample,
                                   const NumericalSample & outputSample);

        /** compute the Sobol' indices given two input samples and a 1d function
         * @deprecated
         * This method is deprecated. Use SensitivityAnalysis instead. */
        static SobolIndicesResult SobolIndices(const SobolIndicesParameters & sobolParameter,
                                               const NumericalSample & firstInputSample,
                                               const NumericalSample & secondInputSample,
                                               const NumericalMathFunction & function) DEPRECATED;

      }; /* class CorrelationAnalysis */

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_CORRELATIONANALYSIS_HXX */
