//                                               -*- C++ -*-
/**
 *  @file  TestResult.cxx
 *  @brief TestResult implements the result of a statistical test
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: TestResult.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "TestResult.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {
      CLASSNAMEINIT(TestResult);

      static Base::Common::Factory< TestResult > RegisteredFactory("TestResult");

      /* Default constructor */
      TestResult::TestResult():
        PersistentObject(),
        testType_(""),
        binaryQualityMeasure_(false),
        pValueThreshold_(0.0),
        pValue_(0.0),
        description_(0)
      {
        // Nothing to do
      }

      /* Parameters constructor */
      TestResult::TestResult(const String & type,
                             const Bool binMeasure,
                             const NumericalScalar pVal,
                             const NumericalScalar pThreshold):
        PersistentObject(),
        testType_(type),
        binaryQualityMeasure_(binMeasure),
        pValueThreshold_(pThreshold),
        pValue_(pVal),
        description_(0)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      TestResult * TestResult::clone() const
      {
        return new TestResult(*this);
      }

      /* Description Accessor */
      void TestResult::setDescription(const Description & description)
      {
        description_ = description;
      }

      TestResult::Description TestResult::getDescription() const
      {
        return description_;
      }

      /* String converter */
      String TestResult::__repr__() const
      {
        return OSS() << "class=" << TestResult::GetClassName()
                     << " name=" << getName()
                     << " type=" << testType_
                     << " binaryQualityMeasure=" << binaryQualityMeasure_
                     << " p-value threshold=" << pValueThreshold_
                     << " p-value=" << pValue_
                     << " description=" << description_;
      }

      /* Elements accessors */
      Bool TestResult::getBinaryQualityMeasure() const
      {
        return binaryQualityMeasure_;
      }

      NumericalScalar TestResult::getPValue() const
      {
        return pValue_;
      }

      NumericalScalar TestResult::getThreshold() const
      {
        return pValueThreshold_;
      }

      String TestResult::getTestType() const
      {
        return testType_;
      }

      /* Comparison operator */
      Bool TestResult::operator == (const TestResult & other) const
      {
        if (this == &other) return true;
        return (binaryQualityMeasure_ == other.binaryQualityMeasure_) && (pValue_ == other.pValue_) && (pValueThreshold_ == other.pValueThreshold_) && (testType_ == other.testType_);
      }


      /** Method save() stores the object through the StorageManager */
      void TestResult::save(StorageManager::Advocate & adv) const
      {

        Base::Common::PersistentObject::save(adv);
        adv.saveAttribute("testType_", testType_);
        adv.saveAttribute("binaryQualityMeasure_", binaryQualityMeasure_);
        adv.saveAttribute("pValueThreshold_", pValueThreshold_);
        adv.saveAttribute("pValue_", pValue_);
        adv.saveAttribute("description_", description_);
      }

      /** Method load() reloads the object from the StorageManager */
      void TestResult::load(StorageManager::Advocate & adv)
      {
        Base::Common::PersistentObject::load(adv);
        adv.loadAttribute("testType_", testType_);
        adv.loadAttribute("binaryQualityMeasure_", binaryQualityMeasure_);
        adv.loadAttribute("pValueThreshold_", pValueThreshold_);
        adv.loadAttribute("pValue_", pValue_);
        adv.loadAttribute("description_", description_);
      }

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
