//                                               -*- C++ -*-
/**
 *  @file  StrongMaximumTest.hxx
 *  @brief StrongMaxTest implements an algorithm to check if a given design point
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: StrongMaximumTest.hxx 1866 2011-04-11 10:32:27Z schueller $
 */
#ifndef OPENTURNS_STRONGMAXTEST_HXX
#define OPENTURNS_STRONGMAXTEST_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "StandardEvent.hxx"
#include "NumericalSample.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      /**
       * @class StrongMaximumTest
       * StrongMaximumTest allows to validate a design point
       */
      class StrongMaximumTest :
        public Base::Common::PersistentObject
      {
        static const NumericalScalar DefaultDeltaPrecision;
        static const NumericalScalar Epsilon;

        CLASSNAME;
      public:
        typedef Base::Stat::NumericalSample            NumericalSample;
        typedef Base::Type::NumericalPoint             NumericalPoint;
        typedef Model::StandardEvent                   StandardEvent;
        typedef Base::Common::InvalidRangeException    InvalidRangeException;
        typedef Base::Common::InvalidArgumentException InvalidArgumentException;
        typedef Base::Common::InternalException        InternalException;
        typedef Base::Common::StorageManager           StorageManager;

        /** Default constructor */
        StrongMaximumTest();

        /** Standard constructor */
        StrongMaximumTest(const StandardEvent & event,
                          const NumericalPoint & standardSpaceDesignPoint,
                          const NumericalScalar importanceLevel,
                          const NumericalScalar accuracyLevel,
                          const NumericalScalar confidenceLevel)
          /* throw(InvalidArgumentException) */;

        /** Standard constructor */
        StrongMaximumTest(const StandardEvent & event,
                          const NumericalPoint & standardSpaceDesignPoint,
                          const NumericalScalar importanceLevel,
                          const NumericalScalar accuracyLevel,
                          const UnsignedLong pointNumber)
          /* throw(InvalidArgumentException) */;


        /** Virtual constructor */
        virtual StrongMaximumTest * clone() const;

        /** standardSpaceDesignPoint accessor */
        NumericalPoint getStandardSpaceDesignPoint() const;

        /** Event accessor */
        StandardEvent getEvent() const;

        /** ImportanceLevel accessor */
        NumericalScalar getImportanceLevel() const;

        /** AccuracyLevel accessor */
        NumericalScalar getAccuracyLevel() const;

        /** ConfidenceLevel accessor */
        NumericalScalar getConfidenceLevel() const ;

        /** DesignPointVicinity accessor */
        NumericalScalar  getDesignPointVicinity() const;

        /** PointNumber accessor */
        UnsignedLong getPointNumber() const;

        /** DeltaEpsilon accessor */
        NumericalScalar getDeltaEpsilon() const;

        /** The function that runs the Strong Max Test */
        void run()
          /* throw(InvalidArgumentException, InternalException) */;

        /** NearDesignPointVerifyingEventPoints accessor */
        NumericalSample getNearDesignPointVerifyingEventPoints() const;

        /** FarDesignPointVerifyingEventPoints accessor */
        NumericalSample getFarDesignPointVerifyingEventPoints() const;

        /** NearDesignPointViolatingEventPoints accessor */
        NumericalSample getNearDesignPointViolatingEventPoints() const;

        /** FarDesignPointViolatingEventPoints accessor */
        NumericalSample getFarDesignPointViolatingEventPoints() const;

        /** NearDesignPointVerifyingEventValues accessor */
        NumericalSample getNearDesignPointVerifyingEventValues() const;

        /** FarDesignPointVerifyingEventValues accessor */
        NumericalSample getFarDesignPointVerifyingEventValues() const;

        /** NearDesignPointViolatingEventValues accessor */
        NumericalSample getNearDesignPointViolatingEventValues() const;

        /** FarDesignPointViolatingEventValues accessor */
        NumericalSample getFarDesignPointViolatingEventValues() const;

        /** String converter */
        String __repr__() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

      private:

        /** AccuracyLevel accessor */
        void setAccuracyLevel(const NumericalScalar accuracyLevel)
          /* throw(InvalidRangeException) */;

        /** ConfidenceLevel accessor */
        void setConfidenceLevel( const NumericalScalar confidenceLevel)
          /* throw(InvalidRangeException) */;

        /** PointNumber accessor */
        void setPointNumber(const UnsignedLong pointNumber)
          /* throw(InvalidRangeException) */;

        /** ImportanceLevel accessor */
        void setImportanceLevel(const NumericalScalar importanceLevel)
          /* throw(InvalidRangeException) */;

        /** standardSpaceDesignPoint accessor */
        void setStandardSpaceDesignPoint(const NumericalPoint & standardSpaceDesignPoint );

        /** DesignPointVicinity accessor */
        void setDesignPointVicinity(const NumericalScalar designPointVicinity)
          /* throw(InvalidRangeException) */;

        /** Event accessor */
        void setEvent(const StandardEvent & event);

        /** Initialize Strong Max Test Parameters : method 1 */
        void initializeParametersGivenConfidenceLevel()
          /* throw(InvalidRangeException) */;

        /** Initialize Strong Max Test Parameters : method 2 */
        void initializeParametersGivenPointNumber()
          /* throw(InvalidRangeException) */;


        /** the function that samples the sphere (radius) with N points */
        NumericalSample sampleSphere(const NumericalScalar radius,
                                     const UnsignedLong dimension,
                                     const UnsignedLong pointNumber) const;

        /**  the function that evaluates the HyperSphereSurfaceRatio (see documentation) */
        NumericalScalar computeHyperSphereSurfaceRatio();

        /**  the function that evaluates delta_epsilon (see documentation) */
        NumericalScalar computeDeltaEpsilon();

        /** the function that evaluates if a point is in the vicinity of the design point */
        Bool isInTheVicinityOfTheDesignPoint(const NumericalPoint & numericalPoint);

        StandardEvent event_;
        NumericalPoint standardSpaceDesignPoint_;
        NumericalScalar importanceLevel_;
        NumericalScalar accuracyLevel_;
        NumericalScalar confidenceLevel_;
        NumericalScalar designPointVicinity_; // cosinus of the cone around the standard design point
        UnsignedLong pointNumber_;
        NumericalScalar deltaEpsilon_;
        NumericalSample nearDesignPointVerifyingEventPoints_;
        NumericalSample nearDesignPointVerifyingEventValues_;
        NumericalSample farDesignPointVerifyingEventPoints_;
        NumericalSample farDesignPointVerifyingEventValues_;
        NumericalSample nearDesignPointViolatingEventPoints_;
        NumericalSample nearDesignPointViolatingEventValues_;
        NumericalSample farDesignPointViolatingEventPoints_;
        NumericalSample farDesignPointViolatingEventValues_;

      }; // class StrongMaximumTest

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_STRONGMAXTEST_HXX */
