//                                               -*- C++ -*-
/**
 *  @file  Factorial.cxx
 *  @brief Implementation of the factorial experiment plane
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: Factorial.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "OTprivate.hxx"
#include "Factorial.hxx"
#include <cmath>

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(Factorial);

      /* Default constructor */
      Factorial::Factorial(const String & name):
        StratifiedExperiment(name)
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      Factorial::Factorial(const NumericalPoint & center,
                           const NumericalPoint & levels,
                           const String & name):
        StratifiedExperiment(center, levels, name)
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      Factorial::Factorial(const UnsignedLong dimension,
                           const NumericalPoint & levels,
                           const String & name):
        StratifiedExperiment(NumericalPoint(dimension, 0.0), levels, name)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      Factorial * Factorial::clone() const
      {
        return new Factorial(*this);
      }

      /* Experiment plane generation */
      Factorial::NumericalSample Factorial::generate()
      {
        /* Dimension of the realizations */
        UnsignedLong dimension(center_.getDimension());
        /* Hypercube number of vertices */
        UnsignedLong verticesNumber((UnsignedLong)round(pow(2, dimension)));
        /* Number of levels to be generated */
        UnsignedLong levelNumber(levels_.getDimension());
        /* Size of the sample to be generated: 1 + number of levels x 2^dimension */
        UnsignedLong size(1 + levelNumber * verticesNumber);
        NumericalSample factorialPlane(size, center_);
        factorialPlane.setName("Factorial plane");
        UnsignedLong index(1);
        /* For each level of the factorial plane */
        for(UnsignedLong levelIndex = 0; levelIndex < levelNumber; ++levelIndex)
          {
            NumericalScalar levelValue(levels_[levelIndex]);
            /* For each vertex of the current level */
            for(UnsignedLong vertex = 0; vertex < verticesNumber; ++vertex)
              {
                /* Compute the coordinates of the current vertex */
                UnsignedLong binaryIndex(vertex);
                for(UnsignedLong coordinate = 0; coordinate < dimension; ++coordinate)
                  {
                    /* If the current bit of binaryIndex is 1, set the coordinate to center[coordinate]+levelValue, else set it to center[coordinate]-levelValue */
                    factorialPlane[index][coordinate] += (2.0 * (binaryIndex % 2) - 1.0) * levelValue;
                    binaryIndex /= 2;
                  } // coordinate
                ++index;
              } // vertex
          } // levelIndex
        return factorialPlane;
      } // generate()

      /* String converter */
      String Factorial::__repr__() const
      {
        OSS oss;
        oss << "class=" << GetClassName()
            << " name=" << getName()
            << " center=" << center_
            << " levels=" << levels_;
        return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
