//                                               -*- C++ -*-
/**
 * @file  SVMKernelRegressionGradient.cxx
 * @brief Implementation of SVM regression gradient
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-05 08:50:15 +0200 (lun, 05 mai 2008) $
 *  Id:      $Id: SVMKernelRegressionGradient.cxx 805 2008-05-05 06:50:15Z schueller $
 */

#include "SVMKernelRegressionGradient.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(SVMKernelRegressionGradient);


      /* Default constructor */
      SVMKernelRegressionGradient::SVMKernelRegressionGradient() :
        Base::Func::NumericalMathGradientImplementation()
      {
        // nothing to do
      }


      /* Constructor with parameters */
      SVMKernelRegressionGradient::SVMKernelRegressionGradient(const SVMKernel & kernel,
                                                               const NumericalPoint & lagrangeMultiplier,
                                                               const NumericalSample & dataIn,
                                                               const NumericalScalar constant) :
        kernel_(kernel),
        lagrangeMultiplier_(lagrangeMultiplier),
        dataIn_(dataIn),
        constant_(constant),
        p_svmEvaluation_()
      {
        // nothing to do
      }


      /* Constructor from SVMKernelRegressionEvaluation */
      SVMKernelRegressionGradient::SVMKernelRegressionGradient(const SVMEvaluation & p_svmEvaluation) :
        Base::Func::NumericalMathGradientImplementation(),
        p_svmEvaluation_(p_svmEvaluation)
      {
        // nothing to do
      }


      /* Virtual constructor */
      SVMKernelRegressionGradient * SVMKernelRegressionGradient::clone() const
      {
        return new SVMKernelRegressionGradient(*this);
      }

      /* Comparison operator */
      Bool SVMKernelRegressionGradient::operator==(const SVMKernelRegressionGradient & other) const
      {
        return true;
      }

      /* String converter */
      String SVMKernelRegressionGradient::__repr__() const {
        OSS oss;
        oss << "class=" << SVMKernelRegressionGradient::GetClassName()
            << " name=" << getName()
            << " kernel=" << kernel_
            << " lagrange multipliers=" << lagrangeMultiplier_
            << " data in=" << dataIn_
            << " constant=" << constant_;
        return oss;
      }

      /* Test for actual implementation */
      Bool SVMKernelRegressionGradient::isActualImplementation() const
      {
        return true;
      }

      /* Gradient method */
      SVMKernelRegressionGradient::Matrix SVMKernelRegressionGradient::gradient(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        ++ callsNumber_;

        UnsignedLong dimension = inP.getDimension();
        if(dimension != dataIn_.getDimension())
          throw InvalidArgumentException(HERE) << "Invalid input dimension";

        // compute the sum of the partial gradients
        UnsignedLong size = dataIn_.getSize();
        NumericalPoint partialGradient(dimension, 0.0);
        for(UnsignedLong i = 0; i < size; ++ i)
          {
            if(lagrangeMultiplier_[i] != 0.0)
              partialGradient += lagrangeMultiplier_[i] * kernel_.partialGradient(inP, dataIn_[i]);
          }

        // return the gradient into a column Matrix
        Matrix gradient(dimension, 1);
        for(UnsignedLong i = 0; i < dimension; ++ i)
          {
            gradient(i, 0) = partialGradient[i];
          }
        return gradient;
      }

      /* Accessor for input point dimension */
      UnsignedLong SVMKernelRegressionGradient::getInputDimension() const
      /* throw(InternalException) */
      {
        return dataIn_.getDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong SVMKernelRegressionGradient::getOutputDimension() const
      /* throw(InternalException) */
      {
        return 1;
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
