//                                               -*- C++ -*-
/**
 *  @file  EnumerateFunction.cxx
 *  @brief The bijective function to select polynomials in the orthogonal basis
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "EnumerateFunction.hxx"
#include "LinearEnumerateFunction.hxx"
#include "HyperbolicAnisotropicEnumerateFunction.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(EnumerateFunction);

      /* Constructor with parameters */
      EnumerateFunction::EnumerateFunction(const EnumerateFunctionImplementation & implementation)
        : Base::Common::TypedInterfaceObject<EnumerateFunctionImplementation>(implementation.clone())
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      EnumerateFunction::EnumerateFunction(Implementation & p_implementation)
        : Base::Common::TypedInterfaceObject<EnumerateFunctionImplementation>(p_implementation)
      {
        // Nothing to do
      }

      /* Parameter constructor */
      EnumerateFunction::EnumerateFunction(const UnsignedLong dimension)
        : Base::Common::TypedInterfaceObject<EnumerateFunctionImplementation>(new LinearEnumerateFunction(dimension))
      {
        // Nothing to do
      }

      /* Parameter constructor */
      EnumerateFunction::EnumerateFunction(const UnsignedLong dimension,
                                           const NumericalScalar q)
        : Base::Common::TypedInterfaceObject<EnumerateFunctionImplementation>(new HyperbolicAnisotropicEnumerateFunction(dimension, q))
      {
        // Nothing to do
      }

      /* Parameter constructor */
      EnumerateFunction::EnumerateFunction(const NumericalPoint & weight,
                                           const NumericalScalar q)
        : Base::Common::TypedInterfaceObject<EnumerateFunctionImplementation>(new HyperbolicAnisotropicEnumerateFunction(weight, q))
      {
        // Nothing to do
      }

      /* String converter */
      String EnumerateFunction::__repr__() const
      {
        return getImplementation()->__repr__();
      }

      String EnumerateFunction::__str__() const
      {
        return getImplementation()->__str__();
      }

      /* The bijective association between an integer and a set of indices */
      EnumerateFunction::Indices EnumerateFunction::operator() (const UnsignedLong index) const
      {
        return getImplementation()->operator()( index );
      }

      /* The inverse of the association */
      UnsignedLong EnumerateFunction::inverse(const Indices & indices) const
      {
        return getImplementation()->inverse( indices );
      }

      /* The cardinal of the given strata */
      UnsignedLong EnumerateFunction::getStrataCardinal(const UnsignedLong strataIndex) const
      {
        return getImplementation()->getStrataCardinal( strataIndex );
      }

      /* The cardinal of the cumulated strata above or equal to the given strata */
      UnsignedLong EnumerateFunction::getStrataCumulatedCardinal(const UnsignedLong strataIndex) const
      {
        return getImplementation()->getStrataCumulatedCardinal( strataIndex );
      }

      /* Dimension accessor */
      void EnumerateFunction::setDimension(const UnsignedLong dimension)
      {
        getImplementation()->setDimension( dimension );
      }

      UnsignedLong EnumerateFunction::getDimension() const
      {
        return getImplementation()->getDimension();
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
