//                                               -*- C++ -*-
/**
 *  @file  LaguerreFactory.cxx
 *  @brief Laguerre polynomial factory
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "LaguerreFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Gamma.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(LaguerreFactory);

      static Base::Common::Factory<LaguerreFactory> RegisteredFactory("LaguerreFactory");

      typedef Distribution::Gamma                    Gamma;
      typedef Base::Common::InvalidArgumentException InvalidArgumentException;


      /* Default constructor, associated with the default Gamma distribution which is equal to the Exponential distribution */
      LaguerreFactory::LaguerreFactory()
        : OrthogonalUniVariatePolynomialFactory(Gamma()), k_(0.0)
      {
        initializeCache();
      }


      /* Parameter constructor: k is the order of the generalized Laguerre polynomial, associated with the Gamma(k+1, 1, 0) distribution */
      LaguerreFactory::LaguerreFactory(const NumericalScalar k)
        : OrthogonalUniVariatePolynomialFactory( Gamma(k + 1.0, 1.0, 0.0) ), k_(k)
      {
        if (k <= -1.0) throw InvalidArgumentException(HERE) << "Error: must have k>-1 to build Laguerre polynomials.";
        initializeCache();
      }

      /* Virtual constructor */
      LaguerreFactory * LaguerreFactory::clone() const
      {
        return new LaguerreFactory(*this);
      }


      /* Calculate the coefficients of recurrence a0n, a1n, a2n such that
         Pn+1(x) = (a0n * x + a1n) * Pn(x) + a2n * Pn-1(x) */
      LaguerreFactory::Coefficients LaguerreFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
        Coefficients recurrenceCoefficients(3, 0.0);
        if (n == 0)
          {
            const NumericalScalar factor(sqrt(k_ + 1.0));
            recurrenceCoefficients[0] = 1.0 / factor;
            recurrenceCoefficients[1] = -factor;
            // Conventional value of 0.0 for recurrenceCoefficients[2]
            return recurrenceCoefficients;
          }
        const NumericalScalar factor(1.0 / sqrt((n + 1.0) * (n + 1.0 + k_)));
        recurrenceCoefficients[0] = factor;
        recurrenceCoefficients[1] = -(2.0 * n + 1.0 + k_) * factor;
        recurrenceCoefficients[2] = -sqrt((n + k_) * n) * factor;
        return recurrenceCoefficients;
      }

      /* K accessor */
      NumericalScalar LaguerreFactory::getK() const
      {
        return k_;
      }

      /* String converter */
      String LaguerreFactory::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " k=" << k_
                     << " measure=" << measure_;
      }


      /* Method save() stores the object through the StorageManager */
      void LaguerreFactory::save(StorageManager::Advocate & adv) const
      {
        OrthogonalUniVariatePolynomialFactory::save(adv);
        adv.saveAttribute( "k_", k_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void LaguerreFactory::load(StorageManager::Advocate & adv)
      {
        OrthogonalUniVariatePolynomialFactory::load(adv);
        adv.loadAttribute( "k_", k_ );
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
