//                                               -*- C++ -*-
/**
 *  @file  StandardDistributionPolynomialFactory.cxx
 *  @brief OrthogonalUniVariatePolynomialStandardDistribution polynomial factory
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */

#include "StandardDistributionPolynomialFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Uniform.hxx"
#include "GramSchmidtAlgorithm.hxx"
#include "CharlierFactory.hxx"
#include "HermiteFactory.hxx"
#include "JacobiFactory.hxx"
#include "KrawtchoukFactory.hxx"
#include "LaguerreFactory.hxx"
#include "LegendreFactory.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(StandardDistributionPolynomialFactory);

      static Base::Common::Factory<StandardDistributionPolynomialFactory> RegisteredFactory("StandardDistributionPolynomialFactory");

      typedef Distribution::Uniform Uniform;

      /* Default constructor */
      StandardDistributionPolynomialFactory::StandardDistributionPolynomialFactory()
        : OrthogonalUniVariatePolynomialFactory(Uniform()),
          orthonormalizationAlgorithm_(GramSchmidtAlgorithm(Uniform()))
      {
        // Initialize the coefficient cache
        initializeCache();
      }


      /* Parameter constructor */
      StandardDistributionPolynomialFactory::StandardDistributionPolynomialFactory(const Distribution & measure)
        : OrthogonalUniVariatePolynomialFactory(measure),
          orthonormalizationAlgorithm_(GramSchmidtAlgorithm(measure))
      {
        initializeCache();
      }


      /* Parameter constructor */
      StandardDistributionPolynomialFactory::StandardDistributionPolynomialFactory(const OrthonormalizationAlgorithm & orthonormalizationAlgorithm)
        : OrthogonalUniVariatePolynomialFactory(orthonormalizationAlgorithm.getMeasure()),
          orthonormalizationAlgorithm_(orthonormalizationAlgorithm)
      {
        initializeCache();
      }


      /* Virtual constructor */
      StandardDistributionPolynomialFactory * StandardDistributionPolynomialFactory::clone() const
      {
        return new StandardDistributionPolynomialFactory(*this);
      }


      /* Calculate the coefficients of recurrence a0n, a1n, a2n such that
         Pn+1(x) = (a0n * x + a1n) * Pn(x) + a2n * Pn-1(x) */
      StandardDistributionPolynomialFactory::Coefficients StandardDistributionPolynomialFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
	// Check for special cases. Need a more elegant conception and implementation. Regis.
	const String measureType(measure_.getImplementation()->getClassName());
	// Legendre factory
	if (measureType == "Uniform") return LegendreFactory().getRecurrenceCoefficients(n);
	// Hermite factory
	if (measureType == "Normal") return HermiteFactory().getRecurrenceCoefficients(n);
	// Jacobi factory
	if (measureType == "Beta")
	  {
	    const NumericalPoint parameters(measure_.getParametersCollection()[0]);
	    return JacobiFactory(parameters[1] - parameters[0] - 1.0, parameters[0] - 1.0).getRecurrenceCoefficients(n);
	  }
	// Laguerre factory
	if (measureType == "Gamma")
	  {
	    const NumericalPoint parameters(measure_.getParametersCollection()[0]);
	    return LaguerreFactory(parameters[0] - 1.0).getRecurrenceCoefficients(n);
	  }
	// Charlier factory
	if (measureType == "Poisson")
	  {
	    const NumericalPoint parameters(measure_.getParametersCollection()[0]);
	    return CharlierFactory(parameters[0]).getRecurrenceCoefficients(n);
	  }
	// Krawtchouk factory
	if (measureType == "Binomial")
	  {
	    const NumericalPoint parameters(measure_.getParametersCollection()[0]);
	    return KrawtchoukFactory(parameters[0], parameters[1]).getRecurrenceCoefficients(n);
	  }
        return orthonormalizationAlgorithm_.getRecurrenceCoefficients(n);
      }

      /* String converter */
      String StandardDistributionPolynomialFactory::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " orthonormalization algorithm=" << orthonormalizationAlgorithm_;
      }

      /* Method save() stores the object through the StorageManager */
      void StandardDistributionPolynomialFactory::save(StorageManager::Advocate & adv) const
      {
        OrthogonalUniVariatePolynomialFactory::save(adv);
        adv.saveAttribute( "orthonormalizationAlgorithm_", orthonormalizationAlgorithm_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void StandardDistributionPolynomialFactory::load(StorageManager::Advocate & adv)
      {
        OrthogonalUniVariatePolynomialFactory::load(adv);
        adv.loadAttribute( "orthonormalizationAlgorithm_", orthonormalizationAlgorithm_ );
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
