//                                               -*- C++ -*-
/**
 *  @file  RootStrategyImplementation.cxx
 *  @brief
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: RootStrategyImplementation.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include <cstdlib>

#include "RootStrategyImplementation.hxx"
#include "Brent.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      /**
       * @class RootStrategyImplementation
       */

      CLASSNAMEINIT(RootStrategyImplementation);

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::ResourceMap                ResourceMap;
      typedef Base::Solver::Brent                      Brent;

      /* Distance from the origin at which the radial density is negligible */
      const NumericalScalar RootStrategyImplementation::DefaultMaximumDistance = ResourceMap::GetAsNumericalScalar( "RootStrategyImplementation-DefaultMaximumDistance" );
      /* Step-size for the detection of the sign changes along a direction */
      const NumericalScalar RootStrategyImplementation::DefaultStepSize = ResourceMap::GetAsNumericalScalar( "RootStrategyImplementation-DefaultStepSize" );

      /* Default constructor */
      RootStrategyImplementation::RootStrategyImplementation()
        : Base::Common::PersistentObject(),
          solver_(new Brent()),
          maximumDistance_(DefaultMaximumDistance),
          stepSize_(DefaultStepSize),
          isAlreadyComputedOriginValue_(false),
          originValue_(0.0)
      {
        // Nothing to do
      }

      /* Parameters constructor */
      RootStrategyImplementation::RootStrategyImplementation(const Solver & solver)
        : Base::Common::PersistentObject(),
          solver_(solver),
          maximumDistance_(DefaultMaximumDistance),
          stepSize_(DefaultStepSize),
          isAlreadyComputedOriginValue_(false),
          originValue_(0.0)
      {
        // Nothing to do
      }

      /* Parameters constructor */
      RootStrategyImplementation::RootStrategyImplementation(const Solver & solver,
                                                             const NumericalScalar maximumDistance,
                                                             const NumericalScalar stepSize)
        : Base::Common::PersistentObject(),
          solver_(solver),
          maximumDistance_(maximumDistance),
          stepSize_(stepSize),
          isAlreadyComputedOriginValue_(false),
          originValue_(0.0)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      RootStrategyImplementation * RootStrategyImplementation::clone() const
      {
        return new RootStrategyImplementation(*this);
      }

      /* Solve gives all the roots found applying the root strategy */
      RootStrategyImplementation::NumericalScalarCollection RootStrategyImplementation::solve(const NumericalMathFunction & function,
                                                                                              const NumericalScalar value)
      {
        throw NotYetImplementedException(HERE);
      }

      /* Solver accessor */
      void RootStrategyImplementation::setSolver(const Solver & solver)
      {
        solver_ = solver;
      }

      RootStrategyImplementation::Solver RootStrategyImplementation::getSolver() const
      {
        return solver_;
      }

      /* Maximum distance accessor */
      void RootStrategyImplementation::setMaximumDistance(const NumericalScalar maximumDistance)
      {
        maximumDistance_ = maximumDistance;
      }

      NumericalScalar RootStrategyImplementation::getMaximumDistance() const
      {
        return maximumDistance_;
      }

      /* Step size accessor */
      void RootStrategyImplementation::setStepSize(const NumericalScalar stepSize)
      {
        stepSize_ = stepSize;
      }

      NumericalScalar RootStrategyImplementation::getStepSize() const
      {
        return stepSize_;
      }

      /* Value of the performance function at the origin accessor */
      void RootStrategyImplementation::setOriginValue(const NumericalScalar originValue)
      {
        originValue_ = originValue;
        isAlreadyComputedOriginValue_ = true;
      }

      /* We have to store the value of the performance function at the origin for two reasons:
         + to save computations, as this value will be reused for each direction
         + to check if the origin is in the failure space or not, wich change the meaning of the roots
      */
      NumericalScalar RootStrategyImplementation::getOriginValue() const
      {
        if (!isAlreadyComputedOriginValue_) throw NotDefinedException(HERE);
        return originValue_;
      }

      /* String converter */
      String RootStrategyImplementation::__repr__() const
      {
        OSS oss;
        oss << "class=" << RootStrategyImplementation::GetClassName()
            << " solver=" << solver_
            << " maximumDistance=" << maximumDistance_
            << " stepSize=" << stepSize_;
        return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
