//                                               -*- C++ -*-
/**
 *  @file  ArcsineFactory.cxx
 *  @brief Factory for Arcsine distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2009-09-30 17:51:57 +0200 (mer. 30 sept. 2009) $
 *  Id:      $Id: ArcsineFactory.cxx 1357 2009-09-30 15:51:57Z schueller $
 */
#include "ArcsineFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      ArcsineFactory::ArcsineFactory()
        : DistributionImplementationFactory()
      {
        // Initialize any other class members here
        // At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      ArcsineFactory * ArcsineFactory::clone() const
      {
        return new ArcsineFactory(*this);
      }


      /* Here is the interface that all derived class must implement */

      Arcsine * ArcsineFactory::build(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Arcsine distribution from an empty sample";
        if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Arcsine distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
        NumericalScalar mean(sample.computeMean()[0]);
        NumericalScalar standardDeviation(sample.computeStandardDeviationPerComponent()[0]);
        return Arcsine(mean, standardDeviation, Arcsine::MUSIGMA).clone();
      }

      Arcsine * ArcsineFactory::build(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
        try {
          Arcsine distribution;
          distribution.setParametersCollection(parameters);
          return distribution.clone();
        }
        catch (InvalidArgumentException & ex)
          {
            throw InternalException(HERE) << "Error: cannot build a Arcsine distribution from the given parameters";
          }
      }

      Arcsine * ArcsineFactory::build() const /* throw(InvalidArgumentException, InternalException) */
      {
        return Arcsine().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
