//                                               -*- C++ -*-
/**
 *  @file  UserDefined.hxx
 *  @brief The UserDefined distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: UserDefined.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_USERDEFINED_HXX
#define OPENTURNS_USERDEFINED_HXX

#include "OTprivate.hxx"
#include "DiscreteDistribution.hxx"
#include "Exception.hxx"
#include "PersistentCollection.hxx"
#include "PersistentObject.hxx"
#include "UserDefinedPair.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class UserDefined
       *
       * The UserDefined distribution.
       */
      class UserDefined
        : public Model::DiscreteDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::DiscreteDistribution                          DiscreteDistribution; // required by SWIG
        typedef DiscreteDistribution::NumericalPoint                 NumericalPoint;
        typedef DiscreteDistribution::NumericalSample                NumericalSample;
        typedef DiscreteDistribution::CovarianceMatrix               CovarianceMatrix;
        typedef DiscreteDistribution::NumericalPointWithDescriptionCollection       NumericalPointWithDescriptionCollection;
        typedef DiscreteDistribution::NotDefinedException            NotDefinedException;

        typedef Base::Type::Collection<UserDefinedPair>              UserDefinedPairCollection;
        typedef Base::Type::PersistentCollection<UserDefinedPair>    UserDefinedPairPersistentCollection;
        typedef Base::Type::Collection<NumericalScalar>              NumericalScalarCollection;
        typedef Base::Type::PersistentCollection<NumericalScalar>    NumericalScalarPersistentCollection;

        /** Default constructor */
        UserDefined();

        /** Constructor from PairCollection */
        explicit UserDefined(const UserDefinedPairCollection & collection)
          /* throw (InvalidArgumentException) */;

        /** Constructor from a sample */
        explicit UserDefined(const NumericalSample & sample)
          /* throw (InvalidArgumentException) */;

        /** Constructor from a sample and associated weights */
        explicit UserDefined(const NumericalSample & sample,
                             const NumericalPoint & weights)
          /* throw (InvalidArgumentException) */;


        /** Comparison operator */
        Bool operator ==(const UserDefined & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual UserDefined * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the PDF of the distribution */
        using DiscreteDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using DiscreteDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point,
                                   const Bool tail = false) const;

        /** Get the PDFGradient of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** Get the support of a discrete distribution that intersect a given interval */
        NumericalSample getSupport(const Interval & interval) const;

        /** Tell if the distribution is integer valued */
        Bool isIntegral() const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;

        /* Interface specific to UserDefined */

        /** Pair collection accessor */
        void setPairCollection(const UserDefinedPairCollection & collection)
          /* throw (InvalidArgumentException) */;
        UserDefinedPairCollection getPairCollection() const;


        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

      private:

        /** Compute the mean of the distribution */
        void computeMean() const;

        /** Compute the covariance of the distribution */
        void computeCovariance() const;

        /** Quantile computation for dimension=1 */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** The collection of couple (xi,pi) of the UserDefined distribution */
        UserDefinedPairPersistentCollection collection_;

        /** The cumulative probabilities si = sum(pk, k=0..i) */
        NumericalScalarPersistentCollection cumulativeProbabilities_;

        /** Flag to accelerate computations in case of uniform weights */
        mutable Bool hasUniformWeights_;

      }; /* class UserDefined */

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_USERDEFINED_HXX */
