//                                               -*- C++ -*-
/**
 *  @file  WeibullFactory.cxx
 *  @brief Factory for Weibull distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: WeibullFactory.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include "WeibullFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Model::DistributionImplementationFactory::NumericalPoint NumericalPoint;

      /* Default constructor */
      WeibullFactory::WeibullFactory():
        DistributionImplementationFactory()
      {
        // Nothing to do
      }

      /* Virtual constructor */
      WeibullFactory * WeibullFactory::clone() const
      {
        return new WeibullFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      Weibull * WeibullFactory::build(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        const NumericalScalar size(sample.getSize());
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Weibull distribution from an empty sample";
        if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Weibull distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
        const NumericalScalar xMin(sample.getMin()[0]);
        const NumericalScalar gamma(xMin - fabs(xMin) / (2.0 + size));
        const NumericalScalar mean(sample.computeMean()[0]);
        const NumericalScalar sigma(sample.computeStandardDeviationPerComponent()[0]);
        try {
          return Weibull(mean, sigma, gamma, Weibull::MUSIGMA).clone();
        }
        catch (InvalidArgumentException & ex)
          {
            throw InternalException(HERE) << "Error: cannot estimate parameters of a Weibull distribution from the given sample";
          }
      }

      Weibull * WeibullFactory::build(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
        try {
          Weibull distribution;
          distribution.setParametersCollection(parameters);
          return distribution.clone();
        }
        catch (InvalidArgumentException & ex)
          {
            throw InternalException(HERE) << "Error: cannot build a Weibull distribution from the given parameters";
          }
      }

      Weibull * WeibullFactory::build() const /* throw(InvalidArgumentException, InternalException) */
      {
        return Weibull().clone();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
