//                                               -*- C++ -*-
/**
 *  @file  ZipfMandelbrot.hxx
 *  @brief The ZipfMandelbrot distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-07-12 15:45:44 +0200 (lun. 12 juil. 2010) $
 *  Id:      $Id: ZipfMandelbrot.hxx 1581 2010-07-12 13:45:44Z dutka $
 */
#ifndef OPENTURNS_ZIPFMANDELBROT_HXX
#define OPENTURNS_ZIPFMANDELBROT_HXX

#include "OTprivate.hxx"
#include "DiscreteDistribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class ZipfMandelbrot
       *
       * The ZipfMandelbrot distribution.
       */
      class ZipfMandelbrot
        : public Model::DiscreteDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::DiscreteDistribution                          DiscreteDistribution;  // required by SWIG
        typedef DiscreteDistribution::NumericalPoint                 NumericalPoint;
        typedef DiscreteDistribution::NumericalSample                NumericalSample;
        typedef DiscreteDistribution::CovarianceMatrix               CovarianceMatrix;
        typedef DiscreteDistribution::NumericalPointWithDescriptionCollection       NumericalPointWithDescriptionCollection;
        typedef DiscreteDistribution::NotDefinedException            NotDefinedException;
        typedef DiscreteDistribution::StorageManager                 StorageManager;
        typedef Base::Type::Collection<NumericalScalar>              NumericalScalarCollection;


        /** Default constructor */
        ZipfMandelbrot();

        /** Parameters constructor */
        ZipfMandelbrot(const UnsignedLong n,
                       const NumericalScalar q,
                       const NumericalScalar s);

        /** Comparison operator */
        Bool operator ==(const ZipfMandelbrot & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual ZipfMandelbrot * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the PDF of the distribution */
        using DiscreteDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using DiscreteDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /* Interface specific to ZipfMandelbrot */

        /** Q accessor */
        void setQ(const NumericalScalar q);
        NumericalScalar getQ() const;

        /** S accessor */
        void setS(const NumericalScalar s);
        NumericalScalar getS() const;

        /** N accessor */
        void setN(const UnsignedLong n);
        UnsignedLong getN() const;

        /** Get the support of a discrete distribution that intersect a given interval */
        NumericalSample getSupport(const Interval & interval) const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using DiscreteDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

        /* Compute generalized harmonic numbers */
        NumericalScalar getHarmonicNumbers(UnsignedLong const) const;
        void computeHarmonicNumbers() const;

        mutable Bool isAlreadyComputedHarmonicNumbers_;
        mutable NumericalScalarCollection harmonicNumbers_;


      private:

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** The maximum value of the ZipfMandelbrot distribution */
        UnsignedLong n_;

        /** The q probability parameter of the ZipfMandelbrot distribution */
        NumericalScalar q_;

        /** The s probability parameter of the ZipfMandelbrot distribution */
        NumericalScalar s_;

      }; /* class ZipfMandelbrot */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_ZIPFMANDELBROT_HXX */
