//                                               -*- C++ -*-
/**
 *  @file  DiscreteDistribution.cxx
 *  @brief Abstract top-level class for Discrete distributions
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: DiscreteDistribution.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include <cstdlib>

#include "DiscreteDistribution.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"
#include "Curve.hxx"
#include "Staircase.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      CLASSNAMEINIT(DiscreteDistribution);

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::ResourceMap                ResourceMap;
      typedef Base::Graph::Curve                       Curve;
      typedef Base::Graph::Staircase                   Staircase;

      const NumericalScalar DiscreteDistribution::SupportEpsilon = ResourceMap::GetAsNumericalScalar( "DiscreteDistribution-SupportEpsilon" );

      /* Default constructor */
      DiscreteDistribution::DiscreteDistribution(const String & name)
        : UsualDistribution(name)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      DiscreteDistribution * DiscreteDistribution::clone() const
      {
        return new DiscreteDistribution(*this);
      }

      /* Comparison operator */
      Bool DiscreteDistribution::operator ==(const DiscreteDistribution & other) const {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          // sameObject = ...
        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String DiscreteDistribution::__repr__() const {
        OSS oss;
        oss << "class=" << DiscreteDistribution::GetClassName();
        return oss;
      }

      /* Tell if the distribution is continuous */
      Bool DiscreteDistribution::isContinuous() const
      {
        return false;
      }

      /* Tell if the distribution is integer valued */
      Bool DiscreteDistribution::isIntegral() const
      {
        return true;
      }

      /* Draw the PDF of the distribution */
      DiscreteDistribution::Graph DiscreteDistribution::drawPDF(const NumericalScalar xMin,
                                                                const NumericalScalar xMax,
                                                                const UnsignedLong pointNumber) const
      {
        if (getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot draw the PDF of a multidimensional discrete distribution this way.";
        if (xMax <= xMin) throw InvalidArgumentException(HERE) << "Error: cannot draw a PDF with xMax >= xMin, here xmin=" << xMin << " and xmax=" << xMax;
        const String title(OSS() << getName() << " PDF");
        const NumericalSample support(getSupport(Interval(xMin, xMax)));
        // First the vertical bars
        const String xName(getDescription()[0]);
        Graph graphPDF(title, xName, "PDF", true, "topright");
        NumericalPoint point(2);
        point[0] = xMin;
        NumericalSample data(0, 2);
        data.add(point);
        for (UnsignedLong i = 0; i < support.getSize(); ++i)
          {
            point[0] = support[i][0];
            data.add(point);
            point[1] = computePDF(point[0]);
            data.add(point);
            point[1] = 0.0;
            data.add(point);
          }
        point[0] = xMax;
        point[1] = 0.0;
        data.add(point);
        graphPDF.add(Curve(data, "red", "solid", 2, title));
        return graphPDF;
      }

      /* Draw the CDF of the distribution */
      DiscreteDistribution::Graph DiscreteDistribution::drawCDF(const NumericalScalar xMin,
                                                                const NumericalScalar xMax,
                                                                const UnsignedLong pointNumber) const
      {
        // Value :    0    1/5  2/5  3/5    4/5    1
        // Data  : ------+-----+---+------+----+---------
        // Case 1: ------------------------------[----]--
        // Case 2: ------------------[---]---------------
        //         -[--]---------------------------------
        // Case 3: ----------[---]-----------------------
        //         ---[-----------------------------]----
        //         -------[-----------------]------------
        if (getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot draw the CDF of a multidimensional discrete distribution this way.";
        if (xMax <= xMin) throw InvalidArgumentException(HERE) << "Error: cannot draw a PDF with xMax >= xMin, here xmin=" << xMin << " and xmax=" << xMax;
        // Create the graph that will store the staircase representing the empirical CDF
        const String title(OSS() << getName() << " CDF");
        const NumericalSample support(getSupport(Interval(xMin, xMax)));
        const UnsignedLong size(support.getSize());
        const String xName(getDescription()[0]);
        Graph graphCDF(title, xName, "CDF", true, "topleft");
        NumericalSample data(size + 2, 2);
        data[0][0] = xMin;
        data[0][1] = computeCDF(xMin);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalScalar x(support[i][0]);
            data[i + 1][0] = x;
            data[i + 1][1] = computeCDF(x);
          }
        data[size + 1][0] = xMax;
        data[size + 1][1] = computeCDF(xMax);
        graphCDF.add(Staircase(data, "red", "solid", 2, "s", title));
        return graphCDF;
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
