//                                               -*- C++ -*-
/**
 *  @file  DistributionImplementationFactory.cxx
 *  @brief Top-level class for all distribution factories
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-28 18:31:43 +0200 (Thu, 28 Jul 2011) $
 *  Id:      $Id: DistributionImplementationFactory.cxx 2046 2011-07-28 16:31:43Z schueller $
 */
#include <cstdlib>
#include <iomanip>
#include <fstream>
#include "DistributionImplementationFactory.hxx"
#include "Description.hxx"
#include "Path.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"
#include "OTconfig.hxx"
#include "Log.hxx"
#include "Os.hxx"
#include "BootstrapExperiment.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      CLASSNAMEINIT(DistributionImplementationFactory);

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::Path                       Path;
      typedef Base::Common::Log                        Log;
      typedef Base::Common::ResourceMap                ResourceMap;
      typedef Base::Common::Os                         Os;
      typedef Base::Type::Description                  Description;
      typedef Algorithm::BootstrapExperiment           BootstrapExperiment;

      const UnsignedLong DistributionImplementationFactory::DefaultBootstrapSize = ResourceMap::GetAsUnsignedLong( "DistributionImplementationFactory-DefaultBootstrapSize" );

      /* Default constructor */
      DistributionImplementationFactory::DistributionImplementationFactory(const UnsignedLong bootstrapSize,
                                                                           const String & name)
        : PersistentObject(name),
          bootstrapSize_(bootstrapSize)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      DistributionImplementationFactory * DistributionImplementationFactory::clone() const
      {
        return new DistributionImplementationFactory(*this);
      }

      /* String converter */
      String DistributionImplementationFactory::__repr__() const
      {
        OSS oss;
        oss << "class=" << DistributionImplementationFactory::GetClassName();
        return oss;
      }

      /* String converter */
      String DistributionImplementationFactory::__str__(const String & offset) const
      {
        return this->getClassName();
      }


      /* Here is the interface that all derived class must implement */

      /* Build a distribution based on a sample */
      DistributionImplementation * DistributionImplementationFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        LOGWARN(OSS() << "The method buildImplementation(const NumericalSample & sample) is deprecated and should not be used anymore. Use build(const NumericalSample & sample) instead.");
        return build(sample);
      }
      DistributionImplementation * DistributionImplementationFactory::build(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        throw NotYetImplementedException(HERE);
      }

      /* Build a distribution based on a sample and gives the covariance matrix of the estimate */
      DistributionImplementation * DistributionImplementationFactory::buildImplementation(const NumericalSample & sample,
                                                                                          CovarianceMatrix & covariance) const /* throw(InvalidArgumentException, InternalException) */
      {
        LOGWARN(OSS() << "The method buildImplementation(const NumericalSample & sample, CovarianceMatrix & covariance) is deprecated and should not be used anymore. Use build(const NumericalSample & sample) instead.");
        return build(sample, covariance);
      }
      DistributionImplementation * DistributionImplementationFactory::build(const NumericalSample & sample,
                                                                            CovarianceMatrix & covariance) const /* throw(InvalidArgumentException, InternalException) */
      {
        /* The bootstrap sampler */
        BootstrapExperiment bootstrap(sample);
        /* Build the distribution based on the given sample */
        DistributionImplementation * distribution(build(sample));
        NumericalSample parametersSample(DefaultBootstrapSize, ParametersAsNumericalPoint(distribution->getParametersCollection()));
        for (UnsignedLong i = 1; i < DefaultBootstrapSize; ++i)
          {
            /* Draw a bootstrap sample */
            const NumericalSample bootstrapSample(bootstrap.generate());
            /* Build the associated distribution */
            const DistributionImplementation::Implementation newDistribution(build(bootstrapSample));
            /* Add the parameters to the parameters sample */
            parametersSample.add(ParametersAsNumericalPoint(newDistribution->getParametersCollection()));
          }
        /* Compute the bootstrap covariance */
        covariance = parametersSample.computeCovariance();
        return distribution;
      }


      /* Build a distribution based on a set of parameters */
      DistributionImplementation * DistributionImplementationFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
        LOGWARN(OSS() << "The method buildImplementation(const NumericalPointCollection & parameters) is deprecated and should not be used anymore. Use build(const NumericalPointCollection & parameters) instead.");
        return build(parameters);
      }
      DistributionImplementation * DistributionImplementationFactory::build(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
        throw NotYetImplementedException(HERE);
      }
      DistributionImplementation * DistributionImplementationFactory::build(const NumericalPointWithDescriptionCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	const UnsignedLong size(parameters.getSize());
	NumericalPointCollection collection(size);
	for (UnsignedLong i = 0; i < size; ++i) collection[i] = parameters[i];
        return build(collection);
      }

      /* Build a distribution using its default constructor */
      DistributionImplementation * DistributionImplementationFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
        LOGWARN(OSS() << "The method buildImplementation() is deprecated and should not be used anymore. Use build() instead.");
        return build();
      }
      DistributionImplementation * DistributionImplementationFactory::build() const /* throw(InvalidArgumentException, InternalException) */
      {
        throw NotYetImplementedException(HERE);
      }

      /* Execute a R script */
      DistributionImplementationFactory::NumericalPoint DistributionImplementationFactory::runRFactory(const NumericalSample & sample,
                                                                                                       const DistributionImplementation & distribution) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        const String distributionName(distribution.getClassName());
        if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a distribution of type " << distributionName << " using R methods";
        const String dataFileName(sample.storeToTemporaryFile());
        const String resultFileName(Path::BuildTemporaryFileName("RResult.txt.XXXXXX"));
        const String commandFileName(Path::BuildTemporaryFileName("RCmd.R.XXXXXX"));
        std::ofstream cmdFile(commandFileName.c_str(), std::ios::out);
        // Fill-in the command file
        cmdFile << "library(rotRPackage)" << std::endl;
        cmdFile << "options(digits=17)" << std::endl;
        cmdFile << "options(warn=-1)" << std::endl;
        cmdFile << "sample <- data.matrix(read.table(\"" << dataFileName << "\"))" << std::endl;
        cmdFile << "res <- estimate" << distributionName << "Parameters(sample)" << std::endl;
        cmdFile << "f <- file(\"" << resultFileName << "\",\"wt\")" << std::endl;
        cmdFile << "cat(";
        const Description parametersDescription(distribution.getParametersCollection()[0].getDescription());
        const UnsignedLong parametersNumber(parametersDescription.getSize());
        for (UnsignedLong i = 0; i < parametersNumber; ++i) cmdFile << "res$" << parametersDescription[i] << ", ";
        cmdFile << "sep=\"\\n\", file=f)" << std::endl;
        cmdFile << "close(f)" << std::endl;
        cmdFile.close();

        OSS systemCommand;
#ifdef R_EXECUTABLE
        systemCommand << ResourceMap::Get("R-executable-command") << " --no-save --silent < \"" << commandFileName << "\"" << Os::GetDeleteCommandOutput();
#else
        throw NotYetImplementedException(HERE) << "DistributionImplementationFactory::runRFactory() need R";
#endif
        int returnCode(system(String(systemCommand).c_str()));
        if (returnCode != 0) throw InternalException(HERE) << "Error: unable to execute the system command " << String(systemCommand) << " returned code is " << returnCode;
        // Parse result file
        std::ifstream resultFile(resultFileName.c_str(), std::ios::in);
        NumericalPoint parametersValue(parametersNumber);
        for (UnsignedLong i = 0; i < parametersNumber; ++i) resultFile >> parametersValue[i];
        if (remove(dataFileName.c_str()) == -1) LOGWARN(OSS() << "Warning: cannot delete file " << dataFileName);
        if (remove(resultFileName.c_str()) == -1) LOGWARN(OSS() << "Warning: cannot delete file " << resultFileName);
        if (remove(commandFileName.c_str()) == -1) LOGWARN(OSS() << "Warning: cannot delete file " << commandFileName);
        return parametersValue;
      }

      /* Convert a parameters collection into a NumericalPoint */
      DistributionImplementationFactory::NumericalPoint DistributionImplementationFactory::ParametersAsNumericalPoint(const NumericalPointWithDescriptionCollection & parameters)
      {
        const UnsignedLong size(parameters.getSize());
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot convert an empty collection of parameters.";
        // Early exit if dimension 1
        if (size == 1) return parameters[0];
        // Concatenate the parameters
        NumericalPoint allParameters(0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalPoint marginalParameters(parameters[i]);
            const UnsignedLong marginalSize(marginalParameters.getSize());
            for (UnsignedLong j = 0; j < marginalSize; ++j) allParameters.add(marginalParameters[j]);
          }
        return allParameters;
      }

      /* Bootstrap size accessor */
      UnsignedLong DistributionImplementationFactory::getBootstrapSize() const
      {
        return bootstrapSize_;
      }

      void DistributionImplementationFactory::setBootstrapSize(const UnsignedLong bootstrapSize)
      {
        if (bootstrapSize == 0) throw InvalidArgumentException(HERE) << "Error: the bootstrap size must be > 0.";
        bootstrapSize_ = bootstrapSize;
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
