//                                               -*- C++ -*-
/**
 *  @file  VisualTest.hxx
 *  @brief StatTest implements statistical tests
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: VisualTest.hxx 1910 2011-05-24 17:30:41Z schueller $
 */
#ifndef OPENTURNS_VISUALTEST_HXX
#define OPENTURNS_VISUALTEST_HXX

#include "OTprivate.hxx"
#include "Exception.hxx"
#include "NumericalSample.hxx"
#include "LinearModel.hxx"
#include "Distribution.hxx"
#include "Copula.hxx"
#include "DistributionImplementation.hxx"
#include "Normal.hxx"
#include "Graph.hxx"

namespace OpenTURNS
{
  namespace Uncertainty
  {
    namespace StatTest
    {

      /**
       * @class Test
       *
       */

      class VisualTest
      {
      public:

        typedef Base::Common::InvalidArgumentException  InvalidArgumentException;
        typedef Base::Common::InvalidDimensionException InvalidDimensionException;
        typedef Base::Graph::Graph                      Graph;
        typedef Base::Stat::NumericalSample             NumericalSample;
        typedef Base::Stat::LinearModel                 LinearModel;
        typedef Distribution::Normal                    Normal;
        typedef Model::Distribution                     Distribution;
        typedef Model::Copula                           Copula;
        typedef Model::DistributionImplementation       DistributionImplementation;

        /** Default constructor, needed by SWIG */
        VisualTest();

        /** Draw the empirical CDF of the Sample when its dimension is 1 */
        static Graph DrawEmpiricalCDF(const NumericalSample & sample,
                                      const NumericalScalar xMin,
                                      const NumericalScalar xMax)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;


        /** Draw the Histogram of the Sample when its dimension is 1 */
        static Graph DrawHistogram(const NumericalSample & sample,
                                   const UnsignedLong BarNumber)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the Histogram of the Sample when its dimension is 1, using the Normal empirical rule */
        static Graph DrawHistogram(const NumericalSample & sample)
          /* throw(InvalidDimensionException) */;

        /** Draw the QQplot of two Samples when its dimension is 1 */
        static Graph DrawQQplot(const NumericalSample & sample1,
                                const NumericalSample & sample2,
                                const UnsignedLong pointNumber = DistributionImplementation::DefaultPointNumber)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;


        /** Draw the QQplot of one Sample and one Distribution when its dimension is 1 */
        static Graph DrawQQplot(const NumericalSample & sample,
                                const Distribution & dist,
                                const UnsignedLong pointNumber = DistributionImplementation::DefaultPointNumber)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the Henry curve for one Sample when its dimension is 1 */
        static Graph DrawHenryLine(const NumericalSample & sample)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the clouds of one Sample and one model when its dimension is 2 */
        static Graph DrawClouds(const NumericalSample & sample1,
                                const Distribution & dist)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the clouds of two Samples when its dimension is 2 */
        static Graph DrawClouds(const NumericalSample & sample1,
                                const NumericalSample & sample2)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the visual test for the LinearModel when its dimension is 1
         * @deprecated
         * This method is deprecated. Use DrawLinearModel() instead.
         */
        static Graph DrawLMVisualTest(const NumericalSample & sample1,
                                      const NumericalSample & sample2,
                                      const LinearModel & linearModel) DEPRECATED
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the visual test for the LinearModel when its dimension is 1 */
        static Graph DrawLinearModel(const NumericalSample & sample1,
                                     const NumericalSample & sample2,
                                     const LinearModel & linearModel)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the visual test for the LinearModel residuals when its dimension is 1
         * @deprecated
         * This method is deprecated. Use DrawLinearModelResidual() instead.
         */
        static Graph DrawLMResidualTest(const NumericalSample & sample1,
                                        const NumericalSample & sample2,
                                        const LinearModel & linearModel) DEPRECATED
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the visual test for the LinearModel residuals when its dimension is 1 */
        static Graph DrawLinearModelResidual(const NumericalSample & sample1,
                                             const NumericalSample & sample2,
                                             const LinearModel & linearModel)
          /* throw(InvalidDimensionException, InvalidArgumentException) */;

        /** Draw the CobWeb visual test */
        static Graph DrawCobWeb(const NumericalSample & inputSample,
                                const NumericalSample & outputSample,
                                const NumericalScalar minValue,
                                const NumericalScalar maxValue,
                                const String & color,
                                const Bool quantileScale = true);

        /** Draw the Kendall plot to assess a copula for a bidimensional sample */
        static Graph DrawKendallPlot(const NumericalSample & data,
                                     const Copula & copula);

        /** Draw the Kendall plot to assess if two bidimensional samples share the same copula */
        static Graph DrawKendallPlot(const NumericalSample & firstSample,
                                     const NumericalSample & secondSample);

      private:

        /** Compute the Kendall plot empirical statistic associated with a bidimensional sample */
        static NumericalSample ComputeKendallPlotEmpiricalStatistics(const NumericalSample & sample);

        /** Compute the Kendall plot theoretical statistic associated with a bidimensional copula */
        static NumericalSample ComputeKendallPlotTheoreticalStatistics(const Copula & copula,
                                                                       const UnsignedLong size);


      }; /* class VisualTest */

    } /* namespace StatTest */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
#endif /* OPENTURNS_VISUALTEST_HXX */
