//                                               -*- C++ -*-
/**
 *  @file  t_MonteCarlo_std.cxx
 *  @brief The test file of MonteCarlo class
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: t_MonteCarlo_std.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include <iostream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Event.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "NumericalMathFunction.hxx"
#include "Simulation.hxx"
#include "MonteCarlo.hxx"
#include "IdentityMatrix.hxx"
#include "Normal.hxx"
#include "NumericalSample.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Func;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;
using namespace OT::Uncertainty::Algorithm;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    /* We create a numerical math function */
    NumericalMathFunction myFunction("poutre");

    UnsignedLong dim(myFunction.getInputDimension());
    /* We create a normal distribution point of dimension 1 */
    NumericalPoint mean(dim, 0.0);
    mean[0] = 50.0; // E
    mean[1] =  1.0; // F
    mean[2] = 10.0; // L
    mean[3] =  5.0; // I
    NumericalPoint sigma(dim, 1.0);
    IdentityMatrix R(dim);
    Normal myDistribution(mean, sigma, R);

    /* We create a 'usual' RandomVector from the Distribution */
    RandomVector vect(myDistribution);

    /* We create a composite random vector */
    RandomVector output(myFunction, vect);

    /* We create an Event from this RandomVector */
    Event myEvent(output, Less(), -3.0);

    /* We create a Monte Carlo algorithm */
    MonteCarlo myAlgo(myEvent);
    myAlgo.setMaximumOuterSampling(250);
    myAlgo.setBlockSize(4);
    myAlgo.setMaximumCoefficientOfVariation(0.1);

    fullprint << "MonteCarlo=" << myAlgo << std::endl;

    /* Perform the simulation */
    myAlgo.run();

    /* Stream out the result */
    fullprint << "MonteCarlo result=" << myAlgo.getResult() << std::endl;

    /* Use the standard deviation as a stoping rule */
    myAlgo = MonteCarlo(myEvent);
    myAlgo.setMaximumOuterSampling(250);
    myAlgo.setBlockSize(4);
    myAlgo.setMaximumCoefficientOfVariation(0.0);
    myAlgo.setMaximumStandardDeviation(0.1);

    fullprint << "MonteCarlo=" << myAlgo << std::endl;

    /* Perform the simulation */
    myAlgo.run();

    /* Stream out the result */
    fullprint << "MonteCarlo result=" << myAlgo.getResult() << std::endl;

    /* Analyse the input sample */
    NumericalSample inputSample(myAlgo.getInputStrategy().getSample());
    fullprint << "Input sample size=" << inputSample.getSize() << " dimension=" << inputSample.getDimension() << " first=" << inputSample[0] << " last=" << inputSample[inputSample.getSize()-1] << std::endl;
    NumericalSample outputSample(myAlgo.getOutputStrategy().getSample());
    fullprint << "Output sample size=" << outputSample.getSize() << " dimension=" << outputSample.getDimension() << " first=" << outputSample[0] << " last=" << outputSample[outputSample.getSize()-1] << std::endl;


    Study myStudy;
    myStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
    myStudy.add("myMc", myAlgo);
    myStudy.save();

    Study myStudy2;
    myStudy2.setStorageManager(XMLStorageManager("testStudy.ot"));
    myStudy2.load();
    MonteCarlo mc;
    myStudy2.fillObject("myMc", mc);
    fullprint << "loaded MonteCarlo=" << mc << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
