// Copyright 2010-2018 Google LLC
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <algorithm>
#include <iterator>
#include <map>
#include <memory>
#include <numeric>
#include <set>
#include <string>
#include <utility>
#include <vector>

#include "absl/container/flat_hash_map.h"
#include "absl/container/flat_hash_set.h"
#include "absl/memory/memory.h"
#include "absl/random/distributions.h"
#include "absl/random/random.h"
#include "absl/strings/str_cat.h"
#include "ortools/base/commandlineflags.h"
#include "ortools/base/hash.h"
#include "ortools/base/integral_types.h"
#include "ortools/base/iterator_adaptors.h"
#include "ortools/base/logging.h"
#include "ortools/base/macros.h"
#include "ortools/base/map_util.h"
#include "ortools/constraint_solver/constraint_solver.h"
#include "ortools/constraint_solver/constraint_solveri.h"
#include "ortools/graph/hamiltonian_path.h"
#include "ortools/util/saturated_arithmetic.h"

ABSL_FLAG(int, cp_local_search_sync_frequency, 16,
          "Frequency of checks for better solutions in the solution pool.");

ABSL_FLAG(int, cp_local_search_tsp_opt_size, 13,
          "Size of TSPs solved in the TSPOpt operator.");

ABSL_FLAG(int, cp_local_search_tsp_lns_size, 10,
          "Size of TSPs solved in the TSPLns operator.");

ABSL_FLAG(bool, cp_use_empty_path_symmetry_breaker, true,
          "If true, equivalent empty paths are removed from the neighborhood "
          "of PathOperators");

namespace operations_research {

// Utility methods to ensure the communication between local search and the
// search.

// Returns true if a local optimum has been reached and cannot be improved.
bool LocalOptimumReached(Search* const search);

// Returns true if the search accepts the delta (actually checking this by
// calling AcceptDelta on the monitors of the search).
bool AcceptDelta(Search* const search, Assignment* delta,
                 Assignment* deltadelta);

// Notifies the search that a neighbor has been accepted by local search.
void AcceptNeighbor(Search* const search);
void AcceptUncheckedNeighbor(Search* const search);

// ----- Base operator class for operators manipulating IntVars -----

bool IntVarLocalSearchOperator::MakeNextNeighbor(Assignment* delta,
                                                 Assignment* deltadelta) {
  CHECK(delta != nullptr);
  VLOG(2) << DebugString() << "::MakeNextNeighbor(delta=("
          << delta->DebugString() << "), deltadelta=("
          << (deltadelta ? deltadelta->DebugString() : std::string("nullptr"));
  while (true) {
    RevertChanges(true);

    if (!MakeOneNeighbor()) {
      return false;
    }

    if (ApplyChanges(delta, deltadelta)) {
      VLOG(2) << "Delta (" << DebugString() << ") = " << delta->DebugString();
      return true;
    }
  }
  return false;
}
// TODO(user): Make this a pure virtual.
bool IntVarLocalSearchOperator::MakeOneNeighbor() { return true; }

// ----- Base Large Neighborhood Search operator -----

BaseLns::BaseLns(const std::vector<IntVar*>& vars)
    : IntVarLocalSearchOperator(vars) {}

BaseLns::~BaseLns() {}

bool BaseLns::MakeOneNeighbor() {
  fragment_.clear();
  if (NextFragment()) {
    for (int candidate : fragment_) {
      Deactivate(candidate);
    }
    return true;
  }
  return false;
}

void BaseLns::OnStart() { InitFragments(); }

void BaseLns::InitFragments() {}

void BaseLns::AppendToFragment(int index) {
  if (index >= 0 && index < Size()) {
    fragment_.push_back(index);
  }
}

int BaseLns::FragmentSize() const { return fragment_.size(); }

// ----- Simple Large Neighborhood Search operator -----

// Frees number_of_variables (contiguous in vars) variables.

namespace {
class SimpleLns : public BaseLns {
 public:
  SimpleLns(const std::vector<IntVar*>& vars, int number_of_variables)
      : BaseLns(vars), index_(0), number_of_variables_(number_of_variables) {
    CHECK_GT(number_of_variables_, 0);
  }
  ~SimpleLns() override {}
  void InitFragments() override { index_ = 0; }
  bool NextFragment() override;
  std::string DebugString() const override { return "SimpleLns"; }

 private:
  int index_;
  const int number_of_variables_;
};

bool SimpleLns::NextFragment() {
  const int size = Size();
  if (index_ < size) {
    for (int i = index_; i < index_ + number_of_variables_; ++i) {
      AppendToFragment(i % size);
    }
    ++index_;
    return true;
  }
  return false;
}

// ----- Random Large Neighborhood Search operator -----

// Frees up to number_of_variables random variables.

class RandomLns : public BaseLns {
 public:
  RandomLns(const std::vector<IntVar*>& vars, int number_of_variables,
            int32 seed)
      : BaseLns(vars), rand_(seed), number_of_variables_(number_of_variables) {
    CHECK_GT(number_of_variables_, 0);
    CHECK_LE(number_of_variables_, Size());
  }
  ~RandomLns() override {}
  bool NextFragment() override;

  std::string DebugString() const override { return "RandomLns"; }

 private:
  std::mt19937 rand_;
  const int number_of_variables_;
};

bool RandomLns::NextFragment() {
  DCHECK_GT(Size(), 0);
  for (int i = 0; i < number_of_variables_; ++i) {
    AppendToFragment(absl::Uniform<int>(rand_, 0, Size()));
  }
  return true;
}
}  // namespace

LocalSearchOperator* Solver::MakeRandomLnsOperator(
    const std::vector<IntVar*>& vars, int number_of_variables) {
  return MakeRandomLnsOperator(vars, number_of_variables, CpRandomSeed());
}

LocalSearchOperator* Solver::MakeRandomLnsOperator(
    const std::vector<IntVar*>& vars, int number_of_variables, int32 seed) {
  return RevAlloc(new RandomLns(vars, number_of_variables, seed));
}

// ----- Move Toward Target Local Search operator -----

// A local search operator that compares the current assignment with a target
// one, and that generates neighbors corresponding to a single variable being
// changed from its current value to its target value.
namespace {
class MoveTowardTargetLS : public IntVarLocalSearchOperator {
 public:
  MoveTowardTargetLS(const std::vector<IntVar*>& variables,
                     const std::vector<int64>& target_values)
      : IntVarLocalSearchOperator(variables),
        target_(target_values),
        // Initialize variable_index_ at the number of the of variables minus
        // one, so that the first to be tried (after one increment) is the one
        // of index 0.
        variable_index_(Size() - 1) {
    CHECK_EQ(target_values.size(), variables.size()) << "Illegal arguments.";
  }

  ~MoveTowardTargetLS() override {}

  std::string DebugString() const override { return "MoveTowardTargetLS"; }

 protected:
  // Make a neighbor assigning one variable to its target value.
  bool MakeOneNeighbor() override {
    while (num_var_since_last_start_ < Size()) {
      ++num_var_since_last_start_;
      variable_index_ = (variable_index_ + 1) % Size();
      const int64 target_value = target_.at(variable_index_);
      const int64 current_value = OldValue(variable_index_);
      if (current_value != target_value) {
        SetValue(variable_index_, target_value);
        return true;
      }
    }
    return false;
  }

 private:
  void OnStart() override {
    // Do not change the value of variable_index_: this way, we keep going from
    // where we last modified something. This is because we expect that most
    // often, the variables we have just checked are less likely to be able
    // to be changed to their target values than the ones we have not yet
    // checked.
    //
    // Consider the case where oddly indexed variables can be assigned to their
    // target values (no matter in what order they are considered), while even
    // indexed ones cannot. Restarting at index 0 each time an odd-indexed
    // variable is modified will cause a total of Theta(n^2) neighbors to be
    // generated, while not restarting will produce only Theta(n) neighbors.
    CHECK_GE(variable_index_, 0);
    CHECK_LT(variable_index_, Size());
    num_var_since_last_start_ = 0;
  }

  // Target values
  const std::vector<int64> target_;

  // Index of the next variable to try to restore
  int64 variable_index_;

  // Number of variables checked since the last call to OnStart().
  int64 num_var_since_last_start_;
};
}  // namespace

LocalSearchOperator* Solver::MakeMoveTowardTargetOperator(
    const Assignment& target) {
  typedef std::vector<IntVarElement> Elements;
  const Elements& elements = target.IntVarContainer().elements();
  // Copy target values and construct the vector of variables
  std::vector<IntVar*> vars;
  std::vector<int64> values;
  vars.reserve(target.NumIntVars());
  values.reserve(target.NumIntVars());
  for (const auto& it : elements) {
    vars.push_back(it.Var());
    values.push_back(it.Value());
  }
  return MakeMoveTowardTargetOperator(vars, values);
}

LocalSearchOperator* Solver::MakeMoveTowardTargetOperator(
    const std::vector<IntVar*>& variables,
    const std::vector<int64>& target_values) {
  return RevAlloc(new MoveTowardTargetLS(variables, target_values));
}

// ----- ChangeValue Operators -----

ChangeValue::ChangeValue(const std::vector<IntVar*>& vars)
    : IntVarLocalSearchOperator(vars), index_(0) {}

ChangeValue::~ChangeValue() {}

bool ChangeValue::MakeOneNeighbor() {
  const int size = Size();
  while (index_ < size) {
    const int64 value = ModifyValue(index_, Value(index_));
    SetValue(index_, value);
    ++index_;
    return true;
  }
  return false;
}

void ChangeValue::OnStart() { index_ = 0; }

// Increments the current value of variables.

namespace {
class IncrementValue : public ChangeValue {
 public:
  explicit IncrementValue(const std::vector<IntVar*>& vars)
      : ChangeValue(vars) {}
  ~IncrementValue() override {}
  int64 ModifyValue(int64 index, int64 value) override { return value + 1; }

  std::string DebugString() const override { return "IncrementValue"; }
};

// Decrements the current value of variables.

class DecrementValue : public ChangeValue {
 public:
  explicit DecrementValue(const std::vector<IntVar*>& vars)
      : ChangeValue(vars) {}
  ~DecrementValue() override {}
  int64 ModifyValue(int64 index, int64 value) override { return value - 1; }

  std::string DebugString() const override { return "DecrementValue"; }
};
}  // namespace

// ----- Path-based Operators -----

PathOperator::PathOperator(const std::vector<IntVar*>& next_vars,
                           const std::vector<IntVar*>& path_vars,
                           int number_of_base_nodes,
                           bool skip_locally_optimal_paths,
                           bool accept_path_end_base,
                           std::function<int(int64)> start_empty_path_class)
    : IntVarLocalSearchOperator(next_vars, true),
      number_of_nexts_(next_vars.size()),
      ignore_path_vars_(path_vars.empty()),
      next_base_to_increment_(number_of_base_nodes),
      base_nodes_(number_of_base_nodes),
      base_alternatives_(number_of_base_nodes),
      base_sibling_alternatives_(number_of_base_nodes),
      end_nodes_(number_of_base_nodes),
      base_paths_(number_of_base_nodes),
      just_started_(false),
      first_start_(true),
      accept_path_end_base_(accept_path_end_base),
      start_empty_path_class_(std::move(start_empty_path_class)),
      skip_locally_optimal_paths_(skip_locally_optimal_paths),
      optimal_paths_enabled_(false),
      alternative_index_(next_vars.size(), -1) {
  DCHECK_GT(number_of_base_nodes, 0);
  if (!ignore_path_vars_) {
    AddVars(path_vars);
  }
  path_basis_.push_back(0);
  for (int i = 1; i < base_nodes_.size(); ++i) {
    if (!OnSamePathAsPreviousBase(i)) path_basis_.push_back(i);
  }
  if ((path_basis_.size() > 2) ||
      (!next_vars.empty() && !next_vars.back()
                                  ->solver()
                                  ->parameters()
                                  .skip_locally_optimal_paths())) {
    skip_locally_optimal_paths_ = false;
  }
}

void PathOperator::Reset() { optimal_paths_.clear(); }

void PathOperator::OnStart() {
  optimal_paths_enabled_ = false;
  InitializeBaseNodes();
  InitializeAlternatives();
  OnNodeInitialization();
}

bool PathOperator::MakeOneNeighbor() {
  while (IncrementPosition()) {
    // Need to revert changes here since MakeNeighbor might have returned false
    // and have done changes in the previous iteration.
    RevertChanges(true);
    if (MakeNeighbor()) {
      return true;
    }
  }
  return false;
}

bool PathOperator::SkipUnchanged(int index) const {
  if (ignore_path_vars_) {
    return true;
  }
  if (index < number_of_nexts_) {
    int path_index = index + number_of_nexts_;
    return Value(path_index) == OldValue(path_index);
  }
  int next_index = index - number_of_nexts_;
  return Value(next_index) == OldValue(next_index);
}

bool PathOperator::MoveChain(int64 before_chain, int64 chain_end,
                             int64 destination) {
  if (destination == before_chain || destination == chain_end) return false;
  DCHECK(CheckChainValidity(before_chain, chain_end, destination) &&
         !IsPathEnd(chain_end) && !IsPathEnd(destination));
  const int64 destination_path = Path(destination);
  const int64 after_chain = Next(chain_end);
  SetNext(chain_end, Next(destination), destination_path);
  if (!ignore_path_vars_) {
    int current = destination;
    int next = Next(before_chain);
    while (current != chain_end) {
      SetNext(current, next, destination_path);
      current = next;
      next = Next(next);
    }
  } else {
    SetNext(destination, Next(before_chain), destination_path);
  }
  SetNext(before_chain, after_chain, Path(before_chain));
  return true;
}

bool PathOperator::ReverseChain(int64 before_chain, int64 after_chain,
                                int64* chain_last) {
  if (CheckChainValidity(before_chain, after_chain, -1)) {
    int64 path = Path(before_chain);
    int64 current = Next(before_chain);
    if (current == after_chain) {
      return false;
    }
    int64 current_next = Next(current);
    SetNext(current, after_chain, path);
    while (current_next != after_chain) {
      const int64 next = Next(current_next);
      SetNext(current_next, current, path);
      current = current_next;
      current_next = next;
    }
    SetNext(before_chain, current, path);
    *chain_last = current;
    return true;
  }
  return false;
}

bool PathOperator::MakeActive(int64 node, int64 destination) {
  if (!IsPathEnd(destination)) {
    int64 destination_path = Path(destination);
    SetNext(node, Next(destination), destination_path);
    SetNext(destination, node, destination_path);
    return true;
  }
  return false;
}

bool PathOperator::MakeChainInactive(int64 before_chain, int64 chain_end) {
  const int64 kNoPath = -1;
  if (CheckChainValidity(before_chain, chain_end, -1) &&
      !IsPathEnd(chain_end)) {
    const int64 after_chain = Next(chain_end);
    int64 current = Next(before_chain);
    while (current != after_chain) {
      const int64 next = Next(current);
      SetNext(current, current, kNoPath);
      current = next;
    }
    SetNext(before_chain, after_chain, Path(before_chain));
    return true;
  }
  return false;
}

bool PathOperator::SwapActiveAndInactive(int64 active, int64 inactive) {
  if (active == inactive) return false;
  const int64 prev = Prev(active);
  return MakeChainInactive(prev, active) && MakeActive(inactive, prev);
}

bool PathOperator::IncrementPosition() {
  const int base_node_size = base_nodes_.size();

  if (!just_started_) {
    const int number_of_paths = path_starts_.size();
    // Finding next base node positions.
    // Increment the position of inner base nodes first (higher index nodes);
    // if a base node is at the end of a path, reposition it at the start
    // of the path and increment the position of the preceding base node (this
    // action is called a restart).
    int last_restarted = base_node_size;
    for (int i = base_node_size - 1; i >= 0; --i) {
      if (base_nodes_[i] < number_of_nexts_ && i <= next_base_to_increment_) {
        if (ConsiderAlternatives(i)) {
          // Iterate on sibling alternatives.
          const int sibling_alternative_index =
              GetSiblingAlternativeIndex(base_nodes_[i]);
          if (sibling_alternative_index >= 0) {
            if (base_sibling_alternatives_[i] <
                alternative_sets_[sibling_alternative_index].size() - 1) {
              ++base_sibling_alternatives_[i];
              break;
            }
            base_sibling_alternatives_[i] = 0;
          }
          // Iterate on base alternatives.
          const int alternative_index = alternative_index_[base_nodes_[i]];
          if (alternative_index >= 0) {
            if (base_alternatives_[i] <
                alternative_sets_[alternative_index].size() - 1) {
              ++base_alternatives_[i];
              break;
            }
            base_alternatives_[i] = 0;
            base_sibling_alternatives_[i] = 0;
          }
        }
        base_alternatives_[i] = 0;
        base_sibling_alternatives_[i] = 0;
        base_nodes_[i] = OldNext(base_nodes_[i]);
        if (accept_path_end_base_ || !IsPathEnd(base_nodes_[i])) break;
      }
      base_alternatives_[i] = 0;
      base_sibling_alternatives_[i] = 0;
      base_nodes_[i] = StartNode(i);
      last_restarted = i;
    }
    next_base_to_increment_ = base_node_size;
    // At the end of the loop, base nodes with indexes in
    // [last_restarted, base_node_size[ have been restarted.
    // Restarted base nodes are then repositioned by the virtual
    // GetBaseNodeRestartPosition to reflect position constraints between
    // base nodes (by default GetBaseNodeRestartPosition leaves the nodes
    // at the start of the path).
    // Base nodes are repositioned in ascending order to ensure that all
    // base nodes "below" the node being repositioned have their final
    // position.
    for (int i = last_restarted; i < base_node_size; ++i) {
      base_alternatives_[i] = 0;
      base_sibling_alternatives_[i] = 0;
      base_nodes_[i] = GetBaseNodeRestartPosition(i);
    }
    if (last_restarted > 0) {
      return CheckEnds();
    }
    // If all base nodes have been restarted, base nodes are moved to new paths.
    // First we mark the current paths as locally optimal if they have been
    // completely explored.
    if (optimal_paths_enabled_ && skip_locally_optimal_paths_) {
      if (path_basis_.size() > 1) {
        for (int i = 1; i < path_basis_.size(); ++i) {
          optimal_paths_[num_paths_ *
                             start_to_path_[StartNode(path_basis_[i - 1])] +
                         start_to_path_[StartNode(path_basis_[i])]] = true;
        }
      } else {
        optimal_paths_[num_paths_ * start_to_path_[StartNode(path_basis_[0])] +
                       start_to_path_[StartNode(path_basis_[0])]] = true;
      }
    }
    std::vector<int> current_starts(base_node_size);
    for (int i = 0; i < base_node_size; ++i) {
      current_starts[i] = StartNode(i);
    }
    // Exploration of next paths can lead to locally optimal paths since we are
    // exploring them from scratch.
    optimal_paths_enabled_ = true;
    while (true) {
      for (int i = base_node_size - 1; i >= 0; --i) {
        const int next_path_index = base_paths_[i] + 1;
        if (next_path_index < number_of_paths) {
          base_paths_[i] = next_path_index;
          base_alternatives_[i] = 0;
          base_sibling_alternatives_[i] = 0;
          base_nodes_[i] = path_starts_[next_path_index];
          if (i == 0 || !OnSamePathAsPreviousBase(i)) {
            break;
          }
        } else {
          base_paths_[i] = 0;
          base_alternatives_[i] = 0;
          base_sibling_alternatives_[i] = 0;
          base_nodes_[i] = path_starts_[0];
        }
      }
      if (!skip_locally_optimal_paths_) return CheckEnds();
      // If the new paths have already been completely explored, we can
      // skip them from now on.
      if (path_basis_.size() > 1) {
        for (int j = 1; j < path_basis_.size(); ++j) {
          if (!optimal_paths_[num_paths_ * start_to_path_[StartNode(
                                               path_basis_[j - 1])] +
                              start_to_path_[StartNode(path_basis_[j])]]) {
            return CheckEnds();
          }
        }
      } else {
        if (!optimal_paths_[num_paths_ *
                                start_to_path_[StartNode(path_basis_[0])] +
                            start_to_path_[StartNode(path_basis_[0])]]) {
          return CheckEnds();
        }
      }
      // If we are back to paths we just iterated on or have reached the end
      // of the neighborhood search space, we can stop.
      if (!CheckEnds()) return false;
      bool stop = true;
      for (int i = 0; i < base_node_size; ++i) {
        if (StartNode(i) != current_starts[i]) {
          stop = false;
          break;
        }
      }
      if (stop) return false;
    }
  } else {
    just_started_ = false;
    return true;
  }
  return CheckEnds();
}

void PathOperator::InitializePathStarts() {
  // Detect nodes which do not have any possible predecessor in a path; these
  // nodes are path starts.
  int max_next = -1;
  std::vector<bool> has_prevs(number_of_nexts_, false);
  for (int i = 0; i < number_of_nexts_; ++i) {
    const int next = OldNext(i);
    if (next < number_of_nexts_) {
      has_prevs[next] = true;
    }
    max_next = std::max(max_next, next);
  }
  // Update locally optimal paths.
  if (optimal_paths_.empty() && skip_locally_optimal_paths_) {
    num_paths_ = 0;
    start_to_path_.clear();
    start_to_path_.resize(number_of_nexts_, -1);
    for (int i = 0; i < number_of_nexts_; ++i) {
      if (!has_prevs[i]) {
        start_to_path_[i] = num_paths_;
        ++num_paths_;
      }
    }
    optimal_paths_.resize(num_paths_ * num_paths_, false);
  }
  if (skip_locally_optimal_paths_) {
    for (int i = 0; i < number_of_nexts_; ++i) {
      if (!has_prevs[i]) {
        int current = i;
        while (!IsPathEnd(current)) {
          if ((OldNext(current) != prev_values_[current])) {
            for (int j = 0; j < num_paths_; ++j) {
              optimal_paths_[num_paths_ * start_to_path_[i] + j] = false;
              optimal_paths_[num_paths_ * j + start_to_path_[i]] = false;
            }
            break;
          }
          current = OldNext(current);
        }
      }
    }
  }
  // Create a list of path starts, dropping equivalent path starts of
  // currently empty paths.
  std::vector<bool> empty_found(number_of_nexts_, false);
  std::vector<int64> new_path_starts;
  const bool use_empty_path_symmetry_breaker =
      absl::GetFlag(FLAGS_cp_use_empty_path_symmetry_breaker);
  for (int i = 0; i < number_of_nexts_; ++i) {
    if (!has_prevs[i]) {
      if (use_empty_path_symmetry_breaker && IsPathEnd(OldNext(i))) {
        if (start_empty_path_class_ != nullptr) {
          if (empty_found[start_empty_path_class_(i)]) continue;
          empty_found[start_empty_path_class_(i)] = true;
        }
      }
      new_path_starts.push_back(i);
    }
  }
  if (!first_start_) {
    // Synchronizing base_paths_ with base node positions. When the last move
    // was performed a base node could have been moved to a new route in which
    // case base_paths_ needs to be updated. This needs to be done on the path
    // starts before we re-adjust base nodes for new path starts.
    std::vector<int> node_paths(max_next + 1, -1);
    for (int i = 0; i < path_starts_.size(); ++i) {
      int node = path_starts_[i];
      while (!IsPathEnd(node)) {
        node_paths[node] = i;
        node = OldNext(node);
      }
      node_paths[node] = i;
    }
    for (int j = 0; j < base_nodes_.size(); ++j) {
      // Always restart from first alternative.
      base_alternatives_[j] = 0;
      base_sibling_alternatives_[j] = 0;
      if (IsInactive(base_nodes_[j]) || node_paths[base_nodes_[j]] == -1) {
        // Base node was made inactive or was moved to a new path, reposition
        // the base node to the start of the path on which it was.
        base_nodes_[j] = path_starts_[base_paths_[j]];
      } else {
        base_paths_[j] = node_paths[base_nodes_[j]];
      }
    }
    // Re-adjust current base_nodes and base_paths to take into account new
    // path starts (there could be fewer if a new path was made empty, or more
    // if nodes were added to a formerly empty path).
    int new_index = 0;
    absl::flat_hash_set<int> found_bases;
    for (int i = 0; i < path_starts_.size(); ++i) {
      int index = new_index;
      // Note: old and new path starts are sorted by construction.
      while (index < new_path_starts.size() &&
             new_path_starts[index] < path_starts_[i]) {
        ++index;
      }
      const bool found = (index < new_path_starts.size() &&
                          new_path_starts[index] == path_starts_[i]);
      if (found) {
        new_index = index;
      }
      for (int j = 0; j < base_nodes_.size(); ++j) {
        if (base_paths_[j] == i && !gtl::ContainsKey(found_bases, j)) {
          found_bases.insert(j);
          base_paths_[j] = new_index;
          // If the current position of the base node is a removed empty path,
          // readjusting it to the last visited path start.
          if (!found) {
            base_nodes_[j] = new_path_starts[new_index];
          }
        }
      }
    }
  }
  path_starts_.swap(new_path_starts);
}

void PathOperator::InitializeInactives() {
  inactives_.clear();
  for (int i = 0; i < number_of_nexts_; ++i) {
    inactives_.push_back(OldNext(i) == i);
  }
}

void PathOperator::InitializeBaseNodes() {
  // Inactive nodes must be detected before determining new path starts.
  InitializeInactives();
  InitializePathStarts();
  if (first_start_ || InitPosition()) {
    // Only do this once since the following starts will continue from the
    // preceding position
    for (int i = 0; i < base_nodes_.size(); ++i) {
      base_paths_[i] = 0;
      base_nodes_[i] = path_starts_[0];
    }
    first_start_ = false;
  }
  for (int i = 0; i < base_nodes_.size(); ++i) {
    // If base node has been made inactive, restart from path start.
    int64 base_node = base_nodes_[i];
    if (RestartAtPathStartOnSynchronize() || IsInactive(base_node)) {
      base_node = path_starts_[base_paths_[i]];
      base_nodes_[i] = base_node;
    }
    end_nodes_[i] = base_node;
  }
  // Repair end_nodes_ in case some must be on the same path and are not anymore
  // (due to other operators moving these nodes).
  for (int i = 1; i < base_nodes_.size(); ++i) {
    if (OnSamePathAsPreviousBase(i) &&
        !OnSamePath(base_nodes_[i - 1], base_nodes_[i])) {
      const int64 base_node = base_nodes_[i - 1];
      base_nodes_[i] = base_node;
      end_nodes_[i] = base_node;
      base_paths_[i] = base_paths_[i - 1];
    }
  }
  for (int i = 0; i < base_nodes_.size(); ++i) {
    base_alternatives_[i] = 0;
    base_sibling_alternatives_[i] = 0;
  }
  just_started_ = true;
}

void PathOperator::InitializeAlternatives() {
  active_in_alternative_set_.resize(alternative_sets_.size(), -1);
  for (int i = 0; i < alternative_sets_.size(); ++i) {
    const int64 current_active = active_in_alternative_set_[i];
    if (current_active >= 0 && !IsInactive(current_active)) continue;
    for (int64 index : alternative_sets_[i]) {
      if (!IsInactive(index)) {
        active_in_alternative_set_[i] = index;
        break;
      }
    }
  }
}

bool PathOperator::OnSamePath(int64 node1, int64 node2) const {
  if (IsInactive(node1) != IsInactive(node2)) {
    return false;
  }
  for (int node = node1; !IsPathEnd(node); node = OldNext(node)) {
    if (node == node2) {
      return true;
    }
  }
  for (int node = node2; !IsPathEnd(node); node = OldNext(node)) {
    if (node == node1) {
      return true;
    }
  }
  return false;
}

// Rejects chain if chain_end is not after before_chain on the path or if
// the chain contains exclude. Given before_chain is the node before the
// chain, if before_chain and chain_end are the same the chain is rejected too.
// Also rejects cycles (cycle detection is detected through chain length
//  overflow).
bool PathOperator::CheckChainValidity(int64 before_chain, int64 chain_end,
                                      int64 exclude) const {
  if (before_chain == chain_end || before_chain == exclude) return false;
  int64 current = before_chain;
  int chain_size = 0;
  while (current != chain_end) {
    if (chain_size > number_of_nexts_) {
      return false;
    }
    if (IsPathEnd(current)) {
      return false;
    }
    current = Next(current);
    ++chain_size;
    if (current == exclude) {
      return false;
    }
  }
  return true;
}

// ----- 2Opt -----

// Reverses a sub-chain of a path. It is called 2Opt because it breaks
// 2 arcs on the path; resulting paths are called 2-optimal.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 -> 5
// (where (1, 5) are first and last nodes of the path and can therefore not be
// moved):
// 1 -> 3 -> 2 -> 4 -> 5
// 1 -> 4 -> 3 -> 2 -> 5
// 1 -> 2 -> 4 -> 3 -> 5
class TwoOpt : public PathOperator {
 public:
  TwoOpt(const std::vector<IntVar*>& vars,
         const std::vector<IntVar*>& secondary_vars,
         std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, 2, true, true,
                     std::move(start_empty_path_class)),
        last_base_(-1),
        last_(-1) {}
  ~TwoOpt() override {}
  bool MakeNeighbor() override;
  bool IsIncremental() const override { return true; }

  std::string DebugString() const override { return "TwoOpt"; }

 protected:
  bool OnSamePathAsPreviousBase(int64 base_index) override {
    // Both base nodes have to be on the same path.
    return true;
  }
  int64 GetBaseNodeRestartPosition(int base_index) override {
    return (base_index == 0) ? StartNode(0) : BaseNode(0);
  }

 private:
  void OnNodeInitialization() override { last_ = -1; }

  int64 last_base_;
  int64 last_;
};

bool TwoOpt::MakeNeighbor() {
  DCHECK_EQ(StartNode(0), StartNode(1));
  if (last_base_ != BaseNode(0) || last_ == -1) {
    RevertChanges(false);
    if (IsPathEnd(BaseNode(0))) {
      last_ = -1;
      return false;
    }
    last_base_ = BaseNode(0);
    last_ = Next(BaseNode(0));
    int64 chain_last;
    if (ReverseChain(BaseNode(0), BaseNode(1), &chain_last)
        // Check there are more than one node in the chain (reversing a
        // single node is a NOP).
        && last_ != chain_last) {
      return true;
    }
    last_ = -1;
    return false;
  }
  const int64 to_move = Next(last_);
  DCHECK_EQ(Next(to_move), BaseNode(1));
  return MoveChain(last_, to_move, BaseNode(0));
}

// ----- Relocate -----

// Moves a sub-chain of a path to another position; the specified chain length
// is the fixed length of the chains being moved. When this length is 1 the
// operator simply moves a node to another position.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 -> 5, for a chain length
// of 2 (where (1, 5) are first and last nodes of the path and can
// therefore not be moved):
// 1 -> 4 -> 2 -> 3 -> 5
// 1 -> 3 -> 4 -> 2 -> 5
//
// Using Relocate with chain lengths of 1, 2 and 3 together is equivalent to
// the OrOpt operator on a path. The OrOpt operator is a limited version of
// 3Opt (breaks 3 arcs on a path).

class Relocate : public PathOperator {
 public:
  Relocate(const std::vector<IntVar*>& vars,
           const std::vector<IntVar*>& secondary_vars, const std::string& name,
           std::function<int(int64)> start_empty_path_class,
           int64 chain_length = 1LL, bool single_path = false)
      : PathOperator(vars, secondary_vars, 2, true, false,
                     std::move(start_empty_path_class)),
        chain_length_(chain_length),
        single_path_(single_path),
        name_(name) {
    CHECK_GT(chain_length_, 0);
  }
  Relocate(const std::vector<IntVar*>& vars,
           const std::vector<IntVar*>& secondary_vars,
           std::function<int(int64)> start_empty_path_class,
           int64 chain_length = 1LL, bool single_path = false)
      : Relocate(vars, secondary_vars,
                 absl::StrCat("Relocate<", chain_length, ">"),
                 std::move(start_empty_path_class), chain_length, single_path) {
  }
  ~Relocate() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return name_; }

 protected:
  bool OnSamePathAsPreviousBase(int64 base_index) override {
    // Both base nodes have to be on the same path when it's the single path
    // version.
    return single_path_;
  }

 private:
  const int64 chain_length_;
  const bool single_path_;
  const std::string name_;
};

bool Relocate::MakeNeighbor() {
  DCHECK(!single_path_ || StartNode(0) == StartNode(1));
  const int64 destination = BaseNode(1);
  DCHECK(!IsPathEnd(destination));
  const int64 before_chain = BaseNode(0);
  int64 chain_end = before_chain;
  for (int i = 0; i < chain_length_; ++i) {
    if (IsPathEnd(chain_end) || chain_end == destination) {
      return false;
    }
    chain_end = Next(chain_end);
  }
  return !IsPathEnd(chain_end) &&
         MoveChain(before_chain, chain_end, destination);
}

// ----- Exchange -----

// Exchanges the positions of two nodes.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 -> 5
// (where (1, 5) are first and last nodes of the path and can therefore not
// be moved):
// 1 -> 3 -> 2 -> 4 -> 5
// 1 -> 4 -> 3 -> 2 -> 5
// 1 -> 2 -> 4 -> 3 -> 5

class Exchange : public PathOperator {
 public:
  Exchange(const std::vector<IntVar*>& vars,
           const std::vector<IntVar*>& secondary_vars,
           std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, 2, true, false,
                     std::move(start_empty_path_class)) {}
  ~Exchange() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "Exchange"; }
};

bool Exchange::MakeNeighbor() {
  const int64 prev_node0 = BaseNode(0);
  const int64 node0 = Next(prev_node0);
  if (IsPathEnd(node0)) return false;
  const int64 prev_node1 = BaseNode(1);
  const int64 node1 = Next(prev_node1);
  if (IsPathEnd(node1)) return false;
  const bool ok = MoveChain(prev_node0, node0, prev_node1);
  return MoveChain(Prev(node1), node1, prev_node0) || ok;
}

// ----- Cross -----

// Cross echanges the starting chains of 2 paths, including exchanging the
// whole paths.
// First and last nodes are not moved.
// Possible neighbors for the paths 1 -> 2 -> 3 -> 4 -> 5 and 6 -> 7 -> 8
// (where (1, 5) and (6, 8) are first and last nodes of the paths and can
// therefore not be moved):
// 1 -> 7 -> 3 -> 4 -> 5  6 -> 2 -> 8
// 1 -> 7 -> 4 -> 5       6 -> 2 -> 3 -> 8
// 1 -> 7 -> 5            6 -> 2 -> 3 -> 4 -> 8

class Cross : public PathOperator {
 public:
  Cross(const std::vector<IntVar*>& vars,
        const std::vector<IntVar*>& secondary_vars,
        std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, 2, true, true,
                     std::move(start_empty_path_class)) {}
  ~Cross() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "Cross"; }
};

bool Cross::MakeNeighbor() {
  const int64 start0 = StartNode(0);
  const int64 start1 = StartNode(1);
  if (start1 == start0) return false;
  const int64 node0 = BaseNode(0);
  if (node0 == start0) return false;
  const int64 node1 = BaseNode(1);
  if (node1 == start1) return false;
  if (!IsPathEnd(node0) && !IsPathEnd(node1)) {
    // If two paths are equivalent don't exchange them.
    if (PathClass(0) == PathClass(1) && IsPathEnd(Next(node0)) &&
        IsPathEnd(Next(node1))) {
      return false;
    }
    return MoveChain(start0, node0, start1) && MoveChain(node0, node1, start0);
  }
  if (!IsPathEnd(node0)) return MoveChain(start0, node0, start1);
  if (!IsPathEnd(node1)) return MoveChain(start1, node1, start0);
  return false;
}

// ----- BaseInactiveNodeToPathOperator -----
// Base class of path operators which make inactive nodes active.

class BaseInactiveNodeToPathOperator : public PathOperator {
 public:
  BaseInactiveNodeToPathOperator(
      const std::vector<IntVar*>& vars,
      const std::vector<IntVar*>& secondary_vars, int number_of_base_nodes,
      std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, number_of_base_nodes, false, false,
                     std::move(start_empty_path_class)),
        inactive_node_(0) {
    // TODO(user): Activate skipping optimal paths.
  }
  ~BaseInactiveNodeToPathOperator() override {}

 protected:
  bool MakeOneNeighbor() override;
  int64 GetInactiveNode() const { return inactive_node_; }

 private:
  void OnNodeInitialization() override;

  int inactive_node_;
};

void BaseInactiveNodeToPathOperator::OnNodeInitialization() {
  for (int i = 0; i < Size(); ++i) {
    if (IsInactive(i)) {
      inactive_node_ = i;
      return;
    }
  }
  inactive_node_ = Size();
}

bool BaseInactiveNodeToPathOperator::MakeOneNeighbor() {
  while (inactive_node_ < Size()) {
    if (!IsInactive(inactive_node_) || !PathOperator::MakeOneNeighbor()) {
      ResetPosition();
      ++inactive_node_;
    } else {
      return true;
    }
  }
  return false;
}

// ----- MakeActiveOperator -----

// MakeActiveOperator inserts an inactive node into a path.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 with 5 inactive (where 1 and
// 4 are first and last nodes of the path) are:
// 1 -> 5 -> 2 -> 3 -> 4
// 1 -> 2 -> 5 -> 3 -> 4
// 1 -> 2 -> 3 -> 5 -> 4

class MakeActiveOperator : public BaseInactiveNodeToPathOperator {
 public:
  MakeActiveOperator(const std::vector<IntVar*>& vars,
                     const std::vector<IntVar*>& secondary_vars,
                     std::function<int(int64)> start_empty_path_class)
      : BaseInactiveNodeToPathOperator(vars, secondary_vars, 1,
                                       std::move(start_empty_path_class)) {}
  ~MakeActiveOperator() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "MakeActiveOperator"; }
};

bool MakeActiveOperator::MakeNeighbor() {
  return MakeActive(GetInactiveNode(), BaseNode(0));
}

// ---- RelocateAndMakeActiveOperator -----

// RelocateAndMakeActiveOperator relocates a node and replaces it by an inactive
// node.
// The idea is to make room for inactive nodes.
// Possible neighbor for paths 0 -> 4, 1 -> 2 -> 5 and 3 inactive is:
// 0 -> 2 -> 4, 1 -> 3 -> 5.
// TODO(user): Naming is close to MakeActiveAndRelocate but this one is
// correct; rename MakeActiveAndRelocate if it is actually used.
class RelocateAndMakeActiveOperator : public BaseInactiveNodeToPathOperator {
 public:
  RelocateAndMakeActiveOperator(
      const std::vector<IntVar*>& vars,
      const std::vector<IntVar*>& secondary_vars,
      std::function<int(int64)> start_empty_path_class)
      : BaseInactiveNodeToPathOperator(vars, secondary_vars, 2,
                                       std::move(start_empty_path_class)) {}
  ~RelocateAndMakeActiveOperator() override {}
  bool MakeNeighbor() override {
    const int64 before_node_to_move = BaseNode(1);
    const int64 node = Next(before_node_to_move);
    return !IsPathEnd(node) &&
           MoveChain(before_node_to_move, node, BaseNode(0)) &&
           MakeActive(GetInactiveNode(), before_node_to_move);
  }

  std::string DebugString() const override {
    return "RelocateAndMakeActiveOpertor";
  }
};

// ----- MakeActiveAndRelocate -----

// MakeActiveAndRelocate makes a node active next to a node being relocated.
// Possible neighbor for paths 0 -> 4, 1 -> 2 -> 5 and 3 inactive is:
// 0 -> 3 -> 2 -> 4, 1 -> 5.

class MakeActiveAndRelocate : public BaseInactiveNodeToPathOperator {
 public:
  MakeActiveAndRelocate(const std::vector<IntVar*>& vars,
                        const std::vector<IntVar*>& secondary_vars,
                        std::function<int(int64)> start_empty_path_class)
      : BaseInactiveNodeToPathOperator(vars, secondary_vars, 2,
                                       std::move(start_empty_path_class)) {}
  ~MakeActiveAndRelocate() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override {
    return "MakeActiveAndRelocateOperator";
  }
};

bool MakeActiveAndRelocate::MakeNeighbor() {
  const int64 before_chain = BaseNode(1);
  const int64 chain_end = Next(before_chain);
  const int64 destination = BaseNode(0);
  return !IsPathEnd(chain_end) &&
         MoveChain(before_chain, chain_end, destination) &&
         MakeActive(GetInactiveNode(), destination);
}

// ----- MakeInactiveOperator -----

// MakeInactiveOperator makes path nodes inactive.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 (where 1 and 4 are first
// and last nodes of the path) are:
// 1 -> 3 -> 4 & 2 inactive
// 1 -> 2 -> 4 & 3 inactive

class MakeInactiveOperator : public PathOperator {
 public:
  MakeInactiveOperator(const std::vector<IntVar*>& vars,
                       const std::vector<IntVar*>& secondary_vars,
                       std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, 1, true, false,
                     std::move(start_empty_path_class)) {}
  ~MakeInactiveOperator() override {}
  bool MakeNeighbor() override {
    const int64 base = BaseNode(0);
    return MakeChainInactive(base, Next(base));
  }

  std::string DebugString() const override { return "MakeInactiveOperator"; }
};

// ----- RelocateAndMakeInactiveOperator -----

// RelocateAndMakeInactiveOperator relocates a node to a new position and makes
// the node which was at that position inactive.
// Possible neighbors for paths 0 -> 2 -> 4, 1 -> 3 -> 5 are:
// 0 -> 3 -> 4, 1 -> 5 & 2 inactive
// 0 -> 4, 1 -> 2 -> 5 & 3 inactive

class RelocateAndMakeInactiveOperator : public PathOperator {
 public:
  RelocateAndMakeInactiveOperator(
      const std::vector<IntVar*>& vars,
      const std::vector<IntVar*>& secondary_vars,
      std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, 2, true, false,
                     std::move(start_empty_path_class)) {}
  ~RelocateAndMakeInactiveOperator() override {}
  bool MakeNeighbor() override {
    const int64 destination = BaseNode(1);
    const int64 before_to_move = BaseNode(0);
    const int64 node_to_inactivate = Next(destination);
    if (node_to_inactivate == before_to_move || IsPathEnd(node_to_inactivate) ||
        !MakeChainInactive(destination, node_to_inactivate)) {
      return false;
    }
    const int64 node = Next(before_to_move);
    return !IsPathEnd(node) && MoveChain(before_to_move, node, destination);
  }

  std::string DebugString() const override {
    return "RelocateAndMakeInactiveOperator";
  }
};

// ----- MakeChainInactiveOperator -----

// Operator which makes a "chain" of path nodes inactive.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 (where 1 and 4 are first
// and last nodes of the path) are:
//   1 -> 3 -> 4 with 2 inactive
//   1 -> 2 -> 4 with 3 inactive
//   1 -> 4 with 2 and 3 inactive

class MakeChainInactiveOperator : public PathOperator {
 public:
  MakeChainInactiveOperator(const std::vector<IntVar*>& vars,
                            const std::vector<IntVar*>& secondary_vars,
                            std::function<int(int64)> start_empty_path_class)
      : PathOperator(vars, secondary_vars, 2, true, false,
                     std::move(start_empty_path_class)) {}
  ~MakeChainInactiveOperator() override {}
  bool MakeNeighbor() override {
    return MakeChainInactive(BaseNode(0), BaseNode(1));
  }

  std::string DebugString() const override {
    return "MakeChainInactiveOperator";
  }

 protected:
  bool OnSamePathAsPreviousBase(int64 base_index) override {
    // Start and end of chain (defined by both base nodes) must be on the same
    // path.
    return true;
  }

  int64 GetBaseNodeRestartPosition(int base_index) override {
    // Base node 1 must be after base node 0.
    return (base_index == 0) ? StartNode(base_index) : BaseNode(base_index - 1);
  }
};

// ----- SwapActiveOperator -----

// SwapActiveOperator replaces an active node by an inactive one.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 with 5 inactive (where 1 and
// 4 are first and last nodes of the path) are:
// 1 -> 5 -> 3 -> 4 & 2 inactive
// 1 -> 2 -> 5 -> 4 & 3 inactive

class SwapActiveOperator : public BaseInactiveNodeToPathOperator {
 public:
  SwapActiveOperator(const std::vector<IntVar*>& vars,
                     const std::vector<IntVar*>& secondary_vars,
                     std::function<int(int64)> start_empty_path_class)
      : BaseInactiveNodeToPathOperator(vars, secondary_vars, 1,
                                       std::move(start_empty_path_class)) {}
  ~SwapActiveOperator() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "SwapActiveOperator"; }
};

bool SwapActiveOperator::MakeNeighbor() {
  const int64 base = BaseNode(0);
  return MakeChainInactive(base, Next(base)) &&
         MakeActive(GetInactiveNode(), base);
}

// ----- ExtendedSwapActiveOperator -----

// ExtendedSwapActiveOperator makes an inactive node active and an active one
// inactive. It is similar to SwapActiveOperator excepts that it tries to
// insert the inactive node in all possible positions instead of just the
// position of the node made inactive.
// Possible neighbors for the path 1 -> 2 -> 3 -> 4 with 5 inactive (where 1 and
// 4 are first and last nodes of the path) are:
// 1 -> 5 -> 3 -> 4 & 2 inactive
// 1 -> 3 -> 5 -> 4 & 2 inactive
// 1 -> 5 -> 2 -> 4 & 3 inactive
// 1 -> 2 -> 5 -> 4 & 3 inactive

class ExtendedSwapActiveOperator : public BaseInactiveNodeToPathOperator {
 public:
  ExtendedSwapActiveOperator(const std::vector<IntVar*>& vars,
                             const std::vector<IntVar*>& secondary_vars,
                             std::function<int(int64)> start_empty_path_class)
      : BaseInactiveNodeToPathOperator(vars, secondary_vars, 2,
                                       std::move(start_empty_path_class)) {}
  ~ExtendedSwapActiveOperator() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override {
    return "ExtendedSwapActiveOperator";
  }
};

bool ExtendedSwapActiveOperator::MakeNeighbor() {
  const int64 base0 = BaseNode(0);
  const int64 base1 = BaseNode(1);
  if (Next(base0) == base1) {
    return false;
  }
  return MakeChainInactive(base0, Next(base0)) &&
         MakeActive(GetInactiveNode(), base1);
}

// ----- TSP-based operators -----

// Sliding TSP operator
// Uses an exact dynamic programming algorithm to solve the TSP corresponding
// to path sub-chains.
// For a subchain 1 -> 2 -> 3 -> 4 -> 5 -> 6, solves the TSP on nodes A, 2, 3,
// 4, 5, where A is a merger of nodes 1 and 6 such that cost(A,i) = cost(1,i)
// and cost(i,A) = cost(i,6).

class TSPOpt : public PathOperator {
 public:
  TSPOpt(const std::vector<IntVar*>& vars,
         const std::vector<IntVar*>& secondary_vars,
         Solver::IndexEvaluator3 evaluator, int chain_length);
  ~TSPOpt() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "TSPOpt"; }

 private:
  std::vector<std::vector<int64>> cost_;
  HamiltonianPathSolver<int64, std::vector<std::vector<int64>>>
      hamiltonian_path_solver_;
  Solver::IndexEvaluator3 evaluator_;
  const int chain_length_;
};

TSPOpt::TSPOpt(const std::vector<IntVar*>& vars,
               const std::vector<IntVar*>& secondary_vars,
               Solver::IndexEvaluator3 evaluator, int chain_length)
    : PathOperator(vars, secondary_vars, 1, true, false, nullptr),
      hamiltonian_path_solver_(cost_),
      evaluator_(std::move(evaluator)),
      chain_length_(chain_length) {}

bool TSPOpt::MakeNeighbor() {
  std::vector<int64> nodes;
  int64 chain_end = BaseNode(0);
  for (int i = 0; i < chain_length_ + 1; ++i) {
    nodes.push_back(chain_end);
    if (IsPathEnd(chain_end)) {
      break;
    }
    chain_end = Next(chain_end);
  }
  if (nodes.size() <= 3) {
    return false;
  }
  int64 chain_path = Path(BaseNode(0));
  const int size = nodes.size() - 1;
  cost_.resize(size);
  for (int i = 0; i < size; ++i) {
    cost_[i].resize(size);
    cost_[i][0] = evaluator_(nodes[i], nodes[size], chain_path);
    for (int j = 1; j < size; ++j) {
      cost_[i][j] = evaluator_(nodes[i], nodes[j], chain_path);
    }
  }
  hamiltonian_path_solver_.ChangeCostMatrix(cost_);
  std::vector<PathNodeIndex> path;
  hamiltonian_path_solver_.TravelingSalesmanPath(&path);
  CHECK_EQ(size + 1, path.size());
  for (int i = 0; i < size - 1; ++i) {
    SetNext(nodes[path[i]], nodes[path[i + 1]], chain_path);
  }
  SetNext(nodes[path[size - 1]], nodes[size], chain_path);
  return true;
}

// TSP-base lns
// Randomly merge consecutive nodes until n "meta"-nodes remain and solve the
// corresponding TSP. This can be seen as a large neighborhood search operator
// although decisions are taken with the operator.
// This is an "unlimited" neighborhood which must be stopped by search limits.
// To force diversification, the operator iteratively forces each node to serve
// as base of a meta-node.

class TSPLns : public PathOperator {
 public:
  TSPLns(const std::vector<IntVar*>& vars,
         const std::vector<IntVar*>& secondary_vars,
         Solver::IndexEvaluator3 evaluator, int tsp_size);
  ~TSPLns() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "TSPLns"; }

 protected:
  bool MakeOneNeighbor() override;

 private:
  void OnNodeInitialization() override {
    // NOTE: Avoid any computations if there are no vars added.
    has_long_enough_paths_ = Size() != 0;
  }

  std::vector<std::vector<int64>> cost_;
  HamiltonianPathSolver<int64, std::vector<std::vector<int64>>>
      hamiltonian_path_solver_;
  Solver::IndexEvaluator3 evaluator_;
  const int tsp_size_;
  std::mt19937 rand_;
  bool has_long_enough_paths_;
};

TSPLns::TSPLns(const std::vector<IntVar*>& vars,
               const std::vector<IntVar*>& secondary_vars,
               Solver::IndexEvaluator3 evaluator, int tsp_size)
    : PathOperator(vars, secondary_vars, 1, true, false, nullptr),
      hamiltonian_path_solver_(cost_),
      evaluator_(std::move(evaluator)),
      tsp_size_(tsp_size),
      rand_(CpRandomSeed()),
      has_long_enough_paths_(true) {
  CHECK_GE(tsp_size_, 0);
  cost_.resize(tsp_size_);
  for (int i = 0; i < tsp_size_; ++i) {
    cost_[i].resize(tsp_size_);
  }
}

bool TSPLns::MakeOneNeighbor() {
  while (has_long_enough_paths_) {
    has_long_enough_paths_ = false;
    if (PathOperator::MakeOneNeighbor()) {
      return true;
    }
    Var(0)->solver()->TopPeriodicCheck();
  }
  return false;
}

bool TSPLns::MakeNeighbor() {
  const int64 base_node = BaseNode(0);
  std::vector<int64> nodes;
  for (int64 node = StartNode(0); !IsPathEnd(node); node = Next(node)) {
    nodes.push_back(node);
  }
  if (nodes.size() <= tsp_size_) {
    return false;
  }
  has_long_enough_paths_ = true;
  // Randomly select break nodes (final nodes of a meta-node, after which
  // an arc is relaxed.
  absl::flat_hash_set<int64> breaks_set;
  // Always add base node to break nodes (diversification)
  breaks_set.insert(base_node);
  CHECK(!nodes.empty());  // Should have been caught earlier.
  while (breaks_set.size() < tsp_size_) {
    breaks_set.insert(nodes[absl::Uniform<int>(rand_, 0, nodes.size())]);
  }
  CHECK_EQ(breaks_set.size(), tsp_size_);
  // Setup break node indexing and internal meta-node cost (cost of partial
  // route starting at first node of the meta-node and ending at its last node);
  // this cost has to be added to the TSP matrix cost in order to respect the
  // triangle inequality.
  std::vector<int> breaks;
  std::vector<int64> meta_node_costs;
  int64 cost = 0;
  int64 node = StartNode(0);
  int64 node_path = Path(node);
  while (!IsPathEnd(node)) {
    int64 next = Next(node);
    if (gtl::ContainsKey(breaks_set, node)) {
      breaks.push_back(node);
      meta_node_costs.push_back(cost);
      cost = 0;
    } else {
      cost = CapAdd(cost, evaluator_(node, next, node_path));
    }
    node = next;
  }
  meta_node_costs[0] += cost;
  CHECK_EQ(breaks.size(), tsp_size_);
  // Setup TSP cost matrix
  CHECK_EQ(meta_node_costs.size(), tsp_size_);
  for (int i = 0; i < tsp_size_; ++i) {
    cost_[i][0] =
        CapAdd(meta_node_costs[i],
               evaluator_(breaks[i], Next(breaks[tsp_size_ - 1]), node_path));
    for (int j = 1; j < tsp_size_; ++j) {
      cost_[i][j] =
          CapAdd(meta_node_costs[i],
                 evaluator_(breaks[i], Next(breaks[j - 1]), node_path));
    }
    cost_[i][i] = 0;
  }
  // Solve TSP and inject solution in delta (only if it leads to a new solution)
  hamiltonian_path_solver_.ChangeCostMatrix(cost_);
  std::vector<PathNodeIndex> path;
  hamiltonian_path_solver_.TravelingSalesmanPath(&path);
  bool nochange = true;
  for (int i = 0; i < path.size() - 1; ++i) {
    if (path[i] != i) {
      nochange = false;
      break;
    }
  }
  if (nochange) {
    return false;
  }
  CHECK_EQ(0, path[path.size() - 1]);
  for (int i = 0; i < tsp_size_ - 1; ++i) {
    SetNext(breaks[path[i]], OldNext(breaks[path[i + 1] - 1]), node_path);
  }
  SetNext(breaks[path[tsp_size_ - 1]], OldNext(breaks[tsp_size_ - 1]),
          node_path);
  return true;
}

// ----- Lin Kernighan -----

// For each variable in vars, stores the 'size' pairs(i,j) with the smallest
// value according to evaluator, where i is the index of the variable in vars
// and j is in the domain of the variable.
// Note that the resulting pairs are sorted.
// Works in O(size) per variable on average (same approach as qsort)

class NearestNeighbors {
 public:
  NearestNeighbors(Solver::IndexEvaluator3 evaluator,
                   const PathOperator& path_operator, int size);
  virtual ~NearestNeighbors() {}
  void Initialize();
  const std::vector<int>& Neighbors(int index) const;

  virtual std::string DebugString() const { return "NearestNeighbors"; }

 private:
  void ComputeNearest(int row);

  std::vector<std::vector<int>> neighbors_;
  Solver::IndexEvaluator3 evaluator_;
  const PathOperator& path_operator_;
  const int size_;
  bool initialized_;

  DISALLOW_COPY_AND_ASSIGN(NearestNeighbors);
};

NearestNeighbors::NearestNeighbors(Solver::IndexEvaluator3 evaluator,
                                   const PathOperator& path_operator, int size)
    : evaluator_(std::move(evaluator)),
      path_operator_(path_operator),
      size_(size),
      initialized_(false) {}

void NearestNeighbors::Initialize() {
  // TODO(user): recompute if node changes path ?
  if (!initialized_) {
    initialized_ = true;
    for (int i = 0; i < path_operator_.number_of_nexts(); ++i) {
      neighbors_.push_back(std::vector<int>());
      ComputeNearest(i);
    }
  }
}

const std::vector<int>& NearestNeighbors::Neighbors(int index) const {
  return neighbors_[index];
}

void NearestNeighbors::ComputeNearest(int row) {
  // Find size_ nearest neighbors for row of index 'row'.
  const int path = path_operator_.Path(row);
  const IntVar* var = path_operator_.Var(row);
  const int64 var_min = var->Min();
  const int var_size = var->Max() - var_min + 1;
  using ValuedIndex = std::pair<int64 /*value*/, int /*index*/>;
  std::vector<ValuedIndex> neighbors(var_size);
  for (int i = 0; i < var_size; ++i) {
    const int index = i + var_min;
    neighbors[i] = std::make_pair(evaluator_(row, index, path), index);
  }
  if (var_size > size_) {
    std::nth_element(neighbors.begin(), neighbors.begin() + size_ - 1,
                     neighbors.end());
  }

  // Setup global neighbor matrix for row row_index
  for (int i = 0; i < std::min(size_, var_size); ++i) {
    neighbors_[row].push_back(neighbors[i].second);
  }
  std::sort(neighbors_[row].begin(), neighbors_[row].end());
}

class LinKernighan : public PathOperator {
 public:
  LinKernighan(const std::vector<IntVar*>& vars,
               const std::vector<IntVar*>& secondary_vars,
               const Solver::IndexEvaluator3& evaluator, bool topt);
  ~LinKernighan() override;
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "LinKernighan"; }

 private:
  void OnNodeInitialization() override;

  static const int kNeighbors;

  bool InFromOut(int64 in_i, int64 in_j, int64* out, int64* gain);

  Solver::IndexEvaluator3 const evaluator_;
  NearestNeighbors neighbors_;
  absl::flat_hash_set<int64> marked_;
  const bool topt_;
};

// While the accumulated local gain is positive, perform a 2opt or a 3opt move
// followed by a series of 2opt moves. Return a neighbor for which the global
// gain is positive.

LinKernighan::LinKernighan(const std::vector<IntVar*>& vars,
                           const std::vector<IntVar*>& secondary_vars,
                           const Solver::IndexEvaluator3& evaluator, bool topt)
    : PathOperator(vars, secondary_vars, 1, true, false, nullptr),
      evaluator_(evaluator),
      neighbors_(evaluator, *this, kNeighbors),
      topt_(topt) {}

LinKernighan::~LinKernighan() {}

void LinKernighan::OnNodeInitialization() { neighbors_.Initialize(); }

bool LinKernighan::MakeNeighbor() {
  marked_.clear();
  int64 node = BaseNode(0);
  int64 path = Path(node);
  int64 base = node;
  int64 next = Next(node);
  if (IsPathEnd(next)) return false;
  int64 out = -1;
  int64 gain = 0;
  marked_.insert(node);
  if (topt_) {  // Try a 3opt first
    if (!InFromOut(node, next, &out, &gain)) return false;
    marked_.insert(next);
    marked_.insert(out);
    const int64 node1 = out;
    if (IsPathEnd(node1)) return false;
    const int64 next1 = Next(node1);
    if (IsPathEnd(next1)) return false;
    if (!InFromOut(node1, next1, &out, &gain)) return false;
    marked_.insert(next1);
    marked_.insert(out);
    if (!CheckChainValidity(out, node1, node) || !MoveChain(out, node1, node)) {
      return false;
    }
    const int64 next_out = Next(out);
    const int64 in_cost = evaluator_(node, next_out, path);
    const int64 out_cost = evaluator_(out, next_out, path);
    if (CapAdd(CapSub(gain, in_cost), out_cost) > 0) return true;
    node = out;
    if (IsPathEnd(node)) return false;
    next = next_out;
    if (IsPathEnd(next)) return false;
  }
  // Try 2opts
  while (InFromOut(node, next, &out, &gain)) {
    marked_.insert(next);
    marked_.insert(out);
    int64 chain_last;
    if (!ReverseChain(node, out, &chain_last)) {
      return false;
    }
    int64 in_cost = evaluator_(base, chain_last, path);
    int64 out_cost = evaluator_(chain_last, out, path);
    if (CapAdd(CapSub(gain, in_cost), out_cost) > 0) {
      return true;
    }
    node = chain_last;
    if (IsPathEnd(node)) {
      return false;
    }
    next = out;
    if (IsPathEnd(next)) {
      return false;
    }
  }
  return false;
}

const int LinKernighan::kNeighbors = 5 + 1;

bool LinKernighan::InFromOut(int64 in_i, int64 in_j, int64* out, int64* gain) {
  const std::vector<int>& nexts = neighbors_.Neighbors(in_j);
  int64 best_gain = kint64min;
  int64 path = Path(in_i);
  int64 out_cost = evaluator_(in_i, in_j, path);
  const int64 current_gain = CapAdd(*gain, out_cost);
  for (int k = 0; k < nexts.size(); ++k) {
    const int64 next = nexts[k];
    if (next != in_j) {
      int64 in_cost = evaluator_(in_j, next, path);
      int64 new_gain = CapSub(current_gain, in_cost);
      if (new_gain > 0 && next != Next(in_j) && marked_.count(in_j) == 0 &&
          marked_.count(next) == 0) {
        if (best_gain < new_gain) {
          *out = next;
          best_gain = new_gain;
        }
      }
    }
  }
  *gain = best_gain;
  return (best_gain > kint64min);
}

// ----- Path-based Large Neighborhood Search -----

// Breaks "number_of_chunks" chains of "chunk_size" arcs, and deactivate all
// inactive nodes if "unactive_fragments" is true.
// As a special case, if chunk_size=0, then we break full paths.

class PathLns : public PathOperator {
 public:
  PathLns(const std::vector<IntVar*>& vars,
          const std::vector<IntVar*>& secondary_vars, int number_of_chunks,
          int chunk_size, bool unactive_fragments)
      : PathOperator(vars, secondary_vars, number_of_chunks, true, true,
                     nullptr),
        number_of_chunks_(number_of_chunks),
        chunk_size_(chunk_size),
        unactive_fragments_(unactive_fragments) {
    CHECK_GE(chunk_size_, 0);
  }
  ~PathLns() override {}
  bool MakeNeighbor() override;

  std::string DebugString() const override { return "PathLns"; }
  bool HasFragments() const override { return true; }

 private:
  inline bool ChainsAreFullPaths() const { return chunk_size_ == 0; }
  void DeactivateChain(int64 node);
  void DeactivateUnactives();

  const int number_of_chunks_;
  const int chunk_size_;
  const bool unactive_fragments_;
};

bool PathLns::MakeNeighbor() {
  if (ChainsAreFullPaths()) {
    // Reject the current position as a neighbor if any of its base node
    // isn't at the start of a path.
    // TODO(user): make this more efficient.
    for (int i = 0; i < number_of_chunks_; ++i) {
      if (BaseNode(i) != StartNode(i)) return false;
    }
  }
  for (int i = 0; i < number_of_chunks_; ++i) {
    DeactivateChain(BaseNode(i));
  }
  DeactivateUnactives();
  return true;
}

void PathLns::DeactivateChain(int64 node) {
  for (int i = 0, current = node;
       (ChainsAreFullPaths() || i < chunk_size_) && !IsPathEnd(current);
       ++i, current = Next(current)) {
    Deactivate(current);
    if (!ignore_path_vars_) {
      Deactivate(number_of_nexts_ + current);
    }
  }
}

void PathLns::DeactivateUnactives() {
  if (unactive_fragments_) {
    for (int i = 0; i < Size(); ++i) {
      if (IsInactive(i)) {
        Deactivate(i);
        if (!ignore_path_vars_) {
          Deactivate(number_of_nexts_ + i);
        }
      }
    }
  }
}

// ----- Limit the number of neighborhoods explored -----

class NeighborhoodLimit : public LocalSearchOperator {
 public:
  NeighborhoodLimit(LocalSearchOperator* const op, int64 limit)
      : operator_(op), limit_(limit), next_neighborhood_calls_(0) {
    CHECK(op != nullptr);
    CHECK_GT(limit, 0);
  }

  void Start(const Assignment* assignment) override {
    next_neighborhood_calls_ = 0;
    operator_->Start(assignment);
  }

  bool MakeNextNeighbor(Assignment* delta, Assignment* deltadelta) override {
    if (next_neighborhood_calls_ >= limit_) {
      return false;
    }
    ++next_neighborhood_calls_;
    return operator_->MakeNextNeighbor(delta, deltadelta);
  }

  bool HoldsDelta() const override { return operator_->HoldsDelta(); }

  std::string DebugString() const override { return "NeighborhoodLimit"; }

 private:
  LocalSearchOperator* const operator_;
  const int64 limit_;
  int64 next_neighborhood_calls_;
};

LocalSearchOperator* Solver::MakeNeighborhoodLimit(
    LocalSearchOperator* const op, int64 limit) {
  return RevAlloc(new NeighborhoodLimit(op, limit));
}

// ----- Concatenation of operators -----

namespace {
class CompoundOperator : public LocalSearchOperator {
 public:
  CompoundOperator(std::vector<LocalSearchOperator*> operators,
                   std::function<int64(int, int)> evaluator);
  ~CompoundOperator() override {}
  void Reset() override;
  void Start(const Assignment* assignment) override;
  bool MakeNextNeighbor(Assignment* delta, Assignment* deltadelta) override;
  bool HasFragments() const override { return has_fragments_; }
  bool HoldsDelta() const override { return true; }

  std::string DebugString() const override {
    return operators_.empty()
               ? ""
               : operators_[operator_indices_[index_]]->DebugString();
  }
  const LocalSearchOperator* Self() const override {
    return operators_.empty() ? this
                              : operators_[operator_indices_[index_]]->Self();
  }

 private:
  class OperatorComparator {
   public:
    OperatorComparator(std::function<int64(int, int)> evaluator,
                       int active_operator)
        : evaluator_(std::move(evaluator)), active_operator_(active_operator) {}
    bool operator()(int lhs, int rhs) const {
      const int64 lhs_value = Evaluate(lhs);
      const int64 rhs_value = Evaluate(rhs);
      return lhs_value < rhs_value || (lhs_value == rhs_value && lhs < rhs);
    }

   private:
    int64 Evaluate(int operator_index) const {
      return evaluator_(active_operator_, operator_index);
    }

    std::function<int64(int, int)> evaluator_;
    const int active_operator_;
  };

  int64 index_;
  std::vector<LocalSearchOperator*> operators_;
  std::vector<int> operator_indices_;
  std::function<int64(int, int)> evaluator_;
  Bitset64<> started_;
  const Assignment* start_assignment_;
  bool has_fragments_;
};

CompoundOperator::CompoundOperator(std::vector<LocalSearchOperator*> operators,
                                   std::function<int64(int, int)> evaluator)
    : index_(0),
      operators_(std::move(operators)),
      evaluator_(std::move(evaluator)),
      started_(operators_.size()),
      start_assignment_(nullptr),
      has_fragments_(false) {
  operators_.erase(std::remove(operators_.begin(), operators_.end(), nullptr),
                   operators_.end());
  operator_indices_.resize(operators_.size());
  std::iota(operator_indices_.begin(), operator_indices_.end(), 0);
  for (LocalSearchOperator* const op : operators_) {
    if (op->HasFragments()) {
      has_fragments_ = true;
      break;
    }
  }
}

void CompoundOperator::Reset() {
  for (LocalSearchOperator* const op : operators_) {
    op->Reset();
  }
}

void CompoundOperator::Start(const Assignment* assignment) {
  start_assignment_ = assignment;
  started_.ClearAll();
  if (!operators_.empty()) {
    OperatorComparator comparator(evaluator_, operator_indices_[index_]);
    std::sort(operator_indices_.begin(), operator_indices_.end(), comparator);
    index_ = 0;
  }
}

bool CompoundOperator::MakeNextNeighbor(Assignment* delta,
                                        Assignment* deltadelta) {
  if (!operators_.empty()) {
    do {
      // TODO(user): keep copy of delta in case MakeNextNeighbor
      // pollutes delta on a fail.
      const int64 operator_index = operator_indices_[index_];
      if (!started_[operator_index]) {
        operators_[operator_index]->Start(start_assignment_);
        started_.Set(operator_index);
      }
      if (!operators_[operator_index]->HoldsDelta()) {
        delta->Clear();
      }
      if (operators_[operator_index]->MakeNextNeighbor(delta, deltadelta)) {
        return true;
      }
      ++index_;
      delta->Clear();
      if (index_ == operators_.size()) {
        index_ = 0;
      }
    } while (index_ != 0);
  }
  return false;
}

int64 CompoundOperatorNoRestart(int size, int active_index,
                                int operator_index) {
  return (operator_index < active_index) ? size + operator_index - active_index
                                         : operator_index - active_index;
}

int64 CompoundOperatorRestart(int active_index, int operator_index) {
  return 0;
}
}  // namespace

LocalSearchOperator* Solver::ConcatenateOperators(
    const std::vector<LocalSearchOperator*>& ops) {
  return ConcatenateOperators(ops, false);
}

LocalSearchOperator* Solver::ConcatenateOperators(
    const std::vector<LocalSearchOperator*>& ops, bool restart) {
  if (restart) {
    std::function<int64(int, int)> eval = CompoundOperatorRestart;
    return ConcatenateOperators(ops, eval);
  }
  const int size = ops.size();
  return ConcatenateOperators(ops, [size](int i, int j) {
    return CompoundOperatorNoRestart(size, i, j);
  });
}

LocalSearchOperator* Solver::ConcatenateOperators(
    const std::vector<LocalSearchOperator*>& ops,
    std::function<int64(int, int)> evaluator) {
  return RevAlloc(new CompoundOperator(ops, std::move(evaluator)));
}

namespace {
class RandomCompoundOperator : public LocalSearchOperator {
 public:
  explicit RandomCompoundOperator(std::vector<LocalSearchOperator*> operators);
  RandomCompoundOperator(std::vector<LocalSearchOperator*> operators,
                         int32 seed);
  ~RandomCompoundOperator() override {}
  void Reset() override;
  void Start(const Assignment* assignment) override;
  bool MakeNextNeighbor(Assignment* delta, Assignment* deltadelta) override;
  bool HoldsDelta() const override { return true; }

  std::string DebugString() const override { return "RandomCompoundOperator"; }
  // TODO(user): define Self method.

 private:
  std::mt19937 rand_;
  const std::vector<LocalSearchOperator*> operators_;
  bool has_fragments_;
};

void RandomCompoundOperator::Start(const Assignment* assignment) {
  for (LocalSearchOperator* const op : operators_) {
    op->Start(assignment);
  }
}

RandomCompoundOperator::RandomCompoundOperator(
    std::vector<LocalSearchOperator*> operators)
    : RandomCompoundOperator(std::move(operators), CpRandomSeed()) {}

RandomCompoundOperator::RandomCompoundOperator(
    std::vector<LocalSearchOperator*> operators, int32 seed)
    : rand_(seed), operators_(std::move(operators)), has_fragments_(false) {
  for (LocalSearchOperator* const op : operators_) {
    if (op->HasFragments()) {
      has_fragments_ = true;
      break;
    }
  }
}

void RandomCompoundOperator::Reset() {
  for (LocalSearchOperator* const op : operators_) {
    op->Reset();
  }
}

bool RandomCompoundOperator::MakeNextNeighbor(Assignment* delta,
                                              Assignment* deltadelta) {
  const int size = operators_.size();
  std::vector<int> indices(size);
  std::iota(indices.begin(), indices.end(), 0);
  std::shuffle(indices.begin(), indices.end(), rand_);
  for (int index : indices) {
    if (!operators_[index]->HoldsDelta()) {
      delta->Clear();
    }
    if (operators_[index]->MakeNextNeighbor(delta, deltadelta)) {
      return true;
    }
    delta->Clear();
  }
  return false;
}
}  // namespace

LocalSearchOperator* Solver::RandomConcatenateOperators(
    const std::vector<LocalSearchOperator*>& ops) {
  return RevAlloc(new RandomCompoundOperator(ops));
}

LocalSearchOperator* Solver::RandomConcatenateOperators(
    const std::vector<LocalSearchOperator*>& ops, int32 seed) {
  return RevAlloc(new RandomCompoundOperator(ops, seed));
}

namespace {
class MultiArmedBanditCompoundOperator : public LocalSearchOperator {
 public:
  explicit MultiArmedBanditCompoundOperator(
      std::vector<LocalSearchOperator*> operators, double memory_coefficient,
      double exploration_coefficient, bool maximize);
  ~MultiArmedBanditCompoundOperator() override {}
  void Reset() override;
  void Start(const Assignment* assignment) override;
  bool MakeNextNeighbor(Assignment* delta, Assignment* deltadelta) override;
  bool HoldsDelta() const override { return true; }

  std::string DebugString() const override {
    return operators_.empty()
               ? ""
               : operators_[operator_indices_[index_]]->DebugString();
  }
  const LocalSearchOperator* Self() const override {
    return operators_.empty() ? this
                              : operators_[operator_indices_[index_]]->Self();
  }

 private:
  double Score(int index);
  int index_;
  std::vector<LocalSearchOperator*> operators_;
  Bitset64<> started_;
  const Assignment* start_assignment_;
  bool has_fragments_;
  std::vector<int> operator_indices_;
  int64 last_objective_;
  std::vector<double> avg_improvement_;
  int num_neighbors_;
  std::vector<double> num_neighbors_per_operator_;
  const bool maximize_;
  // Sets how much the objective improvement of previous accepted neighbors
  // influence the current average improvement. The formula is
  //  avg_improvement +=
  //   memory_coefficient * (current_improvement - avg_improvement).
  const double memory_coefficient_;
  // Sets how often we explore rarely used and unsuccessful in the past
  // operators. Operators are sorted by
  //  avg_improvement_[i] + exploration_coefficient_ *
  //   sqrt(2 * log(1 + num_neighbors_) / (1 + num_neighbors_per_operator_[i]));
  const double exploration_coefficient_;
};

MultiArmedBanditCompoundOperator::MultiArmedBanditCompoundOperator(
    std::vector<LocalSearchOperator*> operators, double memory_coefficient,
    double exploration_coefficient, bool maximize)
    : index_(0),
      operators_(std::move(operators)),
      started_(operators_.size()),
      start_assignment_(nullptr),
      has_fragments_(false),
      last_objective_(kint64max),
      num_neighbors_(0),
      maximize_(maximize),
      memory_coefficient_(memory_coefficient),
      exploration_coefficient_(exploration_coefficient) {
  DCHECK_GE(memory_coefficient_, 0);
  DCHECK_LE(memory_coefficient_, 1);
  DCHECK_GE(exploration_coefficient_, 0);
  operators_.erase(std::remove(operators_.begin(), operators_.end(), nullptr),
                   operators_.end());
  operator_indices_.resize(operators_.size());
  std::iota(operator_indices_.begin(), operator_indices_.end(), 0);
  num_neighbors_per_operator_.resize(operators_.size(), 0);
  avg_improvement_.resize(operators_.size(), 0);
  for (LocalSearchOperator* const op : operators_) {
    if (op->HasFragments()) {
      has_fragments_ = true;
      break;
    }
  }
}

void MultiArmedBanditCompoundOperator::Reset() {
  for (LocalSearchOperator* const op : operators_) {
    op->Reset();
  }
}

double MultiArmedBanditCompoundOperator::Score(int index) {
  return avg_improvement_[index] +
         exploration_coefficient_ *
             sqrt(2 * log(1 + num_neighbors_) /
                  (1 + num_neighbors_per_operator_[index]));
}

void MultiArmedBanditCompoundOperator::Start(const Assignment* assignment) {
  start_assignment_ = assignment;
  started_.ClearAll();
  if (operators_.empty()) return;

  const double objective = assignment->ObjectiveValue();

  if (objective == last_objective_) return;
  // Skip a neighbor evaluation if last_objective_ hasn't been set yet.
  if (last_objective_ == kint64max) {
    last_objective_ = objective;
    return;
  }

  const double improvement =
      maximize_ ? objective - last_objective_ : last_objective_ - objective;
  if (improvement < 0) {
    return;
  }
  last_objective_ = objective;
  avg_improvement_[operator_indices_[index_]] +=
      memory_coefficient_ *
      (improvement - avg_improvement_[operator_indices_[index_]]);

  std::sort(operator_indices_.begin(), operator_indices_.end(),
            [this](int lhs, int rhs) {
              const double lhs_score = Score(lhs);
              const double rhs_score = Score(rhs);
              return lhs_score > rhs_score ||
                     (lhs_score == rhs_score && lhs < rhs);
            });

  index_ = 0;
}

bool MultiArmedBanditCompoundOperator::MakeNextNeighbor(
    Assignment* delta, Assignment* deltadelta) {
  if (operators_.empty()) return false;
  do {
    const int operator_index = operator_indices_[index_];
    if (!started_[operator_index]) {
      operators_[operator_index]->Start(start_assignment_);
      started_.Set(operator_index);
    }
    if (!operators_[operator_index]->HoldsDelta()) {
      delta->Clear();
    }
    if (operators_[operator_index]->MakeNextNeighbor(delta, deltadelta)) {
      ++num_neighbors_;
      ++num_neighbors_per_operator_[operator_index];
      return true;
    }
    ++index_;
    delta->Clear();
    if (index_ == operators_.size()) {
      index_ = 0;
    }
  } while (index_ != 0);
  return false;
}
}  // namespace

LocalSearchOperator* Solver::MultiArmedBanditConcatenateOperators(
    const std::vector<LocalSearchOperator*>& ops, double memory_coefficient,
    double exploration_coefficient, bool maximize) {
  return RevAlloc(new MultiArmedBanditCompoundOperator(
      ops, memory_coefficient, exploration_coefficient, maximize));
}

// ----- Operator factory -----

template <class T>
LocalSearchOperator* MakeLocalSearchOperator(
    Solver* solver, const std::vector<IntVar*>& vars,
    const std::vector<IntVar*>& secondary_vars,
    std::function<int(int64)> start_empty_path_class) {
  return solver->RevAlloc(
      new T(vars, secondary_vars, std::move(start_empty_path_class)));
}

#define MAKE_LOCAL_SEARCH_OPERATOR(OperatorClass)                  \
  template <>                                                      \
  LocalSearchOperator* MakeLocalSearchOperator<OperatorClass>(     \
      Solver * solver, const std::vector<IntVar*>& vars,           \
      const std::vector<IntVar*>& secondary_vars,                  \
      std::function<int(int64)> start_empty_path_class) {          \
    return solver->RevAlloc(new OperatorClass(                     \
        vars, secondary_vars, std::move(start_empty_path_class))); \
  }

MAKE_LOCAL_SEARCH_OPERATOR(TwoOpt)
MAKE_LOCAL_SEARCH_OPERATOR(Relocate)
MAKE_LOCAL_SEARCH_OPERATOR(Exchange)
MAKE_LOCAL_SEARCH_OPERATOR(Cross)
MAKE_LOCAL_SEARCH_OPERATOR(MakeActiveOperator)
MAKE_LOCAL_SEARCH_OPERATOR(MakeInactiveOperator)
MAKE_LOCAL_SEARCH_OPERATOR(MakeChainInactiveOperator)
MAKE_LOCAL_SEARCH_OPERATOR(SwapActiveOperator)
MAKE_LOCAL_SEARCH_OPERATOR(ExtendedSwapActiveOperator)
MAKE_LOCAL_SEARCH_OPERATOR(MakeActiveAndRelocate)
MAKE_LOCAL_SEARCH_OPERATOR(RelocateAndMakeActiveOperator)
MAKE_LOCAL_SEARCH_OPERATOR(RelocateAndMakeInactiveOperator)

#undef MAKE_LOCAL_SEARCH_OPERATOR

LocalSearchOperator* Solver::MakeOperator(const std::vector<IntVar*>& vars,
                                          Solver::LocalSearchOperators op) {
  return MakeOperator(vars, std::vector<IntVar*>(), op);
}

LocalSearchOperator* Solver::MakeOperator(
    const std::vector<IntVar*>& vars,
    const std::vector<IntVar*>& secondary_vars,
    Solver::LocalSearchOperators op) {
  LocalSearchOperator* result = nullptr;
  switch (op) {
    case Solver::TWOOPT: {
      result = RevAlloc(new TwoOpt(vars, secondary_vars, nullptr));
      break;
    }
    case Solver::OROPT: {
      std::vector<LocalSearchOperator*> operators;
      for (int i = 1; i < 4; ++i) {
        operators.push_back(RevAlloc(
            new Relocate(vars, secondary_vars, absl::StrCat("OrOpt<", i, ">"),
                         nullptr, i, true)));
      }
      result = ConcatenateOperators(operators);
      break;
    }
    case Solver::RELOCATE: {
      result = MakeLocalSearchOperator<Relocate>(this, vars, secondary_vars,
                                                 nullptr);
      break;
    }
    case Solver::EXCHANGE: {
      result = MakeLocalSearchOperator<Exchange>(this, vars, secondary_vars,
                                                 nullptr);
      break;
    }
    case Solver::CROSS: {
      result =
          MakeLocalSearchOperator<Cross>(this, vars, secondary_vars, nullptr);
      break;
    }
    case Solver::MAKEACTIVE: {
      result = MakeLocalSearchOperator<MakeActiveOperator>(
          this, vars, secondary_vars, nullptr);
      break;
    }
    case Solver::MAKEINACTIVE: {
      result = MakeLocalSearchOperator<MakeInactiveOperator>(
          this, vars, secondary_vars, nullptr);
      break;
    }
    case Solver::MAKECHAININACTIVE: {
      result = MakeLocalSearchOperator<MakeChainInactiveOperator>(
          this, vars, secondary_vars, nullptr);
      break;
    }
    case Solver::SWAPACTIVE: {
      result = MakeLocalSearchOperator<SwapActiveOperator>(
          this, vars, secondary_vars, nullptr);
      break;
    }
    case Solver::EXTENDEDSWAPACTIVE: {
      result = MakeLocalSearchOperator<ExtendedSwapActiveOperator>(
          this, vars, secondary_vars, nullptr);
      break;
    }
    case Solver::PATHLNS: {
      result = RevAlloc(new PathLns(vars, secondary_vars, 2, 3, false));
      break;
    }
    case Solver::FULLPATHLNS: {
      result = RevAlloc(new PathLns(vars, secondary_vars,
                                    /*number_of_chunks=*/1,
                                    /*chunk_size=*/0,
                                    /*unactive_fragments=*/true));
      break;
    }
    case Solver::UNACTIVELNS: {
      result = RevAlloc(new PathLns(vars, secondary_vars, 1, 6, true));
      break;
    }
    case Solver::INCREMENT: {
      if (secondary_vars.empty()) {
        result = RevAlloc(new IncrementValue(vars));
      } else {
        LOG(FATAL) << "Operator " << op
                   << " does not support secondary variables";
      }
      break;
    }
    case Solver::DECREMENT: {
      if (secondary_vars.empty()) {
        result = RevAlloc(new DecrementValue(vars));
      } else {
        LOG(FATAL) << "Operator " << op
                   << " does not support secondary variables";
      }
      break;
    }
    case Solver::SIMPLELNS: {
      if (secondary_vars.empty()) {
        result = RevAlloc(new SimpleLns(vars, 1));
      } else {
        LOG(FATAL) << "Operator " << op
                   << " does not support secondary variables";
      }
      break;
    }
    default:
      LOG(FATAL) << "Unknown operator " << op;
  }
  return result;
}

LocalSearchOperator* Solver::MakeOperator(
    const std::vector<IntVar*>& vars, Solver::IndexEvaluator3 evaluator,
    Solver::EvaluatorLocalSearchOperators op) {
  return MakeOperator(vars, std::vector<IntVar*>(), std::move(evaluator), op);
}

LocalSearchOperator* Solver::MakeOperator(
    const std::vector<IntVar*>& vars,
    const std::vector<IntVar*>& secondary_vars,
    Solver::IndexEvaluator3 evaluator,
    Solver::EvaluatorLocalSearchOperators op) {
  LocalSearchOperator* result = nullptr;
  switch (op) {
    case Solver::LK: {
      std::vector<LocalSearchOperator*> operators;
      operators.push_back(RevAlloc(
          new LinKernighan(vars, secondary_vars, evaluator, /*topt=*/false)));
      operators.push_back(RevAlloc(
          new LinKernighan(vars, secondary_vars, evaluator, /*topt=*/true)));
      result = ConcatenateOperators(operators);
      break;
    }
    case Solver::TSPOPT: {
      result = RevAlloc(
          new TSPOpt(vars, secondary_vars, evaluator,
                     absl::GetFlag(FLAGS_cp_local_search_tsp_opt_size)));
      break;
    }
    case Solver::TSPLNS: {
      result = RevAlloc(
          new TSPLns(vars, secondary_vars, evaluator,
                     absl::GetFlag(FLAGS_cp_local_search_tsp_lns_size)));
      break;
    }
    default:
      LOG(FATAL) << "Unknown operator " << op;
  }
  return result;
}

namespace {
// Classes for Local Search Operation used in Local Search filters.

class SumOperation {
 public:
  SumOperation() : value_(0) {}
  void Init() { value_ = 0; }
  void Update(int64 update) { value_ = CapAdd(value_, update); }
  void Remove(int64 remove) { value_ = CapSub(value_, remove); }
  int64 value() const { return value_; }
  void set_value(int64 new_value) { value_ = new_value; }

 private:
  int64 value_;
};

class ProductOperation {
 public:
  ProductOperation() : value_(1) {}
  void Init() { value_ = 1; }
  void Update(int64 update) { value_ *= update; }
  void Remove(int64 remove) {
    if (remove != 0) {
      value_ /= remove;
    }
  }
  int64 value() const { return value_; }
  void set_value(int64 new_value) { value_ = new_value; }

 private:
  int64 value_;
};

class MinOperation {
 public:
  void Init() { values_set_.clear(); }
  void Update(int64 update) { values_set_.insert(update); }
  void Remove(int64 remove) { values_set_.erase(remove); }
  int64 value() const {
    return (!values_set_.empty()) ? *values_set_.begin() : 0;
  }
  void set_value(int64 new_value) {}

 private:
  std::set<int64> values_set_;
};

class MaxOperation {
 public:
  void Init() { values_set_.clear(); }
  void Update(int64 update) { values_set_.insert(update); }
  void Remove(int64 remove) { values_set_.erase(remove); }
  int64 value() const {
    return (!values_set_.empty()) ? *values_set_.rbegin() : 0;
  }
  void set_value(int64 new_value) {}

 private:
  std::set<int64> values_set_;
};

// Always accepts deltas, cost 0.
class AcceptFilter : public LocalSearchFilter {
 public:
  std::string DebugString() const override { return "AcceptFilter"; }
  bool Accept(const Assignment* delta, const Assignment* deltadelta,
              int64 obj_min, int64 obj_max) override {
    return true;
  }
  void Synchronize(const Assignment* assignment,
                   const Assignment* delta) override {}
};
}  // namespace

LocalSearchFilter* Solver::MakeAcceptFilter() {
  return RevAlloc(new AcceptFilter());
}

namespace {
// Never accepts deltas, cost 0.
class RejectFilter : public LocalSearchFilter {
 public:
  std::string DebugString() const override { return "RejectFilter"; }
  bool Accept(const Assignment* delta, const Assignment* deltadelta,
              int64 obj_min, int64 obj_max) override {
    return false;
  }
  void Synchronize(const Assignment* assignment,
                   const Assignment* delta) override {}
};
}  // namespace

LocalSearchFilter* Solver::MakeRejectFilter() {
  return RevAlloc(new RejectFilter());
}

PathState::PathState(int num_nodes, std::vector<int> path_start,
                     std::vector<int> path_end)
    : num_nodes_(num_nodes),
      num_paths_(path_start.size()),
      num_nodes_threshold_(std::max(16, 4 * num_nodes_))  // Arbitrary value.
{
  DCHECK_EQ(path_start.size(), num_paths_);
  DCHECK_EQ(path_end.size(), num_paths_);
  for (int p = 0; p < num_paths_; ++p) {
    path_start_end_.push_back({path_start[p], path_end[p]});
  }
  // Initial state is all unperformed: paths go from start to end directly.
  committed_index_.assign(num_nodes_, -1);
  committed_nodes_.assign(2 * num_paths_, {-1, -1});
  chains_.assign(num_paths_ + 1, {-1, -1});  // Reserve 1 more for sentinel.
  paths_.assign(num_paths_, {-1, -1});
  for (int path = 0; path < num_paths_; ++path) {
    const int index = 2 * path;
    const PathStartEnd start_end = path_start_end_[path];
    committed_index_[start_end.start] = index;
    committed_index_[start_end.end] = index + 1;

    committed_nodes_[index] = {start_end.start, path};
    committed_nodes_[index + 1] = {start_end.end, path};

    chains_[path] = {index, index + 2};
    paths_[path] = {path, path + 1};
  }
  chains_[num_paths_] = {0, 0};  // Sentinel.
  // Nodes that are not starts or ends are loops.
  for (int node = 0; node < num_nodes_; ++node) {
    if (committed_index_[node] != -1) continue;  // node is start or end.
    committed_index_[node] = committed_nodes_.size();
    committed_nodes_.push_back({node, -1});
  }
  path_has_changed_.assign(num_paths_, false);
}

PathState::ChainRange PathState::Chains(int path) const {
  const PathBounds bounds = paths_[path];
  return PathState::ChainRange(chains_.data() + bounds.begin_index,
                               chains_.data() + bounds.end_index,
                               committed_nodes_.data());
}

PathState::NodeRange PathState::Nodes(int path) const {
  const PathBounds bounds = paths_[path];
  return PathState::NodeRange(chains_.data() + bounds.begin_index,
                              chains_.data() + bounds.end_index,
                              committed_nodes_.data());
}

void PathState::MakeChainsFromChangedPathsAndArcsWithSelectionAlgorithm() {
  int num_visited_changed_arcs = 0;
  const int num_changed_arcs = tail_head_indices_.size();
  const int num_committed_nodes = committed_nodes_.size();
  // For every path, find all its chains.
  for (const int path : changed_paths_) {
    const int old_chain_size = chains_.size();
    const ChainBounds bounds = chains_[paths_[path].begin_index];
    const int start_index = bounds.begin_index;
    const int end_index = bounds.end_index - 1;
    int current_index = start_index;
    while (true) {
      // Look for smallest non-visited tail_index that is no smaller than
      // current_index.
      int selected_arc = -1;
      int selected_tail_index = num_committed_nodes;
      for (int i = num_visited_changed_arcs; i < num_changed_arcs; ++i) {
        const int tail_index = tail_head_indices_[i].tail_index;
        if (current_index <= tail_index && tail_index < selected_tail_index) {
          selected_arc = i;
          selected_tail_index = tail_index;
        }
      }
      // If there is no such tail index, or more generally if the next chain
      // would be cut by end of path,
      // stack {current_index, end_index + 1} in chains_, and go to next path.
      // Otherwise, stack {current_index, tail_index+1} in chains_,
      // set current_index = head_index, set pair to visited.
      if (start_index <= current_index && current_index <= end_index &&
          end_index < selected_tail_index) {
        chains_.emplace_back(current_index, end_index + 1);
        break;
      } else {
        chains_.emplace_back(current_index, selected_tail_index + 1);
        current_index = tail_head_indices_[selected_arc].head_index;
        std::swap(tail_head_indices_[num_visited_changed_arcs],
                  tail_head_indices_[selected_arc]);
        ++num_visited_changed_arcs;
      }
    }
    const int new_chain_size = chains_.size();
    paths_[path] = {old_chain_size, new_chain_size};
  }
  chains_.emplace_back(0, 0);  // Sentinel.
}

void PathState::MakeChainsFromChangedPathsAndArcsWithGenericAlgorithm() {
  // TRICKY: For each changed path, we want to generate a sequence of chains
  // that represents the path in the changed state.
  // First, notice that if we add a fake end->start arc for each changed path,
  // then all chains will be from the head of an arc to the tail of an arc.
  // A way to generate the changed chains and paths would be, for each path,
  // to start from a fake arc's head (the path start), go down the path until
  // the tail of an arc, and go to the next arc until we return on the fake arc,
  // enqueuing the [head, tail] chains as we go.
  // In turn, to do that, we need to know which arc to go to.
  // If we sort all heads and tails by index in two separate arrays,
  // the head_index and tail_index at the same rank are such that
  // [head_index, tail_index] is a chain. Moreover, the arc that must be visited
  // after head_index's arc is tail_index's arc.

  // Add a fake end->start arc for each path.
  for (const int path : changed_paths_) {
    const PathStartEnd start_end = path_start_end_[path];
    tail_head_indices_.push_back(
        {committed_index_[start_end.end], committed_index_[start_end.start]});
  }

  // Generate pairs (tail_index, arc) and (head_index, arc) for all arcs,
  // sort those pairs by index.
  const int num_arc_indices = tail_head_indices_.size();
  arcs_by_tail_index_.resize(num_arc_indices);
  arcs_by_head_index_.resize(num_arc_indices);
  for (int i = 0; i < num_arc_indices; ++i) {
    arcs_by_tail_index_[i] = {tail_head_indices_[i].tail_index, i};
    arcs_by_head_index_[i] = {tail_head_indices_[i].head_index, i};
  }
  std::sort(arcs_by_tail_index_.begin(), arcs_by_tail_index_.end());
  std::sort(arcs_by_head_index_.begin(), arcs_by_head_index_.end());
  // Generate the map from arc to next arc in path.
  next_arc_.resize(num_arc_indices);
  for (int i = 0; i < num_arc_indices; ++i) {
    next_arc_[arcs_by_head_index_[i].arc] = arcs_by_tail_index_[i].arc;
  }

  // Generate chains: for every changed path, start from its fake arc,
  // jump to next_arc_ until going back to fake arc,
  // enqueuing chains as we go.
  const int first_fake_arc = num_arc_indices - changed_paths_.size();
  for (int fake_arc = first_fake_arc; fake_arc < num_arc_indices; ++fake_arc) {
    const int new_path_begin = chains_.size();
    int32 arc = fake_arc;
    do {
      const int chain_begin = tail_head_indices_[arc].head_index;
      arc = next_arc_[arc];
      const int chain_end = tail_head_indices_[arc].tail_index + 1;
      chains_.emplace_back(chain_begin, chain_end);
    } while (arc != fake_arc);
    const int path = changed_paths_[fake_arc - first_fake_arc];
    const int new_path_end = chains_.size();
    paths_[path] = {new_path_begin, new_path_end};
  }
  chains_.emplace_back(0, 0);  // Sentinel.
}

void PathState::CutChains() {
  if (is_invalid_) return;
  // Filter out unchanged arcs from changed_arcs_,
  // translate changed arcs to changed arc indices.
  // Fill changed_paths_ while we hold node_path.
  DCHECK_EQ(chains_.size(), num_paths_ + 1);  // One per path + sentinel.
  DCHECK(changed_paths_.empty());
  tail_head_indices_.clear();
  int num_changed_arcs = 0;
  for (const auto& arc : changed_arcs_) {
    int node, next;
    std::tie(node, next) = arc;
    const int node_index = committed_index_[node];
    const int next_index = committed_index_[next];
    const int node_path = committed_nodes_[node_index].path;
    if (next != node &&
        (next_index != node_index + 1 || node_path == -1)) {  // New arc.
      tail_head_indices_.push_back({node_index, next_index});
      changed_arcs_[num_changed_arcs++] = {node, next};
      if (node_path != -1 && !path_has_changed_[node_path]) {
        path_has_changed_[node_path] = true;
        changed_paths_.push_back(node_path);
      }
    } else if (node == next && node_path != -1) {  // New loop.
      changed_arcs_[num_changed_arcs++] = {node, node};
    }
  }
  changed_arcs_.resize(num_changed_arcs);

  if (tail_head_indices_.size() + changed_paths_.size() <= 8) {
    MakeChainsFromChangedPathsAndArcsWithSelectionAlgorithm();
  } else {
    MakeChainsFromChangedPathsAndArcsWithGenericAlgorithm();
  }
}

void PathState::Commit() {
  DCHECK(!IsInvalid());
  if (committed_nodes_.size() < num_nodes_threshold_) {
    IncrementalCommit();
  } else {
    FullCommit();
  }
}

void PathState::Revert() {
  is_invalid_ = false;
  chains_.resize(num_paths_ + 1);  // One per path + sentinel.
  for (const int path : changed_paths_) {
    paths_[path] = {path, path + 1};
    path_has_changed_[path] = false;
  }
  changed_paths_.clear();
  changed_arcs_.clear();
}

void PathState::CopyNewPathAtEndOfNodes(int path) {
  // Copy path's nodes, chain by chain.
  const int new_path_begin_index = committed_nodes_.size();
  const PathBounds path_bounds = paths_[path];
  for (int i = path_bounds.begin_index; i < path_bounds.end_index; ++i) {
    const ChainBounds chain_bounds = chains_[i];
    committed_nodes_.insert(committed_nodes_.end(),
                            committed_nodes_.data() + chain_bounds.begin_index,
                            committed_nodes_.data() + chain_bounds.end_index);
  }
  const int new_path_end_index = committed_nodes_.size();
  // Set new nodes' path member to path.
  for (int i = new_path_begin_index; i < new_path_end_index; ++i) {
    committed_nodes_[i].path = path;
  }
}

// TODO(user): Instead of copying paths at the end systematically,
// reuse some of the memory when possible.
void PathState::IncrementalCommit() {
  const int new_nodes_begin = committed_nodes_.size();
  for (const int path : ChangedPaths()) {
    const int chain_begin = committed_nodes_.size();
    CopyNewPathAtEndOfNodes(path);
    const int chain_end = committed_nodes_.size();
    chains_[path] = {chain_begin, chain_end};
  }
  // Re-index all copied nodes.
  const int new_nodes_end = committed_nodes_.size();
  for (int i = new_nodes_begin; i < new_nodes_end; ++i) {
    committed_index_[committed_nodes_[i].node] = i;
  }
  // New loops stay in place: only change their path to -1,
  // committed_index_ does not change.
  for (const auto& arc : ChangedArcs()) {
    int node, next;
    std::tie(node, next) = arc;
    if (node != next) continue;
    const int index = committed_index_[node];
    committed_nodes_[index].path = -1;
  }
  // Committed part of the state is set up, erase incremental changes.
  Revert();
}

void PathState::FullCommit() {
  // Copy all paths at the end of committed_nodes_,
  // then remove all old committed_nodes_.
  const int old_num_nodes = committed_nodes_.size();
  for (int path = 0; path < num_paths_; ++path) {
    const int new_path_begin = committed_nodes_.size() - old_num_nodes;
    CopyNewPathAtEndOfNodes(path);
    const int new_path_end = committed_nodes_.size() - old_num_nodes;
    chains_[path] = {new_path_begin, new_path_end};
  }
  committed_nodes_.erase(committed_nodes_.begin(),
                         committed_nodes_.begin() + old_num_nodes);

  // Reindex path nodes, then loop nodes.
  constexpr int kUnindexed = -1;
  committed_index_.assign(num_nodes_, kUnindexed);
  int index = 0;
  for (const CommittedNode committed_node : committed_nodes_) {
    committed_index_[committed_node.node] = index++;
  }
  for (int node = 0; node < num_nodes_; ++node) {
    if (committed_index_[node] != kUnindexed) continue;
    committed_index_[node] = index++;
    committed_nodes_.push_back({node, -1});
  }
  // Committed part of the state is set up, erase incremental changes.
  Revert();
}

namespace {

class PathStateFilter : public LocalSearchFilter {
 public:
  std::string DebugString() const override { return "PathStateFilter"; }
  PathStateFilter(std::unique_ptr<PathState> path_state,
                  const std::vector<IntVar*>& nexts);
  void Relax(const Assignment* delta, const Assignment* deltadelta) override;
  bool Accept(const Assignment* delta, const Assignment* deltadelta,
              int64 objective_min, int64 objective_max) override {
    return true;
  }
  void Synchronize(const Assignment* delta,
                   const Assignment* deltadelta) override{};
  void Commit(const Assignment* assignment, const Assignment* delta) override;
  void Revert() override;
  void Reset() override;

 private:
  const std::unique_ptr<PathState> path_state_;
  // Map IntVar* index to node, offset by the min index in nexts.
  std::vector<int> variable_index_to_node_;
  int index_offset_;
  // Used only in Reset(), this is a member variable to avoid reallocation.
  std::vector<bool> node_is_assigned_;
};

PathStateFilter::PathStateFilter(std::unique_ptr<PathState> path_state,
                                 const std::vector<IntVar*>& nexts)
    : path_state_(std::move(path_state)) {
  {
    int min_index = std::numeric_limits<int>::max();
    int max_index = std::numeric_limits<int>::min();
    for (const IntVar* next : nexts) {
      const int index = next->index();
      min_index = std::min<int>(min_index, index);
      max_index = std::max<int>(max_index, index);
    }
    variable_index_to_node_.resize(max_index - min_index + 1, -1);
    index_offset_ = min_index;
  }

  for (int node = 0; node < nexts.size(); ++node) {
    const int index = nexts[node]->index() - index_offset_;
    variable_index_to_node_[index] = node;
  }
}

void PathStateFilter::Relax(const Assignment* delta,
                            const Assignment* deltadelta) {
  path_state_->Revert();
  for (const IntVarElement& var_value : delta->IntVarContainer().elements()) {
    if (var_value.Var() == nullptr) continue;
    const int index = var_value.Var()->index() - index_offset_;
    if (index < 0 || variable_index_to_node_.size() <= index) continue;
    const int node = variable_index_to_node_[index];
    if (node == -1) continue;
    if (var_value.Bound()) {
      path_state_->ChangeNext(node, var_value.Value());
    } else {
      path_state_->Revert();
      path_state_->SetInvalid();
      break;
    }
  }
  path_state_->CutChains();
}

void PathStateFilter::Reset() {
  path_state_->Revert();
  // Set all paths of path state to empty start -> end paths,
  // and all nonstart/nonend nodes to node -> node loops.
  const int num_nodes = path_state_->NumNodes();
  node_is_assigned_.assign(num_nodes, false);
  const int num_paths = path_state_->NumPaths();
  for (int path = 0; path < num_paths; ++path) {
    const int start = path_state_->Start(path);
    const int end = path_state_->End(path);
    path_state_->ChangeNext(start, end);
    node_is_assigned_[start] = true;
    node_is_assigned_[end] = true;
  }
  for (int node = 0; node < num_nodes; ++node) {
    if (!node_is_assigned_[node]) path_state_->ChangeNext(node, node);
  }
  path_state_->CutChains();
  path_state_->Commit();
}

// The solver does not guarantee that a given Commit() corresponds to
// the previous Relax() (or that there has been a call to Relax()),
// so we replay the full change call sequence.
void PathStateFilter::Commit(const Assignment* assignment,
                             const Assignment* delta) {
  path_state_->Revert();
  if (delta == nullptr || delta->Empty()) {
    Relax(assignment, nullptr);
  } else {
    Relax(delta, nullptr);
  }
  path_state_->Commit();
}

void PathStateFilter::Revert() { path_state_->Revert(); }

}  // namespace

LocalSearchFilter* MakePathStateFilter(Solver* solver,
                                       std::unique_ptr<PathState> path_state,
                                       const std::vector<IntVar*>& nexts) {
  PathStateFilter* filter = new PathStateFilter(std::move(path_state), nexts);
  return solver->RevAlloc(filter);
}

UnaryDimensionChecker::UnaryDimensionChecker(
    const PathState* path_state, std::vector<Interval> path_capacity,
    std::vector<int> path_class, std::vector<std::vector<Interval>> demand,
    std::vector<Interval> node_capacity)
    : path_state_(path_state),
      path_capacity_(std::move(path_capacity)),
      path_class_(std::move(path_class)),
      demand_(std::move(demand)),
      node_capacity_(std::move(node_capacity)),
      index_(path_state_->NumNodes(), 0),
      maximum_partial_demand_layer_size_(
          std::max(16, 4 * path_state_->NumNodes()))  // 16 and 4 are arbitrary.
{
  const int num_nodes = path_state_->NumNodes();
  const int num_paths = path_state_->NumPaths();
  DCHECK_EQ(num_paths, path_capacity_.size());
  DCHECK_EQ(num_paths, path_class_.size());
  const int maximum_rmq_exponent = MostSignificantBitPosition32(num_nodes);
  partial_demand_sums_rmq_.resize(maximum_rmq_exponent + 1);
  previous_nontrivial_index_.reserve(maximum_partial_demand_layer_size_);
  FullCommit();
}

bool UnaryDimensionChecker::Check() const {
  if (path_state_->IsInvalid()) return true;
  for (const int path : path_state_->ChangedPaths()) {
    const Interval path_capacity = path_capacity_[path];
    if (path_capacity.min == kint64min && path_capacity.max == kint64max) {
      continue;
    }
    const int path_class = path_class_[path];
    // Loop invariant: capacity_used is nonempty and within path_capacity.
    Interval capacity_used = node_capacity_[path_state_->Start(path)];
    capacity_used = {std::max(capacity_used.min, path_capacity.min),
                     std::min(capacity_used.max, path_capacity.max)};
    if (capacity_used.min > capacity_used.max) return false;

    for (const auto chain : path_state_->Chains(path)) {
      const int first_node = chain.First();
      const int last_node = chain.Last();

      const int first_index = index_[first_node];
      const int last_index = index_[last_node];

      const int chain_path = path_state_->Path(first_node);
      const int chain_path_class =
          chain_path == -1 ? -1 : path_class_[chain_path];
      // Call the RMQ if the chain size is large enough;
      // the optimal value was found with the associated benchmark in tests,
      // in particular BM_UnaryDimensionChecker<ChangeSparsity::kSparse, *>.
      constexpr int kMinRangeSizeForRMQ = 4;
      if (last_index - first_index > kMinRangeSizeForRMQ &&
          path_class == chain_path_class &&
          SubpathOnlyHasTrivialNodes(first_index, last_index)) {
        // Compute feasible values of capacity_used that will not violate
        // path_capacity. This is done by considering the worst cases
        // using a range min/max query.
        const Interval min_max =
            GetMinMaxPartialDemandSum(first_index, last_index);
        const Interval prev_sum = partial_demand_sums_rmq_[0][first_index - 1];
        const Interval min_max_delta = {CapSub(min_max.min, prev_sum.min),
                                        CapSub(min_max.max, prev_sum.max)};
        capacity_used = {
            std::max(capacity_used.min,
                     CapSub(path_capacity.min, min_max_delta.min)),
            std::min(capacity_used.max,
                     CapSub(path_capacity.max, min_max_delta.max))};
        if (capacity_used.min > capacity_used.max) return false;
        // Move to last node's state, which is valid since we did not return.
        const Interval last_sum = partial_demand_sums_rmq_[0][last_index];
        capacity_used = {
            CapSub(CapAdd(capacity_used.min, last_sum.min), prev_sum.min),
            CapSub(CapAdd(capacity_used.max, last_sum.max), prev_sum.max)};
      } else {
        for (const int node : chain) {
          const Interval node_capacity = node_capacity_[node];
          capacity_used = {std::max(capacity_used.min, node_capacity.min),
                           std::min(capacity_used.max, node_capacity.max)};
          if (capacity_used.min > capacity_used.max) return false;
          const Interval demand = demand_[path_class][node];
          capacity_used = {CapAdd(capacity_used.min, demand.min),
                           CapAdd(capacity_used.max, demand.max)};
          capacity_used = {std::max(capacity_used.min, path_capacity.min),
                           std::min(capacity_used.max, path_capacity.max)};
        }
      }
    }
    if (std::max(capacity_used.min, path_capacity.min) >
        std::min(capacity_used.max, path_capacity.max)) {
      return false;
    }
  }
  return true;
}

void UnaryDimensionChecker::Commit() {
  const int current_layer_size = partial_demand_sums_rmq_[0].size();
  int change_size = path_state_->ChangedPaths().size();
  for (const int path : path_state_->ChangedPaths()) {
    for (const auto chain : path_state_->Chains(path)) {
      change_size += chain.NumNodes();
    }
  }
  if (current_layer_size + change_size <= maximum_partial_demand_layer_size_) {
    IncrementalCommit();
  } else {
    FullCommit();
  }
}

void UnaryDimensionChecker::IncrementalCommit() {
  for (const int path : path_state_->ChangedPaths()) {
    const int begin_index = partial_demand_sums_rmq_[0].size();
    AppendPathDemandsToSums(path);
    UpdateRMQStructure(begin_index, partial_demand_sums_rmq_[0].size());
  }
}

void UnaryDimensionChecker::FullCommit() {
  // Clear all structures.
  previous_nontrivial_index_.clear();
  for (auto& sums : partial_demand_sums_rmq_) sums.clear();
  // Append all paths.
  const int num_paths = path_state_->NumPaths();
  for (int path = 0; path < num_paths; ++path) {
    const int begin_index = partial_demand_sums_rmq_[0].size();
    AppendPathDemandsToSums(path);
    UpdateRMQStructure(begin_index, partial_demand_sums_rmq_[0].size());
  }
}

void UnaryDimensionChecker::AppendPathDemandsToSums(int path) {
  const int path_class = path_class_[path];
  Interval demand_sum = {0, 0};
  int previous_nontrivial_index = -1;
  int index = partial_demand_sums_rmq_[0].size();
  // Value of partial_demand_sums_rmq_ at node_index-1 must be the sum
  // of all demands of nodes before node.
  partial_demand_sums_rmq_[0].push_back(demand_sum);
  previous_nontrivial_index_.push_back(-1);
  ++index;

  for (const int node : path_state_->Nodes(path)) {
    index_[node] = index;
    const Interval demand = demand_[path_class][node];
    demand_sum = {CapAdd(demand_sum.min, demand.min),
                  CapAdd(demand_sum.max, demand.max)};
    partial_demand_sums_rmq_[0].push_back(demand_sum);

    const Interval node_capacity = node_capacity_[node];
    if (demand.min != demand.max || node_capacity.min != kint64min ||
        node_capacity.max != kint64max) {
      previous_nontrivial_index = index;
    }
    previous_nontrivial_index_.push_back(previous_nontrivial_index);
    ++index;
  }
}

void UnaryDimensionChecker::UpdateRMQStructure(int begin_index, int end_index) {
  // The max layer is the one used by
  // GetMinMaxPartialDemandSum(begin_index, end_index - 1).
  const int maximum_rmq_exponent =
      MostSignificantBitPosition32(end_index - 1 - begin_index);
  for (int layer = 1, window_size = 1; layer <= maximum_rmq_exponent;
       ++layer, window_size *= 2) {
    partial_demand_sums_rmq_[layer].resize(end_index);
    for (int i = begin_index; i < end_index - window_size; ++i) {
      const Interval i1 = partial_demand_sums_rmq_[layer - 1][i];
      const Interval i2 = partial_demand_sums_rmq_[layer - 1][i + window_size];
      partial_demand_sums_rmq_[layer][i] = {std::min(i1.min, i2.min),
                                            std::max(i1.max, i2.max)};
    }
    std::copy(
        partial_demand_sums_rmq_[layer - 1].begin() + end_index - window_size,
        partial_demand_sums_rmq_[layer - 1].begin() + end_index,
        partial_demand_sums_rmq_[layer].begin() + end_index - window_size);
  }
}

// TODO(user): since this is called only when
// last_node_index - first_node_index is large enough,
// the lower layers of partial_demand_sums_rmq_ are never used.
// For instance, if this is only called when the range size is > 4
// and paths are <= 32 nodes long, then we only need layers 0, 2, 3, and 4.
// To compare, on a 512 < #nodes <= 1024 problem, this uses layers in [0, 10].
UnaryDimensionChecker::Interval
UnaryDimensionChecker::GetMinMaxPartialDemandSum(int first_node_index,
                                                 int last_node_index) const {
  DCHECK_LE(0, first_node_index);
  DCHECK_LT(first_node_index, last_node_index);
  DCHECK_LT(last_node_index, partial_demand_sums_rmq_[0].size());
  // Find largest window_size = 2^layer such that
  // first_node_index < last_node_index - window_size + 1.
  const int layer =
      MostSignificantBitPosition32(last_node_index - first_node_index);
  const int window_size = 1 << layer;
  // Classical range min/max query in O(1).
  const Interval i1 = partial_demand_sums_rmq_[layer][first_node_index];
  const Interval i2 =
      partial_demand_sums_rmq_[layer][last_node_index - window_size + 1];
  return {std::min(i1.min, i2.min), std::max(i1.max, i2.max)};
}

bool UnaryDimensionChecker::SubpathOnlyHasTrivialNodes(
    int first_node_index, int last_node_index) const {
  DCHECK_LE(0, first_node_index);
  DCHECK_LT(first_node_index, last_node_index);
  DCHECK_LT(last_node_index, previous_nontrivial_index_.size());
  return first_node_index > previous_nontrivial_index_[last_node_index];
}

namespace {

class UnaryDimensionFilter : public LocalSearchFilter {
 public:
  std::string DebugString() const override { return "UnaryDimensionFilter"; }
  explicit UnaryDimensionFilter(std::unique_ptr<UnaryDimensionChecker> checker)
      : checker_(std::move(checker)) {}

  bool Accept(const Assignment* delta, const Assignment* deltadelta,
              int64 objective_min, int64 objective_max) override {
    return checker_->Check();
  }

  void Synchronize(const Assignment* assignment,
                   const Assignment* delta) override {
    checker_->Commit();
  }

 private:
  std::unique_ptr<UnaryDimensionChecker> checker_;
};

}  // namespace

LocalSearchFilter* MakeUnaryDimensionFilter(
    Solver* solver, std::unique_ptr<UnaryDimensionChecker> checker) {
  UnaryDimensionFilter* filter = new UnaryDimensionFilter(std::move(checker));
  return solver->RevAlloc(filter);
}

namespace {
// ----- Variable domain filter -----
// Rejects assignments to values outside the domain of variables

class VariableDomainFilter : public LocalSearchFilter {
 public:
  VariableDomainFilter() {}
  ~VariableDomainFilter() override {}
  bool Accept(const Assignment* delta, const Assignment* deltadelta,
              int64 objective_min, int64 objective_max) override;
  void Synchronize(const Assignment* assignment,
                   const Assignment* delta) override {}

  std::string DebugString() const override { return "VariableDomainFilter"; }
};

bool VariableDomainFilter::Accept(const Assignment* delta,
                                  const Assignment* deltadelta,
                                  int64 objective_min, int64 objective_max) {
  const Assignment::IntContainer& container = delta->IntVarContainer();
  const int size = container.Size();
  for (int i = 0; i < size; ++i) {
    const IntVarElement& element = container.Element(i);
    if (element.Activated() && !element.Var()->Contains(element.Value())) {
      return false;
    }
  }
  return true;
}
}  // namespace

LocalSearchFilter* Solver::MakeVariableDomainFilter() {
  return RevAlloc(new VariableDomainFilter());
}

// ----- IntVarLocalSearchFilter -----

const int IntVarLocalSearchFilter::kUnassigned = -1;

IntVarLocalSearchFilter::IntVarLocalSearchFilter(
    const std::vector<IntVar*>& vars) {
  AddVars(vars);
}

void IntVarLocalSearchFilter::AddVars(const std::vector<IntVar*>& vars) {
  if (!vars.empty()) {
    for (int i = 0; i < vars.size(); ++i) {
      const int index = vars[i]->index();
      if (index >= var_index_to_index_.size()) {
        var_index_to_index_.resize(index + 1, kUnassigned);
      }
      var_index_to_index_[index] = i + vars_.size();
    }
    vars_.insert(vars_.end(), vars.begin(), vars.end());
    values_.resize(vars_.size(), /*junk*/ 0);
    var_synced_.resize(vars_.size(), false);
  }
}

IntVarLocalSearchFilter::~IntVarLocalSearchFilter() {}

void IntVarLocalSearchFilter::Synchronize(const Assignment* assignment,
                                          const Assignment* delta) {
  if (delta == nullptr || delta->Empty()) {
    var_synced_.assign(var_synced_.size(), false);
    SynchronizeOnAssignment(assignment);
  } else {
    SynchronizeOnAssignment(delta);
  }
  OnSynchronize(delta);
}

void IntVarLocalSearchFilter::SynchronizeOnAssignment(
    const Assignment* assignment) {
  const Assignment::IntContainer& container = assignment->IntVarContainer();
  const int size = container.Size();
  for (int i = 0; i < size; ++i) {
    const IntVarElement& element = container.Element(i);
    IntVar* const var = element.Var();
    if (var != nullptr) {
      if (i < vars_.size() && vars_[i] == var) {
        values_[i] = element.Value();
        var_synced_[i] = true;
      } else {
        const int64 kUnallocated = -1;
        int64 index = kUnallocated;
        if (FindIndex(var, &index)) {
          values_[index] = element.Value();
          var_synced_[index] = true;
        }
      }
    }
  }
}

// ----- Sum Objective filter ------
// Maintains the sum of costs of variables, where the subclass implements
// CostOfSynchronizedVariable() and FillCostOfBoundDeltaVariable() to compute
// the cost of a variable depending on its value.
// An assignment is accepted by this filter if the total cost is allowed
// depending on the relation defined by filter_enum:
// - Solver::LE -> total_cost <= objective_max.
// - Solver::GE -> total_cost >= objective_min.
// - Solver::EQ -> the conjunction of LE and GE.
namespace {
class SumObjectiveFilter : public IntVarLocalSearchFilter {
 public:
  SumObjectiveFilter(const std::vector<IntVar*>& vars,
                     Solver::LocalSearchFilterBound filter_enum)
      : IntVarLocalSearchFilter(vars),
        primary_vars_size_(vars.size()),
        synchronized_costs_(new int64[vars.size()]),
        delta_costs_(new int64[vars.size()]),
        filter_enum_(filter_enum),
        synchronized_sum_(kint64min),
        delta_sum_(kint64min),
        incremental_(false) {
    for (int i = 0; i < vars.size(); ++i) {
      synchronized_costs_[i] = 0;
      delta_costs_[i] = 0;
    }
  }
  ~SumObjectiveFilter() override {
    delete[] synchronized_costs_;
    delete[] delta_costs_;
  }
  bool Accept(const Assignment* delta, const Assignment* deltadelta,
              int64 objective_min, int64 objective_max) override {
    if (delta == nullptr) {
      return false;
    }
    if (deltadelta->Empty()) {
      if (incremental_) {
        for (int i = 0; i < primary_vars_size_; ++i) {
          delta_costs_[i] = synchronized_costs_[i];
        }
        delta_sum_ = synchronized_sum_;
      }
      incremental_ = false;
      delta_sum_ = CapAdd(synchronized_sum_,
                          CostOfChanges(delta, synchronized_costs_, false));
    } else {
      if (incremental_) {
        delta_sum_ =
            CapAdd(delta_sum_, CostOfChanges(deltadelta, delta_costs_, true));
      } else {
        delta_sum_ = CapAdd(synchronized_sum_,
                            CostOfChanges(delta, synchronized_costs_, true));
      }
      incremental_ = true;
    }
    switch (filter_enum_) {
      case Solver::LE: {
        return delta_sum_ <= objective_max;
      }
      case Solver::GE: {
        return delta_sum_ >= objective_min;
      }
      case Solver::EQ: {
        return objective_min <= delta_sum_ && delta_sum_ <= objective_max;
      }
      default: {
        LOG(ERROR) << "Unknown local search filter enum value";
        return false;
      }
    }
  }
  // If the variable is synchronized, returns its associated cost, otherwise
  // returns 0.
  virtual int64 CostOfSynchronizedVariable(int64 index) = 0;
  // If the variable is bound, fills new_cost with the cost associated to the
  // variable's valuation in container, and returns true. Otherwise, fills
  // new_cost with 0, and returns false.
  virtual bool FillCostOfBoundDeltaVariable(
      const Assignment::IntContainer& container, int index,
      int* container_index, int64* new_cost) = 0;
  bool IsIncremental() const override { return true; }

  std::string DebugString() const override { return "SumObjectiveFilter"; }

  int64 GetSynchronizedObjectiveValue() const override {
    return synchronized_sum_;
  }
  int64 GetAcceptedObjectiveValue() const override { return delta_sum_; }

 protected:
  const int primary_vars_size_;
  int64* const synchronized_costs_;
  int64* const delta_costs_;
  Solver::LocalSearchFilterBound filter_enum_;
  int64 synchronized_sum_;
  int64 delta_sum_;
  bool incremental_;

 private:
  void OnSynchronize(const Assignment* delta) override {
    synchronized_sum_ = 0;
    for (int i = 0; i < primary_vars_size_; ++i) {
      const int64 cost = CostOfSynchronizedVariable(i);
      synchronized_costs_[i] = cost;
      delta_costs_[i] = cost;
      synchronized_sum_ = CapAdd(synchronized_sum_, cost);
    }
    delta_sum_ = synchronized_sum_;
    incremental_ = false;
  }
  int64 CostOfChanges(const Assignment* changes, const int64* const old_costs,
                      bool cache_delta_values) {
    int64 total_cost = 0;
    const Assignment::IntContainer& container = changes->IntVarContainer();
    const int size = container.Size();
    for (int i = 0; i < size; ++i) {
      const IntVarElement& new_element = container.Element(i);
      IntVar* const var = new_element.Var();
      int64 index = -1;
      if (FindIndex(var, &index) && index < primary_vars_size_) {
        total_cost = CapSub(total_cost, old_costs[index]);
        int64 new_cost = 0LL;
        if (FillCostOfBoundDeltaVariable(container, index, &i, &new_cost)) {
          total_cost = CapAdd(total_cost, new_cost);
        }
        if (cache_delta_values) {
          delta_costs_[index] = new_cost;
        }
      }
    }
    return total_cost;
  }
};

class BinaryObjectiveFilter : public SumObjectiveFilter {
 public:
  BinaryObjectiveFilter(const std::vector<IntVar*>& vars,
                        Solver::IndexEvaluator2 value_evaluator,
                        Solver::LocalSearchFilterBound filter_enum)
      : SumObjectiveFilter(vars, filter_enum),
        value_evaluator_(std::move(value_evaluator)) {}
  ~BinaryObjectiveFilter() override {}
  int64 CostOfSynchronizedVariable(int64 index) override {
    return IntVarLocalSearchFilter::IsVarSynced(index)
               ? value_evaluator_(index, IntVarLocalSearchFilter::Value(index))
               : 0;
  }
  bool FillCostOfBoundDeltaVariable(const Assignment::IntContainer& container,
                                    int index, int* container_index,
                                    int64* new_cost) override {
    const IntVarElement& element = container.Element(*container_index);
    if (element.Activated()) {
      *new_cost = value_evaluator_(index, element.Value());
      return true;
    }
    const IntVar* var = element.Var();
    if (var->Bound()) {
      *new_cost = value_evaluator_(index, var->Min());
      return true;
    }
    *new_cost = 0;
    return false;
  }

 private:
  Solver::IndexEvaluator2 value_evaluator_;
};

class TernaryObjectiveFilter : public SumObjectiveFilter {
 public:
  TernaryObjectiveFilter(const std::vector<IntVar*>& vars,
                         const std::vector<IntVar*>& secondary_vars,
                         Solver::IndexEvaluator3 value_evaluator,
                         Solver::LocalSearchFilterBound filter_enum)
      : SumObjectiveFilter(vars, filter_enum),
        secondary_vars_offset_(vars.size()),
        value_evaluator_(std::move(value_evaluator)) {
    IntVarLocalSearchFilter::AddVars(secondary_vars);
    CHECK_GE(IntVarLocalSearchFilter::Size(), 0);
  }
  ~TernaryObjectiveFilter() override {}
  int64 CostOfSynchronizedVariable(int64 index) override {
    DCHECK_LT(index, secondary_vars_offset_);
    return IntVarLocalSearchFilter::IsVarSynced(index)
               ? value_evaluator_(index, IntVarLocalSearchFilter::Value(index),
                                  IntVarLocalSearchFilter::Value(
                                      index + secondary_vars_offset_))
               : 0;
  }
  bool FillCostOfBoundDeltaVariable(const Assignment::IntContainer& container,
                                    int index, int* container_index,
                                    int64* new_cost) override {
    DCHECK_LT(index, secondary_vars_offset_);
    *new_cost = 0LL;
    const IntVarElement& element = container.Element(*container_index);
    const IntVar* secondary_var =
        IntVarLocalSearchFilter::Var(index + secondary_vars_offset_);
    if (element.Activated()) {
      const int64 value = element.Value();
      int hint_index = *container_index + 1;
      if (hint_index < container.Size() &&
          secondary_var == container.Element(hint_index).Var()) {
        *new_cost = value_evaluator_(index, value,
                                     container.Element(hint_index).Value());
        *container_index = hint_index;
      } else {
        *new_cost = value_evaluator_(index, value,
                                     container.Element(secondary_var).Value());
      }
      return true;
    }
    const IntVar* var = element.Var();
    if (var->Bound() && secondary_var->Bound()) {
      *new_cost = value_evaluator_(index, var->Min(), secondary_var->Min());
      return true;
    }
    *new_cost = 0;
    return false;
  }

 private:
  int secondary_vars_offset_;
  Solver::IndexEvaluator3 value_evaluator_;
};
}  // namespace

IntVarLocalSearchFilter* Solver::MakeSumObjectiveFilter(
    const std::vector<IntVar*>& vars, Solver::IndexEvaluator2 values,
    Solver::LocalSearchFilterBound filter_enum) {
  return RevAlloc(
      new BinaryObjectiveFilter(vars, std::move(values), filter_enum));
}

IntVarLocalSearchFilter* Solver::MakeSumObjectiveFilter(
    const std::vector<IntVar*>& vars,
    const std::vector<IntVar*>& secondary_vars, Solver::IndexEvaluator3 values,
    Solver::LocalSearchFilterBound filter_enum) {
  return RevAlloc(new TernaryObjectiveFilter(vars, secondary_vars,
                                             std::move(values), filter_enum));
}

LocalSearchVariable LocalSearchState::AddVariable(int64 initial_min,
                                                  int64 initial_max) {
  DCHECK(state_is_valid_);
  DCHECK_LE(initial_min, initial_max);
  initial_variable_bounds_.push_back({initial_min, initial_max});
  variable_bounds_.push_back({initial_min, initial_max});
  variable_is_relaxed_.push_back(false);

  const int variable_index = variable_bounds_.size() - 1;
  return {this, variable_index};
}

void LocalSearchState::RelaxVariableBounds(int variable_index) {
  DCHECK(state_is_valid_);
  DCHECK(0 <= variable_index && variable_index < variable_is_relaxed_.size());
  if (!variable_is_relaxed_[variable_index]) {
    variable_is_relaxed_[variable_index] = true;
    saved_variable_bounds_trail_.emplace_back(variable_bounds_[variable_index],
                                              variable_index);
    variable_bounds_[variable_index] = initial_variable_bounds_[variable_index];
  }
}

int64 LocalSearchState::VariableMin(int variable_index) const {
  DCHECK(state_is_valid_);
  DCHECK(0 <= variable_index && variable_index < variable_bounds_.size());
  return variable_bounds_[variable_index].min;
}

int64 LocalSearchState::VariableMax(int variable_index) const {
  DCHECK(state_is_valid_);
  DCHECK(0 <= variable_index && variable_index < variable_bounds_.size());
  return variable_bounds_[variable_index].max;
}

bool LocalSearchState::TightenVariableMin(int variable_index, int64 min_value) {
  DCHECK(state_is_valid_);
  DCHECK(variable_is_relaxed_[variable_index]);
  DCHECK(0 <= variable_index && variable_index < variable_bounds_.size());
  Bounds& bounds = variable_bounds_[variable_index];
  if (bounds.max < min_value) {
    state_is_valid_ = false;
  }
  bounds.min = std::max(bounds.min, min_value);
  return state_is_valid_;
}

bool LocalSearchState::TightenVariableMax(int variable_index, int64 max_value) {
  DCHECK(state_is_valid_);
  DCHECK(variable_is_relaxed_[variable_index]);
  DCHECK(0 <= variable_index && variable_index < variable_bounds_.size());
  Bounds& bounds = variable_bounds_[variable_index];
  if (bounds.min > max_value) {
    state_is_valid_ = false;
  }
  bounds.max = std::min(bounds.max, max_value);
  return state_is_valid_;
}

// TODO(user): When the class has more users, find a threshold ratio of
// saved/total variables under which a sparse clear would be more efficient
// for both Commit() and Revert().
void LocalSearchState::Commit() {
  DCHECK(state_is_valid_);
  saved_variable_bounds_trail_.clear();
  variable_is_relaxed_.assign(variable_is_relaxed_.size(), false);
}

void LocalSearchState::Revert() {
  for (const auto& bounds_index : saved_variable_bounds_trail_) {
    DCHECK(variable_is_relaxed_[bounds_index.second]);
    variable_bounds_[bounds_index.second] = bounds_index.first;
  }
  saved_variable_bounds_trail_.clear();
  variable_is_relaxed_.assign(variable_is_relaxed_.size(), false);
  state_is_valid_ = true;
}

// ----- LocalSearchProfiler -----

class LocalSearchProfiler : public LocalSearchMonitor {
 public:
  explicit LocalSearchProfiler(Solver* solver) : LocalSearchMonitor(solver) {}
  std::string DebugString() const override { return "LocalSearchProfiler"; }
  void RestartSearch() override {
    operator_stats_.clear();
    filter_stats_.clear();
  }
  void ExitSearch() override {
    // Update times for current operator when the search ends.
    if (solver()->TopLevelSearch() == solver()->ActiveSearch()) {
      UpdateTime();
    }
  }
  LocalSearchStatistics ExportToLocalSearchStatistics() const {
    LocalSearchStatistics statistics_proto;
    std::vector<const LocalSearchOperator*> operators;
    for (const auto& stat : operator_stats_) {
      operators.push_back(stat.first);
    }
    std::sort(
        operators.begin(), operators.end(),
        [this](const LocalSearchOperator* op1, const LocalSearchOperator* op2) {
          return gtl::FindOrDie(operator_stats_, op1).neighbors >
                 gtl::FindOrDie(operator_stats_, op2).neighbors;
        });
    for (const LocalSearchOperator* const op : operators) {
      const OperatorStats& stats = gtl::FindOrDie(operator_stats_, op);
      LocalSearchStatistics::LocalSearchOperatorStatistics* const
          local_search_operator_statistics =
              statistics_proto.add_local_search_operator_statistics();
      local_search_operator_statistics->set_local_search_operator(
          op->DebugString());
      local_search_operator_statistics->set_num_neighbors(stats.neighbors);
      local_search_operator_statistics->set_num_filtered_neighbors(
          stats.filtered_neighbors);
      local_search_operator_statistics->set_num_accepted_neighbors(
          stats.accepted_neighbors);
      local_search_operator_statistics->set_duration_seconds(stats.seconds);
    }
    std::vector<const LocalSearchFilter*> filters;
    for (const auto& stat : filter_stats_) {
      filters.push_back(stat.first);
    }
    std::sort(filters.begin(), filters.end(),
              [this](const LocalSearchFilter* filter1,
                     const LocalSearchFilter* filter2) {
                return gtl::FindOrDie(filter_stats_, filter1).calls >
                       gtl::FindOrDie(filter_stats_, filter2).calls;
              });
    for (const LocalSearchFilter* const filter : filters) {
      const FilterStats& stats = gtl::FindOrDie(filter_stats_, filter);
      LocalSearchStatistics::LocalSearchFilterStatistics* const
          local_search_filter_statistics =
              statistics_proto.add_local_search_filter_statistics();
      local_search_filter_statistics->set_local_search_filter(
          filter->DebugString());
      local_search_filter_statistics->set_num_calls(stats.calls);
      local_search_filter_statistics->set_num_rejects(stats.rejects);
      local_search_filter_statistics->set_duration_seconds(stats.seconds);
    }
    statistics_proto.set_total_num_neighbors(solver()->neighbors());
    statistics_proto.set_total_num_filtered_neighbors(
        solver()->filtered_neighbors());
    statistics_proto.set_total_num_accepted_neighbors(
        solver()->accepted_neighbors());
    return statistics_proto;
  }
  std::string PrintOverview() const {
    size_t max_name_size = 0;
    std::vector<const LocalSearchOperator*> operators;
    for (const auto& stat : operator_stats_) {
      operators.push_back(stat.first);
      max_name_size =
          std::max(max_name_size, stat.first->DebugString().length());
    }
    std::sort(
        operators.begin(), operators.end(),
        [this](const LocalSearchOperator* op1, const LocalSearchOperator* op2) {
          return gtl::FindOrDie(operator_stats_, op1).neighbors >
                 gtl::FindOrDie(operator_stats_, op2).neighbors;
        });
    std::string overview = "Local search operator statistics:\n";
    absl::StrAppendFormat(&overview,
                          "%*s | Neighbors | Filtered | Accepted | Time (s)\n",
                          max_name_size, "");
    OperatorStats total_stats;
    for (const LocalSearchOperator* const op : operators) {
      const OperatorStats& stats = gtl::FindOrDie(operator_stats_, op);
      const std::string& name = op->DebugString();
      absl::StrAppendFormat(&overview, "%*s | %9ld | %8ld | %8ld | %7.2g\n",
                            max_name_size, name, stats.neighbors,
                            stats.filtered_neighbors, stats.accepted_neighbors,
                            stats.seconds);
      total_stats.neighbors += stats.neighbors;
      total_stats.filtered_neighbors += stats.filtered_neighbors;
      total_stats.accepted_neighbors += stats.accepted_neighbors;
      total_stats.seconds += stats.seconds;
    }
    absl::StrAppendFormat(&overview, "%*s | %9ld | %8ld | %8ld | %7.2g\n",
                          max_name_size, "Total", total_stats.neighbors,
                          total_stats.filtered_neighbors,
                          total_stats.accepted_neighbors, total_stats.seconds);
    max_name_size = 0;
    std::vector<const LocalSearchFilter*> filters;
    for (const auto& stat : filter_stats_) {
      filters.push_back(stat.first);
      max_name_size =
          std::max(max_name_size, stat.first->DebugString().length());
    }
    std::sort(filters.begin(), filters.end(),
              [this](const LocalSearchFilter* filter1,
                     const LocalSearchFilter* filter2) {
                return gtl::FindOrDie(filter_stats_, filter1).calls >
                       gtl::FindOrDie(filter_stats_, filter2).calls;
              });
    absl::StrAppendFormat(&overview,
                          "Local search filter statistics:\n%*s |     Calls |  "
                          " Rejects | Time (s) "
                          "| Rejects/s\n",
                          max_name_size, "");
    FilterStats total_filter_stats;
    for (const LocalSearchFilter* const filter : filters) {
      const FilterStats& stats = gtl::FindOrDie(filter_stats_, filter);
      const std::string& name = filter->DebugString();
      absl::StrAppendFormat(&overview, "%*s | %9ld | %9ld | %7.2g  | %7.2g\n",
                            max_name_size, name, stats.calls, stats.rejects,
                            stats.seconds, stats.rejects / stats.seconds);
      total_filter_stats.calls += stats.calls;
      total_filter_stats.rejects += stats.rejects;
      total_filter_stats.seconds += stats.seconds;
    }
    absl::StrAppendFormat(
        &overview, "%*s | %9ld | %9ld | %7.2g  | %7.2g\n", max_name_size,
        "Total", total_filter_stats.calls, total_filter_stats.rejects,
        total_filter_stats.seconds,
        total_filter_stats.rejects / total_filter_stats.seconds);
    return overview;
  }
  void BeginOperatorStart() override {}
  void EndOperatorStart() override {}
  void BeginMakeNextNeighbor(const LocalSearchOperator* op) override {
    if (last_operator_ != op->Self()) {
      UpdateTime();
      last_operator_ = op->Self();
    }
  }
  void EndMakeNextNeighbor(const LocalSearchOperator* op, bool neighbor_found,
                           const Assignment* delta,
                           const Assignment* deltadelta) override {
    if (neighbor_found) {
      operator_stats_[op->Self()].neighbors++;
    }
  }
  void BeginFilterNeighbor(const LocalSearchOperator* op) override {}
  void EndFilterNeighbor(const LocalSearchOperator* op,
                         bool neighbor_found) override {
    if (neighbor_found) {
      operator_stats_[op->Self()].filtered_neighbors++;
    }
  }
  void BeginAcceptNeighbor(const LocalSearchOperator* op) override {}
  void EndAcceptNeighbor(const LocalSearchOperator* op,
                         bool neighbor_found) override {
    if (neighbor_found) {
      operator_stats_[op->Self()].accepted_neighbors++;
    }
  }
  void BeginFiltering(const LocalSearchFilter* filter) override {
    filter_stats_[filter].calls++;
    filter_timer_.Start();
  }
  void EndFiltering(const LocalSearchFilter* filter, bool reject) override {
    filter_timer_.Stop();
    auto& stats = filter_stats_[filter];
    stats.seconds += filter_timer_.Get();
    if (reject) {
      stats.rejects++;
    }
  }
  void Install() override { SearchMonitor::Install(); }

 private:
  void UpdateTime() {
    if (last_operator_ != nullptr) {
      timer_.Stop();
      operator_stats_[last_operator_].seconds += timer_.Get();
    }
    timer_.Start();
  }

  struct OperatorStats {
    int64 neighbors = 0;
    int64 filtered_neighbors = 0;
    int64 accepted_neighbors = 0;
    double seconds = 0;
  };

  struct FilterStats {
    int64 calls = 0;
    int64 rejects = 0;
    double seconds = 0;
  };
  WallTimer timer_;
  WallTimer filter_timer_;
  const LocalSearchOperator* last_operator_ = nullptr;
  absl::flat_hash_map<const LocalSearchOperator*, OperatorStats>
      operator_stats_;
  absl::flat_hash_map<const LocalSearchFilter*, FilterStats> filter_stats_;
};

void InstallLocalSearchProfiler(LocalSearchProfiler* monitor) {
  monitor->Install();
}

LocalSearchProfiler* BuildLocalSearchProfiler(Solver* solver) {
  if (solver->IsLocalSearchProfilingEnabled()) {
    return new LocalSearchProfiler(solver);
  }
  return nullptr;
}

void DeleteLocalSearchProfiler(LocalSearchProfiler* monitor) { delete monitor; }

std::string Solver::LocalSearchProfile() const {
  if (local_search_profiler_ != nullptr) {
    return local_search_profiler_->PrintOverview();
  }
  return "";
}

LocalSearchStatistics Solver::GetLocalSearchStatistics() const {
  if (local_search_profiler_ != nullptr) {
    return local_search_profiler_->ExportToLocalSearchStatistics();
  }
  return LocalSearchStatistics();
}

void LocalSearchFilterManager::InitializeForcedEvents() {
  const int num_events = filter_events_.size();
  int next_forced_event = num_events;
  next_forced_events_.resize(num_events);
  for (int i = num_events - 1; i >= 0; --i) {
    next_forced_events_[i] = next_forced_event;
    if (filter_events_[i].filter->IsIncremental() ||
        (filter_events_[i].event_type == FilterEventType::kRelax &&
         next_forced_event != num_events)) {
      next_forced_event = i;
    }
  }
}

LocalSearchFilterManager::LocalSearchFilterManager(
    std::vector<LocalSearchFilter*> filters)
    : synchronized_value_(kint64min), accepted_value_(kint64min) {
  filter_events_.reserve(2 * filters.size());
  for (LocalSearchFilter* filter : filters) {
    filter_events_.push_back({filter, FilterEventType::kRelax});
  }
  for (LocalSearchFilter* filter : filters) {
    filter_events_.push_back({filter, FilterEventType::kAccept});
  }
  InitializeForcedEvents();
}

LocalSearchFilterManager::LocalSearchFilterManager(
    std::vector<FilterEvent> filter_events)
    : filter_events_(std::move(filter_events)),
      synchronized_value_(kint64min),
      accepted_value_(kint64min) {
  InitializeForcedEvents();
}

// Filters' Revert() must be called in the reverse order in which their
// Relax() was called.
void LocalSearchFilterManager::Revert() {
  for (int i = last_event_called_; i >= 0; --i) {
    auto [filter, event_type] = filter_events_[i];
    if (event_type == FilterEventType::kRelax) filter->Revert();
  }
  last_event_called_ = -1;
}

// TODO(user): the behaviour of Accept relies on the initial order of
// filters having at most one filter with negative objective values,
// this could be fixed by having filters return their general bounds.
bool LocalSearchFilterManager::Accept(LocalSearchMonitor* const monitor,
                                      const Assignment* delta,
                                      const Assignment* deltadelta,
                                      int64 objective_min,
                                      int64 objective_max) {
  Revert();
  accepted_value_ = 0;
  bool ok = true;
  const int num_events = filter_events_.size();
  for (int i = 0; i < num_events;) {
    last_event_called_ = i;
    auto [filter, event_type] = filter_events_[last_event_called_];
    switch (event_type) {
      case FilterEventType::kAccept: {
        if (monitor != nullptr) monitor->BeginFiltering(filter);
        const bool accept = filter->Accept(
            delta, deltadelta, CapSub(objective_min, accepted_value_),
            CapSub(objective_max, accepted_value_));
        ok &= accept;
        if (monitor != nullptr) monitor->EndFiltering(filter, !accept);
        if (ok) {
          accepted_value_ =
              CapAdd(accepted_value_, filter->GetAcceptedObjectiveValue());
          // TODO(user): handle objective min.
          ok = accepted_value_ <= objective_max;
        }
        break;
      }
      case FilterEventType::kRelax: {
        filter->Relax(delta, deltadelta);
        break;
      }
      default:
        LOG(FATAL) << "Unknown filter event type.";
    }
    // If the candidate is rejected, forced events must still be called.
    if (ok) {
      ++i;
    } else {
      i = next_forced_events_[i];
    }
  }
  return ok;
}

void LocalSearchFilterManager::Synchronize(const Assignment* assignment,
                                           const Assignment* delta) {
  // If delta is nullptr or empty, then assignment may be a partial solution.
  // Send a signal to Relaxing filters to inform them,
  // so they can show the partial solution as a change from the empty solution.
  const bool reset_to_assignment = delta == nullptr || delta->Empty();
  // Relax in the forward direction.
  for (auto [filter, event_type] : filter_events_) {
    switch (event_type) {
      case FilterEventType::kAccept: {
        break;
      }
      case FilterEventType::kRelax: {
        if (reset_to_assignment) {
          filter->Reset();
          filter->Relax(assignment, nullptr);
        } else {
          filter->Relax(delta, nullptr);
        }
        break;
      }
      default:
        LOG(FATAL) << "Unknown filter event type.";
    }
  }
  // Synchronize/Commit backwards, so filters can read changes from their
  // dependencies before those are synchronized/committed.
  synchronized_value_ = 0;
  for (auto [filter, event_type] : ::gtl::reversed_view(filter_events_)) {
    switch (event_type) {
      case FilterEventType::kAccept: {
        filter->Synchronize(assignment, delta);
        synchronized_value_ = CapAdd(synchronized_value_,
                                     filter->GetSynchronizedObjectiveValue());
        break;
      }
      case FilterEventType::kRelax: {
        filter->Commit(assignment, delta);
        break;
      }
      default:
        LOG(FATAL) << "Unknown filter event type.";
    }
  }
}

// ----- Finds a neighbor of the assignment passed -----

class FindOneNeighbor : public DecisionBuilder {
 public:
  FindOneNeighbor(Assignment* const assignment, IntVar* objective,
                  SolutionPool* const pool,
                  LocalSearchOperator* const ls_operator,
                  DecisionBuilder* const sub_decision_builder,
                  const RegularLimit* const limit,
                  LocalSearchFilterManager* filter_manager);
  ~FindOneNeighbor() override {}
  Decision* Next(Solver* const solver) override;
  std::string DebugString() const override { return "FindOneNeighbor"; }

 private:
  bool FilterAccept(Solver* solver, Assignment* delta, Assignment* deltadelta,
                    int64 objective_min, int64 objective_max);
  void SynchronizeAll(Solver* solver, bool synchronize_filters = true);

  Assignment* const assignment_;
  IntVar* const objective_;
  std::unique_ptr<Assignment> reference_assignment_;
  SolutionPool* const pool_;
  LocalSearchOperator* const ls_operator_;
  DecisionBuilder* const sub_decision_builder_;
  RegularLimit* limit_;
  const RegularLimit* const original_limit_;
  bool neighbor_found_;
  LocalSearchFilterManager* const filter_manager_;
  int64 solutions_since_last_check_;
  int64 check_period_;
  Assignment last_checked_assignment_;
  bool has_checked_assignment_ = false;
};

// reference_assignment_ is used to keep track of the last assignment on which
// operators were started, assignment_ corresponding to the last successful
// neighbor.
FindOneNeighbor::FindOneNeighbor(Assignment* const assignment,
                                 IntVar* objective, SolutionPool* const pool,
                                 LocalSearchOperator* const ls_operator,
                                 DecisionBuilder* const sub_decision_builder,
                                 const RegularLimit* const limit,
                                 LocalSearchFilterManager* filter_manager)
    : assignment_(assignment),
      objective_(objective),
      reference_assignment_(new Assignment(assignment_)),
      pool_(pool),
      ls_operator_(ls_operator),
      sub_decision_builder_(sub_decision_builder),
      limit_(nullptr),
      original_limit_(limit),
      neighbor_found_(false),
      filter_manager_(filter_manager),
      solutions_since_last_check_(0),
      check_period_(
          assignment_->solver()->parameters().check_solution_period()),
      last_checked_assignment_(assignment) {
  CHECK(nullptr != assignment);
  CHECK(nullptr != ls_operator);

  Solver* const solver = assignment_->solver();
  // If limit is nullptr, default limit is 1 solution
  if (nullptr == limit) {
    limit_ = solver->MakeSolutionsLimit(1);
  } else {
    limit_ = limit->MakeIdenticalClone();
    // TODO(user): Support skipping neighborhood checks for limits accepting
    // more than one solution (e.g. best accept). For now re-enabling systematic
    // checks.
    if (limit_->solutions() != 1) {
      VLOG(1) << "Disabling neighbor-check skipping outside of first accept.";
      check_period_ = 1;
    }
  }
  // TODO(user): Support skipping neighborhood checks with LNS (at least on
  // the non-LNS operators).
  if (ls_operator->HasFragments()) {
    VLOG(1) << "Disabling neighbor-check skipping for LNS.";
    check_period_ = 1;
  }

  if (!reference_assignment_->HasObjective()) {
    reference_assignment_->AddObjective(objective_);
  }
}

Decision* FindOneNeighbor::Next(Solver* const solver) {
  CHECK(nullptr != solver);

  if (original_limit_ != nullptr) {
    limit_->Copy(original_limit_);
  }

  if (!last_checked_assignment_.HasObjective()) {
    last_checked_assignment_.AddObjective(assignment_->Objective());
  }

  if (!neighbor_found_) {
    // Only called on the first call to Next(), reference_assignment_ has not
    // been synced with assignment_ yet

    // Keeping the code in case a performance problem forces us to
    // use the old code with a zero test on pool_.
    // reference_assignment_->CopyIntersection(assignment_);
    pool_->Initialize(assignment_);
    SynchronizeAll(solver, /*synchronize_filters*/ false);
  }

  {
    // Another assignment is needed to apply the delta
    Assignment* assignment_copy =
        solver->MakeAssignment(reference_assignment_.get());
    int counter = 0;

    DecisionBuilder* restore = solver->MakeRestoreAssignment(assignment_copy);
    if (sub_decision_builder_) {
      restore = solver->Compose(restore, sub_decision_builder_);
    }
    Assignment* delta = solver->MakeAssignment();
    Assignment* deltadelta = solver->MakeAssignment();
    while (true) {
      if (!ls_operator_->HoldsDelta()) {
        delta->Clear();
      }
      delta->ClearObjective();
      deltadelta->Clear();
      solver->TopPeriodicCheck();
      if (++counter >= absl::GetFlag(FLAGS_cp_local_search_sync_frequency) &&
          pool_->SyncNeeded(reference_assignment_.get())) {
        // TODO(user) : SyncNeed(assignment_) ?
        counter = 0;
        SynchronizeAll(solver);
      }

      bool has_neighbor = false;
      if (!limit_->Check()) {
        solver->GetLocalSearchMonitor()->BeginMakeNextNeighbor(ls_operator_);
        has_neighbor = ls_operator_->MakeNextNeighbor(delta, deltadelta);
        solver->GetLocalSearchMonitor()->EndMakeNextNeighbor(
            ls_operator_, has_neighbor, delta, deltadelta);
      }

      if (has_neighbor && !solver->IsUncheckedSolutionLimitReached()) {
        solver->neighbors_ += 1;
        // All filters must be called for incrementality reasons.
        // Empty deltas must also be sent to incremental filters; can be needed
        // to resync filters on non-incremental (empty) moves.
        // TODO(user): Don't call both if no filter is incremental and one
        // of them returned false.
        solver->GetLocalSearchMonitor()->BeginFilterNeighbor(ls_operator_);
        const bool mh_filter =
            AcceptDelta(solver->ParentSearch(), delta, deltadelta);
        int64 objective_min = kint64min;
        int64 objective_max = kint64max;
        if (objective_) {
          objective_min = objective_->Min();
          objective_max = objective_->Max();
        }
        if (delta->HasObjective() && delta->Objective() == objective_) {
          objective_min = std::max(objective_min, delta->ObjectiveMin());
          objective_max = std::min(objective_max, delta->ObjectiveMax());
        }
        const bool move_filter = FilterAccept(solver, delta, deltadelta,
                                              objective_min, objective_max);
        solver->GetLocalSearchMonitor()->EndFilterNeighbor(
            ls_operator_, mh_filter && move_filter);
        if (!mh_filter || !move_filter) {
          if (filter_manager_ != nullptr) filter_manager_->Revert();
          continue;
        }
        solver->filtered_neighbors_ += 1;
        if (delta->HasObjective()) {
          if (!assignment_copy->HasObjective()) {
            assignment_copy->AddObjective(delta->Objective());
          }
          if (!assignment_->HasObjective()) {
            assignment_->AddObjective(delta->Objective());
            last_checked_assignment_.AddObjective(delta->Objective());
          }
        }
        assignment_copy->CopyIntersection(reference_assignment_.get());
        assignment_copy->CopyIntersection(delta);
        solver->GetLocalSearchMonitor()->BeginAcceptNeighbor(ls_operator_);
        const bool check_solution = (solutions_since_last_check_ == 0) ||
                                    !solver->UseFastLocalSearch() ||
                                    // LNS deltas need to be restored
                                    !delta->AreAllElementsBound();
        if (has_checked_assignment_) solutions_since_last_check_++;
        if (solutions_since_last_check_ >= check_period_) {
          solutions_since_last_check_ = 0;
        }
        const bool accept = !check_solution || solver->SolveAndCommit(restore);
        solver->GetLocalSearchMonitor()->EndAcceptNeighbor(ls_operator_,
                                                           accept);
        if (accept) {
          solver->accepted_neighbors_ += 1;
          if (check_solution) {
            solver->SetSearchContext(solver->ParentSearch(),
                                     ls_operator_->DebugString());
            assignment_->Store();
            last_checked_assignment_.CopyIntersection(assignment_);
            neighbor_found_ = true;
            has_checked_assignment_ = true;
            return nullptr;
          }
          solver->SetSearchContext(solver->ActiveSearch(),
                                   ls_operator_->DebugString());
          assignment_->CopyIntersection(assignment_copy);
          assignment_->SetObjectiveValue(
              filter_manager_ ? filter_manager_->GetAcceptedObjectiveValue()
                              : 0);
          // Advancing local search to the current solution without
          // checking.
          // TODO(user): support the case were limit_ accepts more than
          // one solution (e.g. best accept).
          AcceptUncheckedNeighbor(solver->ParentSearch());
          solver->IncrementUncheckedSolutionCounter();
          pool_->RegisterNewSolution(assignment_);
          SynchronizeAll(solver);
          // NOTE: SynchronizeAll() sets neighbor_found_ to false, force it
          // back to true when skipping checks.
          neighbor_found_ = true;
        } else {
          if (filter_manager_ != nullptr) filter_manager_->Revert();
          if (check_period_ > 1 && has_checked_assignment_) {
            // Filtering is not perfect, disabling fast local search and
            // resynchronizing with the last checked solution.
            // TODO(user): Restore state of local search operators to
            // make sure we are exploring neighbors in the same order. This can
            // affect the local optimum found.
            VLOG(1) << "Imperfect filtering detected, backtracking to last "
                       "checked solution and checking all solutions.";
            check_period_ = 1;
            solutions_since_last_check_ = 0;
            pool_->RegisterNewSolution(&last_checked_assignment_);
            SynchronizeAll(solver);
            assignment_->CopyIntersection(&last_checked_assignment_);
          }
        }
      } else {
        if (neighbor_found_) {
          // In case the last checked assignment isn't the current one, restore
          // it to make sure the solver knows about it, especially if this is
          // the end of the search.
          // TODO(user): Compare assignments in addition to their cost.
          if (last_checked_assignment_.ObjectiveValue() !=
              assignment_->ObjectiveValue()) {
            // If restoring fails this means filtering is not perfect and the
            // solver will consider the last checked assignment.
            assignment_copy->CopyIntersection(assignment_);
            if (!solver->SolveAndCommit(restore)) solver->Fail();
            last_checked_assignment_.CopyIntersection(assignment_);
            has_checked_assignment_ = true;
            return nullptr;
          }
          AcceptNeighbor(solver->ParentSearch());
          // Keeping the code in case a performance problem forces us to
          // use the old code with a zero test on pool_.
          //          reference_assignment_->CopyIntersection(assignment_);
          pool_->RegisterNewSolution(assignment_);
          SynchronizeAll(solver);
        } else {
          break;
        }
      }
    }
  }
  solver->Fail();
  return nullptr;
}

bool FindOneNeighbor::FilterAccept(Solver* solver, Assignment* delta,
                                   Assignment* deltadelta, int64 objective_min,
                                   int64 objective_max) {
  if (filter_manager_ == nullptr) return true;
  LocalSearchMonitor* const monitor = solver->GetLocalSearchMonitor();
  return filter_manager_->Accept(monitor, delta, deltadelta, objective_min,
                                 objective_max);
}

void FindOneNeighbor::SynchronizeAll(Solver* solver, bool synchronize_filters) {
  pool_->GetNextSolution(reference_assignment_.get());
  neighbor_found_ = false;
  limit_->Init();
  solver->GetLocalSearchMonitor()->BeginOperatorStart();
  ls_operator_->Start(reference_assignment_.get());
  if (synchronize_filters && filter_manager_ != nullptr) {
    filter_manager_->Synchronize(reference_assignment_.get(), nullptr);
  }
  solver->GetLocalSearchMonitor()->EndOperatorStart();
}

// ---------- Local Search Phase Parameters ----------

class LocalSearchPhaseParameters : public BaseObject {
 public:
  LocalSearchPhaseParameters(IntVar* objective, SolutionPool* const pool,
                             LocalSearchOperator* ls_operator,
                             DecisionBuilder* sub_decision_builder,
                             RegularLimit* const limit,
                             LocalSearchFilterManager* filter_manager)
      : objective_(objective),
        solution_pool_(pool),
        ls_operator_(ls_operator),
        sub_decision_builder_(sub_decision_builder),
        limit_(limit),
        filter_manager_(filter_manager) {}
  ~LocalSearchPhaseParameters() override {}
  std::string DebugString() const override {
    return "LocalSearchPhaseParameters";
  }

  IntVar* objective() const { return objective_; }
  SolutionPool* solution_pool() const { return solution_pool_; }
  LocalSearchOperator* ls_operator() const { return ls_operator_; }
  DecisionBuilder* sub_decision_builder() const {
    return sub_decision_builder_;
  }
  RegularLimit* limit() const { return limit_; }
  LocalSearchFilterManager* const filter_manager() const {
    return filter_manager_;
  }

 private:
  IntVar* const objective_;
  SolutionPool* const solution_pool_;
  LocalSearchOperator* const ls_operator_;
  DecisionBuilder* const sub_decision_builder_;
  RegularLimit* const limit_;
  LocalSearchFilterManager* const filter_manager_;
};

LocalSearchPhaseParameters* Solver::MakeLocalSearchPhaseParameters(
    IntVar* objective, LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder) {
  return MakeLocalSearchPhaseParameters(objective, MakeDefaultSolutionPool(),
                                        ls_operator, sub_decision_builder,
                                        nullptr, nullptr);
}

LocalSearchPhaseParameters* Solver::MakeLocalSearchPhaseParameters(
    IntVar* objective, LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder, RegularLimit* const limit) {
  return MakeLocalSearchPhaseParameters(objective, MakeDefaultSolutionPool(),
                                        ls_operator, sub_decision_builder,
                                        limit, nullptr);
}

LocalSearchPhaseParameters* Solver::MakeLocalSearchPhaseParameters(
    IntVar* objective, LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder, RegularLimit* const limit,
    LocalSearchFilterManager* filter_manager) {
  return MakeLocalSearchPhaseParameters(objective, MakeDefaultSolutionPool(),
                                        ls_operator, sub_decision_builder,
                                        limit, filter_manager);
}

LocalSearchPhaseParameters* Solver::MakeLocalSearchPhaseParameters(
    IntVar* objective, SolutionPool* const pool,
    LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder) {
  return MakeLocalSearchPhaseParameters(objective, pool, ls_operator,
                                        sub_decision_builder, nullptr, nullptr);
}

LocalSearchPhaseParameters* Solver::MakeLocalSearchPhaseParameters(
    IntVar* objective, SolutionPool* const pool,
    LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder, RegularLimit* const limit) {
  return MakeLocalSearchPhaseParameters(objective, pool, ls_operator,
                                        sub_decision_builder, limit, nullptr);
}

LocalSearchPhaseParameters* Solver::MakeLocalSearchPhaseParameters(
    IntVar* objective, SolutionPool* const pool,
    LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder, RegularLimit* const limit,
    LocalSearchFilterManager* filter_manager) {
  return RevAlloc(new LocalSearchPhaseParameters(objective, pool, ls_operator,
                                                 sub_decision_builder, limit,
                                                 filter_manager));
}

namespace {
// ----- NestedSolve decision wrapper -----

// This decision calls a nested Solve on the given DecisionBuilder in its
// left branch; does nothing in the left branch.
// The state of the decision corresponds to the result of the nested Solve:
// DECISION_PENDING - Nested Solve not called yet
// DECISION_FAILED  - Nested Solve failed
// DECISION_FOUND   - Nested Solve succeeded

class NestedSolveDecision : public Decision {
 public:
  // This enum is used internally to tag states in the local search tree.
  enum StateType { DECISION_PENDING, DECISION_FAILED, DECISION_FOUND };

  NestedSolveDecision(DecisionBuilder* const db, bool restore,
                      const std::vector<SearchMonitor*>& monitors);
  NestedSolveDecision(DecisionBuilder* const db, bool restore);
  ~NestedSolveDecision() override {}
  void Apply(Solver* const solver) override;
  void Refute(Solver* const solver) override;
  std::string DebugString() const override { return "NestedSolveDecision"; }
  int state() const { return state_; }

 private:
  DecisionBuilder* const db_;
  bool restore_;
  std::vector<SearchMonitor*> monitors_;
  int state_;
};

NestedSolveDecision::NestedSolveDecision(
    DecisionBuilder* const db, bool restore,
    const std::vector<SearchMonitor*>& monitors)
    : db_(db),
      restore_(restore),
      monitors_(monitors),
      state_(DECISION_PENDING) {
  CHECK(nullptr != db);
}

NestedSolveDecision::NestedSolveDecision(DecisionBuilder* const db,
                                         bool restore)
    : db_(db), restore_(restore), state_(DECISION_PENDING) {
  CHECK(nullptr != db);
}

void NestedSolveDecision::Apply(Solver* const solver) {
  CHECK(nullptr != solver);
  if (restore_) {
    if (solver->Solve(db_, monitors_)) {
      solver->SaveAndSetValue(&state_, static_cast<int>(DECISION_FOUND));
    } else {
      solver->SaveAndSetValue(&state_, static_cast<int>(DECISION_FAILED));
    }
  } else {
    if (solver->SolveAndCommit(db_, monitors_)) {
      solver->SaveAndSetValue(&state_, static_cast<int>(DECISION_FOUND));
    } else {
      solver->SaveAndSetValue(&state_, static_cast<int>(DECISION_FAILED));
    }
  }
}

void NestedSolveDecision::Refute(Solver* const solver) {}

// ----- Local search decision builder -----

// Given a first solution (resulting from either an initial assignment or the
// result of a decision builder), it searches for neighbors using a local
// search operator. The first solution corresponds to the first leaf of the
// search.
// The local search applies to the variables contained either in the assignment
// or the vector of variables passed.

class LocalSearch : public DecisionBuilder {
 public:
  LocalSearch(Assignment* const assignment, IntVar* objective,
              SolutionPool* const pool, LocalSearchOperator* const ls_operator,
              DecisionBuilder* const sub_decision_builder,
              RegularLimit* const limit,
              LocalSearchFilterManager* filter_manager);
  // TODO(user): find a way to not have to pass vars here: redundant with
  // variables in operators
  LocalSearch(const std::vector<IntVar*>& vars, IntVar* objective,
              SolutionPool* const pool, DecisionBuilder* const first_solution,
              LocalSearchOperator* const ls_operator,
              DecisionBuilder* const sub_decision_builder,
              RegularLimit* const limit,
              LocalSearchFilterManager* filter_manager);
  LocalSearch(const std::vector<IntVar*>& vars, IntVar* objective,
              SolutionPool* const pool, DecisionBuilder* const first_solution,
              DecisionBuilder* const first_solution_sub_decision_builder,
              LocalSearchOperator* const ls_operator,
              DecisionBuilder* const sub_decision_builder,
              RegularLimit* const limit,
              LocalSearchFilterManager* filter_manager);
  LocalSearch(const std::vector<SequenceVar*>& vars, IntVar* objective,
              SolutionPool* const pool, DecisionBuilder* const first_solution,
              LocalSearchOperator* const ls_operator,
              DecisionBuilder* const sub_decision_builder,
              RegularLimit* const limit,
              LocalSearchFilterManager* filter_manager);
  ~LocalSearch() override;
  Decision* Next(Solver* const solver) override;
  std::string DebugString() const override { return "LocalSearch"; }
  void Accept(ModelVisitor* const visitor) const override;

 protected:
  void PushFirstSolutionDecision(DecisionBuilder* first_solution);
  void PushLocalSearchDecision();

 private:
  Assignment* assignment_;
  IntVar* const objective_ = nullptr;
  SolutionPool* const pool_;
  LocalSearchOperator* const ls_operator_;
  DecisionBuilder* const first_solution_sub_decision_builder_;
  DecisionBuilder* const sub_decision_builder_;
  std::vector<NestedSolveDecision*> nested_decisions_;
  int nested_decision_index_;
  RegularLimit* const limit_;
  LocalSearchFilterManager* const filter_manager_;
  bool has_started_;
};

LocalSearch::LocalSearch(Assignment* const assignment, IntVar* objective,
                         SolutionPool* const pool,
                         LocalSearchOperator* const ls_operator,
                         DecisionBuilder* const sub_decision_builder,
                         RegularLimit* const limit,
                         LocalSearchFilterManager* filter_manager)
    : assignment_(nullptr),
      objective_(objective),
      pool_(pool),
      ls_operator_(ls_operator),
      first_solution_sub_decision_builder_(sub_decision_builder),
      sub_decision_builder_(sub_decision_builder),
      nested_decision_index_(0),
      limit_(limit),
      filter_manager_(filter_manager),
      has_started_(false) {
  CHECK(nullptr != assignment);
  CHECK(nullptr != ls_operator);
  Solver* const solver = assignment->solver();
  assignment_ = solver->GetOrCreateLocalSearchState();
  assignment_->Copy(assignment);
  DecisionBuilder* restore = solver->MakeRestoreAssignment(assignment);
  PushFirstSolutionDecision(restore);
  PushLocalSearchDecision();
}

LocalSearch::LocalSearch(const std::vector<IntVar*>& vars, IntVar* objective,
                         SolutionPool* const pool,
                         DecisionBuilder* const first_solution,
                         LocalSearchOperator* const ls_operator,
                         DecisionBuilder* const sub_decision_builder,
                         RegularLimit* const limit,
                         LocalSearchFilterManager* filter_manager)
    : assignment_(nullptr),
      objective_(objective),
      pool_(pool),
      ls_operator_(ls_operator),
      first_solution_sub_decision_builder_(sub_decision_builder),
      sub_decision_builder_(sub_decision_builder),
      nested_decision_index_(0),
      limit_(limit),
      filter_manager_(filter_manager),
      has_started_(false) {
  CHECK(nullptr != first_solution);
  CHECK(nullptr != ls_operator);
  CHECK(!vars.empty());
  Solver* const solver = vars[0]->solver();
  assignment_ = solver->GetOrCreateLocalSearchState();
  assignment_->Add(vars);
  PushFirstSolutionDecision(first_solution);
  PushLocalSearchDecision();
}

LocalSearch::LocalSearch(
    const std::vector<IntVar*>& vars, IntVar* objective,
    SolutionPool* const pool, DecisionBuilder* const first_solution,
    DecisionBuilder* const first_solution_sub_decision_builder,
    LocalSearchOperator* const ls_operator,
    DecisionBuilder* const sub_decision_builder, RegularLimit* const limit,
    LocalSearchFilterManager* filter_manager)
    : assignment_(nullptr),
      objective_(objective),
      pool_(pool),
      ls_operator_(ls_operator),
      first_solution_sub_decision_builder_(first_solution_sub_decision_builder),
      sub_decision_builder_(sub_decision_builder),
      nested_decision_index_(0),
      limit_(limit),
      filter_manager_(filter_manager),
      has_started_(false) {
  CHECK(nullptr != first_solution);
  CHECK(nullptr != ls_operator);
  CHECK(!vars.empty());
  Solver* const solver = vars[0]->solver();
  assignment_ = solver->GetOrCreateLocalSearchState();
  assignment_->Add(vars);
  PushFirstSolutionDecision(first_solution);
  PushLocalSearchDecision();
}

LocalSearch::LocalSearch(const std::vector<SequenceVar*>& vars,
                         IntVar* objective, SolutionPool* const pool,
                         DecisionBuilder* const first_solution,
                         LocalSearchOperator* const ls_operator,
                         DecisionBuilder* const sub_decision_builder,
                         RegularLimit* const limit,
                         LocalSearchFilterManager* filter_manager)
    : assignment_(nullptr),
      objective_(objective),
      pool_(pool),
      ls_operator_(ls_operator),
      first_solution_sub_decision_builder_(sub_decision_builder),
      sub_decision_builder_(sub_decision_builder),
      nested_decision_index_(0),
      limit_(limit),
      filter_manager_(filter_manager),
      has_started_(false) {
  CHECK(nullptr != first_solution);
  CHECK(nullptr != ls_operator);
  CHECK(!vars.empty());
  Solver* const solver = vars[0]->solver();
  assignment_ = solver->GetOrCreateLocalSearchState();
  assignment_->Add(vars);
  PushFirstSolutionDecision(first_solution);
  PushLocalSearchDecision();
}

LocalSearch::~LocalSearch() {}

// Model Visitor support.
void LocalSearch::Accept(ModelVisitor* const visitor) const {
  DCHECK(assignment_ != nullptr);
  visitor->BeginVisitExtension(ModelVisitor::kVariableGroupExtension);
  // We collect decision variables from the assignment.
  const std::vector<IntVarElement>& elements =
      assignment_->IntVarContainer().elements();
  if (!elements.empty()) {
    std::vector<IntVar*> vars;
    for (const IntVarElement& elem : elements) {
      vars.push_back(elem.Var());
    }
    visitor->VisitIntegerVariableArrayArgument(ModelVisitor::kVarsArgument,
                                               vars);
  }
  const std::vector<IntervalVarElement>& interval_elements =
      assignment_->IntervalVarContainer().elements();
  if (!interval_elements.empty()) {
    std::vector<IntervalVar*> interval_vars;
    for (const IntervalVarElement& elem : interval_elements) {
      interval_vars.push_back(elem.Var());
    }
    visitor->VisitIntervalArrayArgument(ModelVisitor::kIntervalsArgument,
                                        interval_vars);
  }
  visitor->EndVisitExtension(ModelVisitor::kVariableGroupExtension);
}

// This is equivalent to a multi-restart decision builder
// TODO(user): abstract this from the local search part
// TODO(user): handle the case where the tree depth is not enough to hold
//                all solutions.

Decision* LocalSearch::Next(Solver* const solver) {
  CHECK(nullptr != solver);
  CHECK_LT(0, nested_decisions_.size());
  if (!has_started_) {
    nested_decision_index_ = 0;
    solver->SaveAndSetValue(&has_started_, true);
  } else if (nested_decision_index_ < 0) {
    solver->Fail();
  }
  NestedSolveDecision* decision = nested_decisions_[nested_decision_index_];
  const int state = decision->state();
  switch (state) {
    case NestedSolveDecision::DECISION_FAILED: {
      // A local optimum has been reached. The search will continue only if we
      // accept up-hill moves (due to metaheuristics). In this case we need to
      // reset neighborhood optimal routes.
      ls_operator_->Reset();
      if (!LocalOptimumReached(solver->ActiveSearch())) {
        nested_decision_index_ = -1;  // Stop the search
      }
      solver->Fail();
      return nullptr;
    }
    case NestedSolveDecision::DECISION_PENDING: {
      // TODO(user): Find a way to make this balancing invisible to the
      // user (no increase in branch or fail counts for instance).
      const int32 kLocalSearchBalancedTreeDepth = 32;
      const int depth = solver->SearchDepth();
      if (depth < kLocalSearchBalancedTreeDepth) {
        return solver->balancing_decision();
      }
      if (depth > kLocalSearchBalancedTreeDepth) {
        solver->Fail();
      }
      return decision;
    }
    case NestedSolveDecision::DECISION_FOUND: {
      // Next time go to next decision
      if (nested_decision_index_ + 1 < nested_decisions_.size()) {
        ++nested_decision_index_;
      }
      return nullptr;
    }
    default: {
      LOG(ERROR) << "Unknown local search state";
      return nullptr;
    }
  }
  return nullptr;
}

namespace {
class SynchronizeFiltersDecisionBuilder : public DecisionBuilder {
 public:
  SynchronizeFiltersDecisionBuilder(Assignment* assignment,
                                    LocalSearchFilterManager* filter_manager)
      : assignment_(assignment), filter_manager_(filter_manager) {}

  Decision* Next(Solver* const solver) override {
    if (filter_manager_ != nullptr) {
      filter_manager_->Synchronize(assignment_, nullptr);
    }
    return nullptr;
  }

 private:
  Assignment* const assignment_;
  LocalSearchFilterManager* const filter_manager_;
};
}  // namespace

void LocalSearch::PushFirstSolutionDecision(DecisionBuilder* first_solution) {
  CHECK(first_solution);
  Solver* const solver = assignment_->solver();
  DecisionBuilder* store = solver->MakeStoreAssignment(assignment_);
  DecisionBuilder* synchronize = solver->RevAlloc(
      new SynchronizeFiltersDecisionBuilder(assignment_, filter_manager_));
  DecisionBuilder* first_solution_and_store = solver->Compose(
      first_solution, first_solution_sub_decision_builder_, store, synchronize);
  std::vector<SearchMonitor*> monitor;
  monitor.push_back(limit_);
  nested_decisions_.push_back(solver->RevAlloc(
      new NestedSolveDecision(first_solution_and_store, false, monitor)));
}

void LocalSearch::PushLocalSearchDecision() {
  Solver* const solver = assignment_->solver();
  DecisionBuilder* find_neighbors = solver->RevAlloc(
      new FindOneNeighbor(assignment_, objective_, pool_, ls_operator_,
                          sub_decision_builder_, limit_, filter_manager_));
  nested_decisions_.push_back(
      solver->RevAlloc(new NestedSolveDecision(find_neighbors, false)));
}

class DefaultSolutionPool : public SolutionPool {
 public:
  DefaultSolutionPool() {}

  ~DefaultSolutionPool() override {}

  void Initialize(Assignment* const assignment) override {
    reference_assignment_ = absl::make_unique<Assignment>(assignment);
  }

  void RegisterNewSolution(Assignment* const assignment) override {
    reference_assignment_->CopyIntersection(assignment);
  }

  void GetNextSolution(Assignment* const assignment) override {
    assignment->CopyIntersection(reference_assignment_.get());
  }

  bool SyncNeeded(Assignment* const local_assignment) override { return false; }

  std::string DebugString() const override { return "DefaultSolutionPool"; }

 private:
  std::unique_ptr<Assignment> reference_assignment_;
};
}  // namespace

SolutionPool* Solver::MakeDefaultSolutionPool() {
  return RevAlloc(new DefaultSolutionPool());
}

DecisionBuilder* Solver::MakeLocalSearchPhase(
    Assignment* assignment, LocalSearchPhaseParameters* parameters) {
  return RevAlloc(new LocalSearch(
      assignment, parameters->objective(), parameters->solution_pool(),
      parameters->ls_operator(), parameters->sub_decision_builder(),
      parameters->limit(), parameters->filter_manager()));
}

DecisionBuilder* Solver::MakeLocalSearchPhase(
    const std::vector<IntVar*>& vars, DecisionBuilder* first_solution,
    LocalSearchPhaseParameters* parameters) {
  return RevAlloc(new LocalSearch(
      vars, parameters->objective(), parameters->solution_pool(),
      first_solution, parameters->ls_operator(),
      parameters->sub_decision_builder(), parameters->limit(),
      parameters->filter_manager()));
}

DecisionBuilder* Solver::MakeLocalSearchPhase(
    const std::vector<IntVar*>& vars, DecisionBuilder* first_solution,
    DecisionBuilder* first_solution_sub_decision_builder,
    LocalSearchPhaseParameters* parameters) {
  return RevAlloc(new LocalSearch(
      vars, parameters->objective(), parameters->solution_pool(),
      first_solution, first_solution_sub_decision_builder,
      parameters->ls_operator(), parameters->sub_decision_builder(),
      parameters->limit(), parameters->filter_manager()));
}

DecisionBuilder* Solver::MakeLocalSearchPhase(
    const std::vector<SequenceVar*>& vars, DecisionBuilder* first_solution,
    LocalSearchPhaseParameters* parameters) {
  return RevAlloc(new LocalSearch(
      vars, parameters->objective(), parameters->solution_pool(),
      first_solution, parameters->ls_operator(),
      parameters->sub_decision_builder(), parameters->limit(),
      parameters->filter_manager()));
}
}  // namespace operations_research
