/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2003 Robert Osfield 
 *
 * This application is open source and may be redistributed and/or modified   
 * freely and without restriction, both in commericial and non commericial applications,
 * as long as this copyright notice is maintained.
 * 
 * This application is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/
/*
 *
 * Copyright (C) 2005 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#ifdef USE_NPROFILE
#include <nprofile/profile.h>
#endif

#include <fcntl.h>

#include <glib.h>

#include <osg/Version>
#include <osgDB/ReadFile>
#include <osgDB/FileUtils>
#include <osgText/Text>
#include <osgProducer/Viewer>
#include <osg/CoordinateSystemNode>
#include <osg/MatrixTransform>
#include <osg/Projection>
#include <osgGA/TrackballManipulator>

#include <osgCal/Model>

#ifndef DEFAULT_FILES
#define DEFAULT_FILES "franck/cal3d.xfg", "paladin/cal3d.cfg"
#endif // DEFAULT_FILES

#define DEFAULT_DATA_DIR "data"
#ifndef DATA_DIR
#define DATA_DIR "."
#endif // DATA_DIR

static std::string read_file(const std::string& path) {
  gchar* outfit_bytes = NULL;
  GError* error = NULL;
  if(!g_file_get_contents(path.c_str(), &outfit_bytes, NULL, &error)) {
    osg::notify(osg::FATAL) << path << (error ? error->message : "unknown error");
    return "";
  }
  std::string result = outfit_bytes;
  g_free(outfit_bytes);
  return result;
}

class MyManipulator: public osgGA::TrackballManipulator {
public:

  MyManipulator(osgCal::Model *model) {
    this->model=model;
    nMov=0;
  }
  /** handle events, return true if handled, false otherwise.*/
  virtual bool handle(const osgGA::GUIEventAdapter& ea,osgGA::GUIActionAdapter& us) {
    if (ea.getEventType() == osgGA::GUIEventAdapter::KEYDOWN) {
			
      // Key to switch between animations
      if (ea.getKey() == 'm') {
	model->getCalModel()->getMixer()->clearCycle(nMov, 3);
	nMov = ++nMov%6;
	printf("Movement No.%d (Blend time 3sec)\n", nMov);
	model->getCalModel()->getMixer()->blendCycle(nMov, 1, 3);
	return true;
      } else if (ea.getKey() == 'n') {
        _current_outfit++;
        if(_current_outfit >= (int)_outfits.size()) _current_outfit = 0;
        model->installOutfitFromXMLString(read_file(_outfits[_current_outfit]));
        _comment->setText(model->getOutfit()->_comment);
      } else if (ea.getKey() == 'p') {
        _current_outfit--;
        if(_current_outfit < 0) _current_outfit = _outfits.size() - 1;
        model->installOutfitFromXMLString(read_file(_outfits[_current_outfit]));
        _comment->setText(model->getOutfit()->_comment);
      }
			
    }
    return TrackballManipulator::handle(ea,us);
  }

  typedef std::vector<std::string> Outfits;

  void setOutfits(const Outfits& outfits) {
    _outfits = outfits;
    _current_outfit = 0;
  }

  void setComment(osgText::Text* comment) { _comment = comment; }

private:
  Outfits _outfits;
  int _current_outfit;
  osg::ref_ptr<osgText::Text> _comment;
  osgCal::Model* model;
  int nMov;
};

int main( int argc, char **argv )
{
  std::string default_files;
  {
    const char* file_strings[] = { DEFAULT_FILES, NULL };
    const char** pointer;
    for(pointer = file_strings; *pointer; pointer++)
      default_files += std::string(*pointer) + "|";
  }

  {
    osgDB::FilePathList& path = osgDB::getDataFilePathList();
    path.push_front(DATA_DIR);
    path.push_front(DEFAULT_DATA_DIR);
  }

  // use an ArgumentParser object to manage the program arguments.
  osg::ArgumentParser arguments(&argc,argv);
    
  // set up the usage document, in case we need to print out how to use this program.
  arguments.getApplicationUsage()->setApplicationName(arguments.getApplicationName());
  arguments.getApplicationUsage()->setDescription(arguments.getApplicationName()+" view an osgCal model and interact with it.");
  arguments.getApplicationUsage()->setCommandLineUsage(arguments.getApplicationName()+" [options] {" + default_files + "..}");
  arguments.getApplicationUsage()->addCommandLineOption("--image <filename>","Load an image and render it on a quad");
  arguments.getApplicationUsage()->addCommandLineOption("--dem <filename>","Load an image/DEM and render it on a HeightField");
  arguments.getApplicationUsage()->addCommandLineOption("-h or --help","Display command line paramters");
  arguments.getApplicationUsage()->addCommandLineOption("--help-env","Display environmental variables available");
  arguments.getApplicationUsage()->addCommandLineOption("--help-keys","Display keyboard & mouse bindings available");
  arguments.getApplicationUsage()->addCommandLineOption("--help-all","Display all command line, env vars and keyboard & mouse bindigs.");
  arguments.getApplicationUsage()->addCommandLineOption("--benchmark","run non interactive benchmark and output results.");
    

  // construct the viewer.
  osgProducer::Viewer viewer(arguments);

  // set up the value with sensible default event handlers.
  viewer.setUpViewer(osgProducer::Viewer::STANDARD_SETTINGS);

  // get details on keyboard and mouse bindings used by the viewer.
  viewer.getUsage(*arguments.getApplicationUsage());

  // if user request help write it out to cout.
  bool helpAll = arguments.read("--help-all");
  unsigned int helpType = ((helpAll || arguments.read("-h") || arguments.read("--help"))? osg::ApplicationUsage::COMMAND_LINE_OPTION : 0 ) |
    ((helpAll ||  arguments.read("--help-env"))? osg::ApplicationUsage::ENVIRONMENTAL_VARIABLE : 0 ) |
    ((helpAll ||  arguments.read("--help-keys"))? osg::ApplicationUsage::KEYBOARD_MOUSE_BINDING : 0 );
  if (helpType)
    {
      arguments.getApplicationUsage()->write(std::cout, helpType);
      return 1;
    }

  // report any errors if they have occured when parsing the program aguments.
  if (arguments.errors())
    {
      arguments.writeErrorMessages(std::cout);
      return 1;
    }
    
  if (arguments.argc()<=1)
    {
      arguments.getApplicationUsage()->write(std::cout, osg::ApplicationUsage::COMMAND_LINE_OPTION);
      return 1;
    }

  std::vector<std::string> positional_arguments;
  for(int pos = 1; pos < arguments.argc(); ++pos) 
    if(!arguments.isOption(pos))
      positional_arguments.push_back(arguments[pos]);

  osg::Timer_t start_tick = osg::Timer::instance()->tick();

  std::string path = positional_arguments.front();
  positional_arguments.erase(positional_arguments.begin());

  std::string suffix;
  {
    bool error_occurred = true;
    unsigned int dot = path.rfind('.');
    std::string lcpath = path;
    std::transform(lcpath.begin(), lcpath.end(), lcpath.begin(), tolower);
    if(dot != std::string::npos) {
      suffix = lcpath.substr(dot + 1);
      if(suffix == "cfg" || suffix == "xfg") {
        error_occurred = false;
      }
    } 

    if(error_occurred) {
      std::cout << "ERROR: expected a .cfg or .xfg file but got " << path << " instead" << std::endl;
      arguments.getApplicationUsage()->write(std::cout, osg::ApplicationUsage::COMMAND_LINE_OPTION);
      return 1;
    }
  }

  osg::Group* root	= new osg::Group();

  osg::ref_ptr<osgCal::Model> model = new osgCal::Model();
  root->addChild(model.get());

  osg::Object* object	= osgDB::readObjectFile(path);
  osg::ref_ptr<osgCal::CoreModel> coreModelRef = dynamic_cast<osgCal::CoreModel*>( object );
  osgCal::CoreModel* coreModel = coreModelRef.get();
  if(!model->setCoreModel(coreModel)) {
    std::cerr << "setCoreModel: " << CalError::getLastErrorDescription().c_str() << std::endl;
    return -1;
  }

  if(suffix == "cfg") {
    for(int coreMeshId = 0; coreMeshId < model->getCalCoreModel()->getCoreMeshCount(); coreMeshId++) {
      model->getCalModel()->attachMesh(coreMeshId);
    }

    // set the material set of the whole model
    model->getCalModel()->setMaterialSet(0);

    // Creating a concrete model using the core template
    if(!model->create()) {
      std::cerr << "create: " << CalError::getLastErrorDescription().c_str() << std::endl;
      return -1;
    }

  } else if(suffix == "xfg") {
    osg::Group* fxgroup = new osg::Group();
    fxgroup->setName("fxGroup");
    fxgroup->getOrCreateStateSet()->setMode(GL_ALPHA_TEST, osg::StateAttribute::OFF);
    fxgroup->getOrCreateStateSet()->setMode(GL_DITHER, osg::StateAttribute::ON);
    fxgroup->getOrCreateStateSet()->setRenderBinDetails(100, "RenderBin");

    root->addChild(fxgroup);

    model->setFXGroup(fxgroup);

    if(positional_arguments.size() <= 0) {
      std::string directory = g_dirname(path.c_str());
      std::string outfit = directory + "/outfit.xfg";

      std::string outfit_path = osgDB::findDataFile(outfit);
      if(outfit_path == "") {
        std::cerr << "unable to find file " << outfit << " in OSG path" << std::endl;
        return 1;
      }
      positional_arguments.push_back(outfit_path);
    }

    std::string outfit = read_file(positional_arguments.front());
    if(outfit == "") return 1;
      
    //
    // Apply outfit parameters
    //
    if(!model->initOutfitFromXMLString(outfit))
      return 1;

    if(!model->create()) {
      std::cerr << "create failed " << CalError::getLastErrorDescription() << std::endl;
      return 1;
    }
  }

  // if no model has been successfully loaded report failure.
  if (!model) 
    {
      std::cout << arguments.getApplicationName() <<": No data loaded" << std::endl;
      return 1;
    }

  // any option left unread are converted into errors to write out later.
  arguments.reportRemainingOptionsAsUnrecognized();

  // report any errors if they have occured when parsing the program aguments.
  if (arguments.errors())
    {
      arguments.writeErrorMessages(std::cout);
    }

  osg::Timer_t end_tick = osg::Timer::instance()->tick();

  std::cout << "Time to load = "<<osg::Timer::instance()->delta_s(start_tick,end_tick)<<std::endl;

  // pass the loaded scene graph to the viewer.
  viewer.setSceneData(root);
  
  // do not start in full screen by default
  viewer.getCamera(0)->getRenderSurface()->setWindowRectangle(200,300,800,600);

  // Create new manipulator. 
  MyManipulator* tm	= new MyManipulator(model.get());

  // attach manipulator to loadedModel node. 
  tm->setNode( model.get() );
  if(suffix == "xfg")
    tm->setOutfits(positional_arguments);

  // Add new manipulator to viewer
  unsigned int id = viewer.addCameraManipulator(tm);

  // Select this manipulator
  viewer.selectCameraManipulator(id);

  // create the windows and run the threads.
  viewer.realize();

  viewer.getCullSettings().setDefaults();
  viewer.getCullSettings().setComputeNearFarMode( osg::CullSettings::DO_NOT_COMPUTE_NEAR_FAR );
  viewer.getCullSettings().setCullingMode( osg::CullSettings::DEFAULT_CULLING );
	
  osgProducer::OsgCameraGroup::SceneHandlerList &shlist = viewer.getSceneHandlerList();
  osgUtil::SceneView* scene_view = (*shlist.begin())->getSceneView();
  // Retrieve global state set
  osg::StateSet* globalStateSet = scene_view->getGlobalStateSet();
  if (!globalStateSet) {
    globalStateSet = new osg::StateSet;
    scene_view->setGlobalStateSet(globalStateSet);
  }
  // disable back face culling else the both cape sides are not rendered 
  globalStateSet->setMode(GL_CULL_FACE, osg::StateAttribute::OFF  | osg::StateAttribute::OVERRIDE );

  if(suffix == "xfg") {
    //
    // Black magic that *must* disapear urgently (09/2005)
    //
    model->setFXState(scene_view->getState());
    if(!model->applyParameterFromOutfitDescription())
      return 1;
  }

  osgText::Text* text = new osgText::Text;
  text->setPosition(osg::Vec3(5.f, 5.f, 0.f));
  text->setAlignment(osgText::Text::LEFT_BOTTOM);
  text->setCharacterSize(12.f);
  {
    osg::Projection* projection = new osg::Projection;
    projection->setMatrix(osg::Matrix::ortho2D(0,800,0,600));

    osg::MatrixTransform* modelview_abs = new osg::MatrixTransform;
    modelview_abs->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
    modelview_abs->setMatrix(osg::Matrix::identity());

    osg::Geode* text_geode = new osg::Geode;
    text_geode->addDrawable(text);

    modelview_abs->addChild(text_geode);

    projection->addChild(modelview_abs);

    root->addChild(projection);

  }
  tm->setComment(text);


  {
    NPROFILE_SAMPLE("osgcal");
    if(arguments.read("--benchmark")) {

#ifndef USE_NPROFILE
      osg::notify(osg::FATAL) << "recompile with --enable-nprofile to enable --benchmark" << std::endl;
      exit(1);
#endif // USE_NPROFILE

      bool done = false;
      do {
        for(int i = 0; i < 5; i++) {
          viewer.sync();
          viewer.update();
          viewer.frame();
        }
        if(positional_arguments.size() > 0 && suffix == "xfg") {
          const std::string& outfit = positional_arguments.front();
          model->installOutfitFromXMLString(read_file(outfit));
          positional_arguments.erase(positional_arguments.begin());
        } else {
          done = true;
        }
      } while(!done);
      
    } else {

      while( !viewer.done() ) {
        // wait for all cull and draw threads to complete.
        viewer.sync();

        // update the scene by traversing it with the the update visitor which will
        // call all node update callbacks and animations.
        viewer.update();
         
        // fire off the cull and draw traversals of the scene.
        viewer.frame();
        
      }
    }
  }
    
  // wait for all cull and draw threads to complete before exit.
  viewer.sync();

#if OSG_VERSION_RELEASE >= 9
  // run a clean up frame to delete all OpenGL objects.
  viewer.cleanup_frame();

  // wait for all the clean up frame to complete.
  viewer.sync();
#endif // OSG_VERSION_RELEASE >= 9

#ifdef USE_NPROFILE
  nprf::GetProfiler()->EndProfile();
  nprf::GetProfiler()->GetRootNode()->DisplayFlatStats(std::cout);
  nprf::GetProfiler()->GetRootNode()->DisplayFlatStats(std::cout, nprf::MACHINE_READABLE);
  std::cout << std::endl;
#endif // USE_NPROFILE

  return 0;
}

/*
 * Local variables:
 * compile-command: "make && libtool --mode=execute valgrind --leak-check=yes --show-reachable=yes --suppressions=r200.supp  --suppressions=xorg.supp osgcal franck/cal3d.xfg"
 * End:
 */
