
/******************************************************************************
**
**  Copyright (C) 2005 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/******************************************************************************
**
**    File:      filter_user.c
**    Author:    Brian Wotring
**
**    Date:      March 14, 2002.
**    Project:   osiris
**
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
#include <windows.h>
#include <accctrl.h>
#include <aclapi.h>
#else
#include <pwd.h>
#endif

#include "utilities.h"
#include "scan_record.h"
#include "filter.h"

#ifdef WIN32

osi_bool filter_user( const char *file_path, FILTER_CONTEXT *context )
{
    DWORD dwRtnCode = 0;
    PSID pSidOwner;
    BOOL result        = TRUE;
    LPTSTR AcctName    = NULL;
    DWORD dwAcctName   = 1;
    DWORD dwDomainName = 1;
    SID_NAME_USE eUse  = SidTypeUnknown;

    PSECURITY_DESCRIPTOR pSD;

    unsigned char psd[sizeof(PSECURITY_DESCRIPTOR)];
    unsigned char psid[sizeof(PSID)];

    pSD = (PSECURITY_DESCRIPTOR)psd;
    pSidOwner = (PSID)psid;

    if( ( context != NULL ) && ( file_path != NULL ) )
    {
        // Get the owner SID of the file.
        dwRtnCode = GetNamedSecurityInfo(
                    (LPTSTR)file_path,
                    SE_FILE_OBJECT,
                    OWNER_SECURITY_INFORMATION,
                    &pSidOwner,
                    NULL,
                    NULL,
                    NULL,
                    &pSD);

        // Check GetLastError for GetSecurityInfo error condition.
        if( dwRtnCode != ERROR_SUCCESS ) 
        {
            result = FALSE;
            goto exit_error;
        }
        
        // First call to LookupAccountSid to get the buffer sizes.
        result = LookupAccountSid(
                        NULL,           // local computer
                        pSidOwner,
                        AcctName,
                        (LPDWORD)&dwAcctName,
                        NULL,
                        (LPDWORD)&dwDomainName,
                        &eUse);
        
        // Reallocate memory for the buffers.
        AcctName = (char *)GlobalAlloc( GMEM_FIXED, dwAcctName );
        
        // Check GetLastError for GlobalAlloc error condition.
        if( AcctName == NULL ) 
        {
            result = FALSE;
            goto exit_error;
        }
        
        // Second call to LookupAccountSid to get the account name.
        result = LookupAccountSid(
                NULL,                  // name of local or remote computer
                pSidOwner,             // security identifier
                AcctName,              // account name buffer
                (LPDWORD)&dwAcctName,  // size of account name buffer
                NULL,			       // domain name
                (LPDWORD)&dwDomainName,    // size of domain name buffer
                &eUse);                    // SID type
        
        if( ( result ) == TRUE && ( context->filter_argument != NULL ) )
        {
            if( strcmp( context->filter_argument, AcctName ) == 0 )
            {
                result = TRUE;
                goto exit_gracefully;
            }
        }
    }

exit_gracefully:

    GlobalFree( (HGLOBAL)AcctName );

exit_error:

    GlobalFree( (HGLOBAL)pSD );
    GlobalFree( (HGLOBAL)pSidOwner );

    return result;
}

#else

osi_bool filter_user( const char *file_path, FILTER_CONTEXT *context )
{
    struct passwd *file_owner;

    if( context != NULL )
    {
        file_owner = getpwuid( context->file_stats->st_uid );

        if( ( file_owner != NULL ) && ( context->filter_argument != NULL ) )
        {
            if( strcmp( file_owner->pw_name, context->filter_argument ) == 0 )
            {
                return TRUE;
            }
        }
    }

    return FALSE;
}

#endif

