///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file StandardLinearControllers.h
 * \brief Contains the definition of the linear interpolation controller classes.
 */

#ifndef __OVITO_STD_LINEAR_CONTROLLERS_H
#define __OVITO_STD_LINEAR_CONTROLLERS_H

#include <core/Core.h>
#include "StandardKeyedControllers.h"

namespace Core {


/**
 * \brief Implementation of the key interpolator concept that performs linear interpolation.
 *
 * This class calculates a value in the [0,1] range based on the input time and the times of the two input keys.
 * The returned value specifies the weighting of the two key values.
 *
 * This class is used with the linear interpolation controller.
 *
 * \author Alexander Stukowski
 */
template<typename ValueType>
struct LinearKeyInterpolator {
	ValueType operator()(const TimeTicks& time,
						 const pair<TimeTicks, ValueType>& key1,
						 const pair<TimeTicks, ValueType>& key2) const {
		OVITO_ASSERT(key2.first > key1.first);
		FloatType t = (FloatType)(time - key1.first) / (key2.first - key1.first);
		LinearValueInterpolator<ValueType> valueInterpolator;
		return valueInterpolator(t, key1.second, key2.second);
	}
};

/**
 * \brief A keyed controller that interpolates between float values using
 *        a linear interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT LinearFloatController : public StandardKeyedController<FloatController, FloatType, FloatType, FloatType, LinearKeyInterpolator<FloatType> > {
public:
	LinearFloatController(bool isLoading = false) : StandardKeyedController<FloatController, FloatType, FloatType, FloatType, LinearKeyInterpolator<FloatType> >(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LinearFloatController)
};

/**
 * \brief A keyed controller that interpolates between integer values using
 *        a linear interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT LinearIntegerController : public StandardKeyedController<IntegerController, int, int, int, LinearKeyInterpolator<int> > {
public:
	LinearIntegerController(bool isLoading = false) : StandardKeyedController<IntegerController, int, int, int, LinearKeyInterpolator<int> >(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LinearIntegerController)
};

/**
 * \brief A keyed controller that interpolates between Vector3 values using
 *        a linear interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT LinearVectorController : public StandardKeyedController<VectorController, Vector3, Vector3, NullVector, LinearKeyInterpolator<Vector3> > {
public:
	LinearVectorController(bool isLoading = false) : StandardKeyedController<VectorController, Vector3, Vector3, NullVector, LinearKeyInterpolator<Vector3> >(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LinearVectorController)
};

/**
 * \brief A keyed controller that interpolates between position values using
 *        a linear interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT LinearPositionController : public KeyedPositionController<Vector3, LinearKeyInterpolator<Vector3> > {
public:
	LinearPositionController(bool isLoading = false) : KeyedPositionController<Vector3, LinearKeyInterpolator<Vector3> >(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LinearPositionController)
};

/**
 * \brief A keyed controller that interpolates between Base::Rotation values using
 *        a linear interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT LinearRotationController : public KeyedRotationController<Rotation, LinearKeyInterpolator<Rotation> > {
public:
	LinearRotationController(bool isLoading = false) : KeyedRotationController<Rotation, LinearKeyInterpolator<Rotation> >(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LinearRotationController)
};

/**
 * \brief A keyed controller that interpolates between Base::Scaling values using
 *        a linear interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT LinearScalingController : public KeyedScalingController<Scaling, LinearKeyInterpolator<Scaling> > {
public:
	LinearScalingController(bool isLoading = false) : KeyedScalingController<Scaling, LinearKeyInterpolator<Scaling> >(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(LinearScalingController)
};

};

#endif // __OVITO_STD_LINEAR_CONTROLLERS_H
