///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/ModifiedObject.h>
#include <core/undo/UndoManager.h>

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ModifiedObject, SceneObject)
DEFINE_REFERENCE_FIELD(ModifiedObject, SceneObject, "InputObject", _inputObject)
DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(ModifiedObject, ModifierApplication, "ModifierApplications", PROPERTY_FIELD_ALWAYS_CLONE, apps)
SET_PROPERTY_FIELD_LABEL(ModifiedObject, _inputObject, "Input")
SET_PROPERTY_FIELD_LABEL(ModifiedObject, apps, "Modifier Applications")

/******************************************************************************
* Default constructor.
******************************************************************************/
ModifiedObject::ModifiedObject(bool isLoading) : SceneObject(isLoading), pipelineCacheIndex(-1)
{
	INIT_PROPERTY_FIELD(ModifiedObject, _inputObject);
	INIT_PROPERTY_FIELD(ModifiedObject, apps);
}

/******************************************************************************
* Asks the object for its validity interval at the given time.
******************************************************************************/
TimeInterval ModifiedObject::objectValidity(TimeTicks time)
{
	return TimeForever;
}

/******************************************************************************
* Render the object into the viewport.
******************************************************************************/
void ModifiedObject::renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp)
{
	// A ModifiedObject should never be the result of the geometry pipeline.
	OVITO_ASSERT_MSG(false, "ModifiedObject::renderObject()", "A ModifiedObject should not be rendered in the viewports.");
}

/******************************************************************************
* Returns the bounding box of the object in local object coordinates.
******************************************************************************/
Box3 ModifiedObject::boundingBox(TimeTicks time, ObjectNode* contextNode)
{
	// A ModifiedObject should never be the result of the geometry pipeline.
	OVITO_ASSERT_MSG(false, "ModifiedObject::boundingBox()", "A ModifiedObject should not be rendered in the viewports.");
	return Box3();
}

/******************************************************************************
* Asks the object for the result of the geometry pipeline at the given time
* up to a given point in the modifier stack.
* If upToHere is NULL then the complete modifier stack will be evaluated.
* Otherwise only the modifiers in the pipeline before the given point will be applied to the
* result. The 'include' specifies whether the last modifier given by 'upToHere'
* will be applied.
******************************************************************************/
PipelineFlowState ModifiedObject::evalObject(TimeTicks time, ModifierApplication* upToHere, bool including)
{
	UndoSuspender undoSuspender;	// Do not create undo records for this.
	PipelineFlowState state;

	// The index up to which to evaluate the stack.
	int upToHereIndex = modifierApplications().size();
	if(upToHere != NULL) {
		upToHereIndex = modifierApplications().indexOf(upToHere);
		if(including) upToHereIndex++;
	}

	// The index from which on the stack should be evaluated.
	int fromHereIndex = 0;
	// Use the cached result if possible.
	if(pipelineCacheIndex >= 0 && pipelineCacheIndex <= upToHereIndex && pipelineCache.stateValidity().contains(time)) {
		fromHereIndex = pipelineCacheIndex;
		state = pipelineCache;
	}
	else {
		// Evaluate the geometry pipeline of the input object.
		if(!inputObject()) return PipelineFlowState();
		state = inputObject()->evalObject(time);
	}

	// Clear cache and then rebuild below.
	invalidatePipelineCache();

    // Apply the modifiers, one after another.
	int stackIndex;
	for(stackIndex = fromHereIndex; stackIndex < upToHereIndex; stackIndex++) {
    	ModifierApplication* app = modifierApplications()[stackIndex];
    	CHECK_OBJECT_POINTER(app);
		Modifier* mod = app->modifier();
		CHECK_OBJECT_POINTER(mod);

		// Skip disabled modifiers.
		if(!mod->isModifierEnabled()) {
	    	// Reset evaluation status.
	    	app->setStatus(EvaluationStatus());
			continue;
		}

		// Put evaluation result into cache if the next modifier is changing alot (because it is being edited).
		if(mod->modifierValidity(time).isEmpty()) {
			pipelineCache = state;
			pipelineCacheIndex = stackIndex;
		}

		// Apply modifier.
		if(state.result() != NULL) {
			app->setStatus(mod->modifyObject(time, app, state));
		}
		else {
			app->setStatus(EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Modifier did not receive any input object.")));
		}
	}

	// Cache the final result.
	if(pipelineCacheIndex < 0 && state.result() != NULL) {
		pipelineCache = state;
		pipelineCacheIndex = stackIndex;
	}

	return state;
}

/******************************************************************************
* Inserts the given modifier into this object.
******************************************************************************/
ModifierApplication* ModifiedObject::insertModifier(Modifier* modifier, int atIndex)
{
	CHECK_OBJECT_POINTER(modifier);

	// Create a mod app.
	ModifierApplication::SmartPtr modApp(new ModifierApplication(modifier));
	insertModifierApplication(modApp.get(), atIndex);
	return modApp.get();
}

/******************************************************************************
* Inserts the given modifier into this object.
******************************************************************************/
void ModifiedObject::insertModifierApplication(ModifierApplication* modApp, int atIndex)
{
	OVITO_ASSERT(atIndex >= 0);
	CHECK_OBJECT_POINTER(modApp);
	atIndex = min(atIndex, apps.size());
	apps.insert(atIndex, modApp);

	if(modApp->modifier()) {
		modApp->modifier()->initializeModifier(this, modApp);
	}
}

/******************************************************************************
* Removes the given modifier application.
******************************************************************************/
void ModifiedObject::removeModifier(ModifierApplication* app)
{
	CHECK_OBJECT_POINTER(app);
	OVITO_ASSERT(app->modifiedObject() == this);

	int index = apps.indexOf(app);
	OVITO_ASSERT(index >= 0);

	apps.remove(index);
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool ModifiedObject::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == inputObject()) {
		if(msg->type() == REFTARGET_CHANGED) {
			// If the input object has changed then the whole modifier stack needs
			// to be informed of this.
			notifyModifiersInputChanged(apps.size());
		}
	}
	else {
		if(msg->type() == REFTARGET_CHANGED || msg->type() == MODIFIER_ENABLED) {
			// If one of the modifiers has changed then all other modifiers
			// following it in the stack need to be informed.
			int index = apps.indexOf(source);
			if(index != -1) {
				notifyModifiersInputChanged(index);
				if(msg->type() == MODIFIER_ENABLED)
					notifyDependents(REFTARGET_CHANGED);
			}
		}
	}
	return SceneObject::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Is called when the value of a reference field of this RefMaker changes.
******************************************************************************/
void ModifiedObject::onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget)
{
	// If the input object has been replaced then the whole modifier stack needs
	// to be informed.
	if(field == PROPERTY_FIELD_DESCRIPTOR(ModifiedObject, _inputObject)) {
		notifyModifiersInputChanged(apps.size());
	}
	SceneObject::onRefTargetReplaced(field, oldTarget, newTarget);
}

/******************************************************************************
* Is called when a reference target has been added to a list reference field of this RefMaker.
******************************************************************************/
void ModifiedObject::onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex)
{
	// If a new modifier has been inserted into the stack then all
	// modifiers following it in the stack need to be informed.
	if(field == PROPERTY_FIELD_DESCRIPTOR(ModifiedObject, apps)) {
		notifyModifiersInputChanged(listIndex);
	}
	SceneObject::onRefTargetInserted(field, newTarget, listIndex);
}

/******************************************************************************
* Is called when a reference target has been removed from a list reference field of this RefMaker.
******************************************************************************/
void ModifiedObject::onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex)
{
	// If a new modifier has been removed from the stack then all
	// modifiers following it in the stack need to be informed.
	if(field == PROPERTY_FIELD_DESCRIPTOR(ModifiedObject, apps)) {
		notifyModifiersInputChanged(listIndex);
	}
	SceneObject::onRefTargetRemoved(field, oldTarget, listIndex);
}

/******************************************************************************
* Notifies all modifiers from the given index on that their input has changed.
******************************************************************************/
void ModifiedObject::notifyModifiersInputChanged(int changedIndex)
{
	if(getPluginClassFlag(FLAG_OBJ_BEING_LOADED))
		return;	// Do not send messages when the modifiers are being loaded.

	// Invalidate the internal cache if it is affected by the changed modifier.
	if(changedIndex < pipelineCacheIndex || changedIndex == apps.size())
		invalidatePipelineCache();

	// Call the onInputChanged() method for all affected modifiers.
	while(++changedIndex < apps.size()) {
		ModifierApplication* app = apps[changedIndex];
		if(app && app->modifier()) {
			CHECK_POINTER(app->modifier());
			app->modifier()->onInputChanged(app);
		}
	}
}

};
