///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file BoxCreationMode.h 
 * \brief Contains the definition of the StdObjects::BoxCreationMode class. 
 */

#include <core/Core.h>
#include <core/scene/objects/CreationMode.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>

#include <stdobjects/StdObjects.h>
#include "BoxObject.h"

namespace StdObjects {

/**
 * \brief This input mode lets the user creat a procedural BoxObject in the viewports.
 * 
 * \author Alexander Stukowski
 */
class BoxCreationMode : public SimpleCreationMode
{
public:
	
	/// \brief Default constructor.
	BoxCreationMode() : SimpleCreationMode(tr("Create Box"), tr("Box")) {}
	
protected:

	/// \brief This creates the actual scene object.
	/// \return The scene object to be inserted into the scene.
	virtual SceneObject::SmartPtr createObject() {
		return new BoxObject();
	}

	/// \brief Will be called when the user presses the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMousePressed(QMouseEvent* event) {
		SimpleCreationMode::onMousePressed(event);

		if(clickCount() == 1) {
			// Take click point as first box corner.
			cornerPoint1 = cornerPoint2 = event->pos();
			corner1 = ORIGIN;
			if(!viewport()->snapPoint(cornerPoint1, corner1))
				onAbort();
			else
				onMouseDrag(event);	// This moves the new object to the current click point.
		}
		else if(clickCount() == 2) {
			onFinish();
		}
	}

	/// \brief Will be called when the user releases the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMouseReleased(QMouseEvent* event) {
		SimpleCreationMode::onMouseReleased(event);
		if(clickCount() == 2) {
			if(cornerPoint1 != cornerPoint2) {
				beginAdjustOperation();
				SNAPPING_MANAGER.clearLastSnapPoint();
			}
			else
				onAbort();
		}
	}

	/// \brief Will be called when the user moves the mouse while the operation is active.
	/// \param event The mouse event to be handled.
	virtual void onMouseDrag(QMouseEvent* event) {
		SimpleCreationMode::onMouseDrag(event);
		if(clickCount() == 1) {
			// Take mouse position as second box corner.
			cornerPoint2 = event->pos();
			corner2 = ORIGIN;
			if(!viewport()->snapPoint(cornerPoint2, corner2))
				return;
			
			abortAdjustOperation();
			beginAdjustOperation();

			// Compute box center
			Point3 center = (corner1 + corner2) / 2.0;

			// Position scene node.
			AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(center - ORIGIN);			
			objectNode()->transformationController()->setValue(0, tm);

			// Resize box.
			static_object_cast<BoxObject>(object())->widthController()->setValue(0, abs(corner1.X - corner2.X));
			static_object_cast<BoxObject>(object())->lengthController()->setValue(0, abs(corner1.Y - corner2.Y));
		}
		else {
			abortAdjustOperation();
			beginAdjustOperation();

			FloatType h = viewport()->grid().computeConstructionLength(Ray3(corner2, Vector3(0,0,1)), cornerPoint2, event->pos());
			static_object_cast<BoxObject>(object())->heightController()->setValue(0, h);
		}

		// Update viewports immediately.
		VIEWPORT_MANAGER.processViewportUpdates();
	}

private:

	/// The first corner of the box in construction plane coordinates.
    Point3 corner1;

	/// The screen coordinates of the first corner.
	QPoint cornerPoint1;

	/// The second corner of the box in construction plane coordinates.
	Point3 corner2;

	/// The screen coordinates of the second corner.
	QPoint cornerPoint2;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(BoxCreationMode)
};

};
