
/* 
 * Copyright (C) 2000-2004 the xine project
 * Copyright (C) 2004-2005 Maximilian Schwerin
 * 
 * This file is part of oxine a free media player.
 * The code in this file was copied from the xine-ui project.
 * 
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: lang.c 2034 2007-05-14 17:30:04Z mschwerin $
 *
 */
#include "config.h"

#include "string.h"
#include "stdlib.h"

#include "i18n.h"
#include "lang.h"

/*
 * Our LC_MESSAGES handling.
 */

/*
  locale -a tell us (* mean we already handle):
  -----------------
  *  C
  *  POSIX
     af_ZA
     ar_AE
     ar_BH
     ar_DZ
     ar_EG
     ar_IN
     ar_IQ
     ar_JO
     ar_KW
     ar_LB
     ar_LY
     ar_MA
     ar_OM
     ar_QA
     ar_SA
     ar_SD
     ar_SY
     ar_TN
     ar_YE
     be_BY
     bg_BG
     bokmal
     bokmål
     ca_ES
     ca_ES@euro
     catalan
     croatian
  *  cs_CZ
  *  cs_CZ.UTF8
     czech
     da_DK
     danish
     dansk
     de_AT
     de_AT@euro
     de_BE
     de_BE@euro
     de_CH
  *  de_DE
  *  de_DE@euro
     de_LU
     de_LU@euro
  *  deutsch
     dutch
     eesti
     el_GR
     en_AU
     en_BW
     en_CA
     en_DK
  *  en_GB
     en_HK
     en_IE
     en_IE@euro
     en_IN
     en_NZ
     en_PH
     en_SG
     en_US
     en_ZA
     en_ZW
     es_AR
     es_BO
     es_CL
     es_CO
     es_CR
     es_DO
     es_EC
  *  es_ES
  *  es_ES@euro
     es_GT
     es_HN
     es_MX
     es_NI
     es_PA
     es_PE
     es_PR
     es_PY
     es_SV
     es_US
     es_UY
     es_VE
     estonian
     et_EE
     eu_ES
     eu_ES@euro
     fa_IR
     fi_FI
     fi_FI@euro
     finnish
     fo_FO
     fr_BE
     fr_BE@euro
     fr_CA
     fr_CH
  *  fr_FR
  *  fr_FR@euro
     fr_LU
     fr_LU@euro
  *  français
  *  french
     ga_IE
     ga_IE@euro
     galego
     galician
  *  german
     gl_ES
     gl_ES@euro
     greek
     gv_GB
     he_IL
     hebrew
     hi_IN
     hr_HR
     hrvatski
     hu_HU
     hungarian
     icelandic
     id_ID
     is_IS
     it_CH
     it_IT
     it_IT@euro
     italian
     iw_IL
     ja_JP
     ja_JP.eucjp
     ja_JP.ujis
     japanese
     japanese.euc
     japanese.sjis
     kl_GL
     ko_KR
     ko_KR.euckr
     ko_KR.utf8
     korean
     korean.euc
     kw_GB
     lithuanian
     lt_LT
     lv_LV
     mk_MK
     mr_IN
     ms_MY
     mt_MT
     nb_NO
     nb_NO.ISO-8859-1
     nl_BE
     nl_BE@euro
     nl_NL
     nl_NL@euro
     nn_NO
     no_NO
     norwegian
     nynorsk
 *   pl_PL
 *   pl_PL.UTF-8
     polish
     portuguese
  *  pt_BR
     pt_PT
     pt_PT@euro
     ro_RO
     romanian
     ru_RU
     ru_RU.koi8r
     ru_UA
     russian
     sk_SK
     sl_SI
     slovak
     slovene
     slovenian
  *  spanish
     sq_AL
     sr_YU
     sr_YU@cyrillic
     sv_FI
     sv_FI@euro
     sv_SE
     swedish
     ta_IN
     te_IN
     th_TH
     thai
     tr_TR
     turkish
  *  uk_UA
     vi_VN
     zh_CN
     zh_CN.gb18030
     zh_CN.gbk
     zh_HK
     zh_TW
     zh_TW.euctw

*/

/* Global definition */
static langs_t _langs[] = {
    /* English section */
    {"C", ENGLISH, "en", NULL},                                 /* WARNING: This should be the first entry */
    {"POSIX", ENGLISH, "en", NULL},
    {"en_GB", ENGLISH, "en", "ISO-8859-1"},

    /* French section */
    {"fr_FR", FRENCH, "fr", "ISO-8859-1"},
    {"fr_FR@euro", FRENCH, "fr", "ISO-8859-1"},
    {"français", FRENCH, "fr", "ISO-8859-1"},
    {"french", FRENCH, "fr", "ISO-8859-1"},

    /* German section */
    {"de_DE", GERMAN, "de", "ISO-8859-1"},
    {"de_DE@euro", GERMAN, "de", "ISO-8859-1"},
    {"deutsch", GERMAN, "de", "ISO-8859-1"},
    {"german", GERMAN, "de", "ISO-8859-1"},

    /* Portuguese section */
    {"pt_BR", PORTUGUESE, "pt_BR", "ISO-8859-1"},

    /* Spanish section */
    {"es_ES", SPANISH, "es", "ISO-8859-1"},
    {"es_ES@euro", SPANISH, "es", "ISO-8859-1"},
    {"spanish", SPANISH, "es", "ISO-8859-1"},

    /* Polish */
    {"pl_PL", POLISH, "pl", "ISO-8859-2"},
    {"pl_PL.UTF-8", POLISH, "pl", "ISO-8859-2"},

    /* Ukrainian */
    {"uk_UA", UKRAINIAN, "uk", "KOI8-U"},
    {"uk_UA.UTF-8", UKRAINIAN, "uk", "KOI8-U"},

    /* Czech */
    {"cs_CZ", CZECH, "cs", "ISO-8859-2"},
    {"cs_CZ.UTF-8", CZECH, "cs", "ISO-8859-2"},

    /* NORWEGIAN */
    {"nb_NO", NORWEGIAN_BOKMAL, "nb", "ISO-8859-1"},
    {"nb_NO.ISO-8859-1", NORWEGIAN_BOKMAL, "nb", "ISO-8859-1"},

    /* The ultimate solution */
    {NULL, ENGLISH, "en", "ISO-8859-1"}
};

/* ISO 639-1 */
typedef struct {
    char *two_letters;
    char *language;
} iso639_1_t;

/*
 * Return language from iso639-1 two letters.
 */
const char *
get_language_from_iso639_1 (char *two_letters)
{
    iso639_1_t iso639_1[] = {
        {"ab", _("Abkhazian")},
        {"aa", _("Afar")},
        {"af", _("Afrikaans")},
        {"ak", _("Akan")},
        {"sq", _("Albanian")},
        {"am", _("Amharic")},
        {"ar", _("Arabic")},
        {"an", _("Aragonese")},
        {"hy", _("Armenian")},
        {"as", _("Assamese")},
        {"av", _("Avaric")},
        {"ae", _("Avestan")},
        {"ay", _("Aymara")},
        {"az", _("Azerbaijani")},
        {"bm", _("Bambara")},
        {"ba", _("Bashkir")},
        {"eu", _("Basque")},
        {"be", _("Belarusian")},
        {"bn", _("Bengali")},
        {"bh", _("Bihari")},
        {"bi", _("Bislama")},
        {"bs", _("Bosnian")},
        {"br", _("Breton")},
        {"bg", _("Bulgarian")},
        {"my", _("Burmese")},
        {"ca", _("Catalan")},
        {"ch", _("Chamorro")},
        {"ce", _("Chechen")},
        {"zh", _("Chinese")},
        {"cv", _("Chuvash")},
        {"kw", _("Cornish")},
        {"co", _("Corsican")},
        {"cr", _("Cree")},
        {"hr", _("Croatian")},
        {"cs", _("Czech")},
        {"da", _("Danish")},
        {"dv", _("Divehi")},
        {"nl", _("Dutch")},
        {"dz", _("Dzongkha")},
        {"en", _("English")},
        {"eo", _("Esperanto")},
        {"et", _("Estonian")},
        {"ee", _("Ewe")},
        {"fo", _("Faroese")},
        {"fj", _("Fijian")},
        {"fi", _("Finnish")},
        {"fr", _("French")},
        {"fy", _("Frisian")},
        {"ff", _("Fulah")},
        {"gd", _("Gaelic")},
        {"gl", _("Galician")},
        {"lg", _("Ganda")},
        {"ka", _("Georgian")},
        {"de", _("German")},
        {"el", _("Greek")},
        {"kl", _("Greenlandic")},
        {"gn", _("Guarani")},
        {"gu", _("Gujarati")},
        {"ha", _("Hausa")},
        {"he", _("Hebrew")},
        {"hz", _("Herero")},
        {"hi", _("Hindi")},
        {"ho", _("Hiri Motu")},
        {"hu", _("Hungarian")},
        {"is", _("Icelandic")},
        {"io", _("Ido")},
        {"ig", _("Igbo")},
        {"id", _("Indonesian")},
        {"ia", _("Interlingua")},
        {"ie", _("Interlingue")},
        {"iu", _("Inuktitut")},
        {"ik", _("Inupiaq")},
        {"iw", _("Hebrew")},
        {"ga", _("Irish")},
        {"it", _("Italian")},
        {"ja", _("Japanese")},
        {"jv", _("Javanese")},
        {"kn", _("Kannada")},
        {"kr", _("Kanuri")},
        {"ks", _("Kashmiri")},
        {"kk", _("Kazakh")},
        {"km", _("Khmer")},
        {"ki", _("Kikuyu")},
        {"rw", _("Kinyarwanda")},
        {"ky", _("Kirghiz")},
        {"kv", _("Komi")},
        {"kg", _("Kongo")},
        {"ko", _("Korean")},
        {"kj", _("Kuanyama")},
        {"ku", _("Kurdish")},
        {"lo", _("Lao")},
        {"la", _("Latin")},
        {"lv", _("Latvian")},
        {"lb", _("Letzeburgesch")},
        {"li", _("Limburgan")},
        {"ln", _("Lingala")},
        {"lt", _("Lithuanian")},
        {"lu", _("Luba-Katanga")},
        {"mk", _("Macedonian")},
        {"mg", _("Malagasy")},
        {"ms", _("Malay")},
        {"ml", _("Malayalam")},
        {"mt", _("Maltese")},
        {"gv", _("Manx")},
        {"mi", _("Maori")},
        {"mr", _("Marathi")},
        {"mh", _("Marshallese")},
        {"mo", _("Moldavian")},
        {"mn", _("Mongolian")},
        {"na", _("Nauru")},
        {"nv", _("Navaho")},
        {"nd", _("Ndebele North")},
        {"nr", _("Ndebele South")},
        {"ng", _("Ndonga")},
        {"ne", _("Nepali")},
        {"se", _("Northern Sami")},
        {"no", _("Norwegian")},
        {"nb", _("Norwegian (Bokmål)")},
        {"nn", _("Norwegian (Nynorsk)")},
        {"ny", _("Nyanja")},
        {"oc", _("Provençal")},
        {"oj", _("Ojibwa")},
        {"cu", _("Old Bulgarian")},
        {"or", _("Oriya")},
        {"om", _("Oromo")},
        {"os", _("Ossetian")},
        {"pi", _("Pali")},
        {"pa", _("Panjabi")},
        {"fa", _("Persian")},
        {"pl", _("Polish")},
        {"pt", _("Portuguese")},
        {"ps", _("Pushto")},
        {"qu", _("Quechua")},
        {"rm", _("Raeto-Romance")},
        {"ro", _("Romanian")},
        {"rn", _("Rundi")},
        {"ru", _("Russian")},
        {"sm", _("Samoan")},
        {"sg", _("Sango")},
        {"sa", _("Sanskrit")},
        {"sc", _("Sardinian")},
        {"sr", _("Serbian")},
        {"sn", _("Shona")},
        {"ii", _("Sichuan Yi")},
        {"sd", _("Sindhi")},
        {"si", _("Sinhalese")},
        {"sk", _("Slovak")},
        {"sl", _("Slovenian")},
        {"so", _("Somali")},
        {"st", _("Sotho")},
        {"es", _("Spanish")},
        {"su", _("Sundanese")},
        {"sw", _("Swahili")},
        {"ss", _("Swati")},
        {"sv", _("Swedish")},
        {"tl", _("Tagalog")},
        {"ty", _("Tahitian")},
        {"tg", _("Tajik")},
        {"ta", _("Tamil")},
        {"tt", _("Tatar")},
        {"te", _("Telugu")},
        {"th", _("Thai")},
        {"bo", _("Tibetan")},
        {"ti", _("Tigrinya")},
        {"to", _("Tonga")},
        {"ts", _("Tsonga")},
        {"tn", _("Tswana")},
        {"tr", _("Turkish")},
        {"tk", _("Turkmen")},
        {"tw", _("Twi")},
        {"ug", _("Uighur")},
        {"uk", _("Ukrainian")},
        {"ur", _("Urdu")},
        {"uz", _("Uzbek")},
        {"ve", _("Venda")},
        {"vi", _("Vietnamese")},
        {"vo", _("Volapük")},
        {"wa", _("Walloon")},
        {"cy", _("Welsh")},
        {"wo", _("Wolof")},
        {"xh", _("Xhosa")},
        {"yi", _("Yiddish")},
        {"yo", _("Yoruba")},
        {"za", _("Zhuang")},
        {"zu", _("Zulu")},
        {NULL, NULL}
    };

    if (two_letters) {
        char *tl = two_letters;

        while ((*tl == ' ') && (*tl != '\0'))
            tl++;

        if (tl && (strlen (tl) == 2)) {
            int i;

            for (i = 0; iso639_1[i].two_letters; i++) {
                if (!strcmp (iso639_1[i].two_letters, tl)) {
                    return iso639_1[i].language;
                }
            }
        }
    }

    return two_letters ? two_letters : _("Unknown");
}

/*
 * Return a langs_t pointer on static struct langs[];
 */
const langs_t *
get_lang (void)
{
    langs_t *l;
    char *lcmsg = setlocale (LC_MESSAGES, NULL);

    if (lcmsg) {
        for (l = _langs; l->lang != NULL; l++) {
            if (!strncasecmp (l->lang, lcmsg, strlen (lcmsg))) {
                return l;
            }
        }
    }

    return &_langs[0];
}

const langs_t *
get_default_lang (void)
{
    return &_langs[0];
}
