
/*
 * Copyright (C) 2007 Maximilian Schwerin
 *
 * This file is part of wxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_http.c 2644 2007-08-17 14:38:21Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>

#ifdef HAVE_LIBX11
#include <X11/Xlib.h>
#endif

#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "playlist.h"
#include "types.h"
#include "utils.h"

#include "odk.h"
#include "oxine.h"
#include "odk_http.h"

#define QUEUE_LENGTH            5
#define BUFFER_SIZE             2048
#define HTTP_PORT               8080
#define CRLF                    "\r\n"

#define HTTP_VERSION            "HTTP/1.1"

#ifdef HAVE_HTTP
typedef struct {
    odk_input_t class;

    odk_t *odk;
    void (*event_handler) (void *data, oxine_event_t * ev);
} http_input_t;


extern oxine_t *oxine;

static int
http_init (void)
{
    int ret;
    int listen_socket = 0;
    struct sockaddr_in server_addr;

    /* Create socket */
    listen_socket = socket (AF_INET, SOCK_STREAM, 0);
    if (listen_socket == -1) {
        perror ("Server kann Socket nicht erzeugen");
        exit (1);
    }

    /* Socket-Option einstellen: IP/Port-Adresse
     * darf sofort wieder vergeben werden */
    int optval = 1;
    ret = setsockopt (listen_socket, SOL_SOCKET, SO_REUSEADDR,
                      (void *) &optval, sizeof (optval));
    if (ret == -1) {
        perror ("setsockopt() gescheitert");
        exit (1);
    }

    /* Socket mit "Namen", d.h. mit Port verbinden */
    memset ((void *) &server_addr, 0, sizeof (server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = htonl (INADDR_ANY);
    server_addr.sin_port = htons (HTTP_PORT);
    ret = bind (listen_socket, (struct sockaddr *) &server_addr,
                sizeof (server_addr));
    if (ret == -1) {
        perror ("Server kann Port nicht binden");
        exit (1);
    }

    /* Socket fuer eingehende Verbindungen aktivieren */
    ret = listen (listen_socket, QUEUE_LENGTH);
    if (ret == -1) {
        perror ("Server kann nicht an Socket horchen");
        exit (1);
    }

    return listen_socket;
}


static void
http_printf (int client_socket, const char *format, ...)
{
    va_list args;
    char buffer[BUFFER_SIZE];

    va_start (args, format);
    vsnprintf (buffer, BUFFER_SIZE, format, args);
    va_end (args);

    if (write (client_socket, buffer, strlen (buffer)) == -1) {
        perror ("write failed");
        exit (1);
    }

    debug ("> %s", trim_whitespace (buffer));
}


static const char *
http_get_reason_phrase (int status_code)
{
    switch (status_code) {
    case 200:
        return "OK";
    case 303:
        return "See Other";
    case 404:
        return "Not Found";
    case 501:
        return "Not Implemented";
    default:
        return "";
    }
}


static void
http_send_status (int client_socket, int status_code)
{
    http_printf (client_socket,
                 HTTP_VERSION " %d %s" CRLF,
                 status_code, http_get_reason_phrase (status_code));
}


static void
http_send_error (int client_socket, int status_code)
{
    http_send_status (client_socket, status_code);
    http_printf (client_socket,
                 "Content-Type: text/html; charset=UTF-8" CRLF);
    http_printf (client_socket, CRLF);
    http_printf (client_socket,
                 "<html>"
                 "<head><title>wxine - %s</title></head>"
                 "<body>%d - %s</body>"
                 "</html>",
                 http_get_reason_phrase (status_code),
                 status_code, http_get_reason_phrase (status_code));
}


static bool
handle_connection (int client_socket, http_input_t * li)
{
    int ret = 0;
    char *end;
    char buffer[BUFFER_SIZE];

    ret = read (client_socket, buffer, BUFFER_SIZE);
    if (ret == -1) {
        perror ("read failed");
        exit (1);
    }
    buffer[ret] = '\0';

    printf ("%s", buffer);

    char *cmd = strstr (buffer, "cmd=");
    if (cmd) {
        cmd = strdup (&cmd[4]);
        end = index (cmd, ' ');
        if (end) {
            end[0] = '\0';
        }
        end = strstr (cmd, "&gui=");
        if (end) {
            end[0] = '\0';
        }
        end = strstr (cmd, "&prm=");
        if (end) {
            end[0] = '\0';
        }
    }

    char *prm = strstr (buffer, "prm=");
    if (prm) {
        prm = strdup (&prm[4]);
        end = index (prm, ' ');
        if (end) {
            end[0] = '\0';
        }
        end = strstr (prm, "&cmd=");
        if (end) {
            end[0] = '\0';
        }
        end = strstr (prm, "&gui=");
        if (end) {
            end[0] = '\0';
        }
    }

    char *gui = strstr (buffer, "gui=");
    if (gui) {
        gui = strdup (&gui[4]);
        end = index (gui, ' ');
        if (end) {
            end[0] = '\0';
        }
        end = strstr (gui, "&cmd=");
        if (end) {
            end[0] = '\0';
        }
        end = strstr (gui, "&prm=");
        if (end) {
            end[0] = '\0';
        }
    }

    debug ("cmd: %s", cmd);
    debug ("prm: %s", prm);
    debug ("gui: %s", gui);

    if (cmd) {
        if (strcasecmp (cmd, "play_file") == 0) {
            char *mrl = prm;
            char *title = create_title (mrl);

            filelist_t *arglist = NULL;
            filelist_ref_set (&arglist, filelist_new (NULL, NULL, "/",
                                                      ALLOW_FILES_MULTIMEDIA));

            playlist_clear (oxine->rw_playlist);
            playlist_add_fileitem (oxine->rw_playlist,
                                   filelist_add (arglist, title, mrl,
                                                 FILE_TYPE_UNKNOWN));

            filelist_ref_set (&arglist, NULL);

            ho_free (title);

            playlist_play_first (oxine->rw_playlist);
        }
        else if (strcasecmp (cmd, "set_fullscreen") == 0) {
            bool fullscreen = (strcasecmp (prm, "true") == 0);
            odk_window_set_fullscreen (li->odk, fullscreen);
        }
        else if (strcasecmp (cmd, "set_geometry") == 0) {
            int x = 0;
            int y = 0;
            int w = 0;
            int h = 0;
            XParseGeometry (prm, &x, &y,
                            ((unsigned int *) &w), ((unsigned int *) &h));
            odk_window_set_geometry (li->odk, x, y, w, h);
        }
        else if (strcasecmp (cmd, "meta_info") == 0) {
            http_send_status (client_socket, 200);
            http_printf (client_socket,
                         "Content-Type: text/xml; charset=UTF-8" CRLF);
            http_printf (client_socket, CRLF);
            http_printf (client_socket,
                         "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" CRLF);
            http_printf (client_socket, "<wxine_meta_info>" CRLF);
            if (odk_current_is_playback_mode (li->odk)) {
                if (odk_get_meta_info (li->odk, META_INFO_TITLE)) {
                    http_printf (client_socket, "<title>%s</title>",
                                 odk_get_meta_info (li->odk,
                                                    META_INFO_TITLE));
                }
                else {
                    char *title =
                        create_title (odk_current_get_mrl (li->odk));
                    http_printf (client_socket, "<title>%s</title>", title);
                    ho_free (title);
                }
                if (odk_get_meta_info (li->odk, META_INFO_ARTIST)) {
                    http_printf (client_socket, "<artist>%s</artist>",
                                 odk_get_meta_info (li->odk,
                                                    META_INFO_ARTIST));
                }
                if (odk_get_meta_info (li->odk, META_INFO_ALBUM)) {
                    http_printf (client_socket, "<album>%s</album>",
                                 odk_get_meta_info (li->odk,
                                                    META_INFO_ALBUM));
                }
                if (odk_get_meta_info (li->odk, META_INFO_GENRE)) {
                    http_printf (client_socket, "<genre>%s</genre>",
                                 odk_get_meta_info (li->odk,
                                                    META_INFO_GENRE));
                }
                if (odk_get_meta_info (li->odk, META_INFO_YEAR)) {
                    http_printf (client_socket, "<year>%s</year>",
                                 odk_get_meta_info (li->odk, META_INFO_YEAR));
                }
                if (odk_get_meta_info (li->odk, META_INFO_TRACK_NUMBER)) {
                    http_printf (client_socket, "<track>%s</track>",
                                 odk_get_meta_info (li->odk,
                                                    META_INFO_TRACK_NUMBER));
                }
            }
            http_printf (client_socket, "</wxine_meta_info>" CRLF);
            close (client_socket);
            return true;
        }
    }

    if (gui) {
        http_send_status (client_socket, 303);
        http_printf (client_socket, "Location: %s" CRLF, gui);
        http_printf (client_socket, CRLF);
    }
    else {
        http_send_status (client_socket, 200);
        http_printf (client_socket, CRLF CRLF);
        close (client_socket);
        return false;
    }

    close (client_socket);
    return true;
}


static void *
http_thread (void *this)
{
    http_input_t *li = (http_input_t *) this;

    info (_("Successfully started the HTTP input plugin."));
    debug ("       HTTP thread: 0x%X", (int) pthread_self ());

    struct sockaddr_in client_addr;
    int client_addr_size;
    int listen_socket = http_init ();
    int client_socket;

    while (1) {
        client_addr_size = sizeof (client_addr);
        client_socket = accept (listen_socket,
                                (struct sockaddr *) &client_addr,
                                &client_addr_size);
        if (client_socket == -1) {
            perror ("interner Fehler: accept() ging schief");
            exit (1);
        }

        handle_connection (client_socket, li);
    }

    return NULL;
}


static void
http_key_handler (oxine_event_t * ev)
{
    switch (ev->source.key) {
    case OXINE_KEY_SHOW_OSD:
        if (odk_window_is_fullscreen (oxine->odk)) {
            odk_window_set_fullscreen (oxine->odk, false);
        }
        else {
            odk_window_set_fullscreen (oxine->odk, true);
        }
        break;
    default:
        break;
    }
}


static void
http_event_handler (void *p, oxine_event_t * event)
{
    switch (event->type) {
    case OXINE_EVENT_KEY:
        http_key_handler (event);
        break;
    case OXINE_EVENT_PLAYBACK_STOPPED:
    case OXINE_EVENT_PLAYBACK_FINISHED:
        odk_window_set_fullscreen (oxine->odk, false);
        odk_window_set_geometry (oxine->odk, 0, 0, 1, 1);
        break;
    default:
        break;
    }
}


odk_input_t *
start_http (odk_t * odk, event_handler_t event_handler)
{
    http_input_t *li = ho_new (http_input_t);
    li->odk = odk;
    li->event_handler = event_handler;

    odk_add_event_handler (odk, http_event_handler, odk,
                           EVENT_HANDLER_PRIORITY_LOW);

    /* Create the thread to handle HTTP commands. */
    if (pthread_create (&li->class.thread, NULL, http_thread, li) != 0) {
        error (_("Could not create HTTP thread: %s!"), strerror (errno));

        ho_free (li);
        return NULL;
    }

    return (odk_input_t *) li;
}


void
stop_http (odk_input_t * oi)
{
    http_input_t *li = (http_input_t *) oi;
    pthread_cancel (oi->thread);
    ho_free (li);

    info (_("Successfully stopped the HTTP input plugin."));
}


#endif /* HAVE_HTTP */
