
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_image.c 2440 2007-07-05 08:28:13Z mschwerin $
 */
#include "config.h"

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mutex.h"
#include "odk.h"
#include "otk.h"

#ifdef HAVE_OSD_IMAGE

typedef struct {
    otk_widget_t widget;

    char *mrl;
    odk_osd_image_t *odk_image;

    bool show_border;
} otk_image_t;


void
otk_image_set_mrl (otk_widget_t * this, const char *mrl)
{
    otk_image_t *image = (otk_image_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_IMAGE))
        return;

    if (!image->mrl && !mrl)
        return;

    if (image->mrl && mrl && (strcmp (image->mrl, mrl) == 0))
        return;

    ho_free (image->mrl);
    image->mrl = mrl ? ho_strdup (mrl) : NULL;

    odk_osd_image_t *old_image = image->odk_image;
    image->odk_image = NULL;

    this->draw (this);
    odk_osd_hide_image (old_image);
}


void
otk_image_preload (otk_widget_t * this)
{
    otk_image_t *image = (otk_image_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_IMAGE))
        return;

    if (!image->mrl)
        return;

    odk_osd_preload_image (this->odk, image->mrl,
                           this->x, this->y, this->w, this->h,
                           image->show_border, 0x000000, this->alignment);
}


static void
image_hide (otk_widget_t * this)
{
    otk_image_t *image = (otk_image_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_IMAGE))
        return;

    if (!this->is_visible || !image->mrl) {
        odk_osd_hide_image (image->odk_image);
        image->odk_image = NULL;
    }
}


static void
image_draw (otk_widget_t * this)
{
    otk_image_t *image = (otk_image_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_IMAGE))
        return;

    if (!this->is_visible || !image->mrl) {
        odk_osd_hide_image (image->odk_image);
        image->odk_image = NULL;
    }

    else if (!odk_osd_is_image_visible (image->odk_image)) {
        image->odk_image = odk_osd_show_image (this->odk, image->mrl,
                                               this->x, this->y,
                                               this->w, this->h,
                                               image->show_border, 0x000000,
                                               this->alignment);
    }

    this->need_repaint = false;
}


static void
image_destroy (otk_widget_t * this)
{
    otk_image_t *image = (otk_image_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_IMAGE))
        return;

    odk_osd_hide_image (image->odk_image);
    ho_free (image->mrl);
    ho_free (image);
}


otk_widget_t *
otk_image_new (otk_t * otk, int x, int y, int w, int h,
               const char *mrl, int alignment, bool show_border)
{
    otk_image_t *image = ho_new (otk_image_t);

    otk_widget_constructor ((otk_widget_t *) image, otk,
                            OTK_WIDGET_IMAGE, x, y, w, h);

    image->widget.selectable = OTK_SELECTABLE_NONE;

    image->widget.draw = image_draw;
    image->widget.hide = image_hide;
    image->widget.destroy = image_destroy;

    image->widget.alignment = alignment;

    image->mrl = mrl ? ho_strdup (mrl) : NULL;
    image->odk_image = NULL;
    image->show_border = show_border;

    return (otk_widget_t *) image;
}

#endif /* HAVE_OSD_IMAGE */
