#include <qmessagebox.h>
#include <qaction.h>
#include <qpopupmenu.h>
#include <qmutex.h>

#include <unistd.h>		// for getuid and geteuid
#include <sys/types.h>	//

// Tagcoll
#include <ExpressionFilter.h>
#include <FilterChain.h>
#include <TagcollBuilder.h>
#include <HandleMaker.h>
#include <InputMerger.h>
#include <Implications.h>
#include <StdioParserInput.h>
#include <TagcollParser.h>
#include <Exception.h>

//Debtags
#include <Environment.h>
#include <TagDB.h>
#include <Vocabulary.h>

#include "debtagsplugincontainer.h"

#include "helpers.h"

// NApplication
#include "applicationfactory.h"
#include "runcommand.h"

// NPlugin
#include <iprovider.h>
#include <plugincontainer.h>
#include <iprogressobserver.h>
#include "debtagsplugin.h"
#include "relatedplugin.h"
#include "debtagspluginfactory.h"

// NDebtags
#include "stringtagcollbuilder.h"
#include "selectedfacetsmatcher.h"

#include "debtagssettingswidget.h"

extern "C" 
{ 
	NPlugin::PluginContainer* new_debtagsplugin() 
	{
		return new NPlugin::DebtagsPluginContainer;
	} 

	NPlugin::PluginInformation get_pluginInformation()
	{
		return NPlugin::PluginInformation("debtagsplugin", "1.2", "Benjamin Mesing");
	} 
	
// 	void initialize_plugin()
// 	{
// 		static bool debtagsInitialized = false;
// 		if (!debtagsInitialized)
// 		{
// 			std::cerr << "initilizing debtags plugins..."<<endl;
// 			debtagsInitialized = true;
// 			// @warning check if calling the init function can be avoided or centralized
// 			Debtags::Environment::init(false, false);	// initialize the single Environment instance
// 		}
// 	} 

}

/** Initialize the plugin. */
__attribute__ ((constructor)) void init() 
{
	Debtags::Environment::init(false, false);	// initialize the single Environment instance
}
      
      
// __attribute__ ((destructor)) void fini() 
// {
//   /* code here is executed just before dlclose() unloads the module */
// } 


namespace NPlugin
{

DebtagsPluginContainer::DebtagsPluginContainer() :
	_pHandleMaker(0), _pCollection(0)
{
	DebtagsPluginFactory::getInstance()->setContainer(this);
	_pCommand = 0;
	_pRelatedPlugin = 0;
	_pDebtagsPlugin = 0;
	_pSettingsWidget = 0;
	addPlugin("DebtagsPlugin");
	addPlugin("RelatedPlugin");
	_pDebtagsUpdateAction = new QAction(QObject::tr("Debtags Update"), 0, this, "DebtagsUpdateAction");
	
	connect( _pDebtagsUpdateAction, SIGNAL(activated()), SLOT(onDebtagsUpdate()) );
	_debtagsEnabled=false;
	
}
 
DebtagsPluginContainer::~DebtagsPluginContainer()
{
	delete _pDebtagsUpdateAction; 
	delete _pCollection;
	delete _pCommand;
}

/////////////////////////////////////////////////////
// Plugin Container Interface
/////////////////////////////////////////////////////

bool DebtagsPluginContainer::init(IProvider* pProvider)
{
	BasePluginContainer::init(pProvider, DebtagsPluginFactory::getInstance());
	_pHandleMaker = &pProvider->handleMaker();
	_pDebtagsUpdateAction->addTo(provider()->systemMenu());
	// use dynamic cast here because of the virtual base class 
	// (static_cast is not allowed there)
	updateDebtags();
	_pRelatedPlugin = dynamic_cast<RelatedPlugin*>(requestPlugin("RelatedPlugin"));
	_pDebtagsPlugin = dynamic_cast<DebtagsPlugin*>(requestPlugin("DebtagsPlugin"));
	return debtagsEnabled();
}

vector< pair<QString, QAction*> > DebtagsPluginContainer::actions()
{
	vector< pair<QString, QAction*> > result;
	result.push_back(make_pair("System", _pDebtagsUpdateAction));
	return result;
}

QWidget* DebtagsPluginContainer::getSettingsWidget(QWidget* pParent)
{
	_pSettingsWidget = new DebtagsSettingsWidget(pParent, "DebtagsSettingsWidget");
	_pSettingsWidget->init(_hiddenFacets);
 	return _pSettingsWidget;
}

void DebtagsPluginContainer::applySettings()
{
	assert(_pSettingsWidget);
	_hiddenFacets = _pSettingsWidget->hiddenFacets();
	updateVocabulary();
}

void DebtagsPluginContainer::loadSettings(const QDomElement source)
{
	QDomNodeList hiddenFacets = source.elementsByTagName("HiddenFacet");
	for (int i=0; i < hiddenFacets.count(); ++i)
	{
		string hiddenFacet = toString( hiddenFacets.item(i).toElement().text() );
		_hiddenFacets.insert(hiddenFacet);
	}
	if (debtagsEnabled())
		updateVocabulary();
}

void DebtagsPluginContainer::saveSettings(NXml::XmlData& outData, QDomElement parent) const
{
	QDomElement debtagsPlugin = outData.addElement(parent, name());
	for (set<string>::const_iterator it = _hiddenFacets.begin(); it != _hiddenFacets.end(); ++it)
	{
		QDomElement hiddenFacet = outData.addElement(debtagsPlugin, "HiddenFacet");
		outData.addText(hiddenFacet, *it);
	}
}




/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPluginContainer::onDebtagsUpdate()
{
	if ( geteuid() != 0 )	// check if the user have super user priviledges
	{
		QMessageBox::information( provider()->mainWindow(), tr("Need Root Priviledges"), 
			tr("You need super user priviledges to execute this command.\n"
			  "Please start the application as root and try again."));
		return;
	}
	_pDebtagsUpdateAction->setEnabled(false);
	// this will fetch us the update of the db
	NApplication::ApplicationFactory fac;
	_pCommand = fac.getRunCommand("DebtagsUpdateProcess");
	connect(_pCommand, SIGNAL(quit()), SLOT(onDebtagsUpdateFinished()) );
	_pCommand->addArgument("/usr/bin/debtags");
	_pCommand->addArgument("update");
	if ( !_pCommand->start() )
	{
		provider()->reportError( tr("Command not executed"), tr("The command could not be executed.\n"
			"Make sure that /usr/bin/debtags is installed on your system.") );
		_pDebtagsUpdateAction->setEnabled(true);
		delete _pCommand;
		_pCommand = 0;
	}
}

void DebtagsPluginContainer::onDebtagsUpdateFinished()
{
	if (_pCommand->processExitedSuccessful())
	{
		updateDebtags();
	}
	delete _pCommand;
	_pCommand = 0;
	_pDebtagsUpdateAction->setEnabled(true);
}


void DebtagsPluginContainer::updateVocabulary(bool informPlugins)
{
	NDebtags::HiddenFacetsMatcher matcher(_hiddenFacets);
	// get all the facets which are not hidden
	_facets = Debtags::Environment::get().vocabulary().getFacets(matcher);
	if (_pDebtagsPlugin)
		_pDebtagsPlugin->debtagsDataChanged();
	if (_pRelatedPlugin)
		_pRelatedPlugin->debtagsDataChanged();
}

void DebtagsPluginContainer::updateDebtags()
{
	if ( !Debtags::TagDB::hasTagDatabase() )	// not all debtags files are existent
	{
		provider()->reportError(
			tr("Tag Database Not Available" ),
			tr(
				"<p>The tag database is not available and the  debtags plugin was disabled!</p>"
				"<p>"
				"You must execute <tt><b>debtags update</b></tt> as root on the commandline to download "
				"the database. If debtags is not on your system you can install it via "
				"<tt><b>apt-get install debtags</b></tt><br>"
				"After this you can enable the debtags plugin via the plugin menu -> Control Plugins."
				"</p>"
			) 
		);
		/* New text
						tr(
						"<p>The tag database is not available!</p>"
						"The database is required "
						"by this program. Do you want to try downloading it now?<br>"
						"Note: Super user priviledges are needed for this. You can also update by hand "
						"executing <tt>debtags update</tt> on the commandline.<br>"
						"If debtags is not on your system you can get it via <tt>apt-get install debtags</tt>."
						"</p>"
					),
					QMessageBox::Yes, QMessageBox::No )
				== 0
		*/
		setDebtagsEnabled(false);	// disable the debtags system
	}
	else 
		setDebtagsEnabled(true);
	delete _pCollection;
	_pCollection = 0;
	_facets.clear();
	if (debtagsEnabled())
	{
		try
		{
			// this is set to 0 if no listener is available
			NUtil::IProgressObserver* pProgressObserver = provider()->progressObserver();
			if (pProgressObserver)
			{
				pProgressObserver->setText("reading tag database");
			}
			Tagcoll::InputMerger<int, string> merger;
			Tagcoll::TagcollBuilder collBuilder(*_pHandleMaker);
			// load the database into the collBuilder and change it to handles
			Debtags::TagDB::outputSystem(collBuilder, false);
			if (pProgressObserver)
			{
				pProgressObserver->setProgress(25);
			}
			// now sent the collection to the InputMerger
			collBuilder.collection().output(merger);
			if (pProgressObserver)
			{
				pProgressObserver->setProgress(50);
			}
			// output the merger to create a collection
			_pCollection = new Tagcoll::TagCollection<int, string>();
			merger.output(*_pCollection);
			if (pProgressObserver)
			{
				pProgressObserver->setProgress(75);
				pProgressObserver->setText("reading vocabulary");
			}
			///@todo add error handling here, but currently it is not supported because of bad
			/// exception specification in the Environment class
			// load the new vocabulary
			updateVocabulary(false);
			if (pProgressObserver)
			{
				pProgressObserver->setProgress(100);
			}
		}	
		catch (ContextException& e)	// PackageDBException e
			// this might be a FileException, a ParserException or a PackageDBException
		{
			provider()->reportError(
				tr("Loading failed"),
				QObject::tr(
				"<p>Loading the debtags database failed!</p>"
				"<p>Most likely it is malformed.<br>"
				"Try selecting \"System\"->\"Debtags Update\" from the menu or running "
				"<tt>debtags update</tt> on the commandline."
				"</p>"
				"This was the original error message:<br>") 
				+ toQString(e.desc())
			);
		}
	}
	// inform the plugins about the new collections
	if (_pDebtagsPlugin)
		_pDebtagsPlugin->debtagsDataChanged();
	if (_pRelatedPlugin)
		_pRelatedPlugin->debtagsDataChanged();
}

void DebtagsPluginContainer::setDebtagsEnabled(bool enabled)
{
	_debtagsEnabled = enabled;
}


}	// namespace NPlugin

