//
// C++ Implementation: unselectedtagsview
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2007
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "unselectedtagsview.h"

#include <cassert>

#include <QContextMenuEvent>
#include <QMenu>
#include <QAction>

// NTagModel
#include "debtagsplugincontainer.h"
#include "vocabularymodelrole.h"
#include "emptytagfilter.h"

#include <modeltest.h>
#include <helpers.h>


/** @brief A filter which hides top level nodes which do not have any child nodes.
  *
  * All child nodes will be accepted.
  *
  * @author Benjamin Mesing
  */
class TreeFilter : public QSortFilterProxyModel
{ 
public:
	TreeFilter(QObject * pParent = 0) : QSortFilterProxyModel(pParent)
	{}
	virtual bool filterAcceptsRow(int sourceRow, const QModelIndex& sourceParent) const
	{
		QModelIndex current = sourceModel()->index(sourceRow, 0, sourceParent);
		if (sourceParent.isValid())
			return true;
// 		qStrDebug("TreeSortFilter:item: " + data(current).toString());
		if (sourceModel()->rowCount(current) > 0)
		{
// 			qDebug("has children: %d ", sourceModel()->rowCount(current));
			return true;
		}
		return false;
	}
};


namespace NTagModel
{

/** @brief A filter which accepts all facets and tags only if they match the active filter.
  *
  *
  * @author Benjamin Mesing
  */
class TextFilter : public QSortFilterProxyModel
{ 
public:
	TextFilter(QObject * pParent = 0) : QSortFilterProxyModel(pParent)
	{}
	virtual bool filterAcceptsRow(int sourceRow, const QModelIndex& sourceParent) const
	{
		// if we have a facet (i.e. a root item)
		if (!sourceParent.isValid())
			return true;
		return QSortFilterProxyModel::filterAcceptsRow(sourceRow, sourceParent);
	}
};

UnselectedTagsView::UnselectedTagsView(const NPlugin::DebtagsPluginContainer* pContainer, QWidget * parent)
 : QTreeView(parent), _filterSelectedProxyModel(false, this), _hiddenFilterProxyModel(false, this)
{
	connect(this, SIGNAL(doubleClicked(const QModelIndex&)), SLOT(onItemDoubleClicked(const QModelIndex&)));
	
	// build up ProxyModel/Filter chain -> see class documentation for the order of the chain
	_hiddenFilterProxyModel.setDynamicSortFilter(true);

	_pTagFilter = new EmptyTagFilter(pContainer->vocabularyModel(), this);
	_pTagFilter->setDynamicSortFilter(true);

 	_filterSelectedProxyModel.setDynamicSortFilter(true);
	
	
	_pTextFilter = new TextFilter(this);
	// according to QT-Doc setting the column to -1 should search in all columns
	// however this seems not to work
	_pTextFilter->setFilterKeyColumn(1);
	_pTextFilter->setFilterCaseSensitivity(Qt::CaseInsensitive);
	_pTextFilter->setDynamicSortFilter(true);
	
 	
	_filterChain.push_back(&_hiddenFilterProxyModel);
 	_filterChain.push_back(_pTagFilter);
  	_filterChain.push_back(&_filterSelectedProxyModel);
 	_filterChain.push_back(_pTextFilter);
	QSortFilterProxyModel* pTreeFilter = new TreeFilter(this);
	pTreeFilter->setDynamicSortFilter(true);
	_filterChain.push_back(pTreeFilter);
 	
 	vector<QAbstractProxyModel*>::iterator it = _filterChain.begin();
 	vector<QAbstractProxyModel*>::iterator oldIt = _filterChain.begin();
 	++it;
 	for (; it != _filterChain.end(); ++it )
 	{
 		(*it)->setSourceModel(*oldIt);
 		oldIt = it;
 	}
}


UnselectedTagsView::~UnselectedTagsView()
{
}


void UnselectedTagsView::setModel(QAbstractItemModel* pModel)
{
	_filterChain.front()->setSourceModel(pModel);
// this is for debugging purposes, Modeltest are some really convenient classes to 
// detect problems in ones model
// 	for (	vector<QAbstractProxyModel*>::iterator it = _filterChain.begin(); it != _filterChain.end(); ++it )
// 	{
//  		new ModelTest(*it, this);
// 	}
 	QTreeView::setModel(_filterChain.back());
 	resizeColumnToContents(0);
// 	resizeColumnToContents(0);
}


void UnselectedTagsView::onItemDoubleClicked(const QModelIndex& index )
{
	model()->setData(index, true, SelectedRole);
}

void UnselectedTagsView::contextMenuEvent(QContextMenuEvent* pEvent)
{
	QMenu menu(this);
	QModelIndex index = indexAt(pEvent->pos());
	QAction* pSelectAsWanted = 0;
	// if the context menu was requested for a tag
	if ( index.isValid() && (model()->data(index, TypeRole).toInt() == TagTypeItem) )
	{
		pSelectAsWanted = menu.addAction("Add");
	// not yet
	// 	QAction* pSelectAsUnwanted = menu.addAction("Select as unwanted");
	}
	menu.addSeparator();
	QAction* pCollapseAll = menu.addAction("Collapse all");
	QAction* pExpandAll = menu.addAction("Expand all");
	QAction* pAction = menu.exec(pEvent->globalPos());
	if (pAction)
	{
		if (pAction == pCollapseAll)
			collapseAll();
		else if  (pAction == pExpandAll)
			expandAll();
		else if  (pAction == pSelectAsWanted)
		// the index is definitely valid here
			model()->setData(index, true, SelectedRole);
	}
}


void UnselectedTagsView::setFilterFixedString(const QString& string)
{
	_pTextFilter->setFilterFixedString(string);
	// invalidate the TextFilter, so that the following filters are reevaluated.
	// This is required, because the TreeFilter changes depending on the tags in the filter
	_pTextFilter->invalidate();
}

}

