/*   FILE: pam_keyring.c
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 4 June 2004
 * AUTHOR: Jonathan Nettleton <jon.nettleton@gmail.com>
 *   DATE: 13 March 2006
 *
 * Copyright (C) 2004 W. Michael Petullo <mike@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 */

#include <config.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <syslog.h>
#include <glib.h>
#include <pwd.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/types.h>

#include <security/_pam_macros.h>
#include <security/pam_modules.h>

#ifdef HAVE_OLD_PAM
#include "compat.h"
#else
#include <security/pam_ext.h>
#endif

#define PAM_DEBUG_ARG				0x0001
#define PAM_USE_FPASS_ARG			0x0002
#define PAM_TRY_FPASS_ARG			0x0004

/* ============================ preexec_t ================================== */
typedef struct preexec_t {
	const char *user;
	const char *evar;
} preexec_t;

/* ============================ free_cb () ================================= */
/* INPUT: pamh; data, must point to malloced memory of be NULL; errcode
 * SIDE AFFECTS: if data does not point to NULL then it is freed
 * NOTE: this is registered as a PAM callback function */
void free_cb(pam_handle_t * pamh, void *data, int errcode)
{
	if (data) {
		g_free(data);
	}
}

/* ============================ _pam_parse () ========================== */
/* INPUT: argc and argv, standard main()-type arguments
 * SIDE AFFECTS: global args are initialized, based on argc and argv */
static int
_pam_parse (const pam_handle_t *pamh, int argc,
            const char **argv, const char **keyring)
{
	int ctrl;

	*keyring = NULL;
	
	for (ctrl = 0; argc-- > 0; ++argv) {
		if (!strcmp(*argv,"debug"))
			ctrl |= PAM_DEBUG_ARG;
		else if (!strcmp(*argv,"use_first_pass"))
			ctrl |= PAM_USE_FPASS_ARG;
		else if (!strcmp(*argv,"try_first_pass"))
			ctrl |= PAM_TRY_FPASS_ARG;
		else if (!strncasecmp(*argv,"keyring=", 8))
			{
				*keyring = (*argv) + 8;
				if ( **keyring  == '\0' ) {
					*keyring = NULL;
					pam_syslog(pamh , LOG_ERR,
										"keyring= specification missing argument - using default");
				}
			}				
		 else
        	{
          		pam_syslog(pamh, LOG_ERR, "unknown option: %s", *argv);
        	}
	}		
	return ctrl;
}

/* ============================ read_password () =========================== */
/* INPUT: pamh; prompt1
 * SIDE AFFECTS: pass points to PAM's (user's) response to prompt
 * OUTPUT: any PAM error code encountered or PAM_SUCCESS
 * NOTE:   adapted from pam_userdb/pam_userdb.c
 */
static int
obtain_authtok(pam_handle_t *pamh)
{
    char *resp;
    const void *item;
    int retval;

    retval = pam_prompt(pamh, PAM_PROMPT_ECHO_OFF, &resp, "Password: ");

    if (retval != PAM_SUCCESS)
        return retval;

    if (resp == NULL)
        return PAM_CONV_ERR;

    /* set the auth token */
    retval = pam_set_item(pamh, PAM_AUTHTOK, resp);

    /* clean it up */
    _pam_overwrite(resp);
    _pam_drop(resp);

    if ( (retval != PAM_SUCCESS) ||
         (retval = pam_get_item(pamh, PAM_AUTHTOK, &item))
         != PAM_SUCCESS ) {
        return retval;
    }

    return retval;
}


/* ============================ pipewrite () =============================== */
/* INPUT: fd, a valid file descriptor; buf, a buffer of size count
 * SIDE AFFECTS: buf is written to fd
 * OUTPUT: number of bytes written or 0 on error
 * NOTE: SIGPIPE is ignored during this operation to avoid "broken pipe"
 */
int pipewrite(int fd, const void *buf, size_t count)
{
	int fnval;
	struct sigaction ignoresact = {
		.sa_handler = SIG_IGN
	}, oldsact;

	assert(fd >= 0);
	assert(buf != NULL);
	assert(count >= 0);

	/* avoid bomb on command exiting before data written */
	if (sigaction(SIGPIPE, &ignoresact, &oldsact) < 0) {
		fnval = -1;
		goto _return;
	}
	fnval = write(fd, buf, count);
	/* restore old handler */
	if (sigaction(SIGPIPE, &oldsact, NULL) < 0) {
		fnval = -1;
		goto _return;
	}
      _return:
	return fnval;
}

/* ============================ preexec () ================================= */
/* INPUT: user, a valid user name
 * SIDE AFFECTS: the calling process's uid, gid, euid and homedir are set to user's
 * NOTE: this is required because gnome-keyring-daemon restricts connections
 *       to its owner.
 */
static void preexec(const gpointer data)
{
#ifdef HAVE_SELINUX_SELINUX_H
	int selinux_enabled = 0;
#endif
	struct passwd *passwd_ent;

	assert(data);
	assert(((preexec_t *) data)->user);

	passwd_ent = getpwnam((char *) ((preexec_t *) data)->user);
	if (!passwd_ent) {
		syslog(LOG_ERR, "pam_keyring: error looking up user information for %s", (char *) ((preexec_t *) data)->user);
		exit(EXIT_FAILURE);
	}
	/* must exit because g_spawn_sync does not check return value before
	 * executing argument vector */
	if (setgid(passwd_ent->pw_gid) == -1) {
		syslog(LOG_ERR, "pam_keyring: error setting gid (%s)",  strerror(errno));
		exit(EXIT_FAILURE);
	}
	if (setuid(passwd_ent->pw_uid) == -1) {
		syslog (LOG_ERR, "pam_keyring: error setting uid (%s)", strerror(errno));
		exit(EXIT_FAILURE);
	}
	if (seteuid(passwd_ent->pw_uid) == -1) {
		syslog (LOG_ERR, "pam_keyring: error setting euid: %s", strerror(errno));
		exit(EXIT_FAILURE);
	}
	if (setenv( "HOME", passwd_ent->pw_dir, 1) == -1) {
		syslog (LOG_ERR, "pam_keyring: error setting home: %s",  passwd_ent->pw_dir);
		exit(EXIT_FAILURE);
	}
	if (((preexec_t *) data)->evar)
		/* putenv does not want const char *, only char * */
		putenv((char *) ((preexec_t *) data)->evar);
}

/* ============================ keyring_daemon_stop () ===================== */
/* INPUT: data, a valid preexec_t name; pid, the pid of gnome-keyring-daemon
 * SIDE AFFECTS: a SIGTERM is sent to the gnome-keyring-daemon process
 * OUTPUT: TRUE if the SIGTERM is sent, otherwise FALSE
 */
static int keyring_daemon_stop(pam_handle_t *pamh, const preexec_t * data, pid_t pid)
{
	char *cmd_line;
	gchar **argv;
	GError *err = NULL;
	int retval = TRUE, status;

	assert(data != NULL);
	assert(data->user != NULL);

	cmd_line = g_strdup_printf("%s %d", KILL, pid);

	if (!g_shell_parse_argv(cmd_line, NULL, &argv, &err)) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: error parsing %s", cmd_line);
		goto _return;
	}
	if (!g_spawn_sync(NULL, argv, NULL, G_SPAWN_SEARCH_PATH, preexec,
			  (void *) data, NULL, NULL, &status, &err)) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: failed to run kill: %s", err->message);
		g_error_free(err);
		goto _return;
	}
	if (WIFEXITED(status) && WEXITSTATUS(status) != 0) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: failed to execute code, exit code: %d",
							 WEXITSTATUS(status));
		retval = FALSE;
		goto _return;
	}
      _return:
	return retval;
}

/* ============================ keyring_daemon_start () ==================== */
/* INPUT: pamh, a valid pam_handle_t *; data, a valid user preexec_t
 * SIDE AFFECTS: gnome-keyring-daemon is started
 * OUTPUT: gnome-keyring-daemon's PID or 0 on error
 *         data->evar is set to g-k-d's i/o socket
 */
static pid_t keyring_daemon_start(pam_handle_t * pamh, preexec_t * data)
{
	/* FIXME: for some reason, gnome-keyring-daemon runs until pam_sm_authenticate
	   returns and then stops before the user's session starts when using with
	   login (not, ie., su).  Is the login process exiting and causing this?
	 */
	GError *err = NULL;
	char *standard_out = NULL;
	char **lines;
	gint status = 0;
	long pid;
	pid_t fnval = 0;
	char *pid_str, *end;
	gchar **argv = NULL;

	assert(pamh != NULL);
	assert(data != NULL);
	assert(data->user != NULL);
	if (!g_shell_parse_argv(GNOME_KEYRING_DAEMON, NULL, &argv, &err)) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: error parsing: %s", GNOME_KEYRING_DAEMON);
		goto _return;
	}
	if (!g_spawn_sync(NULL, argv, NULL, G_SPAWN_SEARCH_PATH, preexec,
			  (void *) data, &standard_out, NULL, &status, &err)) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: failed to run gome-keyring-daemon: %s", err->message);
		g_error_free(err);
		goto _return;
	}
	if (WIFEXITED(status) == 0 && standard_out != NULL) {
		lines = g_strsplit(standard_out, "\n", 3);
		if (lines[0] != NULL &&
		    lines[1] != NULL &&
		    g_str_has_prefix(lines[1], "GNOME_KEYRING_PID=")) {
			pid_str = lines[1] + strlen("GNOME_KEYRING_PID=");
			pid = strtol(pid_str, &end, 10);
			if (end != pid_str) {
				fnval = pid;
				/* set variable for ulock operation */
				data->evar = g_strdup(lines[0]);
				/* FIXME: memory leak */
				/* ensure variable will be set in login session */
				if (pam_putenv (pamh, g_strdup(lines[0])) != PAM_SUCCESS) {
					fnval = 0;
					pam_syslog(pamh, LOG_ERR, "pam_keyring: error setting %s", lines[0]);
					goto _return;
				}
			}
		}
		g_strfreev(lines);
	} else {
		/* daemon failed for some reason */
		pam_syslog(pamh, LOG_ERR, "pam_keyring: gnome-keyring-daemon failed to start correctly, exit code: %d", WEXITSTATUS(status));
	}
	g_free(standard_out);
      _return:
	g_strfreev(argv);
	return fnval;
}

/* ============================ unlock () ================================== */
/* INPUT: data, authtok, keyring
 * SIDE AFFECTS: user's GNOME keyring is unlocked
 * OUTPUT: PAM error code on error or PAM_SUCCESS
 */
static int unlock(pam_handle_t * pamh, const preexec_t * data,
		  const void *authtok, const char *keyring)
{
	gchar **argv = NULL;
	int child_exit;
	pid_t pid = -1;
	GError *err = NULL;
	int retval = PAM_SUCCESS;
	int cstdin = -1, cstderr = -1;
	char *cmd_line = NULL;

	assert(data);
	assert(data->user);
	assert(authtok);

	if (keyring == NULL) {
		cmd_line =
	    		g_strconcat(PAM_KEYRING_TOOL, " -u -s", NULL);
	} else {
		cmd_line =
	    		g_strconcat(PAM_KEYRING_TOOL, " -u  -s --keyring=", keyring, NULL);
	}
	
	pam_syslog(pamh, LOG_WARNING, "pam_keyring: going to execute %s", cmd_line);
	if (!g_shell_parse_argv(cmd_line, NULL, &argv, &err)) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: error parsing %s", cmd_line);
		goto _return;
	}
	if (g_spawn_async_with_pipes
	    (NULL, argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD,
	     preexec, (void *) data, &pid, &cstdin, NULL, &cstderr,
	     &err) == FALSE) {
	    pam_syslog(pamh, LOG_ERR, "pam_keyring: error executing %s", cmd_line);
		retval = PAM_SERVICE_ERR;
		goto _return;
	}
	if (pipewrite(cstdin, authtok, strlen(authtok)) != strlen(authtok)) {
		pam_syslog(pamh, LOG_WARNING, "pam_keyring: error writing authtok to gnome-keyring");
		retval = PAM_SERVICE_ERR;
		goto _return;
	}
	close(cstdin);
	if (waitpid(pid, &child_exit, 0) == -1) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: error waiting for child");
		retval = PAM_SERVICE_ERR;
		goto _return;
	}
	close(cstderr);
	retval = WEXITSTATUS(child_exit) ? PAM_SERVICE_ERR : PAM_SUCCESS;
      _return:
      
	g_strfreev(argv);
	g_free(cmd_line);
	return retval;
}

/* ============================ pam_sm_authenticate () ===================== */
/* INPUT: this function is called by PAM
 * SIDE AFFECTS: user's GNOME keyring is unlocked
 * OUTPUT: PAM error code on error or PAM_SUCCESS
 * NOTE:   adapted from pam_userdb/pam_userdb.c
 */
PAM_EXTERN int
pam_sm_authenticate(pam_handle_t * pamh, int flags,
		    int argc, const char **argv)
{
	const char *keyring = NULL;
	const void *password;
	pid_t *daemon_pid;
	int retval = PAM_AUTH_ERR, ctrl;
	preexec_t data = { NULL, NULL };

	ctrl = _pam_parse(pamh, argc, argv, &keyring);

	/* needed because gdm does not prompt for username as login does: */
	retval = pam_get_user(pamh, &data.user, NULL);
	if ((retval != PAM_SUCCESS) || (!data.user)) {
		pam_syslog(pamh, LOG_ERR, "can not get the username");
		retval = PAM_SERVICE_ERR;
		goto _return;
	}
	
	if ((ctrl & PAM_USE_FPASS_ARG) == 0 && (ctrl & PAM_TRY_FPASS_ARG) == 0) {
        /* Converse to obtain a password */
        retval = obtain_authtok(pamh);
        if (retval != PAM_SUCCESS) {
            pam_syslog(pamh, LOG_ERR, "can not obtain password from user");
            goto _return;
        }
	}
	
	/* Check if we got a password */
	retval = pam_get_item(pamh, PAM_AUTHTOK, &password);
	if (retval != PAM_SUCCESS || password == NULL) {
		if ((ctrl & PAM_TRY_FPASS_ARG) != 0) {
            retval = obtain_authtok(pamh);
            if (retval != PAM_SUCCESS) {
                pam_syslog(pamh, LOG_ERR, "can not obtain password from user");
                return retval;
            }
            retval = pam_get_item(pamh, PAM_AUTHTOK, &password);
        }
        if (retval != PAM_SUCCESS || password == NULL) {
            pam_syslog(pamh, LOG_ERR, "can not recover user password");
            return PAM_AUTHTOK_RECOVER_ERR;
        }
     }
	
	if (ctrl & PAM_DEBUG_ARG)
		pam_syslog(pamh, LOG_INFO, "Verify user `%s' with a password",
							data.user);
	
	if ((data.evar = getenv("GNOME_KEYRING_SOCKET")) != NULL) {
		pam_syslog(pamh, LOG_WARNING, "pam_keyring: daemon already exists (%s)", data.evar);
		/* FIXME: g_strconcat = memory leak? */
		/* ensure variable will be passed to next session */
		if (pam_putenv(pamh, g_strconcat("GNOME_KEYRING_SOCKET=",
						 data.evar,
						 NULL)) != PAM_SUCCESS) {
            pam_syslog(pamh, LOG_ERR, "pam_keyring: error setting GNOME_KEYRING_SOCKET");
			retval = PAM_SERVICE_ERR;
			goto _return;
		}
		retval = PAM_SUCCESS;
		goto _return;
	}

	pam_syslog(pamh, LOG_WARNING, "pam_keyring: starting gnome-keyring-daemon");
	daemon_pid = g_new0(pid_t, 1);	// FIXME: memory leak?
	if ((*daemon_pid = keyring_daemon_start(pamh, &data)) == 0) {
		retval = PAM_SERVICE_ERR;
		goto _return;
	}
	if ((retval =
	     pam_set_data(pamh, "pam_keyring_gkd_pid", daemon_pid,
			  free_cb)) != PAM_SUCCESS) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: %s\n", "error trying to save gnome-keyring-daemon PID");
		goto _return;
	}
	pam_syslog(pamh, LOG_WARNING, "pam_keyring: unlocking keyring");
	retval = unlock(pamh, &data, password, keyring);
      _return:
	return retval;
}

/* ============================ pam_sm_open_session () ===================== */
/* NOTE: placeholder function so PAM does not get mad */
PAM_EXTERN int
pam_sm_open_session(pam_handle_t * pamh, int flags,
		    int argc, const char **argv)
{
	return PAM_SUCCESS;
}

/* ============================ pam_sm_close_session () ==================== */
/* INPUT: this function is called by PAM
 * SIDE AFFECTS: user's gnome-keyring-daemon is sent a SIGTERM
 * OUTPUT: PAM error code on error or PAM_SUCCESS
 */
PAM_EXTERN int
pam_sm_close_session(pam_handle_t * pamh, int flags, int argc,
		     const char **argv)
{
	preexec_t data = { NULL, NULL };
	const void *daemon_pid;
	int retval = PAM_SUCCESS;

	assert(pamh);

	pam_syslog(pamh, LOG_WARNING, "pam_keyring: received order to close session");

	if ((retval = pam_get_user(pamh, &data.user, NULL)) != PAM_SUCCESS) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: %s", "could not get user");
		/* do NOT return PAM_SERVICE_ERR or root will not be able 
		 * to su to other users */
		goto _return;
	}
	if (!strcmp(data.user, "root")) {
		pam_syslog(pamh, LOG_WARNING, "pam_keyring: do nothing for root");
		retval = PAM_SUCCESS;
		goto _return;
	}
	if ((retval =
	     pam_get_data(pamh, "pam_keyring_gkd_pid",
			  &daemon_pid)) != PAM_SUCCESS) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: %s", "could not retrieve gnome-keyring-deamon PID");
		retval = PAM_SUCCESS;
		goto _return;
	}
	if (keyring_daemon_stop(pamh, &data, *(int *) daemon_pid) == FALSE) {
		pam_syslog(pamh, LOG_ERR, "pam_keyring: error trying to kill gnome-keyring-daemon (%d)", *(int *) daemon_pid);
		retval = PAM_SERVICE_ERR;
		goto _return;
	}
      _return:
	return retval;
}

/* ============================ pam_sm_setcred () ========================== */
/* NOTE: placeholder function so PAM does not get mad */
PAM_EXTERN int
pam_sm_setcred(pam_handle_t * pamh, int flags, int argc, const char **argv)
{
	return PAM_SUCCESS;
}
