#include <algorithm>

#include "mudclient.h"
#include "AliasEntity.h"
#include "TriggerEntity.h"
#include "EntityGroup.h"

static int EntityCmp(Entity * e1, Entity * e2) {
  return (e1 < e2);
}

EntityGroup::EntityGroup() : Entity() {
  setType(EntityGroupGroup);
  widget = NULL;
  disabled = false;
}

EntityGroup::~EntityGroup() {
  EntityList::iterator i_next;
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i = i_next) {
    i_next = i;
    i_next++;
    removeEntity(*i);
    delete *i;
  }
}

bool EntityGroup::contains(Entity * e) {
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); ++i) {
    if ((*i) == e)
      return true;
  }
  return false;
}

void EntityGroup::addEntity(Entity * e) {


  EntityList::iterator i = std::lower_bound(my_entities.begin(),
					    my_entities.end(),
					    e,
					    EntityCmp);
  my_entities.insert(i, e);
}

bool EntityGroup::removeEntity(Entity * e) {
  EntityList::iterator i = std::lower_bound(my_entities.begin(),
					    my_entities.end(),
					    e,
					    EntityCmp);
  if (i == my_entities.end() || (*i) != e)
    return false;

  my_entities.erase(i);
  return true;
}

bool EntityGroup::findExecute(EntityType t, char * field, char * value, void * data, bool multiple) {

  // If this group is disabled perform no checking for its children.
  if (getDisabled())
    return false;

  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i++)
    if ((*i)->findExecute(t, field, value, data, multiple) && !multiple)
      return true;

  return false;
}

void EntityGroup::exportEntities(FILE * fp, GList * list) {

  bool first_time = true;
  // Are any of this group's entities in the list?

  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i++) {

    GList * element = g_list_find(list, (*i));

    if (element) {

      if (first_time) {

	fprintf(fp, "EntityGroup %s\n", getName());
	
	fprintf(fp, "Description\n");
	fprintf(fp, "%s", getDescription());
	if (strlen(getDescription()) > 0)
	  if (getDescription()[strlen(getDescription()) - 1] != '\n')
	    fprintf(fp, "\n");
	fprintf(fp, "EndDescription\n");

	first_time = false;
      }

      (*i)->save(fp);
    }
  }

  if (!first_time) // Has an element been found and the group created?
    fprintf(fp, "EndEntityGroup\n");

}

void EntityGroup::save(FILE * fp) {

  fprintf(fp, "EntityGroup %s\n", getName());

  fprintf(fp, "Description\n");
  fprintf(fp, "%s", getDescription());
  if (strlen(getDescription()) > 0)
    if (getDescription()[strlen(getDescription()) - 1] != '\n')
      fprintf(fp, "\n");
  fprintf(fp, "EndDescription\n");

  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i++)
    (*i)->save(fp);

  fprintf(fp, "EndEntityGroup\n");
}

bool EntityGroup::load(FILE * fp, char * name) {

  char buf[16384];
  setName(name);

  while (true) {
    if (!fgets(buf, 16384, fp)) {
      perror("fgets");
      return false;
    }

    if (!strncasecmp(buf, "Description", 11)) {
      char * ptr = loadBlock(fp, "EndDescription");
      if (!ptr)
	return false;

      setDescription(ptr);
    }
      
    if (!strncasecmp(buf, "Trigger", 7)) {
      char name[16384];
      if (sscanf(buf, "Trigger %[^\n]\n", name) != 1) {
	perror("sscanf");
	return false;
      }

      TriggerEntity * tmp = new TriggerEntity();
      if (!tmp->load(fp, name)) {
	delete tmp;
	return false;
      }
	
      addEntity(tmp);
      continue;
    }

    if (!strncasecmp(buf, "Alias", 5)) {
      char name[16384];
      if (sscanf(buf, "Alias %[^\n]\n", name) != 1) {
	perror("sscanf");
	return false;
      }

      AliasEntity * tmp = new AliasEntity();
      if (!tmp->load(fp, name)) {
	delete tmp;
	return false;
      }
	
      addEntity(tmp);
      continue;
    }

    if (!strncasecmp(buf, "Disabled", 8)) {
      int tmp;
      if (sscanf(buf, "Disabled %d\n", &tmp) != 1) {
	perror("sscanf");
	return false;
      }

      setDisabled((bool) tmp);
      continue;
    }

    if (!strncasecmp(buf, "EndEntityGroup", 14)) {
      return true;
    }
  }
}

void EntityGroup::apply() {

  gchar * new_desc = NULL;
  gchar * new_name = NULL;

  if (widget) {
    new_name = (gchar *)gtk_entry_get_text(GTK_ENTRY(name_entry));
    setName((char *)new_name);

    GtkTextIter start, end;
    gtk_text_buffer_get_bounds(description_buffer, &start, &end);
    new_desc = gtk_text_buffer_get_text(description_buffer, &start, &end, false);
    setDescription(new_desc);

    setDisabled(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(disabled_toggle)));
  }

  EntityList::iterator i_next;
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i = i_next) {
    i_next = i;
    i_next++;
    (*i)->apply();
  }
}

int EntityGroup::count() {
  int count = 0;
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i = ++i)
    count++;

  return count;
}

extern void entity_selected(GtkWidget *, gpointer);

void EntityGroup::populate_tree_model(GtkTreeStore * store, GtkTreeIter * parent) {

  if (getType() != EntityGroupGroup) {
    debug(_("%s is not an EntityGroup!\n"), getName());
    return;
  }
  
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); i++) {
    char buf[16384], id_buf[1024];
    GtkTreeIter iter;

    Entity * e = (Entity *)(*i);

    switch ((*i)->getType()) {
    case EntityTrigger:
      snprintf(buf, 16384, _("[Trigger] %s"), (*i)->getName());
      break;
      
    case EntitySystemTrigger:
      snprintf(buf, 16384, _("[System ] %s"), (*i)->getName());
      break;

    case EntityAlias:
      snprintf(buf, 16384, _("[ Alias ] %s"), (*i)->getName());
      break;

    default:
      snprintf(buf, 16384, _("[ ERROR ] %s"), (*i)->getName());
      break;
    }

    snprintf(id_buf, 1024, "%d", (*i)->getID());

    gtk_tree_store_insert(GTK_TREE_STORE(store),
			  &iter,
			  parent,
			  0);

    gtk_tree_store_set(GTK_TREE_STORE(store), &iter,
		       0, buf, 1, id_buf, -1);

  }
}


GtkWidget * EntityGroup::getWidgets() {

  //  if (widget)
  //    return widget;

  GtkWidget * frame4, * table3, * label13, * scrolledwindow5, * label14;

  frame4 = gtk_frame_new (_("General"));
  gtk_widget_ref (frame4);
  gtk_widget_show (frame4);
  gtk_container_set_border_width (GTK_CONTAINER (frame4), 2);

  

  table3 = gtk_table_new (2, 2, FALSE);
  gtk_widget_ref (table3);
  gtk_widget_show (table3);
  gtk_container_add (GTK_CONTAINER (frame4), table3);
  gtk_container_set_border_width (GTK_CONTAINER (table3), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table3), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table3), 2);

  label13 = gtk_label_new (_("Name:"));
  gtk_widget_ref (label13);
  gtk_widget_show (label13);
  gtk_table_attach (GTK_TABLE (table3), label13, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label13), 0, 0.5);

  name_entry = gtk_entry_new ();
  gtk_widget_ref (name_entry);
  gtk_widget_show (name_entry);
  gtk_table_attach (GTK_TABLE (table3), name_entry, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  scrolledwindow5 = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_ref (scrolledwindow5);
  gtk_widget_show (scrolledwindow5);
  gtk_table_attach (GTK_TABLE (table3), scrolledwindow5, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow5), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

  description_buffer = gtk_text_buffer_new(NULL);
  description_view = gtk_text_view_new_with_buffer(description_buffer);
  gtk_widget_ref(description_view);
  gtk_widget_show(description_view);
  gtk_container_add(GTK_CONTAINER(scrolledwindow5), description_view);


  label14 = gtk_label_new (_("Description:"));
  gtk_widget_ref (label14);
  gtk_widget_show (label14);
  gtk_table_attach (GTK_TABLE (table3), label14, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label14), 0, 7.45058e-09);

  disabled_toggle = gtk_check_button_new_with_label(_("Disable this group of aliases and triggers (this will also disable any system triggers in the group)"));
  gtk_widget_ref(disabled_toggle);
  gtk_widget_show(disabled_toggle);
  gtk_table_attach (GTK_TABLE (table3), disabled_toggle, 0, 2, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  if (getName()) {
    gtk_entry_set_text(GTK_ENTRY(name_entry), getName());
  }

  if (getDescription()) {
    GtkTextIter start, end;
    gtk_text_buffer_get_bounds(description_buffer, &start, &end);
    gtk_text_buffer_insert(description_buffer, &end, getDescription(), -1);
  }

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(disabled_toggle), getDisabled());

  widget = frame4;
  gtk_widget_ref(widget);
  return widget;
}

Entity * EntityGroup::findByID(int id) {
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); ++i) {
    if ((*i)->getID() == id)
      return (*i);
  }

  return NULL;
}

void EntityGroup::destroyWidgets() {
  for (EntityList::iterator i = my_entities.begin(); i != my_entities.end(); ++i) {
    (*i)->destroyWidgets();
  }
}

void EntityGroup::setDisabled(bool dis) {
  disabled = dis;
}

bool EntityGroup::getDisabled() {
  return disabled;
}
