#include "PreferenceHandler.h"
#include "mudclient.h"
#include "Connection.h"

#include <gmodule.h>

extern Connection * first_connection;
extern Prefs * globalPreferences;

static struct prefs_interface_data prefs_panels[] = {
  { CategoryTerminal, _("Terminal"), NULL },
  { CategoryTerminalOutput, _("Terminal"), _("Output") },
  { CategoryTerminalInput, _("Terminal"), _("Input") },
  { CategoryTerminalHistory, _("Terminal"), _("Command History (and Tab Completion)") },
  { CategoryConnection, _("Connection"), NULL },
  { CategoryConnectionGeneral, _("Connection"), _("General") },
  { CategoryConnectionCompression, _("Connection"), _("Compression") },
  { CategoryProxies, _("Proxies"), NULL },
  { CategoryColours, _("Colours"), NULL },
  { CategoryFonts, _("Fonts"), NULL },
  { CategoryPlugins, _("Plugins"), NULL },
  { CategoryLast, NULL, NULL }
};

PreferenceHandler::PreferenceHandler() {
  xml = NULL;
  mud = NULL;
}

PreferenceHandler::~PreferenceHandler() {
  if (xml)
    g_object_unref(xml);
}

static int EditorCmp(Preference * p1, Preference * p2) {
  return (p1 < p2);
}

void PreferenceHandler::addPreferenceEditor(Preference * editor) {
  PreferenceEditorList::iterator i = std::lower_bound(editors.begin(),
						      editors.end(),
						      editor,
						      EditorCmp);
  editors.insert(i, editor);
}

void PreferenceHandler::removePreferenceEditor(Preference * editor) {
  PreferenceEditorList::iterator i = std::lower_bound(editors.begin(),
						      editors.end(),
						      editor,
						      EditorCmp);
  if (i == editors.end() || (*i) != editor)
    return;

  editors.erase(i);
}

void PreferenceHandler::createWindow(MUD * m) {
  char buf[1024];

  if (xml) {
    GtkWidget * window = glade_xml_get_widget(xml, "preferences_window");
	gtk_window_present(GTK_WINDOW(window));
    return;
  }

  mud = m;

  Prefs * prefs;
  if (mud && mud->getPreferences())
    prefs = mud->getPreferences();
  else
    prefs = globalPreferences;

  snprintf(buf, 1024, "%s/share/papaya/preferences_editor.glade", getPrefix());
  xml = glade_xml_new(buf, NULL, NULL);

  
  
  GtkWidget * notebook = glade_xml_get_widget(xml, "preferences_notebook");
  //  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_LEFT);

  for (PreferenceEditorList::iterator i = editors.begin();
       i != editors.end(); i++) {
    Preference * editor = (*i);
    addDisplayWidgets(editor);
    editor->loadPreferences(prefs);
  }

  glade_xml_signal_autoconnect(xml);

  gtk_widget_show(glade_xml_get_widget(xml, "preferences_window"));
}

void PreferenceHandler::apply() {
  Prefs * prefs;
  if (mud && mud->getPreferences())
    prefs = mud->getPreferences();
  else
    prefs = globalPreferences;

  for (PreferenceEditorList::iterator i = editors.begin();
       i != editors.end(); i++) {
    (*i)->applyPreferences(prefs);
  }

  prefs->save(mud);

  for (Connection * tmp = first_connection; tmp; tmp = tmp->getNext())
    tmp->getVT()->prefsUpdated();
  
}

void PreferenceHandler::destroyWindow() {

  for (PreferenceEditorList::iterator i = editors.begin();
       i != editors.end(); i++) {
    removeDisplayWidgets(*i);
    (*i)->destroyWidget();
  }

  GtkWidget * window = glade_xml_get_widget(xml, "preferences_window");
  gtk_widget_hide(window);
  g_object_unref(xml);
  xml = NULL;

  mud = NULL;
}


/**
 * Adds a preference's display widgets to the preferences dialog.
 * This is generally done when the user requests that preferences be edited.
 */

void PreferenceHandler::addDisplayWidgets(Preference * editor) {
  GtkWidget * child = editor->getWidget();
  if (!child)
    return; // No widgets to display.  Unusual, but not forbidden.

  GtkWidget * notebook_page = getNotebookPage(editor->getCategories(), true);
  if (notebook_page) {
    gtk_box_pack_start(GTK_BOX(notebook_page), child, false, true, 0);
    g_object_ref(G_OBJECT(child));
  }
}

void PreferenceHandler::removeDisplayWidgets(Preference * editor) {
  GtkWidget * child = editor->getWidget();
  if (!child)
    return;

  GtkWidget * notebook_page = getNotebookPage(editor->getCategories(), false);
  if (notebook_page) {
    gtk_container_remove(GTK_CONTAINER(notebook_page), child);
    g_object_unref(child);
  }
}

/**
 * Returns the container widget for the notebook page whose tab matches
 * one of the specified categories.
 *
 * Can return NULL if no appropriate tab is available.
 */

GtkWidget * PreferenceHandler::getNotebookPage(Category * categories, bool create) {

  Category * currentCategory = categories;
  while (*currentCategory != CategoryLast) {
    struct prefs_interface_data * iface_data = findPrefsInterfaceData(*currentCategory);
    if (iface_data) {
      // Look for the notebook page.
      GtkWidget * pane = findNotebookPageByName(iface_data->tab_name, create);
      
      if (iface_data->child_frame == NULL)
	return pane;
      
      GList * frames = gtk_container_get_children(GTK_CONTAINER(pane));
      int num_frames = g_list_length(frames);
      if (num_frames > 0) {
	for (int i = 0; i < num_frames; i++) {
	  GtkWidget * existing_frame = (GtkWidget *) g_list_nth_data(frames, i);
	  
	  GList * children = gtk_container_get_children(GTK_CONTAINER(existing_frame));
	  GtkWidget * label = (GtkWidget *)g_list_nth_data(children, 0);
	  if (!strcmp(gtk_label_get_text(GTK_LABEL(label)), iface_data->child_frame)) {
	    // Get the hbox in the vbox
	    GtkWidget * hbox = (GtkWidget *) g_list_nth_data(children, 1);

	    // get the vbox in the hbox.
	    GList * hbox_children = gtk_container_get_children(GTK_CONTAINER(hbox));
	    return (GtkWidget *) g_list_nth_data(hbox_children, 1);
	  }



	  /*

	  if (!strcmp(iface_data->child_frame, gtk_frame_get_label(GTK_FRAME(existing_frame)))) {

	    GList * children = gtk_container_get_children(GTK_CONTAINER(existing_frame));
	    return (GtkWidget *)g_list_nth_data(children, 0);
	  }
	  */

	}
      }
      
      // Create a VBOX to hold a section of data.
      GtkWidget * vbox = gtk_vbox_new(false, 0);
      gtk_widget_show(vbox);
      gtk_box_set_spacing(GTK_BOX(vbox), 6);
      gtk_box_pack_start(GTK_BOX(pane), vbox, false, true, 0);
      char buf[1024];
      
      // In the first row goes a label, in bold.
      snprintf(buf, 1024, "<span weight=\"bold\">%s</span>", iface_data->child_frame);
      GtkWidget * label = gtk_label_new(NULL);
      gtk_widget_show(label);
      gtk_label_set_markup(GTK_LABEL(label), buf);
      gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
      gtk_box_pack_start(GTK_BOX(vbox), label, false, true, 0);

      // In the second row goes a hbox, two wide.
      GtkWidget * hbox = gtk_hbox_new(false, 0);
      gtk_widget_show(hbox);
      gtk_box_pack_start(GTK_BOX(vbox), hbox, false, true, 0);
      
      // The first column of the hbox is a spacer label: "    "
      label = gtk_label_new("    ");
      gtk_widget_show(label);
      gtk_box_pack_start(GTK_BOX(hbox), label, false, true, 0);

      // The second column of the hbox is a vbox to put the widgets into.
      GtkWidget * real_vbox = gtk_vbox_new(false, 0);
      gtk_widget_show(real_vbox);
      gtk_box_pack_start(GTK_BOX(hbox), real_vbox, true, true, 0);

      // Return the box into which the widgets should be packed.
      return real_vbox;
    }
    currentCategory++;
  }

  return NULL;
}

struct prefs_interface_data * PreferenceHandler::findPrefsInterfaceData(Category category) {

  struct prefs_interface_data * iface_data = prefs_panels;

  while (iface_data->category != CategoryLast) {
    if (iface_data->category == category)
      return iface_data;
    iface_data++;
  }

  return NULL;
}

GtkWidget * PreferenceHandler::findNotebookPageByName(char * name, bool create) {

  GtkWidget * notebook = glade_xml_get_widget(xml, "preferences_notebook");
  if (!notebook)
    return NULL;

  int i = 0;
  while (true) {
    GtkWidget * page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), i);
    if (!page)
      break;

    GtkWidget * label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), page);
    const gchar * label_text = gtk_label_get_text(GTK_LABEL(label));
    if (!strcmp(label_text, name)) {
      return page;
    }

    i++;
  }

  if (create) {
    // This notebook page doesn't yet exist.  Create it.

    // Create the vbox to hold the various bits of data.
    GtkWidget * child_vbox = gtk_vbox_new(false, 0);

    // Spacing between the sections is 18 pixels.
    gtk_box_set_spacing(GTK_BOX(child_vbox), 18);

    // Border width of 12 pixels.
    gtk_container_set_border_width(GTK_CONTAINER(child_vbox), 12);
    gtk_widget_show(child_vbox);

    // Create the page's label.
    GtkWidget * child_label = gtk_label_new(name);
    gtk_widget_show(child_label);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), child_vbox, child_label);
    return child_vbox;
  }

  return NULL;
}

extern PreferenceHandler * preferenceHandler;

extern "C" G_MODULE_EXPORT gint on_preferences_cancel_button_clicked(GtkWidget * widget, gpointer data) {
  preferenceHandler->destroyWindow();
  return 1;
}

extern "C" G_MODULE_EXPORT gint on_preferences_apply_button_clicked(GtkWidget * widget, gpointer data) {
  preferenceHandler->apply();
  return 1;
}

extern "C" G_MODULE_EXPORT gint on_preferences_ok_button_clicked(GtkWidget * widget, gpointer data) {
  on_preferences_apply_button_clicked(widget, data);
  on_preferences_cancel_button_clicked(widget, data);
  return 1;
}

extern "C" G_MODULE_EXPORT gint on_preferences_window_delete_event(GtkWidget * widget, GdkEvent * event, gpointer data) {
  on_preferences_cancel_button_clicked(widget, data);
  return 1;
}
