#include "Server.h"
#include "common.h"
#include "TrException.h"
#include "Reply.h"
#include "Util.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>

/**
 * C callback function used to start the Server thread
 */
void*
server_run_thread (void* arg) {
  Server* server = (Server*)arg;
  server->runThread();
  return NULL;
}

/**
 * Create a new <i>Server</i> to listen incoming message.
 *
 * @param client The class to notify when a message arrives.
 * @param addr The server will listen for incoming messages from this address
 *             (in network endianess).
 * @param port The server will listen for incoming messages on this port.
 * @param protocol The protocol to listen ("icmp", "tcp").
 *
 * @throw TrException An error occured.
 */
Server::Server (Options* opts, const char* protocol) {
  // Initialisation
  this->client   = new Tracert*[opts->threads_count];
  this->client[0]   = NULL;
  this->client_id =  new int[opts->threads_count];
  this->opts     = opts;

  stop_thread    = true;

  // Create the socket
  sock_server = socket(PF_INET, SOCK_RAW, Util::protocol2int(protocol));
  if (sock_server < 0) throw TrException(str_log(ERROR,
			"Cannot create the server : %s", strerror(errno)));

// WHAT FOR ???
#if 0
  // Set timeout on the socket : 5000 usec
  struct timeval tv;
  tv.tv_sec  = 0;
  tv.tv_usec = 5000;
  int res = setsockopt(sock_server,SOL_SOCKET,SO_RCVTIMEO,&tv,sizeof(timeval));
  if (res < 0) throw TrException(str_log(ERROR, "Set timeout failed : %s",
                                                strerror(errno)));
#endif

  // Bind it
  sockaddr_in saddr;
  memset(&saddr, 0, sizeof(sockaddr_in));
  saddr.sin_family      = AF_INET;
  saddr.sin_port        = htons(opts->src_port);
  int res = bind(sock_server, (sockaddr*)&saddr, sizeof(sockaddr_in));
  if (res < 0) throw TrException(str_log(ERROR,
			"Cannot bind the server : %s", strerror(errno)));

  // Create a mutex
  res = pthread_mutex_init(&lock, NULL);
  if (res != 0) throw TrException(str_log(ERROR, "Create a mutex"));
}

/**
 * Stop the server.
 */
Server::~Server () {
  log(INFO, "delete server");
  stopThread();
  log(INFO, "stop thread done");
  //close(sock_server);
  log(INFO, "close socket done");
// STRANGE: uder linux, the recvfrom wont unblock when we  
// close the socket if there is no timeour on the socket. 
#if 0  
  pthread_join(thread, NULL);
  log(INFO, "join done");
#endif  
  pthread_mutex_destroy(&lock);
  log(INFO, "destroy mutex done");
}

/**
 * Start the thread.
 */
void
Server::startThread () {
  // Create and execute the listening thread
  stop_thread = false;
  int res = pthread_create(&thread, NULL, server_run_thread, this);
  if (res != 0) throw TrException(str_log(ERROR, "Create a thread"));
}

/**
 * Stop the thread.
 */
void
Server::stopThread () {
  pthread_mutex_lock(&lock);
  stop_thread = true;
  pthread_mutex_unlock(&lock);
}

/**
 * Set/change the client to notify.
 */
void
Server::setClient (Tracert* client) {
  pthread_mutex_lock(&lock);
  this->client[0] = client;
  pthread_mutex_unlock(&lock);
}

void
Server::addClient (Tracert* client, int i) {
  pthread_mutex_lock(&lock);
  //if (opts->debug)
  //	printf("addClient: %x\n", client); 
  this->client[i] = client;
  pthread_mutex_unlock(&lock);
}

/**
 * Capture all messages and notify the client.
 * 
 * In the constructor of this class, we have specified one protocol. This
 * thread will listen and capture all messages from this protocol. It
 * will then wrap the message into a reply (ICMPReply for the ICMP protocol
 * and TCPReply for the TCP protocol). Finally, it will notify the client that
 * a new probable reply has arrived. It is up to the client to distinguish
 * between a reply of a probe or junk traffic.
 */
void
Server::runThread () {
  pthread_mutex_lock(&lock);
  while (!stop_thread) {
    pthread_mutex_unlock(&lock);

    // Wait a message, TODO: don't limit the reply size to 1024
    sockaddr_in from;
    int         from_len = sizeof(sockaddr_in);
    uint8	data[1024];
    //printf("runThread, recvfrom\n");
    int         data_len = recvfrom(sock_server, data, 1024, 0,
				(sockaddr*)&from, (socklen_t*)&from_len);
    //printf("runThread, recvfom done\n");
    if (data_len > 0) {
      log(DUMP, "Incoming message :");
      dumpRawData(DUMP, data, data_len);

      Reply* reply = Reply::replyFactory(data, data_len);
      log(DUMP, "Incoming message parsed :");
      reply->dump();
      //printf("server locking...\n");
      pthread_mutex_lock(&lock);
      //printf("server locked\n");
      // XXX temp for UDP
      //reply->proc_id = opts->proc_id;
      
      // validate the reply
      uint16 id = reply->getProcId();

#ifdef DEVANOMALIES
			// source port
			id = reply->getID3();
#else  
      id = reply->getID();
#endif
      //id = reply->getIPId();
      //printf("server, id %d\n", id);
//      
//      if (id > 20000) {
//      	printf("dumpraw\n");
//      	reply->dumpRaw();
//      }
      //id = (id & 0xffff) >> (16 - 5);
      
     	//printf("id = 0x%x %d\n", id, id);
     	
#ifdef DEVANOMALIES
			id = (id - 32000) / (32000 / opts->threads_count);
#else
      id = id / (65536 / opts->threads_count);
#endif
      
      //printf("client id = %d\n", id);
      
      if (id < 0 || id >= 32)
      	log(FATAL, "bug, id can't be greater than 31");
      reply->proc_id = opts->proc_id;
      
      //if (opts->debug)
  		//printf("runthread, locked, notifyrelply %d %x\n", id, client[id]);
  		// DEVANOMALIES: the single client will recive all responses
      if (client[id] != NULL) client[id]->notifyReply(reply);
      //printf("runthread, unlocked notifyreply done\n");
      delete(reply);
      pthread_mutex_unlock(&lock);
    }

    pthread_mutex_lock(&lock);
  }
  // Otherwise NetBSD will
  pthread_mutex_unlock(&lock);
}

