/* conff.c - Maintain configuration information
   Copyright (C) 2000, 2001 Thomas Moestl

   With modifications by Paul Rombouts, 2002, 2003, 2004.

This file is part of the pdnsd package.

pdnsd is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

pdnsd is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with pdsnd; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include "ipvers.h"
#include "conff.h"
#include "consts.h"
#include "helpers.h"
#include "conf-parser.h"
#include "servers.h"

#if !defined(lint) && !defined(NO_RCSIDS)
static char rcsid[]="$Id: conff.c,v 1.26 2001/07/02 18:55:27 tmm Exp $";
#endif

globparm_t global={
  perm_cache:        2048,
  cache_dir:         NULL,
  port:              53,
#ifdef ENABLE_IPV4
  a:                 {{INADDR_ANY}},
#else
  a:                 {IN6ADDR_ANY_INIT},
#endif
  max_ttl:           604800,
  min_ttl:           120,
  neg_ttl:           900,
  neg_rrs_pol:       C_AUTH,
  neg_domain_pol:    C_AUTH,
  run_as:            "",
  strict_suid:       1,
  paranoid:          0,
  lndown_kluge:      0,
  onquery:           0,
  rnd_recs:          1,
  ctl_perms:         0600,
  scheme_file:       NULL,
  proc_limit:        20,
  procq_limit:       30,
  tcp_qtimeout:      TCP_TIMEOUT,
  par_queries:       PAR_QUERIES,
  query_port_start:  0,
  query_port_end:    65535,
  deleg_only_zones:  NULL
};

servparm_t serv_presets={
  port:          53,
  uptest:        C_NONE,
  timeout:       120,
  interval:      900,
  ping_timeout:  600,
  scheme:        "",
  uptest_cmd:    NULL,
  uptest_usr:    "",
  interface:     "",
  device:        "",
  label:         NULL,
  purge_cache:   0,
  nocache:       0,
  lean_query:    1,
  is_proxy:      0,
  preset:        1,
  policy:        C_INCLUDED,
  alist:         NULL,
  atup_a:        NULL,
#ifdef ENABLE_IPV4
  ping_a:        {{INADDR_ANY}}
#else
  ping_a:        {IN6ADDR_ANY_INIT}
#endif
};

servparm_array servers=NULL;

void lex_set_io(FILE *in, FILE *out); /* defined in conf.l */
int  yyparse (void);                  /* from yacc/bison output */
static int report_server_stat(int f,int i);


/*
 * Read a configuration file and save the result. This uses the parser generated by yacc/bison
 * from conf.y, which in turn uses the lexer generated lex/flex from conf.l
 */
void read_config_file(char *nm)
{
	FILE *in;
	struct stat sb;
	if (nm==NULL) {
		nm=CONFDIR"/pdnsd.conf";
	} 
	if (stat(nm,&sb)!=0) {
		fprintf(stderr,"Error: Could not stat conf file %s: %s\n",nm,strerror(errno));
		exit(3);
	}
	if (sb.st_uid!=getuid()) {
		fprintf(stderr,"Error: You must own the config file.\n");
		exit(3);
	}
	if ((sb.st_mode&(S_IWGRP|S_IWOTH))) {
		fprintf(stderr,"Error: Bad config file permissions: the file must be only writeable by the user.\n");
		exit(3);
	}
	if (!(in=fopen(nm,"r"))) {
		fprintf(stderr,"Error: Could not open conf file %s: %s\n",nm,strerror(errno));
		exit(3);
	}

	if (!confparse(in))
		exit(3);
	fclose(in);
}

/* Report the current configuration to the file descriptor f (for the status fifo, see status.c) */
int report_conf_stat(int f)
{
	int i,retval=0;
	
	fsprintf_or_return(f,"\nConfiguration:\n==============\nGlobal:\n-------\n");
	fsprintf_or_return(f,"\tCache size: %li kB\n",global.perm_cache);
	fsprintf_or_return(f,"\tServer directory: %s\n",global.cache_dir);
	fsprintf_or_return(f,"\tScheme file (for Linux pcmcia support): %s\n",global.scheme_file);
	fsprintf_or_return(f,"\tServer port: %i\n",global.port);
	{char buf[ADDRSTR_MAXLEN];
	 fsprintf_or_return(f,"\tServer ip (0.0.0.0=any available one): %s\n",pdnsd_a2str(&global.a,buf,ADDRSTR_MAXLEN));}
	fsprintf_or_return(f,"\tIgnore cache when link is down: %s\n",global.lndown_kluge?"on":"off");
	fsprintf_or_return(f,"\tMaximum ttl: %li\n",(long)global.max_ttl);
	fsprintf_or_return(f,"\tMinimum ttl: %li\n",(long)global.min_ttl);
	fsprintf_or_return(f,"\tNegative ttl: %li\n",(long)global.neg_ttl);
	fsprintf_or_return(f,"\tNegative RRS policy: %s\n",const_name(global.neg_rrs_pol));
	fsprintf_or_return(f,"\tNegative domain policy: %s\n",const_name(global.neg_domain_pol));
	fsprintf_or_return(f,"\tRun as: %s\n",global.run_as);
	fsprintf_or_return(f,"\tStrict run as: %s\n",global.strict_suid?"on":"off");
	fsprintf_or_return(f,"\tParanoid mode (cache pollution prevention): %s\n",global.paranoid?"on":"off");
	fsprintf_or_return(f,"\tControl socket permissions (mode): %o\n",global.ctl_perms);
	fsprintf_or_return(f,"\tMaximum parallel queries served: %i\n",global.proc_limit);
	fsprintf_or_return(f,"\tMaximum queries queued for serving: %i\n",global.procq_limit);
	fsprintf_or_return(f,"\tMaximum parallel queries done: %i\n",global.par_queries);
	fsprintf_or_return(f,"\tRandomize records in answer: %s\n",global.rnd_recs?"on":"off");
	fsprintf_or_return(f,"\tQuery port start: %i\n",global.query_port_start);
	fsprintf_or_return(f,"\tQuery port end: %i\n",global.query_port_end);
	fsprintf_or_return(f,"\tDelegation-only zones: ");
	if(global.deleg_only_zones==NULL) {
		fsprintf_or_return(f,"(none)\n");
	}
	else {
		for(i=0;i<DA_NEL(global.deleg_only_zones);++i) {
			unsigned char zstr[256];
			rhn2str(DA_INDEX(global.deleg_only_zones,i),zstr);
			fsprintf_or_return(f,i==0?"%s":", %s",zstr);
		}
		fsprintf_or_return(f,"\n");
	}

	lock_server_data();
	for(i=0;i<DA_NEL(servers);i++) {
		int rv=report_server_stat(f,i);
		if(rv<0) {
			retval=rv;
			break;
		}
	}
	unlock_server_data();

	return retval;
}

/* Report the current status of server i to the file descriptor f.
   Call with locks applied.
*/
static int report_server_stat(int f,int i)
{
	servparm_t *st=&DA_INDEX(servers,i);
	int j;

	fsprintf_or_return(f,"Server %i:\n------\n",i);
	fsprintf_or_return(f,"\tlabel: %s\n",st->label?st->label:"(none)");
	for(j=0;j<DA_NEL(st->atup_a);j++) {
		atup_t *at=&DA_INDEX(st->atup_a,j);
		{char buf[ADDRSTR_MAXLEN];
		 fsprintf_or_return(f,"\tip: %s\n",pdnsd_a2str(&at->a,buf,ADDRSTR_MAXLEN));}
		fsprintf_or_return(f,"\tserver assumed available: %s\n",at->is_up?"yes":"no");
	}		  
	fsprintf_or_return(f,"\tport: %hu\n",st->port);
	fsprintf_or_return(f,"\tuptest: %s\n",const_name(st->uptest));
	fsprintf_or_return(f,"\ttimeout: %li\n",(long)st->timeout);
	if(st->interval>0) {
		fsprintf_or_return(f,"\tuptest interval: %li\n",(long)st->interval);
	} else {
		fsprintf_or_return(f,"\tuptest interval: %s\n",st->interval?"onquery":"(never retest)");
	}
	fsprintf_or_return(f,"\tping timeout: %li\n",(long)st->ping_timeout);
	{char buf[ADDRSTR_MAXLEN];
	 fsprintf_or_return(f,"\tping ip: %s\n",is_inaddr_any(&st->ping_a)?"(using server ip)":pdnsd_a2str(&st->ping_a,buf,ADDRSTR_MAXLEN));}
	fsprintf_or_return(f,"\tinterface: %s\n",st->interface);
	fsprintf_or_return(f,"\tdevice (for special Linux ppp device support): %s\n",st->device);
	fsprintf_or_return(f,"\tuptest command: %s\n",st->uptest_cmd?:"");
	fsprintf_or_return(f,"\tuptest user: %s\n",st->uptest_usr[0]?st->uptest_usr:"(process owner)");
	if (st->scheme[0]!='\0') {
		fsprintf_or_return(f,"\tscheme: %s\n", st->scheme);
	}
	fsprintf_or_return(f,"\tforce cache purging: %s\n",st->purge_cache?"on":"off");
	fsprintf_or_return(f,"\tserver is cached: %s\n",st->nocache?"off":"on");
	fsprintf_or_return(f,"\tlean query: %s\n",st->lean_query?"on":"off");
	fsprintf_or_return(f,"\tUse only proxy?: %s\n",st->is_proxy?"on":"off");
	fsprintf_or_return(f,"\tDefault policy: %s\n",const_name(st->policy));
	fsprintf_or_return(f,"\tPolicies:\n");
	if (st->alist==NULL) {
		fsprintf_or_return(f,"\t\t(none)\n");
	} else {
		for (j=0;j<DA_NEL(st->alist);j++) {
			slist_t *sl=&DA_INDEX(st->alist,j);
			fsprintf_or_return(f,"\t\t%s: %s\n",sl->rule==C_INCLUDED?"include":"exclude",sl->domain);
		}
	}
	return 0;
}
