#include "screen_practice.hh"

#include "util.hh"
#include "fs.hh"
#include "record.hh"
#include "joystick.hh"

ScreenPractice::ScreenPractice(std::string const& name, Audio& audio, Capture& capture):
  Screen(name), m_audio(audio), m_capture(capture)
{}

void ScreenPractice::enter() {
	m_audio.playMusic(getThemePath("practice.ogg"));
	theme.reset(new ThemePractice());
	// draw vu meters
	for (unsigned int i = 0, mics = m_capture.analyzers().size(); i < mics; ++i) {
		ProgressBar* b;
		m_vumeters.push_back(b = new ProgressBar(getThemePath("vumeter_bg.svg"), getThemePath("vumeter_fg.svg"), ProgressBar::VERTICAL, 0.136, 0.023));
		b->dimensions.screenBottom().left(-0.4 + i * 0.2).fixedWidth(0.04);
	}
	unsigned int sr = m_audio.getSR();
	m_samples.push_back(Sample(getPath("sounds/drum_bass.ogg"), sr));
	m_samples.push_back(Sample(getPath("sounds/drum_snare.ogg"), sr));
	m_samples.push_back(Sample(getPath("sounds/drum_hi-hat.ogg"), sr));
	m_samples.push_back(Sample(getPath("sounds/drum_tom1.ogg"), sr));
	m_samples.push_back(Sample(getPath("sounds/drum_cymbal.ogg"), sr));
	//m_samples.push_back(Sample(getPath("sounds/drum_tom2.ogg"), sr));
}

void ScreenPractice::exit() {
	m_vumeters.clear();
	m_samples.clear();
	theme.reset();
}

void ScreenPractice::manageEvent(SDL_Event event) {
	ScreenManager * sm = ScreenManager::getSingletonPtr();
	input::NavButton nav(input::getNav(event));
	if (nav == input::CANCEL || nav == input::START || nav == input::SELECT) sm->activateScreen("Intro");
	else if (nav == input::PAUSE) m_audio.togglePause();
	else if (nav == input::VOLUME_DOWN) --config["audio/music_volume"];
	else if (nav == input::VOLUME_UP) ++config["audio/music_volume"];
	else if (event.type == SDL_JOYBUTTONDOWN // Play drum sounds here
	  && input::Private::devices[event.jbutton.which].type_match(input::DRUMS)) {
		int b = input::buttonFromSDL(input::Private::devices[event.jbutton.which].type(), event.jbutton.button);
		if (b != -1) m_audio.play(m_samples[unsigned(b) % m_samples.size()], "audio/fail_volume");
	}
}

void ScreenPractice::draw() {
	theme->bg.draw();
	this->draw_analyzers();
}

void ScreenPractice::draw_analyzers() {
	MusicalScale scale;
	boost::ptr_vector<Analyzer>& analyzers = m_capture.analyzers();
	if (analyzers.empty()) return;
	double textPower = -getInf();
	double textFreq = 0.0;

	for (unsigned int i = 0; i < analyzers.size(); ++i) {
		Analyzer& analyzer = analyzers[i];
		analyzer.process();
		Tone const* tone = analyzer.findTone();
		double freq = (tone ? tone->freq : 0.0);
		if (tone && tone->db > textPower) {
			textPower = tone->db;
			textFreq = freq;
		}
		// getPeak returns 0.0 when clipping, negative values when not that loud.
		// Normalizing to [0,1], where 0 is -43 dB or less (to match the vumeter graphic)
		m_vumeters[i].draw(analyzer.getPeak() / 43.0 + 1.0);

		if (freq != 0.0) {
			Analyzer::tones_t tones = analyzer.getTones();

			for (Analyzer::tones_t::const_iterator t = tones.begin(); t != tones.end(); ++t) {
				if (t->age < Tone::MINAGE) continue;
				int note = scale.getNoteId(t->freq);
				if (note < 0) continue;
				int octave = note / 12 - 1;
				double noteOffset = scale.getNoteNum(note);
				bool sharp = scale.isSharp(note);
				noteOffset += octave*7;
				noteOffset += 0.4 * scale.getNoteOffset(t->freq);
				float posXnote = -0.25 + 0.2 * i + 0.002 * t->stabledb;
				float posYnote = .075-noteOffset*0.015;

				theme->note.dimensions.left(posXnote).center(posYnote);
				theme->note.draw();
				if (sharp) {
					theme->sharp.dimensions.right(posXnote).center(posYnote);
					theme->sharp.draw();
				}
			}
		}
	}
	// Display note and frequency
	if (textFreq > 0.0) theme->note_txt.draw(scale.getNoteStr(textFreq));
}
