/***************************************************************************
                          phaseshift.cpp  -  description
                             -------------------
    begin                : Sat Sep 18 12:08:54 EDT 1999
    copyright            : (C) 1999 by Edson Pereira
    email                : edson_pereira@harvard.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <termios.h>
#include <math.h>

#include "phaseshift.h"
#include "filesave.xpm"
#include "fileopen.xpm"
//#include "filenew.xpm"
#include "tune.xpm"
#include "bpsk.xpm"
#include "qpsk.xpm"
#include "afc.xpm"
#include "dcd.xpm"
#include "net.xpm"
#include "beacon.xpm"
#include "typeAhead.xpm"
#include "txstop.xpm"
#include "vox.xpm"

#include "phasescope.h"
#include "masterdisplay.h"
#include "fftengine.h"
#include "myqlineedit.h"

#include <qcheckbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qmultilinedit.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qframe.h>
#include <qtimer.h>
#include <qscrollbar.h>
#include <qlistbox.h>
#include <qcursor.h>

#define DMA_BUF_BITS 6

PhaseShift::PhaseShift()
{
    int val;

    // Initialize variables
	transmit = 0;
	noiselevel = 0;
	dcdFlag = 0;
	afcFlag = 1;
	full_duplex = 0;
	qpskFlag = 0;
	lsbFlag = 0;
	netFlag = 0;
    rxFreq = 1000.0;
	txFreq = 1000.0;
    numMacros = 0;
	intermodFlag = 0;
    typeAheadFlag = 0;
    confidenceLevel = 0;
    squelchlevel = 0;
    beaconInterval = 0;
    beaconFlag = 0;
    logOpen = 0;
    logDir[0] = '\0';
    sendingFileFlag = 0;
    bytesSent = 0;
    prependCallsignFlag = 1;

    sprintf(sound_device, "/dev/dsp");
    sprintf(pttDevice, "/dev/ttyS0");

    setCaption("PhaseShift v" VERSION);

    printf("Reading configuration...\n");
    readConfig();
    printf("Done reading configuration\n");

    // Instanciate the sound mixer and set default mixer levels
    mixer = new SoundMixer();
    mixer->setVolume( soundOutputLevel );
    mixer->setPcmLevel( soundPcmLevel );
    mixer->setLineLevel( soundInputLevel );


    ///////////////////////////////////////////////////////////////////
    // call inits to invoke all other construction parts
    printf("Initializing GUI... ");
    initMenuBar();
    initToolBar();
    initStatusBar();
    initView();
    printf("Done!\n");

    srandom(time((time_t *)0));
	
    /* init audio device */
    printf("Initializing Sound Device... ");
    if (( audio_fd = open(sound_device,O_RDWR|O_NONBLOCK)) < 0 )
    {
        perror( "open( sound_device )" );
        exit(1);
    }

    int retval;

    val=8000;
    if ((retval = ioctl(audio_fd, SNDCTL_DSP_SPEED, &val)) < 0 )
    {
        perror("ioctl( SNDCTL_DSP_SPEED )");
        exit( 1 );
    }

    val=AFMT_S16_LE;
    if (( retval = ioctl(audio_fd, SNDCTL_DSP_SETFMT, &val)) < 0 )
    {
        perror("ioctl( SNDCTL_DSP_SETFMT )");
        exit( 1 );
    }

    val=DMA_BUF_BITS;
    if (( retval = ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &val)) < 0 )
    {
        perror("ioctl( SNDCTL_DSP_SETFRAGMENT )");
        exit( 1 );
    }

    val=1;
    if (( retval = ioctl(audio_fd, SNDCTL_DSP_NONBLOCK, &val)) < 0 )
    {
        perror("ioctl( SNDCTL_DSP_NONBLOCK )");
        exit( 1 );
    }


    // Check if the device is operating in full duplex mode
    if (( retval = ioctl(audio_fd, SNDCTL_DSP_GETCAPS, &val)) < 0)
    {
        perror("ioctl( SNDCTL_DSP_GETCAPS )");
        exit( 1 );
    }

    if (val&DSP_CAP_DUPLEX) full_duplex = 1;
    printf("Done!\n");

    printf("Initializing PSK31 coder... ");
    psk31_coder::init_tables();
    printf("Done!\n");

    printf("Initializing PSK31 TX... ");
    psk31tx.set_freq( txFreq );
   	psk31tx.set_audiofd(audio_fd);
   	psk31tx.set_mode (qpskFlag,lsbFlag);
	psk31tx.setFullDuplex( full_duplex );
	connect( &psk31tx, SIGNAL(signalSymbol( int )),
	         phaseScope, SLOT( plotPhase( int )));
	connect( &psk31tx, SIGNAL(signalSample( short )),
	         this, SLOT(processTxSample( short )));
	printf("Done!\n");
	
	printf("Initializing PSK31 RX... ");
	psk31rx.set_freq( rxFreq );
   	psk31rx.set_afc(afcFlag);
   	psk31rx.set_txtrack (netFlag);
   	psk31rx.set_mode (qpskFlag,lsbFlag);
    printf("Done!\n");

    printf("Initializing FFT Engine... ");
    fftEngine = new FftEngine();
    connect( fftEngine, SIGNAL(fftComplete()), this, SLOT(drawSpectrum()));
    connect( &psk31rx, SIGNAL(dcdOn()), this, SLOT(calculateIntermod()));
    printf("Done!\n");

    printf("Initializing FFT Timer... ");
  	QTimer *t0 = new QTimer( this );
	connect( t0, SIGNAL(timeout()), fftEngine, SLOT(setCalculateFlag()) );
	t0->start( 100, FALSE );
	printf("Done!\n");

    printf("Initializing Process Timer... ");
  	QTimer *t1 = new QTimer( this );
	connect( t1, SIGNAL(timeout()), SLOT(process()) );
	t1->start( 0, FALSE );
	printf("Done!\n");

	printf("Initializing Beacon Timer... ");
	QTimer *t2 = new QTimer( this );
	connect( t2, SIGNAL(timeout()), SLOT(doBeacon()));
	t2->start( beaconInterval * 60000, FALSE );
	printf("Done!\n");
	
    printf("Initialization Completed!\n\n");
}

PhaseShift::~PhaseShift(){
}

void PhaseShift::initMenuBar()
{
  ///////////////////////////////////////////////////////////////////
  // MENUBAR  

  ///////////////////////////////////////////////////////////////////
  // menuBar entry file_menu
  file_menu = new QPopupMenu();
  //file_menu->insertItem("&New",this,SLOT(slotFileNew()),CTRL+Key_N, ID_FILE_NEW );
  file_menu->insertItem("&Send File",this,SLOT(slotFileOpen()),CTRL+Key_S, ID_FILE_OPEN );
  file_menu->insertSeparator();
  file_menu->insertItem("&Open Log",this,SLOT(slotFileSave()),CTRL+Key_L, ID_FILE_SAVE );
  file_menu->insertItem("Open Log &as...",this,SLOT(slotFileSaveAs()),CTRL+Key_A, ID_FILE_SAVE_AS);
  file_menu->insertItem("&Close Log",this,SLOT(slotFileClose()),CTRL+Key_C, ID_FILE_CLOSE );
  file_menu->insertSeparator();
  //file_menu->insertItem("&Print",this,SLOT(slotFilePrint()),CTRL+Key_P, ID_FILE_PRINT );
  file_menu->insertSeparator();
  file_menu->insertItem("E&xit",this,SLOT(slotFileQuit()),CTRL+Key_Q, ID_FILE_QUIT );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry edit_menu
  //edit_menu = new QPopupMenu();
  //edit_menu->insertItem("Cu&t",this,SLOT(slotEditCut()),CTRL+Key_X, ID_EDIT_CUT );
  //edit_menu->insertItem("&Copy",this,SLOT(slotEditCopy()),CTRL+Key_C, ID_EDIT_COPY );
  //edit_menu->insertItem("&Paste",this,SLOT(slotEditPaste()),CTRL+Key_V, ID_EDIT_PASTE );
 
  
  ///////////////////////////////////////////////////////////////////
  // menuBar entry view_menu
  options_menu = new QPopupMenu();
  options_menu->setCheckable(true);
  options_menu->insertItem("&Prepend Callsign",this,SLOT(slotPrependCallsign()),0, ID_PREPEND_CALLSIGN);
  //options_menu->insertItem("&Statusbar",this,SLOT(slotViewStatusBar()),0, ID_VIEW_STATUSBAR );

  options_menu->setItemChecked(ID_PREPEND_CALLSIGN, true);
  //options_menu->setItemChecked(ID_VIEW_STATUSBAR,true);

  ///////////////////////////////////////////////////////////////////
  // EDIT YOUR APPLICATION SPECIFIC MENUENTRIES HERE
  
  ///////////////////////////////////////////////////////////////////
  // menuBar entry help_menu
  help_menu = new QPopupMenu();
  help_menu->insertItem("About...",this,SLOT(slotHelpAbout()),0,ID_HELP_ABOUT);


  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // set menuBar() the current menuBar 

  menuBar()->insertItem("&File", file_menu);
  menuBar()->insertItem("&Options", options_menu);
  menuBar()->insertSeparator();
  menuBar()->insertItem("&Help", help_menu);
  
  ///////////////////////////////////////////////////////////////////
  // CONNECT THE SUBMENU SLOTS WITH SIGNALS

  connect(file_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
  connect(options_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
  connect(help_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
  
}
void PhaseShift::initToolBar()
{
    ///////////////////////////////////////////////////////////////////
    // TOOLBAR
    QPixmap openIcon, saveIcon, newIcon, tuneIcon, bpskIcon, qpskIcon, afcIcon,
          dcdIcon, netIcon, beaconIcon, typeAheadIcon, txStopIcon, voxIcon;

    file_toolbar = new QToolBar( this, "file operations" );

    openIcon = QPixmap( fileopen );
    QToolButton * fileOpen = new QToolButton( openIcon, "Send File", 0,this, SLOT(slotFileOpen()),
					    file_toolbar);

    file_toolbar->addSeparator();	
    saveIcon = QPixmap( filesave );
    QToolButton * fileSave = new QToolButton( saveIcon, "Log", 0,this, SLOT(slotFileSave()),
					    file_toolbar);

    file_toolbar->addSeparator();  				
    tuneIcon = QPixmap( tune );
    QToolButton * Tune = new QToolButton( tuneIcon, "Tune", 0,this, SLOT(slotTune()),
					    file_toolbar);

    file_toolbar->addSeparator();
    bpskIcon = QPixmap( bpsk );
    QToolButton *  Bpsk = new QToolButton( bpskIcon, "BPSK", 0,this, SLOT(slotBPSK()),
					    file_toolbar);

    file_toolbar->addSeparator();
    qpskIcon = QPixmap( qpsk );
    QToolButton * Qpsk = new QToolButton( qpskIcon, "QPSK", 0,this, SLOT(slotQPSK()),
					    file_toolbar);

    file_toolbar->addSeparator();
    afcIcon = QPixmap( afc );
    QToolButton * Afc = new QToolButton( afcIcon, "AFC", 0,this, SLOT(slotAFC()),
					    file_toolbar);

    file_toolbar->addSeparator();
    dcdIcon = QPixmap( dcd );
    QToolButton * Dcd = new QToolButton( dcdIcon, "DCD", 0,this, SLOT(slotDCD()),
					    file_toolbar);

    file_toolbar->addSeparator();
    netIcon = QPixmap( net );
    QToolButton * Net = new QToolButton( netIcon, "NET", 0,this, SLOT(slotNET()),
					    file_toolbar);

    file_toolbar->addSeparator();
    beaconIcon = QPixmap( beacon );
    QToolButton * Beacon = new QToolButton( beaconIcon, "Beacon", 0,this, SLOT(slotBEACON()),
              file_toolbar);

    file_toolbar->addSeparator();
    typeAheadIcon = QPixmap( typeAhead );
    QToolButton * TypeAhead = new QToolButton( typeAheadIcon, "TX Line", 0,this, SLOT(slotTypeAhead()),
              file_toolbar);

    file_toolbar->addSeparator();
    voxIcon = QPixmap( vox );
    QToolButton * Vox = new QToolButton( voxIcon, "Toggles Break-in Control", 0,this, SLOT(slotVox()),
              file_toolbar);

    file_toolbar->addSeparator();
    txStopIcon = QPixmap( txstop );
    QToolButton * TxStop = new QToolButton( txStopIcon, "Emergency TX Stop", 0,this, SLOT(slotTxStop()),
              file_toolbar);

    file_toolbar->addSeparator();
    QWidget* widget_p = new QWidget(file_toolbar);
    file_toolbar->setStretchableWidget(widget_p);

    (void)QWhatsThis::whatsThisButton( file_toolbar );
    setRightJustification(true);

    QWhatsThis::add(fileOpen,"Click this button to send a text file.\n\n"
		  "You can also select the Send command from the File menu.");
    QWhatsThis::add(fileSave,"Click this button to open or close the log file.\n\n"
		  "You can also select the command from the File menu.");
	QWhatsThis::add(Tune, "Click this button to transmit a continuous tone"
          " for tuning your transmitter.");
	QWhatsThis::add(Bpsk, "Click this button to switch to BPSK");
	QWhatsThis::add(Qpsk, "Click this button to switch to QPSK");
    QWhatsThis::add(Afc, "Click this button to toggle AFC on/off");
	QWhatsThis::add(Dcd, "Click this button to toggle DCD on/off");
    QWhatsThis::add(Net, "Click this button to toggle NET on/off");
    QWhatsThis::add(Beacon, "Click this button to toggle BEACON on/off");
    QWhatsThis::add(TypeAhead, "Click here to switch between type-ahead and normal TX buffer" );
    QWhatsThis::add(TxStop, "Click here to interrupt a transmission");
    QWhatsThis::add(Vox, "Click here to toggle break-in operation");
}

void PhaseShift::initStatusBar()
{
    ///////////////////////////////////////////////////////////////////
    //STATUSBAR
    statusBar()->message( IDS_DEFAULT, 2000 );
}

void PhaseShift::initView()
{ 
    ////////////////////////////////////////////////////////////////////
    // set the main widget here

  	divider1 = new QLabel( this, "divider1" );
  	divider1->setGeometry( 0, 55, 640, 2 );
  	divider1->setMinimumSize( 0, 0 );
  	divider1->setMaximumSize( 32767, 32767 );
  	divider1->setFocusPolicy( QWidget::NoFocus );
  	divider1->setBackgroundMode( QWidget::PaletteBackground );
  	divider1->setFontPropagation( QWidget::NoChildren );
  	divider1->setPalettePropagation( QWidget::NoChildren );
  	divider1->setFrameStyle( 33 );
  	divider1->setLineWidth( 1 );
  	divider1->setMidLineWidth( 0 );
  	divider1->QFrame::setMargin( 0 );
  	divider1->setText( "" );
  	divider1->setAlignment( 289 );
  	divider1->setMargin( -1 );

  	QLabel *divider2 = new QLabel( this, "divider1" );
  	divider2->setGeometry( 5, 60, 121, 121 );
  	divider2->setMinimumSize( 0, 0 );
  	divider2->setMaximumSize( 32767, 32767 );
  	divider2->setFocusPolicy( QWidget::NoFocus );
  	divider2->setBackgroundMode( QWidget::PaletteBackground );
  	divider2->setFontPropagation( QWidget::NoChildren );
  	divider2->setPalettePropagation( QWidget::NoChildren );
  	divider2->setFrameStyle( 33 );
  	divider2->setLineWidth( 1 );
  	divider2->setMidLineWidth( 0 );
  	divider2->QFrame::setMargin( 0 );
  	divider2->setText( "" );
  	divider2->setAlignment( 289 );
  	divider2->setMargin( -1 );

	// The Phase Scope
	phaseScope = new PhaseScope( divider2, "phaseScope" );
  	phaseScope->setGeometry( 2, 2, 116, 116 );
    QWhatsThis::add( phaseScope, "This shows the phase angle and intensity of the received signal" );

    connect( &psk31rx, SIGNAL(signalStrength()), phaseScope, SLOT(plot()) );					

    // The master display
    masterDisplay = new MasterDisplay( this, "masterDisplay");
    masterDisplay->setGeometry( 130, 60, 200, 60 );
    connect( this, SIGNAL(signalFreq( float )), masterDisplay, SLOT(setRxFreq( float )));

    QLabel* qtarch_Label_11;
    qtarch_Label_11 = new QLabel( this, "Label_11" );
    qtarch_Label_11->setGeometry( 375, 59, 40, 10 );
    qtarch_Label_11->setMinimumSize( 0, 0 );
    qtarch_Label_11->setMaximumSize( 32767, 32767 );
    qtarch_Label_11->setFocusPolicy( QWidget::NoFocus );
    qtarch_Label_11->setBackgroundMode( QWidget::PaletteBackground );
    qtarch_Label_11->setFontPropagation( QWidget::NoChildren );
    qtarch_Label_11->setPalettePropagation( QWidget::NoChildren );
    qtarch_Label_11->setFrameStyle( 0 );
    qtarch_Label_11->setLineWidth( 1 );
    qtarch_Label_11->setMidLineWidth( 0 );
    qtarch_Label_11->QFrame::setMargin( 0 );
    qtarch_Label_11->setText( "Out" );
    qtarch_Label_11->setAlignment( 289 );
    qtarch_Label_11->setMargin( -1 );


    QSlider* outputSlider;
	outputSlider = new QSlider( this, "outputSlider" );
    outputSlider->setGeometry( 415, 60, 150, 10 );
	outputSlider->setMinimumSize( 0, 0 );
	outputSlider->setMaximumSize( 32767, 32767 );
	outputSlider->setFocusPolicy( QWidget::TabFocus );
	outputSlider->setBackgroundMode( QWidget::PaletteMid );
	outputSlider->setFontPropagation( QWidget::NoChildren );
	outputSlider->setPalettePropagation( QWidget::NoChildren );
	outputSlider->setOrientation( QSlider::Horizontal );
	outputSlider->setRange( 0, 99 );
	outputSlider->setSteps( 1, 10 );
	outputSlider->setValue( soundOutputLevel );
	outputSlider->setTracking( TRUE );
	outputSlider->setTickmarks( QSlider::NoMarks );
	outputSlider->setTickInterval( 0 );
    QWhatsThis::add( outputSlider, "Sets the sound card output level" );

	QLabel* outputLabel;
	outputLabel = new QLabel( this, "outputLabel" );
    outputLabel->setGeometry( 570, 59, 20, 10 );
	outputLabel->setMinimumSize( 0, 0 );
	outputLabel->setMaximumSize( 32767, 32767 );
	{
		QFont font( "helvetica", 12, 50, 0 );
		font.setFixedPitch( TRUE );
		font.setStyleHint( (QFont::StyleHint)0 );
		font.setCharSet( (QFont::CharSet)0 );
		outputLabel->setFont( font );
	}
	outputLabel->setFocusPolicy( QWidget::NoFocus );
	outputLabel->setBackgroundMode( QWidget::PaletteBackground );
	outputLabel->setFontPropagation( QWidget::NoChildren );
	outputLabel->setPalettePropagation( QWidget::NoChildren );
	outputLabel->setFrameStyle( 0 );
	outputLabel->setLineWidth( 1 );
	outputLabel->setMidLineWidth( 0 );
	outputLabel->QFrame::setMargin( 0 );
	outputLabel->setNum( soundOutputLevel );
	outputLabel->setAlignment( 289 );
	outputLabel->setMargin( -1 );

    connect( outputSlider, SIGNAL( valueChanged( int )),
           outputLabel, SLOT( setNum( int )));

    connect( outputSlider, SIGNAL( valueChanged( int )),
           mixer, SLOT( setVolume( int )));

    // The Input slider
	QSlider* inputSlider;
	inputSlider = new QSlider( this, "inputSlider" );
    inputSlider->setGeometry( 415, 76, 150, 10 );
	inputSlider->setMinimumSize( 0, 0 );
	inputSlider->setMaximumSize( 32767, 32767 );
	inputSlider->setFocusPolicy( QWidget::TabFocus );
	inputSlider->setBackgroundMode( QWidget::PaletteMid );
	inputSlider->setFontPropagation( QWidget::NoChildren );
	inputSlider->setPalettePropagation( QWidget::NoChildren );
	inputSlider->setOrientation( QSlider::Horizontal );
	inputSlider->setRange( 0, 99 );
	inputSlider->setSteps( 1, 10 );
	inputSlider->setValue( soundInputLevel );
	inputSlider->setTracking( TRUE );
	inputSlider->setTickmarks( QSlider::NoMarks );
	inputSlider->setTickInterval( 0 );
    QWhatsThis::add( inputSlider, "Sets the sound card input level" );

    QLabel* qtarch_Label_12;
    qtarch_Label_12 = new QLabel( this, "Label_12" );
    qtarch_Label_12->setGeometry( 375, 75, 40, 10 );
    qtarch_Label_12->setMinimumSize( 0, 0 );
    qtarch_Label_12->setMaximumSize( 32767, 32767 );
    qtarch_Label_12->setFocusPolicy( QWidget::NoFocus );
    qtarch_Label_12->setBackgroundMode( QWidget::PaletteBackground );
    qtarch_Label_12->setFontPropagation( QWidget::NoChildren );
    qtarch_Label_12->setPalettePropagation( QWidget::NoChildren );
    qtarch_Label_12->setFrameStyle( 0 );
    qtarch_Label_12->setLineWidth( 1 );
    qtarch_Label_12->setMidLineWidth( 0 );
    qtarch_Label_12->QFrame::setMargin( 0 );
    qtarch_Label_12->setText( "In" );
    qtarch_Label_12->setAlignment( 289 );
    qtarch_Label_12->setMargin( -1 );

	QLabel* inputLabel;
	inputLabel = new QLabel( this, "Label_18" );
    inputLabel->setGeometry( 570, 75, 20, 10 );
	inputLabel->setMinimumSize( 0, 0 );
	inputLabel->setMaximumSize( 32767, 32767 );
	{
		QFont font( "helvetica", 12, 50, 0 );
		font.setFixedPitch( TRUE );
		font.setStyleHint( (QFont::StyleHint)0 );
		font.setCharSet( (QFont::CharSet)0 );
		inputLabel->setFont( font );
	}
	inputLabel->setFocusPolicy( QWidget::NoFocus );
	inputLabel->setBackgroundMode( QWidget::PaletteBackground );
	inputLabel->setFontPropagation( QWidget::NoChildren );
	inputLabel->setPalettePropagation( QWidget::NoChildren );
	inputLabel->setFrameStyle( 0 );
	inputLabel->setLineWidth( 1 );
	inputLabel->setMidLineWidth( 0 );
	inputLabel->QFrame::setMargin( 0 );
	inputLabel->setNum( soundInputLevel );
	inputLabel->setAlignment( 289 );
	inputLabel->setMargin( -1 );

    connect( inputSlider, SIGNAL( valueChanged( int )),
           inputLabel, SLOT( setNum( int )));

    connect( inputSlider, SIGNAL( valueChanged( int )),
           mixer, SLOT( setLineLevel( int )));


    QSlider* PCM_slider;
    PCM_slider = new QSlider( this, "Slider_8" );
    PCM_slider->setGeometry( 415, 92, 150, 10 );
    PCM_slider->setMinimumSize( 0, 0 );
    PCM_slider->setMaximumSize( 32767, 32767 );
    PCM_slider->setFocusPolicy( QWidget::TabFocus );
    PCM_slider->setBackgroundMode( QWidget::PaletteMid );
    PCM_slider->setFontPropagation( QWidget::NoChildren );
    PCM_slider->setPalettePropagation( QWidget::NoChildren );
    PCM_slider->setOrientation( QSlider::Horizontal );
    PCM_slider->setRange( 0, 99 );
    PCM_slider->setSteps( 1, 10 );
    PCM_slider->setValue( soundPcmLevel );
    PCM_slider->setTracking( TRUE );
    PCM_slider->setTickmarks( QSlider::NoMarks );
    PCM_slider->setTickInterval( 0 );
    QWhatsThis::add( PCM_slider, "Sets the sound card PCM level" );

	QLabel* qtarch_Label_13;
	qtarch_Label_13 = new QLabel( this, "Label_13" );
	qtarch_Label_13->setGeometry( 375, 91, 40, 10 );
	qtarch_Label_13->setMinimumSize( 0, 0 );
	qtarch_Label_13->setMaximumSize( 32767, 32767 );
	qtarch_Label_13->setFocusPolicy( QWidget::NoFocus );
	qtarch_Label_13->setBackgroundMode( QWidget::PaletteBackground );
	qtarch_Label_13->setFontPropagation( QWidget::NoChildren );
	qtarch_Label_13->setPalettePropagation( QWidget::NoChildren );
	qtarch_Label_13->setFrameStyle( 0 );
	qtarch_Label_13->setLineWidth( 1 );
	qtarch_Label_13->setMidLineWidth( 0 );
	qtarch_Label_13->QFrame::setMargin( 0 );
	qtarch_Label_13->setText( "PCM" );
	qtarch_Label_13->setAlignment( 289 );
	qtarch_Label_13->setMargin( -1 );

	QLabel* qtarch_Label_19;
	qtarch_Label_19 = new QLabel( this, "Label_19" );
	qtarch_Label_19->setGeometry( 570, 91, 20, 10 );
	qtarch_Label_19->setMinimumSize( 0, 0 );
	qtarch_Label_19->setMaximumSize( 32767, 32767 );
	{
		QFont font( "helvetica", 12, 50, 0 );
		font.setFixedPitch( TRUE );
		font.setStyleHint( (QFont::StyleHint)0 );
		font.setCharSet( (QFont::CharSet)0 );
		qtarch_Label_19->setFont( font );
	}
	qtarch_Label_19->setFocusPolicy( QWidget::NoFocus );
	qtarch_Label_19->setBackgroundMode( QWidget::PaletteBackground );
	qtarch_Label_19->setFontPropagation( QWidget::NoChildren );
	qtarch_Label_19->setPalettePropagation( QWidget::NoChildren );
	qtarch_Label_19->setFrameStyle( 0 );
	qtarch_Label_19->setLineWidth( 1 );
	qtarch_Label_19->setMidLineWidth( 0 );
	qtarch_Label_19->QFrame::setMargin( 0 );
	qtarch_Label_19->setNum( soundPcmLevel );
	qtarch_Label_19->setAlignment( 289 );
	qtarch_Label_19->setMargin( -1 );

    connect( PCM_slider, SIGNAL( valueChanged( int )),
           qtarch_Label_19, SLOT( setNum( int )));

    connect( PCM_slider, SIGNAL( valueChanged( int )),
           mixer, SLOT( setPcmLevel( int )));


    QSlider* meterSensitivity;
	meterSensitivity = new QSlider( this, "Slider_2" );
    meterSensitivity->setGeometry( 415, 108, 150, 10 );
	meterSensitivity->setMinimumSize( 0, 0 );
	meterSensitivity->setMaximumSize( 32767, 32767 );
	meterSensitivity->setFocusPolicy( QWidget::TabFocus );
	meterSensitivity->setBackgroundMode( QWidget::PaletteMid );
	meterSensitivity->setFontPropagation( QWidget::NoChildren );
	meterSensitivity->setPalettePropagation( QWidget::NoChildren );
	meterSensitivity->setOrientation( QSlider::Horizontal );
	meterSensitivity->setRange( 0, 99 );
	meterSensitivity->setSteps( 1, 10 );
	meterSensitivity->setValue( squelchlevel );
	meterSensitivity->setTracking( TRUE );
	meterSensitivity->setTickmarks( QSlider::NoMarks );
	meterSensitivity->setTickInterval( 0 );
    QWhatsThis::add( meterSensitivity, "Sets the squelch level" );

	QLabel* qtarch_Label_20;
	qtarch_Label_20 = new QLabel( this, "Label_20" );
	qtarch_Label_20->setGeometry( 570, 107, 20, 10 );
	qtarch_Label_20->setMinimumSize( 0, 0 );
	qtarch_Label_20->setMaximumSize( 32767, 32767 );
	{
		QFont font( "helvetica", 12, 50, 0 );
		font.setFixedPitch( TRUE );
		font.setStyleHint( (QFont::StyleHint)0 );
		font.setCharSet( (QFont::CharSet)0 );
		qtarch_Label_20->setFont( font );
	}
	qtarch_Label_20->setFocusPolicy( QWidget::NoFocus );
	qtarch_Label_20->setBackgroundMode( QWidget::PaletteBackground );
	qtarch_Label_20->setFontPropagation( QWidget::NoChildren );
	qtarch_Label_20->setPalettePropagation( QWidget::NoChildren );
	qtarch_Label_20->setFrameStyle( 0 );
	qtarch_Label_20->setLineWidth( 1 );
	qtarch_Label_20->setMidLineWidth( 0 );
	qtarch_Label_20->QFrame::setMargin( 0 );
	qtarch_Label_20->setNum( squelchlevel );
	qtarch_Label_20->setAlignment( 289 );
	qtarch_Label_20->setMargin( -1 );


	QLabel* qtarch_Label_14;
	qtarch_Label_14 = new QLabel( this, "Label_14" );
	qtarch_Label_14->setGeometry( 375, 107, 40, 10 );
	qtarch_Label_14->setMinimumSize( 0, 0 );
	qtarch_Label_14->setMaximumSize( 32767, 32767 );
	qtarch_Label_14->setFocusPolicy( QWidget::NoFocus );
	qtarch_Label_14->setBackgroundMode( QWidget::PaletteBackground );
	qtarch_Label_14->setFontPropagation( QWidget::NoChildren );
	qtarch_Label_14->setPalettePropagation( QWidget::NoChildren );
	qtarch_Label_14->setFrameStyle( 0 );
	qtarch_Label_14->setLineWidth( 1 );
	qtarch_Label_14->setMidLineWidth( 0 );
	qtarch_Label_14->QFrame::setMargin( 0 );
	qtarch_Label_14->setText( "SQCH" );
	qtarch_Label_14->setAlignment( 289 );
	qtarch_Label_14->setMargin( -1 );

    connect( meterSensitivity, SIGNAL( valueChanged( int )),
           qtarch_Label_20, SLOT( setNum( int )));

    connect( meterSensitivity, SIGNAL( valueChanged( int )),
           this, SLOT( slotSquelchLevel( int )));

    QFont font( "Fixed", 12, 50, 0 );

    rxLine = new QLineEdit( this, "rxLine" );
    rxLine->setGeometry( 5, 185, 642, 30 );
    rxLine->setMinimumSize( 0, 0 );
    rxLine->setMaximumSize( 32767, 32767 );
    rxLine->setFocusPolicy( QWidget::StrongFocus );
	rxLine->setBackgroundMode( QWidget::PaletteBase );
    rxLine->setFontPropagation( QWidget::NoChildren );
    rxLine->setPalettePropagation( QWidget::NoChildren );
    rxLine->setFont( font );
    rxLine->setText( "" );
    rxLine->setMaxLength( 32767 );
    rxLine->setFrame( QLineEdit::Normal );
    rxLine->setFrame( TRUE );
    QWhatsThis::add( rxLine, "This is the receive text area" );

    txLine1 = new MyQLineEdit( this, "txLine_1" );
    txLine1->setGeometry( 5, 220, 642, 30 );
    txLine1->setMinimumSize( 0, 0 );
    txLine1->setMaximumSize( 32767, 32767 );
    txLine1->setFocusPolicy( QWidget::StrongFocus );
    txLine1->setBackgroundMode( QWidget::PaletteBase );
    txLine1->setFontPropagation( QWidget::NoChildren );
    txLine1->setPalettePropagation( QWidget::NoChildren );
    txLine1->setFont( font );
    txLine1->setText( "" );
    txLine1->setMaxLength( 32767 );
    txLine1->setFrame( QLineEdit::Normal );
    txLine1->setFrame( TRUE );
    QWhatsThis::add( txLine1, "This is the transmit text area #1\n"
                              "Each character you type here is sent\n"
                              "to the transmit buffer as you type." );
    txLine = txLine1;

    connect( txLine1, SIGNAL( keyPressed( const char )), this, SLOT( txTextCallback( const char )));
    connect( txLine1, SIGNAL( returnPressed()), this, SLOT(txReturnPressedCallback()));
    connect( txLine1, SIGNAL( CTRL_R_Pressed()), this, SLOT(slotTxStop()));
    connect( txLine1, SIGNAL( CTRL_T_Pressed(bool)), this, SLOT(setTx(bool)));
    connect( txLine1, SIGNAL( CTRL_E_Pressed(bool)), this, SLOT(setTx(bool)));

    txLine2 = new QLineEdit( this, "txLine_2" );
    txLine2->setGeometry( 5, 220, 642, 30 );
    txLine2->setMinimumSize( 0, 0 );
    txLine2->setMaximumSize( 32767, 32767 );
    txLine2->setFocusPolicy( QWidget::StrongFocus );
    txLine2->setBackgroundMode( QWidget::PaletteBase );
    txLine2->setFontPropagation( QWidget::NoChildren );
    txLine2->setPalettePropagation( QWidget::NoChildren );
    txLine2->setFont( font );
    txLine2->setText( "" );
    txLine2->setMaxLength( 32767 );
    txLine2->setFrame( QLineEdit::Normal );
    txLine2->setFrame( TRUE );
    QWhatsThis::add( txLine2, "This is the transmit text area #2.\n"
                              "Here you can type a line of text and it will be sent\n"
                              "to the transmit buffer only when you press return.");

    connect( txLine2, SIGNAL(returnPressed()), this, SLOT(txLineCallback()));
    txLine2->hide();

    spectrumDisplay = new SpectrumFrame( this, "spectrumDisplay");
    spectrumDisplay->setGeometry( 130, 124, WF_SIZE, 56 );
    spectrumDisplay->setMinimumSize( 0, 0 );
    spectrumDisplay->setMaximumSize( 32767, 32767 );
    spectrumDisplay->setFocusPolicy( QWidget::NoFocus );
    spectrumDisplay->setBackgroundMode( QWidget::PaletteBackground );
    spectrumDisplay->setFontPropagation( QWidget::NoChildren );
    spectrumDisplay->setPalettePropagation( QWidget::NoChildren );
    spectrumDisplay->setFrameStyle( 50 );
    spectrumDisplay->setLineWidth( 1 );
    spectrumDisplay->setMidLineWidth( 0 );
    spectrumDisplay->QFrame::setMargin( 0 );
    spectrumDisplay->setCursor( QCursor( UpArrowCursor ) );
    QWhatsThis::add( spectrumDisplay, "This is the spectrum of the received signal" );
    connect( spectrumDisplay, SIGNAL(signalChangeFreq( float )),
             this, SLOT(setFreq(float)));

	cqPushButton = new QPushButton( this, "PushButton_10" );
	cqPushButton->setGeometry( 595, 60, 50, 30 );
	cqPushButton->setMinimumSize( 0, 0 );
	cqPushButton->setMaximumSize( 32767, 32767 );
	cqPushButton->setFocusPolicy( QWidget::TabFocus );
	cqPushButton->setBackgroundMode( QWidget::PaletteBackground );
	cqPushButton->setFontPropagation( QWidget::NoChildren );
	cqPushButton->setPalettePropagation( QWidget::NoChildren );
	cqPushButton->setText( "CQ" );
	cqPushButton->setAutoRepeat( FALSE );
	cqPushButton->setAutoResize( FALSE );
	cqPushButton->setToggleButton( FALSE );
	cqPushButton->setDefault( FALSE );
	cqPushButton->setAutoDefault( FALSE );
	cqPushButton->setIsMenuButton( FALSE );
    QWhatsThis::add( cqPushButton, "Click here to call CQ" );
	
	connect(cqPushButton, SIGNAL(pressed()), this, SLOT(slotCQ()));

	txPushButton = new QPushButton( this, "PushButton_10" );
	txPushButton->setGeometry( 595, 90, 50, 30 );
	txPushButton->setMinimumSize( 0, 0 );
	txPushButton->setMaximumSize( 32767, 32767 );
	txPushButton->setFocusPolicy( QWidget::TabFocus );
	txPushButton->setBackgroundMode( QWidget::PaletteBackground );
	txPushButton->setFontPropagation( QWidget::NoChildren );
	txPushButton->setPalettePropagation( QWidget::NoChildren );
	txPushButton->setText( "TX" );
	txPushButton->setAutoRepeat( FALSE );
	txPushButton->setAutoResize( FALSE );
	txPushButton->setToggleButton( TRUE );
	txPushButton->setDefault( FALSE );
	txPushButton->setAutoDefault( FALSE );
	txPushButton->setIsMenuButton( FALSE );
    QWhatsThis::add( txPushButton, "Click here to activate transmitter" );

	connect( txPushButton, SIGNAL(toggled(bool)), this, SLOT(setTx(bool)));

	// The RX frequency control
	QScrollBar* rxFreqBar = new QScrollBar( 0, 1000, 1, 10, rxFreq / 2, QScrollBar::Vertical, this, "rxFreqBar");
	rxFreqBar->setGeometry( 336, 60, 15, 60 );
    QWhatsThis::add( rxFreqBar, "Sets the RX audiofrequency" );
	connect( rxFreqBar, SIGNAL( valueChanged( int ) ), this, SLOT( setRxFreq( int ) ) );

    // The TX frequency control
	QScrollBar* txFreqBar = new QScrollBar( 0, 1000, 1, 10, txFreq / 2, QScrollBar::Vertical, this, "rxFreqBar");
	txFreqBar->setGeometry( 353, 60, 15, 60 );
    QWhatsThis::add( txFreqBar, "Sets the TX audio frequency" );

	macroList = new QListBox( this, "callList" );
	macroList->setAutoBottomScrollBar( false );
    macroList->setGeometry( 336, 124, 300, 56 );
    QWhatsThis::add( macroList, "This is the list of macros.\n"
       "Chose the macro and double click the mouse to select" );

    for ( int i = 0; i < numMacros; i++ )
	{
  	    macroList->insertItem( macro[ i ] );
    }
    connect( macroList, SIGNAL(selected( int )),
             this, SLOT(slotListItemSelected( int )));

    resize( 642,275 );
    setMinimumSize( 642, 275 );
	setMaximumSize( 800, 275 );
}

bool PhaseShift::queryExit()
{
  int exit=QMessageBox::information(this,"Quit...","Really Quit?",QMessageBox::Ok,QMessageBox::Cancel);

    if(exit==1){
      return true;
    }
    else{
      return false;
    }
}

void PhaseShift::resizeEvent( QResizeEvent * )
{
	int w = width();
	int h = height();

    if ( w < 635 ) w = 635;
    if ( h < 485 ) h = 485;

	divider1->setGeometry( 0, 55, w, 2 );
    cqPushButton->setGeometry( 590, 60, w - 595, 28 );
	txPushButton->setGeometry( 590, 90, w - 595, 28 );
    rxLine->setGeometry( 5, 185, w - 10, 30 );
    txLine1->setGeometry( 5, 220, w - 10, 30 );
    txLine2->setGeometry( 5, 220, w - 10, 30 );
    macroList->setGeometry( 336, 124, w - 341, 56 );
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

//Del by KDevelop: //Del by KDevelop: void PhaseShift::displayRxChar( const char *c )
//Del by KDevelop: //Del by KDevelop: {
//Del by KDevelop: //Del by KDevelop: 	statusArea->append( c );
//Del by KDevelop: //Del by KDevelop: }


void PhaseShift::slotFileOpen()
{
    char fileName[ 256 ];

    sendingFileFlag = 1;

    statusBar()->message("Opening file...");

    QString fn = QFileDialog::getOpenFileName(0,0,this);
    strcpy( fileName, fn );

    if ( !fn.isEmpty() )
    {
        if ((send_fp = fopen(fileName, "r")) == NULL )
        {
            perror("Cannot open file");
            return;
        }
        QString message = "Sending File: " + fn;
        statusBar()->message( message, 2000 );
        psk31tx.send_char( TX_START );
        transmit = 1;
        doPTT (PTT_ON);
        sendFile();
    }
    else
    {
        statusBar()->message( "Send aborted", 2000 );
    }
}


void PhaseShift::slotFileSave()
{
    char logFileName[128];
    time_t t;
    struct tm *_tm;

    t = time( NULL );
    _tm = localtime( &t );

    if ( !logOpen )
    {
        statusBar()->message("Opening Log File...");
        if ( logDir[0] == '\0' )
        {
            printf("Log dir not set in phaseshift.conf\n");
            return;
        }

        sprintf(logFileName, "%s/%s/%04d%02d%02d.log",
                getenv("HOME"),
                logDir,
                _tm->tm_year + 1900,
                _tm->tm_mon + 1,
                _tm->tm_mday );
        statusBar()->message(logFileName);
        if ((logFile = fopen(logFileName, "a")) == NULL )
        {
            perror(logFileName);
            return;
        }
        logOpen = 1;
        fprintf(logFile, "\n\n### Log open on %s\n", ctime( &t ));
    }
    else
    {
        statusBar()->message("LogFile is already open...");
        fprintf(logFile, "\n\n### Log open on %s\n", ctime( &t ));
    }
}

void PhaseShift::slotFileSaveAs()
{
    char logFileName[128];

    if ( !logOpen )
    {
        statusBar()->message("Saving log file under new filename...");
        QString fn = QFileDialog::getSaveFileName(0,0,this);
        if ( !fn.isEmpty() )
        {
            logOpen = 1;
            strcpy( logFileName, fn );

            if ((logFile = fopen(logFileName, "a")) == NULL )
            {
                perror("Cannot open log file");
                return;
            }
        }
        else
        {
            statusBar()->message( "Opening aborted", 2000 );
        }
    }
    else
    {
        statusBar()->message("LogFile is already open...", 2000 );
    }
}

void PhaseShift::slotFileClose()
{
    if ( logOpen )
    {
        statusBar()->message("Closing file...");
        logOpen = 0;
        fclose( logFile );
        statusBar()->message(IDS_DEFAULT);
    }
}

void PhaseShift::slotFilePrint()
{
  statusBar()->message("Printing...");
  QPrinter printer;
  if (printer.setup(this))
  {
      QPainter painter;
      painter.begin( &printer );

      ///////////////////////////////////////////////////////////////////
      // TODO: Define printing by using the QPainter methods here

      painter.end();
  };
  statusBar()->message(IDS_DEFAULT);
}

void PhaseShift::slotFileQuit()
{ 
  ///////////////////////////////////////////////////////////////////
  // exits the Application
  if ( logOpen ) fclose( logFile );
  qApp->quit();
}

void PhaseShift::slotEditCut()
{
  statusBar()->message("Cutting selection...");
  statusBar()->message(IDS_DEFAULT);
}

void PhaseShift::slotEditCopy()
{
  statusBar()->message("Copying selection to Clipboard...");
  statusBar()->message(IDS_DEFAULT);
}

void PhaseShift::slotEditPaste()
{
  statusBar()->message("Inserting Clipboard contents...");
  statusBar()->message(IDS_DEFAULT);
}

void PhaseShift::slotPrependCallsign()
{
  if ( prependCallsignFlag ) {
    prependCallsignFlag = 0;
    options_menu->setItemChecked( ID_PREPEND_CALLSIGN, false);
  } 
  else {
    prependCallsignFlag = 1;
    options_menu->setItemChecked( ID_PREPEND_CALLSIGN, true );
  }
}


void PhaseShift::slotHelpAbout(){
  QMessageBox::about(this,"About...",
    IDS_APP_ABOUT
    " \n(w) 2000 by Edson Pereira, PU1JTE, N1VTN\n\n"
	  "I would like to thank Pawel Jalocha for the creation of the\n"
    "slow BPSK mode; Peter Martinez for the creation of the PSK31 mode;\n"
    "and Hansi Reiser for the creation of the Linux version of PSK31.\n"
    "Without their contribution this work would not be possible\n\n"
    "This program is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public License as published by\n"
    "the Free Software Foundation; either version 2 of the License, or\n"
    "(at your option) any later version.\n\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
    "See the GNU General Public License for more details.\n");
}

void PhaseShift::slotStatusHelpMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}

void PhaseShift::slotTune()
{
    statusBar()->message("Sending Tune Tone...");
    for ( int i = 0; i < 10; i++ )
    {
        psk31tx.tune();
    }
    psk31tx.send_char( TX_END );
    transmit = 1;
    doPTT (PTT_ON);
}

void PhaseShift::slotSquelchLevel( int level )
{
   squelchlevel = level;
}

void PhaseShift::slotBPSK()
{
  psk31rx.set_mode( 0, lsbFlag );
  psk31tx.set_mode( 0, lsbFlag );
  masterDisplay->setMode( 0 );
}

void PhaseShift::slotQPSK()
{
  psk31rx.set_mode( 1, lsbFlag );
  psk31tx.set_mode( 1, lsbFlag );
  masterDisplay->setMode( 1 );
}

void PhaseShift::slotAFC()
{
	afcFlag = !afcFlag;
	psk31rx.set_afc( afcFlag );
    masterDisplay->setAFC( afcFlag );
}

void PhaseShift::slotDCD()
{
    dcdFlag = !dcdFlag;
	psk31rx.set_dcd( dcdFlag );
	masterDisplay->setDCD( dcdFlag );
}

void PhaseShift::slotNET()
{
    netFlag = !netFlag;
	psk31rx.set_txtrack( netFlag );
	masterDisplay->setNET( netFlag );
}

void PhaseShift::slotBEACON()
{
    beaconFlag = !beaconFlag;
    masterDisplay->setBeacon( beaconFlag );
}

void PhaseShift::setRxFreq( int f )
{
    float freq = -1.0 * (float) f + 1500;
    masterDisplay->setRxFreq( freq );
	psk31rx.set_freq( freq );	
}

// This slot is called by the FFT Engine once we have complete the
// computation of a FFT series and we are ready to display it.
void PhaseShift::drawSpectrum()
{
   int x, y, rxDial;
   QRect    r = spectrumDisplay->frameRect();
   r.setWidth( r.width() - 2 );
   r.setHeight( r.height() - 2 );
   QPixmap  pix( r.size());
   QPainter p;
   QPainter tmp;

   pix.setOptimization( QPixmap::BestOptim );

   //pix.fill( this, r.topLeft() );
   pix.fill( QColor( 0, 55, 0) );

   //static QPen pen1( QColor( 105, 105, 235 ), 1 );
   static QPen pen1( QColor( 0, 255, 0 ), 1);  // green
   static QPen pen2( QColor( 255, 100, 100 ), 1 );    // red
   static QPen pen3( QColor( 255, 255, 255 ), 1 );    //
   static QPen pen4( QColor( 0 ,0 ,0 ), 1 );

   static float deltaf = 8000.0 / MAX_SAMPLES / 2;                 /* Hz per filter */
   float fc;                     /* center freq of fft */
   int peak = 0;
   int rxCursor;
   //float signalAvg = 0;

   rxDial = WF_SIZE / 2;
   fc = rxFreq / deltaf;
   int delta = (int)(fc - WF_SIZE );

   // find signal peak within the tuning window (~40 Hz)
   for (x = (int)fc - 12; x < (int)fc + 12; x++ )
   {
       if ( peak < fftEngine->values[ x ] )
       {
           peak = fftEngine->values[ x ];
       }
   }
   rxCursor = 54 - peak;
   if ( rxCursor < 10 ) rxCursor = 10;

   // Sample the spectrum near the tuning window and average the bins.
   // This is a relative (brute force) SNR measurement
/*   for ( x = (int)fc - 24; x < (int)fc - 12; x++ )
   {
        signalAvg += fftEngine->values[ x ];
   }
   for ( x = (int)fc + 12; x < (int)fc + 24; x++ )
   {
        signalAvg += fftEngine->values[ x ];
   }
   signalAvg /= 24.0;
   snr = signalAvg;
*/
   tmp.begin( &pix );

   for (x = 1; x < WF_SIZE - 1; x++)
   {
      int amplitude;

      amplitude =  fftEngine->values[ 2 * x + delta + 1 ];

      if ( amplitude < 0 ) amplitude = 0;

      y = (WF_HEIGHT - amplitude) > 5 ? (WF_HEIGHT - amplitude) : 5;

      tmp.setPen( pen1 );
      tmp.drawLine( x, WF_HEIGHT - 1, x, y - 1 );
      //tmp.setPen( pen4 );
      //tmp.drawLine( x, y - 1, x, 1 );

        if ( x == rxDial )
        {
            tmp.setPen( pen2 );
            tmp.drawLine( rxDial, rxCursor - 3, rxDial, rxCursor - 8 );
        }
        else if ( x == rxDial - 4 || x == rxDial + 4 )
        {
            tmp.setPen( pen3 );
            tmp.drawLine( x, rxCursor - 3, x, rxCursor - 8 );
        }
   }

   // Draw the scale
   //tmp.setPen( pen2 );
/*   for (x = 1; x < WF_SIZE - 2; x++)
   {
      if ( x % 16 == 0 )
      {
          p.drawLine( x, 1, x, 3 );
      }
   }
*/
   tmp.end();

   QPoint point = r.topLeft();
   point.setX( point.x() + 1 );
   point.setY( point.y() + 1 );
   p.begin( spectrumDisplay );
   p.drawPixmap( point, pix );
   p.end();

/* This has to be perfected some time later. The variances are too large */
/*	 if ( intermodFlag )
   {
      intermodLevel = fftEngine->values[ 2 * (rxDial - 4)  + delta ] +
                      fftEngine->values[ 2 * (rxDial + 4)  + delta ] -
                      fftEngine->values[ 2 * (rxDial - 12) + delta ] -
                      fftEngine->values[ 2 * (rxDial + 12) + delta ];
      intermodFlag = 0;
      printf( "intermod = %d\n", intermodLevel );
   }
*/
}
void PhaseShift::statusCallback(int id_){
  switch (id_) {
    
  case ID_FILE_NEW        :  slotStatusHelpMsg("Creates a new document");break;
  case ID_FILE_OPEN       :  slotStatusHelpMsg("Opens an existing document");break;
  case ID_FILE_SAVE       :  slotStatusHelpMsg("Save the actual document");break;
  case ID_FILE_SAVE_AS    :  slotStatusHelpMsg("Save the actual document");break;
  case ID_FILE_CLOSE      :  slotStatusHelpMsg("Closes the actual file");break;
  case ID_FILE_QUIT       :  slotStatusHelpMsg("Exits the program");break;
  case ID_EDIT_CUT        :  slotStatusHelpMsg("Cuts the selected section and puts it to the clipboard")
			      ;break;
  case ID_EDIT_COPY       :  slotStatusHelpMsg("Copys the selected section to the clipboard");break;
  case ID_EDIT_PASTE      :  slotStatusHelpMsg("Pastes the clipboard contents to actual position");break;
  case ID_EDIT_SELECT_ALL :  slotStatusHelpMsg("Selects the whole document contents");break;
  case ID_PREPEND_CALLSIGN:  slotStatusHelpMsg("Enables / disables the prepending of callsign in each line transmitted");break;
  case ID_VIEW_STATUSBAR  :  slotStatusHelpMsg("Enables / disables the Statusbar");break;   
  case ID_HELP_ABOUT      :  slotStatusHelpMsg("Shows an aboutbox");break;  
  }
}

/** Process psk31 events */
void PhaseShift::process()
{
   int res;
   static int strength;
   static int dd = 0, de = 0;
   struct timeval tm;
   static float rxfreqOld;
   int phdelta;
   fd_set rset, wset, eset;
   static double sum = 0;
   static int i = 0;


    /* Here starts the whole thing */

    FD_ZERO(&rset); FD_ZERO(&wset); FD_ZERO(&eset);
    FD_SET(audio_fd, &rset);
    if(transmit)
    {
        FD_SET(audio_fd, &wset);
    }

    tm.tv_sec=0; tm.tv_usec=500;
    res = select( audio_fd + 1, &rset, &wset, &eset, &tm);

    /*********** RECEIVE ***********/
    if(!transmit && FD_ISSET(audio_fd, &rset) )
    {
        /* handle audio input */
        int s; short sample;
        for( ; ; )
        {
            res = read(audio_fd, &sample, 2);

            //s=(int)((random()-RAND_MAX/2) * 2 * ((float)noiselevel/RAND_MAX));

            //if(noiselevel)
            //	s+=sample;
            //else
            s=sample;

            if( res == 2 )
            {
                processSample( s );
                res = psk31rx.process_rx_sample( s );
                fftEngine->addSample( s );
                psk31rx.get_info (NULL, NULL, &rxFreq, NULL, NULL,
                                 &dcdFlag, &phdelta, &strength);

                phaseScope->set( phdelta, strength, dcdFlag );

                sum += (double)(samples[i] * samples[i]);
                i++;
                if ( i > SAMPLESLEN )
                {
                    masterDisplay->setInputLevel( sqrt(sum/SAMPLESLEN) );
                    i = 0; sum = 0;
                }

               if ( netFlag )
               {
                    txFreq = rxFreq;
                    psk31tx.set_freq( txFreq );
               }

               if(res != NO_CHAR
                  && ( dcdFlag || confidenceLevel >=squelchlevel ))
               {
                  /* ignore ctrl chars */
                  if(res<32 && res!='\n' && res!='\r' && res!='\b')
                  {
                     res=' ';
                  }
                  /* Display received char */
                  rxText( res );

               }
               if(  dd++ > 0x0140 )
               {
                    dd = 0;
                    if ( de++ > 0x000f )
                    {
                        de = 0;
                        masterDisplay->setDCD( dcdFlag );
                        if (fabs(rxFreq - rxfreqOld) > .1)
                        {
	                        emit signalFreq( rxFreq );
                            rxfreqOld = rxFreq;
                            if ( netFlag )
                            {
                                masterDisplay->setTxFreq( txFreq );
                            }
                            txLine->setFocus();
                        }


                        int _confidence = 0;
                        for ( int i = 0; i < psk31rx.confidenceLen; i++ )
                        {
                            _confidence += psk31rx.confidence_A[i];
                        }
                        confidenceLevel = (int)((float)_confidence /
                                          (float)CONFLEN * 100.0);

                        masterDisplay->setConfidence( confidenceLevel );
                    }
                }
            }
            else if ( res == 0 )
            {
               break;
            }
            else if ( res < 0 && (errno==EAGAIN || errno==EBUSY))
            {
               break;
            }
            else
            {
                perror( "read" );
                exit(1);
            }
         }
      }

      /*********** TRANSMIT ***********/
      if(transmit)
      {
         for(;;)
         {
            res = psk31tx.processor();
            if(res == TX_BUSY)
            {
                break;
            }
            else if(res == TX_ERROR)
            {
               perror("tx error:");
               break;
            }
            else if( res == TX_END )
            {
               transmit = 0;
               bytesSent = 0;
               if(!full_duplex)
               {
                  ioctl(audio_fd, SNDCTL_DSP_SYNC);
               }
               doPTT(PTT_OFF);          // Turn PTT off
               statusBar()->message("");
               break;
            }

            if ( sendingFileFlag )
            {
                bytesSent++;
                char temp[ 40 ];
                sprintf( temp, "Characters sent: %d", bytesSent );
                statusBar()->message( temp );
                if ( bytesSent % 512 == 0 )
                {
                    sendFile();
                }
            }
            //Display the tx char
            rxText ( (char)res );
        }
    }
}


double PhaseShift::computePower( void )
{
    double sum = 0;
    int len = SAMPLESLEN;

    for( int i = 0; i < len; i++ ) sum += (double)(samples[i] * samples[i]);

    return sqrt(sum/len);
}
#define BUFSIZE 128
#define BUFGAP   32

void PhaseShift::rxText( int c )
{
  static unsigned char buffer[BUFSIZE + 1];
  static int head = 0;
  static int flag = 0;

  char line[BUFSIZE + 2];

  //fprintf(stdout, "%c", c), fflush( stdout );

  if ( logOpen ) fputc( c, logFile );

    // Clear buffer
    if ( !flag )
    {
        for ( int i = 0; i < BUFSIZE; i++ ) buffer[ i ] = ' ';
        flag = 1;
    }
    if ( c == 0x0d || c == 0x0a ) c = 0x20;

    if ( c == 0x08 && head > 0 )
    {
        head--;
    }
    else
    {
        buffer[ head++ ] = (char)c;
        if ( head >= BUFSIZE ) head = 0;
    }

    int j = head;
    for ( int i = 0; i < BUFSIZE; i++ )
    {
        if ( j >= BUFSIZE ) j = 0;
        line[ i ] = buffer[ j++ ];
    }

    line[ BUFSIZE ] = '\0';
    buffer[ BUFSIZE ] = '\0';

    // Autodetect stations calling CQ
    /*
    if ( strncasecmp( line + ( BUFSIZE - 30), "CQ CQ CQ DE ", 12) == 0)
    {
        int p;
        char callsign[ 11 ];

        for ( p = 0; p < 10; p++ )
        {
            if ( line [ (BUFSIZE - 30) + 12 + p ] == ' ' ) break;
        }
        strncpy( callsign, line + (BUFSIZE - 30) + 12, p );
        callsign[ p ] = '\0';
        sprintf( msgbuf,
            "=====> PSK31: %s heard near 14070 kHz. SNR = %f dB\n",
            callsign, snr);
        if ( networked )
        {
            if ( sendto(udp_sockfd, msgbuf,
                strlen( msgbuf ), MSG_DONTWAIT,
                (struct sockaddr *) &udp_serv_addr,
                sizeof(udp_serv_addr)) < 0 )
            {
                perror( "sendto" );
            }

        }
    }
    else if ( strncasecmp( line + ( BUFSIZE - 30), "DE ", 3 ) == 0 )
    {
        printf("DE detected");
    }
    */
    rxLine->setText( line );
    //printf("%02d %02x %c ", head, (unsigned char)c, (unsigned char)c );
    //printf("%s\n", buffer);
}

/** Read the configuration file .phaseshift in the users's home directory */
void PhaseShift::readConfig()
{
    int cfgerr=0;
    int lineno = 0;
    FILE *fp=(FILE *)0;
    char cfgfile[1024], buf[256], *ptr;
    float f;
    int n = 0;

    ptr=getenv("HOME");
    if(ptr)
    {
        strncpy(cfgfile, ptr, 1000);
        strcat(cfgfile, "/.phaseshift.conf");
        fp = fopen(cfgfile, "r");
    }
    if(!fp)
    {
        fprintf(stderr, "Configuration file ~/.phaseshift.conf not found");
        exit(1);
    }

    while(fgets(buf, 128, fp) != (char *)0)
    {
        lineno++;
        if ( buf[0] == '#' ||
             buf[0] == ' ' ||
             buf[0] == '\r'||
             buf[0] == '\n' ) continue;
									
    	if(sscanf(buf, "CALL=\"%[^\"]", callsign) == 1)
    	{
    	    printf("Call = %s\n", callsign );
    	}
    	else if(sscanf(buf, "FREQ=%f", &f) == 1)
    	{
    	    txFreq = rxFreq = f;
    	    printf("Freq = %f\n", f );
    	}
    	else if(sscanf(buf, "AFC=%d", &afcFlag) == 1)
    	{
    	    printf("AFC = %s\n", afcFlag ? "ON" : "OFF" );
    	}
    	else if(sscanf(buf, "SNDDEVICE=\"%[^\"]", sound_device) == 1)
    	{
    	    printf("SoundDevice = %s\n", sound_device );
    	}
        else if(sscanf(buf, "SNDINPUT=%d", &soundInputLevel) == 1)
        {
            printf("SoundInputLevel = %d\n", soundInputLevel );
        }
		else if(sscanf(buf, "SNDOUTPUT=%d", &soundOutputLevel) == 1)
		{
		    printf("SounfOutputLevel = %d\n", soundOutputLevel );
		}
		else if(sscanf(buf, "SNDPCM=%d", &soundPcmLevel) == 1)
		{
		    printf("SoundPcmLevel = %d\n", soundPcmLevel );
		}
		else if(sscanf(buf, "SQLCHLEVEL=%d", &squelchlevel) == 1 )
		{
		    printf("SquelchLevel = %d\n", squelchlevel );
		}
		else if(sscanf(buf, "PTTDEVICE=\"%[^\"]", pttDevice) == 1 )
		{
    		printf("pttDevice = \"%s\"\n", pttDevice );
        }
 		else if(sscanf(buf, "BEACON=\"%[^\"]", beaconText) == 1 )
 		{
 		    printf("BeaconText = \"%s\"\n", beaconText );
 		}
 		else if(sscanf(buf, "BEACONINTERVAL=%d", &beaconInterval) == 1 )
 		{
 		    printf("BeaconInterval = %d\n", beaconInterval );
 		}
		else if(sscanf(buf, "LOGDIR=\"%[^\"]", logDir) == 1 )
		{
		    printf("LogDirectory = %s/%s\n", getenv("HOME"), logDir );
		}
		else if((n = sscanf(buf, "MACRO=\"%[^\"]", macro[ numMacros++ ])) == 1 )
		{
		    printf("Macro #%d = \"%s\"\n", numMacros, macro[ numMacros - 1] );
		}
        else
        {
            fprintf(stderr,"Error on line %d in config file%s:\n%s",
                    lineno, cfgfile, buf);
            cfgerr=1;
        }
    }
    fclose(fp);
    if(cfgerr) exit(1);
}

/**  Calculate the intermod level whenever an idling is detected */
void PhaseShift::calculateIntermod()
{
   intermodFlag = 1;
}
/** Called whenever a text change occurs in the tx text area */
void PhaseShift::txTextCallback( const char c )
{
    if ( voxFlag && !transmit )
    {
        psk31tx.send_char( TX_START );
        transmit = 1;
        doPTT (PTT_ON);
    }
    psk31tx.send_char( c );
}
/**  */
void PhaseShift::slotTypeAhead()
{
   typeAheadFlag = !typeAheadFlag;

   if ( typeAheadFlag )
   {
      txLine = txLine2;
      txLine2->show();
      txLine1->hide();
      statusBar()->message("TX Line in Type-ahead mode...");
   }
   else
   {
      txLine = txLine1;
      txLine1->show();
      txLine2->hide();
      statusBar()->message("TX Line in character mode...");
   }
}
/**  */
void PhaseShift::setTx( bool tx )
{
    int c;

    // if break-in is enabled, disable it. Otherwise confusion...
    if ( voxFlag )
    {
        voxFlag = 0;
        masterDisplay->setVox( voxFlag );
        psk31tx.setBreakIn( 0 );
    }

    if ( tx )
    {
        c = TX_START;
        transmit = 1;
        doPTT (PTT_ON);
    }
    else
    {
        c = TX_END;
        //psk31tx.flush();
    }
    psk31tx.send_char( '\n' );
    psk31tx.send_char( '\n' );
    psk31tx.send_char(c);
}

/**  */
void PhaseShift::doPTT( int state )
{
   static int serfd = 0;
   int arg = TIOCM_RTS;

   if (serfd == 0)                            /* is serfd open? */
   {
      if ((serfd = open (pttDevice, O_RDWR)) == -1)
      {
         perror ("Can't open serial port");
      }
   }

   if (ioctl (serfd, state, &arg) == -1 && serfd)      /* toggle PTT */
   {
      perror("RTS failed");
   }
}

/** Called whenever a return is pressed in the tx text area #1 */
void PhaseShift::txReturnPressedCallback()
{
    txLine1->setText("");
    psk31tx.send_char( '\n' );
}
/** Called whenever return is pressed on txLine #2 */
void PhaseShift::txLineCallback()
{
    psk31tx.send_char( TX_START );
    psk31tx.send_char( '\n' );
    if ( prependCallsignFlag )
    {
        psk31tx.send_string( ">>" );
        psk31tx.send_string( callsign );
        psk31tx.send_string( ": " );
    }
    psk31tx.send_string( (char *)txLine2->text() );
    psk31tx.send_char( TX_END );
    transmit = 1;
    doPTT (PTT_ON);
    txLine2->setText("");
}
/** Called whenever the CQ push button is pressed */
void PhaseShift::slotCQ()
{
    char cqString[80];

    if ( !transmit )
    {
        statusBar()->message("Calling CQ...");

        psk31tx.send_char( TX_START );
        sprintf(cqString, "\n\ncq cq cq de %s %s %s \n", callsign, callsign, callsign);
        psk31tx.send_string( cqString );
        psk31tx.send_char( TX_START );
        sprintf(cqString, "cq cq cq de %s %s %s \n", callsign, callsign, callsign);
        psk31tx.send_string( cqString );
        psk31tx.send_char( TX_START );
        sprintf(cqString, "cq cq cq de %s %s %s \npse k \n", callsign, callsign, callsign);
        psk31tx.send_string( cqString );
        psk31tx.send_char( TX_END );
        transmit = 1;
        doPTT (PTT_ON);
    }
}
/** Called whenever an item is selected on the marco list */
void PhaseShift::slotListItemSelected( int index )
{
    if ( voxFlag && !transmit )
    {
        psk31tx.send_char( TX_START );
        psk31tx.send_string( macro[ index ] );
        psk31tx.send_char( TX_END );
        transmit = 1;
        doPTT (PTT_ON);
    }
    else
    {
        psk31tx.send_string( macro[ index ] );
    }
}
/** Called when the beacon timer expires */
void PhaseShift::doBeacon()
{
    if ( beaconFlag )
    {
        statusBar()->message("Beacon...");

        psk31tx.send_char( TX_START );
        psk31tx.send_char( '\n' );
        psk31tx.send_string( ">>> " );
        psk31tx.send_string( beaconText );
        psk31tx.send_char( '\n' );
        psk31tx.send_char( TX_END );
        transmit = 1;
        doPTT (PTT_ON);
    }
}
/**  */
void PhaseShift::processTxSample( short s )
{
    fftEngine->addSample( s );
}
/** Called periodically when a file is sent */
void PhaseShift::sendFile()
{
    char c = 0;
    int i;

    for ( i = 0; i < 512; i++ )
    {
        c = fgetc( send_fp );
        if ( c == EOF )
        {
            psk31tx.send_char( TX_END );
            fclose( send_fp );
            sendingFileFlag = 0;
            break;
        }
        else
        {
            psk31tx.send_char( c );
        }
    }
}
/**  */
void PhaseShift::slotTxStop()
{
    psk31tx.flush();
    psk31tx.send_char(TX_END);
}
/** Set Rx frequency */
void PhaseShift::setFreq( float delta )
{
      psk31rx.set_freq( rxFreq + delta );
}
/** Toggles the break-in operation */
void PhaseShift::slotVox()
{
    voxFlag = !voxFlag;
    masterDisplay->setVox( voxFlag );
    psk31tx.setBreakIn( voxFlag );
}
