# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/

# Follows PEP8

import os
import re
import sys
from cStringIO import StringIO
from itertools import cycle
from urllib import urlopen

import Image
import ImageDraw
import ImageEnhance

import system

CHECKBOARD = {}
COLOR_MAP = [255] * 128 + [0] * 128
WWW_CACHE = {}

EXT_BY_FORMATS = {
    'JPEG': ['JPG', 'JPEG', 'JPE'],
    'TIFF': ['TIF', 'TIFF'],
    'SVG': ['SVG', 'SVGZ'],
}
FORMATS_BY_EXT = {}
for format, exts in EXT_BY_FORMATS.items():
    for ext in exts:
        FORMATS_BY_EXT[ext] = format


class InvalidWriteFormatError(Exception):
    pass


def get_format(ext):
    """Guess the image format by the file extension.

    :param ext: file extension
    :type ext: string
    :returns: image format
    :rtype: string

    .. warning::

        This is only meant to check before saving files. For existing files
        open the image with PIL and check its format attribute.

    >>> get_format('jpg')
    'JPEG'
    """
    ext = ext.lstrip('.').upper()
    return FORMATS_BY_EXT.get(ext, ext)


def get_format_filename(filename):
    """Guess the image format by the filename.

    :param filename: filename
    :type filename: string
    :returns: image format
    :rtype: string

    .. warning::

        This is only meant to check before saving files. For existing files
        open the image with PIL and check its format attribute.

    >>> get_format_filename('test.tif')
    'TIFF'
    """
    return get_format(system.file_extension(filename))


def open_image(uri):
    """Open local files or remote files over http.

    :param uri: image location
    :type uri: string
    :returns: image
    :rtype: pil.Image
    """
    if system.is_www_file(uri):
        try:
            return WWW_CACHE[uri]
        except KeyError:
            f = urlopen(uri)
            im = WWW_CACHE[uri] = open_image_data(f.read())
            return im
    if uri[:7] == 'file://':
        uri = uri[7:]
    return Image.open(uri)


def open_image_data(data):
    """Open image from format data.

    :param data: image format data
    :type data: string
    :returns: image
    :rtype: pil.Image
    """
    return Image.open(StringIO(data))


def open_image_exif(uri):
    """Open local files or remote files over http and transpose the
    image to its exif orientation.

    :param uri: image location
    :type uri: string
    :returns: image
    :rtype: pil.Image
    """
    return transpose_exif(open_image(uri))


class _ByteCounter:
    """Helper class to count how many bytes are written to a file.

    .. see also:: :func:`get_size`

    >>> bc = _ByteCounter()
    >>> bc.write('12345')
    >>> bc.bytes
    5
    """
    bytes = 0

    def write(self, data):
        self.bytes += len(data)


def get_size(im, format, **options):
    """Gets the size in bytes if the image would be written to a file.

    :param format: image file format (e.g. ``'JPEG'``)
    :type format: string
    :returns: the file size in bytes
    :rtype: int
    """
    try:
        out = _ByteCounter()
        im.save(out, format, **options)
        return out.bytes
    except AttributeError:
        # fall back on full in-memory compression
        out = StringIO()
        im.save(out, format, **options)
        return len(out.getvalue())


def get_quality(im, size, format, down=0, up=100, delta=1000, options=None):
    """Figure out recursively the quality save parameter to obtain a
    certain image size. This mostly used for ``JPEG`` images.

    :param im: image
    :type im: pil.Image
    :param format: image file format (e.g. ``'JPEG'``)
    :type format: string
    :param down: minimum file size in bytes
    :type down: int
    :param up: maximum file size in bytes
    :type up: int
    :param delta: fault tolerance in bytes
    :type delta: int
    :param options: image save options
    :type options: dict
    :returns: save quality
    :rtype: int

    Example::

        filename = '/home/stani/sync/Desktop/IMGA3345.JPG'
        im = Image.open(filename)
        q = get_quality(im, 300000, "JPEG")
        im.save(filename.replace('.jpg', '_sized.jpg'))
    """
    if options is None:
        options = {}
    q = options['quality'] = (down+up)/2
    if q==down or q==up:
        return max(q, 1)
    s = get_size(im, format, **options)
    if abs(s-size)<delta:
        return q
    elif s > size:
        return get_quality(im, size, format, down, up=q, options=options)
    else:
        return get_quality(im, size, format, down=q, up=up, options=options)


def fill_background_color(image, color):
    """Fills given image with background color.

    :param image: source image
    :type image: pil.Image
    :param color: background color
    :type color: tuple of int
    :returns: filled image
    :rtype: pil.Image
    """
    if image.mode == 'LA':
        image = image.convert('RGBA')
    elif image.mode != 'RGBA' and\
        not (image.mode == 'P' and 'transparency' in image.info):
        return image
    if len(color) == 4 and color[-1]!=255:
        mode = 'RGBA'
    else:
        mode = 'RGB'
    back = Image.new(mode, image.size, color)
    if (image.mode == 'P' and mode == 'RGBA'):
        image = image.convert('RGBA')
    if image.mode in ['RGBA', 'LA']:
        back.paste(image, image)
    elif image.mode == 'P':
        palette = image.getpalette()
        index = image.info['transparency']
        palette[index * 3: index * 3 + 3] = color[:3]
        image.putpalette(palette)
        del image.info['transparency']
        back = image
    else:
        back.paste(image)
    return back


def generate_layer(image_size, mark, method,
        horizontal_offset, vertical_offset,
        horizontal_justification, vertical_justification,
        orientation, opacity):
    """Generate new layer for backgrounds or watermarks on which a given
    image ``mark`` can be positioned, scaled or repeated.

    :param image_size: size of the reference image
    :type image_size: tuple of int
    :param mark: image mark
    :type mark: pil.Image
    :param method: ``'Tile'``, ``'Scale'``, ``'By Offset'``
    :type method: string
    :param horizontal_offset: horizontal offset
    :type horizontal_offset: int
    :param vertical_offset: vertical offset
    :type vertical_offset: int
    :param horizontal_justification: ``'Left'``, ``'Middle'``, ``'Right'``
    :type horizontal_justification: string
    :param vertical_justification: ``'Top'``, ``'Middle'``, ``'Bottom'``
    :type vertical_justification: string
    :param orientation: mark orientation (e.g. ``'ROTATE_270'``)
    :type orientation: string
    :param opacity: opacity within ``[0, 1]``
    :type opacity: float
    :returns: generated layer
    :rtype: pil.Image

    .. see also:: :func:`reduce_opacity`
    """
    mark = open_image(mark)
    opacity /= 100.0
    mark = reduce_opacity(mark, opacity)
    layer = Image.new('RGBA', image_size, (0, 0, 0, 0))
    if method == 'Tile':
        for y in range(0, image_size[1], mark.size[1]):
            for x in range(0, image_size[0], mark.size[0]):
                layer.paste(mark, (x, y))
    elif method == 'Scale':
        # scale, but preserve the aspect ratio
        ratio = min(float(image_size[0]) / mark.size[0],
            float(image_size[1]) / mark.size[1])
        w = int(mark.size[0] * ratio)
        h = int(mark.size[1] * ratio)
        mark = mark.resize((w, h))
        layer.paste(mark, ((image_size[0] - w) / 2,
            (image_size[1] - h) / 2))
    elif method == 'By Offset':
        location = calculate_location(
            horizontal_offset, vertical_offset,
            horizontal_justification, vertical_justification,
            image_size, mark.size)
        if orientation:
            orientation_value = getattr(Image, orientation)
            mark = mark.transpose(orientation_value)
        layer.paste(mark, location)
    else:
        raise ValueError('Unknown method "%s" for generate_layer.'%method)
    return layer


def identity_color(image, value=0):
    """Get a color with same color component values.

    >>> im = Image.new('RGB', (1,1))
    >>> identity_color(im, 2)
    (2, 2, 2)
    >>> im = Image.new('L', (1,1))
    >>> identity_color(im, 7)
    7
    """
    bands = image.getbands()
    if len(bands) == 1:
        return value
    return tuple([value for band in bands])


def blend(im1, im2, amount, color=None):
    """Blend two images with each other. If the images differ in size
    the color will be used for undefined pixels.

    :param im1: first image
    :type im1: pil.Image
    :param im2: second image
    :type im2: pil.Image
    :param amount: amount of blending
    :type amount: int
    :param color: color of undefined pixels
    :type color: tuple
    :returns: blended image
    :rtype: pil.Image
    """
    im2 = convert_safe_mode(im2)
    if im1.size == im2.size:
        im1 = convert(im1, im2.mode)
    else:
        if color is None:
            expanded = Image.new(im2.mode, im2.size)
        elif im2.mode in ('1', 'L') and type(color) != int:
            expanded = Image.new(im2.mode, im2.size, color[0])
        else:
            expanded = Image.new(im2.mode, im2.size, color)
        im1 = im1.convert(expanded.mode)
        we, he = expanded.size
        wi, hi = im1.size
        expanded.paste(im1, ((we-wi)/2, (he-hi)/2), im1.convert('RGBA'))
        im1 = expanded
    return Image.blend(im1, im2, amount)


def reduce_opacity(im, opacity):
    """Returns an image with reduced opacity if opacity is
    within ``[0, 1]``.

    :param im: source image
    :type im: pil.Image
    :param opacity: opacity within ``[0, 1]``
    :type opacity: float
    :returns im: image
    :rtype: pil.Image

    >>> im = Image.new('RGBA', (1, 1), (255, 255, 255))
    >>> im = reduce_opacity(im, 0.5)
    >>> im.getpixel((0,0))
    (255, 255, 255, 127)
    """
    if opacity < 0 or opacity > 1:
        return im
    alpha = get_alpha(im)
    alpha = ImageEnhance.Brightness(alpha).enhance(opacity)
    put_alpha(im, alpha)
    return im


def calculate_location(horizontal_offset, vertical_offset,
        horizontal_justification, vertical_justification,
        canvas_size, image_size):
    """Calculate location based on offset and justification. Offsets
    can be positive and negative.

    :param horizontal_offset: horizontal offset
    :type horizontal_offset: int
    :param vertical_offset: vertical offset
    :type vertical_offset: int
    :param horizontal_justification: ``'Left'``, ``'Middle'``, ``'Right'``
    :type horizontal_justification: string
    :param vertical_justification: ``'Top'``, ``'Middle'``, ``'Bottom'``
    :type vertical_justification: string
    :param canvas_size: size of the total canvas
    :type canvas_size: tuple of int
    :param image_size: size of the image/text which needs to be placed
    :type image_size: tuple of int
    :returns: location
    :rtype: tuple of int

    .. see also:: :func:`generate layer`

    >>> calculate_location(50, 50, 'Left', 'Middle', (100,100), (10,10))
    (50, 45)
    """
    canvas_width, canvas_height = canvas_size
    image_width, image_height = image_size

    # check offsets
    if horizontal_offset < 0:
        horizontal_offset += canvas_width
    if vertical_offset < 0:
        vertical_offset += canvas_height

    # check justifications
    if horizontal_justification == 'Left':
        horizontal_delta = 0
    elif horizontal_justification == 'Middle':
        horizontal_delta = -image_width / 2
    elif horizontal_justification == 'Right':
        horizontal_delta = -image_width

    if vertical_justification == 'Top':
        vertical_delta = 0
    elif vertical_justification == 'Middle':
        vertical_delta = -image_height / 2
    elif vertical_justification == 'Bottom':
        vertical_delta = -image_height

    return horizontal_offset + horizontal_delta, \
        vertical_offset + vertical_delta


def checkboard(size, delta=8, fg=(128, 128, 128), bg=(204, 204, 204)):
    """Draw an n x n checkboard, which is often used as background
    for transparent images. The checkboards are stored in the
    ``CHECKBOARD`` cache.

    :param delta: dimension of one square
    :type delta: int
    :param fg: foreground color
    :type fg: tuple of int
    :param bg: background color
    :type bg: tuple of int
    :returns: checkboard image
    :rtype: pil.Image
    """
    if not (size in CHECKBOARD):
        dim = max(size)
        n = int(dim/delta)+1 #FIXME: now acts like square->nx, ny

        def sq_start(i):
            "Return the x/y start coord of the square at column/row i."
            return i * delta

        def square(i, j):
            "Return the square corners"
            return map(sq_start, [i, j, i + 1, j + 1])

        image = Image.new("RGB", size, bg)
        draw_square = ImageDraw.Draw(image).rectangle
        squares = (square(i, j)
           for i_start, j in zip(cycle((0, 1)), range(n))
           for i in range(i_start, n, 2))
        for sq in squares:
            draw_square(sq, fill=fg)
        CHECKBOARD[size] = image
    return CHECKBOARD[size].copy()


####################################
####    PIL helper functions    ####
####################################

def has_alpha(image):
    """Checks if the image has an alpha band.
    i.e. the image mode is either RGBA or LA.
    The transparency in the P mode doesn't count as an alpha band

    :param image: the image to check
    :type image: PIL image object
    :returns: True or False
    :rtype: boolean
    """
    return image.mode in ['RGBA', 'LA']


def has_transparency(image):
    """Checks if the image has transparency.
    The image has an alpha band or a P mode with transparency.

    :param image: the image to check
    :type image: PIL image object
    :returns: True or False
    :rtype: boolean
    """
    return (image.mode == 'P' and 'transparency' in image.info) or\
            has_alpha(image)


def get_alpha(image):
    """Gets the image alpha band. Can handles P mode images with transpareny.
    Returns a band with all values set to 255 if no alpha band exists.

    :param image: input image
    :type image: PIL image object
    :returns: alpha as a band
    :rtype: single band image object
    """
    if has_alpha(image):
        return image.split()[-1]
    if image.mode == 'P' and 'transparency' in image.info:
        return image.convert('LA').split()[-1]
    # No alpha layer, create one.
    return Image.new('L', image.size, 255)


def get_format_data(image, format):
    """Convert the image in the file bytes of the image. By consequence
    this byte data is different for the chosen format (``JPEG``,
    ``TIFF``, ...).

    .. see also:: :func:`thumbnail.get_format_data`

    :param image: source image
    :type impage: pil.Image
    :param format: image file type format
    :type format: string
    :returns: byte data of the image
    """
    f = StringIO()
    convert_save_mode_by_format(image, format).save(f, format)
    return f.getvalue()


def put_alpha(image, alpha):
    """Copies the given band to the alpha layer of the given image.

    :param image: input image
    :type image: PIL image object
    :param alpha: the alpha band to copy
    :type alpha: single band image object
    """
    if image.mode in ['CMYK', 'YCbCr', 'P']:
        image = image.convert('RGBA')
    elif image.mode in ['1', 'F']:
        image = image.convert('LA')
    image.putalpha(alpha)


def remove_alpha(image):
    """Returns a copy of the image after removing the alpha band or
    transparency

    :param image: input image
    :type image: PIL image object
    :returns: the input image after removing the alpha band or transparency
    :rtype: PIL image object
    """
    if image.mode == 'RGBA':
        return image.convert('RGB')
    if image.mode == 'LA':
        return image.convert('L')
    if image.mode == 'P' and 'transparency' in image.info:
        del image.info['transparency']
        return image.convert('RGB')
    return image


def auto_crop(image):
    """Crops all transparent or black background from the image
    :param image: input image
    :type image: PIL image object
    :returns: the cropped image
    :rtype: PIL image object
    """

    alpha = get_alpha(image)
    box = alpha.getbbox()
    return convert_safe_mode(image).crop(box)


def convert(image, mode, *args, **keyw):
    """Returns a converted copy of an image

    :param image: input image
    :type image: PIL image object
    :param mode: the new mode
    :type mode: string
    :param args: extra options
    :type args: tuple of values
    :param keyw: extra keyword options
    :type keyw: dictionary of options
    :returns: the converted image
    :rtype: PIL image object
    """
    if mode == 'P':
        if image.mode == 'P':
            return image
        if image.mode in ['1', 'F']:
            return image.convert('L').convert(mode, *args, **keyw)
        if image.mode in ['RGBA', 'LA']:
            alpha = image.split()[-1]
            output = image.convert('RGB').convert(
                mode, colors=255, *args, **keyw)
            output.paste(
                255, alpha.point(COLOR_MAP))
            output.info['transparency'] = 255
            return output
        return image.convert('RGB').convert(mode, *args, **keyw)
    if image.mode == 'P' and mode == 'LA':
        # A workaround for a PIL bug.
        # Converting from P to LA directly doesn't work.
        return image.convert('RGBA').convert('LA', *args, **keyw)
    if has_transparency(image) and (not mode in ['RGBA', 'LA']):
        if image.mode == 'P':
            image = image.convert('RGBA')
            del image.info['transparency']
        #image = fill_background_color(image, (255, 255, 255, 255))
        image = image.convert(mode, *args, **keyw)
        return image
    return image.convert(mode, *args, **keyw)


def convert_safe_mode(image):
    """Converts image into a processing-safe mode.

    :param image: input image
    :type image: PIL image object
    :returns: the converted image
    :rtype: PIL image object
    """
    if image.mode in ['1', 'F']:
        return image.convert('L')
    if image.mode == 'P' and 'transparency' in image.info:
        del image.info['transparency']
        return image.convert('RGBA')
    if image.mode in ['P', 'YCbCr', 'CMYK', 'RGBX']:
        return image.convert('RGB')
    return image


def convert_save_mode_by_format(image, format):
    """Converts image into a saving-safe mode.

    :param image: input image
    :type image: PIL image object
    :param format: target format
    :type format: string
    :returns: the converted image
    :rtype: PIL image object
    """
    #TODO: Extend this helper function to support other formats as well
    if image.mode == 'P':
        # Make sure P is handled correctly
        if not format in ['GIF', 'PNG', 'TIFF', 'IM', 'PCX']:
            image = remove_alpha(image)
    if format == 'JPEG':
        if image.mode in ['RGBA', 'P']:
            return image.convert('RGB')
        if image.mode in ['LA']:
            return image.convert('L')
    elif format == 'BMP':
        if image.mode in ['LA']:
            return image.convert('L')
        if image.mode in ['P', 'RGBA', 'YCbCr', 'CMYK']:
            return image.convert('RGB')
    elif format == 'DIB':
        if image.mode in ['YCbCr', 'CMYK']:
            return image.convert('RGB')
    elif format == 'EPS':
        if image.mode in ['1', 'LA']:
            return image.convert('L')
        if image.mode in ['P', 'RGBA', 'YCbCr']:
            return image.convert('RGB')
    elif format == 'GIF':
        return convert(image, 'P', palette=Image.ADAPTIVE)
    elif format == 'PBM':
        if image.mode in ['P', 'CMYK', 'YCbCr']:
            return image.convert('RGB')
        if image.mode in ['LA']:
            return image.convert('L')
    elif format == 'PCX':
        if image.mode in ['RGBA', 'CMYK', 'YCbCr']:
            return image.convert('RGB')
        if image.mode in ['LA', '1']:
            return image.convert('L')
    elif format == 'PDF':
        if image.mode in ['LA']:
            return image.convert('L')
        if image.mode in ['RGBA', 'YCbCr']:
            return image.convert('RGB')
    elif format == 'PGM':
        if image.mode in ['P', 'CMYK', 'YCbCr']:
            return image.convert('RGB')
        if image.mode in ['LA']:
            return image.convert('L')
    elif format == 'PPM':
        if image.mode in ['P', 'CMYK', 'YCbCr']:
            return image.convert('RGB')
        if image.mode in ['LA']:
            return image.convert('L')
    elif format == 'PS':
        if image.mode in ['1', 'LA']:
            return image.convert('L')
        if image.mode in ['P', 'RGBA', 'YCbCr']:
            return image.convert('RGB')
    elif format == 'XBM':
        if not image.mode in ['1']:
            return image.convert('1')
    elif format == 'TIFF':
        if image.mode in ['YCbCr']:
            return image.convert('RGB')
    elif format == 'PNG':
        if image.mode in ['CMYK', 'YCbCr']:
            return image.convert('RGB')
    #for consistency return a copy! (thumbnail.py depends on it)
    return image.copy()


def save(image, filename, **options):
    """Saves an image with a filename and raise the specific
    ``InvalidWriteFormatError`` in case of an error instead of a
    ``KeyError``.

    :param image: image
    :type image: pil.Image
    :param filename: image filename
    :type filename: string
    """
    try:
        image.save(filename, **options)
    except KeyError, format:
        raise InvalidWriteFormatError(format)


def save_check_mode(image, filename, **options):
    #save image with pil
    save(image, filename, **options)
    #verify saved file
    image_file = Image.open(filename)
    image_file.verify()
    if image.mode!= image_file.mode:
        return image_file.mode
    return ''


def save_safely(image, filename):
    """Saves an image with a filename and raise the specific
    ``InvalidWriteFormatError`` in case of an error instead of a
    ``KeyError``.

    :param image: image
    :type image: pil.Image
    :param filename: image filename
    :type filename: string
    """
    ext = os.path.splitext(filename)[-1][1:]
    format = get_format(ext)
    image = convert_save_mode_by_format(image, format)
    save(image, filename)


def get_reverse_transposition(transposition):
    """Get the reverse transposition method.

    :param transposition: transpostion, e.g. ``Image.ROTATE_90``
    :returns: inverse transpostion, e.g. ``Image.ROTATE_270``
    """
    if transposition == Image.ROTATE_90:
        return Image.ROTATE_270
    elif transposition == Image.ROTATE_270:
        return Image.ROTATE_90
    return transposition


def get_exif_transposition(orientation):
    """Get the transposition methods necessary to aling the image to
    its exif orientation.

    :param orientation: exif orientation
    :type orientation: int
    :returns: (transposition methods, reverse transpostion methods)
    :rtype: tuple
    """
    #see EXIF.py
    if orientation == 1:
        transposition = transposition_reverse = ()
    elif orientation == 2:
        transposition = Image.FLIP_LEFT_RIGHT,
        transposition_reverse = Image.FLIP_LEFT_RIGHT,
    elif orientation == 3:
        transposition = Image.ROTATE_180,
        transposition_reverse = Image.ROTATE_180,
    elif orientation == 4:
        transposition = Image.FLIP_TOP_BOTTOM,
        transposition_reverse = Image.FLIP_TOP_BOTTOM,
    elif orientation == 5:
        transposition = Image.FLIP_LEFT_RIGHT, \
                                        Image.ROTATE_90
        transposition_reverse = Image.ROTATE_270, \
                                        Image.FLIP_LEFT_RIGHT
    elif orientation == 6:
        transposition = Image.ROTATE_270,
        transposition_reverse = Image.ROTATE_90,
    elif orientation == 7:
        transposition = Image.FLIP_LEFT_RIGHT, \
                                        Image.ROTATE_270
        transposition_reverse = Image.ROTATE_90, \
                                        Image.FLIP_LEFT_RIGHT
    elif orientation == 8:
        transposition = Image.ROTATE_90,
        transposition_reverse = Image.ROTATE_270,
    else:
        transposition = transposition_reverse = ()
    return transposition, transposition_reverse


def get_exif_orientation(image):
    """Gets the exif orientation of an image.

    :param image: image
    :type image: pil.Image
    :returns: orientation
    :rtype: int
    """
    if not hasattr(image, '_getexif'):
        return 1
    try:
        _exif = image._getexif()
        if not _exif:
            return 1
        return _exif[0x0112]
    except KeyError:
        return 1


def transpose(image, methods):
    """Transpose with a sequence of transformations, mainly useful
    for exif.

    :param image: image
    :type image: pil.Image
    :param methods: transposition methods
    :type methods: list
    :returns: transposed image
    :rtype: pil.Image
    """
    for method in methods:
        image = image.transpose(method)
    return image


def transpose_exif(image, reverse=False):
    """Transpose an image to its exif orientation.

    :param image: image
    :type image: pil.Image
    :param reverse: False when opening, True when saving
    :type reverse: bool
    :returns: transposed image
    :rtype: pil.Image
    """
    orientation = get_exif_orientation(image)
    transposition = get_exif_transposition(orientation)[int(reverse)]
    if transposition:
        return transpose(image, transposition)
    return image


def checkboard(size, delta=8, fg=(128, 128, 128), bg=(204, 204, 204)):
    """Draw an n x n checkboard, which is often used as background
    for transparent images. The checkboards are stored in the
    ``CHECKBOARD`` cache.

    :param delta: dimension of one square
    :type delta: int
    :param fg: foreground color
    :type fg: tuple of int
    :param bg: background color
    :type bg: tuple of int
    :returns: checkboard image
    :rtype: pil.Image
    """
    if not (size in CHECKBOARD):
        dim = max(size)
        n = int(dim/delta)+1 #FIXME: now acts like square->nx, ny

        def sq_start(i):
            "Return the x/y start coord of the square at column/row i."
            return i * delta

        def square(i, j):
            "Return the square corners"
            return map(sq_start, [i, j, i + 1, j + 1])

        image = Image.new("RGB", size, bg)
        draw_square = ImageDraw.Draw(image).rectangle
        squares = (square(i, j)
           for i_start, j in zip(cycle((0, 1)), range(n))
           for i in range(i_start, n, 2))
        for sq in squares:
            draw_square(sq, fill=fg)
        CHECKBOARD[size] = image
    return CHECKBOARD[size].copy()


def add_checkboard(image):
    """"If the image has a transparent mask, a RGB checkerboard will be
    drawn in the background.

    .. note::

        In case of a thumbnail, the resulting image can not be used for
        the cache, as it replaces the transparency layer with a non
        transparent checkboard.

    :param image: image
    :type image: pil.Image
    :returns: image, with checkboard if transparant
    :rtype: pil.Image
    """
    if (image.mode == 'P' and 'transparency' in image.info) or\
            image.mode.endswith('A'):
        #transparant image
        image = image.convert('RGBA')
        image_bg = checkboard(image.size)
        image_bg.paste(image, (0, 0), image)
        return image_bg
    else:
        return image


if __name__ == '__main__':
    example()
