#include "phluid.h"
#include "debug.h"

static void draw_rect(Client *);

void
drag_client(Client * c, int mouse_x, int mouse_y)
{
  XEvent ev;
  Client *exposed;
  int startx, starty;

  XDefineCursor(disp, c->frame, move_curs);

  startx = c->x;
  starty = c->y;

#ifdef DEBUG
  fprintf(stdout, "in drag_client: %d,%d\n", mouse_x, mouse_y);
#endif

  for (;;) {
    XMaskEvent(disp, ButtonReleaseMask | PointerMotionMask | ExposureMask, &ev);
    switch (ev.type) {
      case MotionNotify:
#ifdef DEBUG
        fprintf(stdout, "got MotionNotify: %d,%d\n", ev.xmotion.x_root,
                ev.xmotion.y_root);
#endif
        c->x = startx + (ev.xmotion.x_root - mouse_x);
        c->y = starty + (ev.xmotion.y_root - mouse_y);
#ifdef DEBUG
        fprintf(stdout, "c->x, c->y: %d,%d\n", c->x, c->y);
#endif

        if (!MOVE_AFTER_DRAG) {
          XMoveWindow(disp, c->frame, c->x, c->y);
          XFlush(disp);
          if (!SEND_CONFIGURE_AFTER_DRAG)
            icccm_send_configure_event(c);
        }
        break;
      case ButtonRelease:
        XUndefineCursor(disp, c->frame);
        return;
      case Expose:
        exposed = find_client_by_frame(ev.xexpose.window);
        if (exposed) {
          exposed->updates =
              imlib_update_append_rect(exposed->updates, ev.xexpose.x,
                                       ev.xexpose.y, ev.xexpose.width,
                                       ev.xexpose.height);
          redraw_client(exposed);
        }
        break;
    }
  }
}

void
shade_client(Client * c)
{
  if (!c->shaded) {
    c->shaded = 1;
    XResizeWindow(disp, c->frame, c->width + 2 * EDGE_THICKNESS,
                  TITLEBAR_HEIGHT);
    c->ignore_unmap++;
    XUnmapWindow(disp, c->window);
    icccm_set_state(c, IconicState);
  } else {
    c->shaded = 0;
    XMapWindow(disp, c->window);
    XResizeWindow(disp, c->frame, c->width + 2 * EDGE_THICKNESS,
                  c->height + TITLEBAR_HEIGHT + EDGE_THICKNESS);
    icccm_set_state(c, NormalState);
  }
}

void
iconify_client(Client * c)
{
  if (c->visible) {
    c->visible = 0;
    c->ignore_unmap += 2;
    XUnmapWindow(disp, c->frame);
    XUnmapWindow(disp, c->window);
    icccm_set_state(c, IconicState);
  } else {
    c->visible = 1;
    XMapWindow(disp, c->window);
    XMapWindow(disp, c->frame);
    icccm_set_state(c, NormalState);
  }
}

void
resize_client(Client * c, int mouse_x, int mouse_y)
{
  XEvent ev;
  Client *exposed;
  int startw, starth;
  int width_inc, height_inc;
  int chop;
  width_inc = height_inc = 1;

  startw = c->width;
  starth = c->height;

#ifdef DEBUG
  fprintf(stdout, "in resize_client: %d,%d\n", mouse_x, mouse_y);
#endif

	XGrabServer(disp);
  draw_rect(c);
  for (;;) {
    XMaskEvent(disp, ButtonReleaseMask | PointerMotionMask | ExposureMask, &ev);
    switch (ev.type) {
      case MotionNotify:
#ifdef DEBUG
        fprintf(stdout, "got MotionNotify: %d,%d\n", ev.xmotion.x_root,
                ev.xmotion.y_root);
#endif
        draw_rect(c);           // clear old drawing
        if (c->size_hints->flags & PResizeInc) {
          width_inc = c->size_hints->width_inc;
          height_inc = c->size_hints->height_inc;
        }
        chop = (ev.xmotion.x_root - mouse_x) % width_inc;
        c->width = startw + (ev.xmotion.x_root - mouse_x) - chop;
        chop = (ev.xmotion.y_root - mouse_y) % height_inc;
        c->height = starth + (ev.xmotion.y_root - mouse_y) - chop;
        if (c->width <= 0)
          c->width = 0;
        if (c->height <= 0)
          c->height = 0;
        draw_rect(c);
#ifdef DEBUG
        fprintf(stdout, "c->x, c->y: %d,%d\n", c->x, c->y);
#endif

        break;
      case ButtonRelease:
        draw_rect(c);
        XResizeWindow(disp, c->window, c->width, c->height);
        XResizeWindow(disp, c->frame, c->width + 2 * EDGE_THICKNESS,
                      c->height + TITLEBAR_HEIGHT + EDGE_THICKNESS);
        render_title_bar(c);
        icccm_send_configure_event(c);
        XFlush(disp);
				XUngrabServer(disp);
        return;
      case Expose:
        exposed = find_client_by_frame(ev.xexpose.window);
        if (exposed) {
          exposed->updates =
              imlib_update_append_rect(exposed->updates, ev.xexpose.x,
                                       ev.xexpose.y, ev.xexpose.width,
                                       ev.xexpose.height);
          redraw_client(exposed);
        }
        break;
    }
  }
}

static void
draw_rect(Client * c)
{
  XDrawRectangle(disp, root, invert_gc, c->x, c->y,
                 c->width + 2 * EDGE_THICKNESS,
                 c->height + EDGE_THICKNESS + TITLEBAR_HEIGHT);
  XDrawLine(disp, root, invert_gc, c->x,
            c->y + TITLEBAR_HEIGHT,
            c->x + c->width + 2 * EDGE_THICKNESS, c->y + TITLEBAR_HEIGHT);
}
