<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.5 $
 *
 *  ABOUT
 *  -----
 *
 *  A generic class for Kolab object management.
 *
 */

/** Define the possible Kolab object attributes */
define('KOLAB_ATTR_SN',      'sn');
define('KOLAB_ATTR_CN',      'cn');
define('KOLAB_ATTR_FN',      'fn');
define('KOLAB_ATTR_MAIL',    'mail');
define('KOLAB_ATTR_UID',     'uid');
define('KOLAB_ATTR_DELETED', 'kolabDeleteFlag');

/** Define the different Kolab object types */
define('KOLAB_OBJECT_ADDRESS',          'Address');
define('KOLAB_OBJECT_ADMINISTRATOR',    'Administrator');
define('KOLAB_OBJECT_DOMAINMAINTAINER', 'DomainMaintainer');
define('KOLAB_OBJECT_GROUP',            'Group');
define('KOLAB_OBJECT_MAINTAINER',       'Maintainer');
define('KOLAB_OBJECT_SHAREDFOLDER',     'SharedFolder');
define('KOLAB_OBJECT_USER',             'User');

/**
 * This class provides general methods to deal with Kolab objects.
 *
 * $Header: /home/kroupware/jail/kolabrepository/server/php-kolab/Kolab_Webadmin/Webadmin/object.php,v 1.5 2007/09/18 06:08:48 gunnar Exp $
 *
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Webadmin
 */
class KolabObject {

    /**
     * Link into the Kolab LDAP db
     *
     * @var KolabLDAP
     */
    var $_ldap;

    /**
     * DN of this object in the Kolab LDAP db
     *
     * @var string
     */
    var $_dn;

    /**
     * Type of this object (one of KOLAB_OBJECT_*)
     *
     * @var string
     */
    var $type;

    /**
     * The cached LDAP result
     *
     * @var mixed
     */
    var $_cache = false;

    /**
     * May this object log in to the Kolab system?
     *
     * @var boolean
     */
    var $login_allowed = false;

    /**
     * The LDAP filter to retrieve this object type
     *
     * @var string
     */
    var $filter = '';

    /**
     * Sort using this attribute by default
     *
     * @var string
     */
    var $sort = '';

    /**
     * The LDAP attributes fetched for listing
     *
     * @var array
     */
    var $list_attributes = array();

    /**
     * The attributes supported by this class
     *
     * @var array
     */
    var $supported_attributes = array();

    /**
     * Initialize the Kolab Object. Provide either the DN or a
     * LDAP search result
     *
     * @param KolabLDAP $ldap   The link into the Kolab LDAP db
     * @param string    $dn     DN of the object
     * @param string    $result LDAP search result for this object
     */
    function KolabObject($ldap, $dn = null, $result = null)
    {
        $this->_ldap = $ldap;
        if (empty($dn)) {
            if (empty($result)) {
                return PEAR::raiseError(_('Specify either the DN or a search result!'));
            }
            $this->_dn = $result['dn'][0];
            $this->_cache = $result;
        } else {
            $this->_dn = $dn;
        }
    }

    /**
     * Read the object into the cache
     */
    function read()
    {
        $this->_cache = $this->_ldap->read($this->_dn);
    }

    /**
     * Get the DN of this object
     *
     * @return string the DN of this object
     */
    function getDn()
    {
        return $this->_dn;
    }

    /**
     * Get the "sn" attribute of this object
     *
     * @return string the "sn" of this object
     */
    function getSn()
    {
        if (!in_array(KOLAB_ATTR_SN, $this->_supported_attributes)) {
            return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                    KOLAB_ATTR_SN));
        }
        if (!$this->_cache) {
            $this->read();
        }
        if (isset($this->_cache[KOLAB_ATTR_SN])) {
            return $this->_cache[KOLAB_ATTR_SN][0];
        }
        return '';
    }

    /**
     * Get the "cn" attribute of this object
     *
     * @return string the "cn" of this object
     */
    function getCn()
    {
        if (!in_array(KOLAB_ATTR_CN, $this->_supported_attributes)) {
            return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                    KOLAB_ATTR_CN));
        }
        if (!$this->_cache) {
            $this->read();
        }
        if (isset($this->_cache[KOLAB_ATTR_CN])) {
            return $this->_cache[KOLAB_ATTR_CN][0];
        }
        return '';
    }

    /**
     * Get the "first name" attribute of this object
     *
     * @return string the "first name" of this object
     */
    function getFn()
    {
        if (!in_array(KOLAB_ATTR_FN, $this->_supported_attributes)) {
            return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                            KOLAB_ATTR_FN));
        }
        if (!$this->_cache) {
            $this->read();
        }
        $sn = '';
        if (isset($this->_cache[KOLAB_ATTR_SN])) {
            $sn = $this->_cache[KOLAB_ATTR_SN][0];
        }
        $cn = '';
        if (isset($this->_cache[KOLAB_ATTR_CN])) {
            $cn = $this->_cache[KOLAB_ATTR_CN][0];
        }
        
        return $this->_ldap->getFirstName($sn, $cn);
    }
    
    /**
     * Get the "mail" attribute of this object
     *
     * @return string the "mail" of this object
     */
    function getMail()
    {
        if (!in_array(KOLAB_ATTR_MAIL, $this->_supported_attributes)) {
            return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                    KOLAB_ATTR_MAIL));
        }
        if (!$this->_cache) {
            $this->read();
        }
        if (isset($this->_cache[KOLAB_ATTR_MAIL])) {
            return $this->_cache[KOLAB_ATTR_MAIL][0];
        }
        return '';
    }

    /**
     * Get the "uid" attribute of this object
     *
     * @return string the "uid" of this object
     */
    function getUid()
    {
        if (!in_array(KOLAB_ATTR_UID, $this->_supported_attributes)) {
            return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                    KOLAB_ATTR_UID));
        }
        if (!$this->_cache) {
            $this->read();
        }
        if (isset($this->_cache[KOLAB_ATTR_UID])) {
            return $this->_cache[KOLAB_ATTR_UID][0];
        }
        return '';
    }

    /**
     * Get the "deleted" attribute of this object
     *
     * @return string the "deleted" state of this object
     */
    function getDeleted()
    {
        if (!in_array(KOLAB_ATTR_DELETED, $this->_supported_attributes)) {
            return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                    KOLAB_ATTR_DELETED));
        }
        if (!$this->_cache) {
            $this->read();
        }
        if (isset($this->_cache[KOLAB_ATTR_DELETED])) {
            return true;
        }
        return false;
    }

    /**
     * Get the group of this object
     *
     * @return string the group of this object
     */
    function group()
    {
        return substr(get_class($this), 5);
    }

    
    /**
     * Return a list view renderer for this type of object
     *
     * @return 
     */
    function getListView()
    {
        $class = get_class($this) . 'ListView';
        $renderer = &new $class($this);
        return $renderer;
    }
    
};

class KolabObjectListView {

    /**
     * The object to display
     *
     * @var KolabObject
     */
    var $_object;

    /**
     * Constructor
     */
    function KolabObjectListView($object)
    {
        $this->_object = $object;
    }
    
    /**
     * Return a list title for this type of object
     *
     * @return string The title
     */
    function getTitle()
    {
        return PEAR::raiseError(_("Not implemented!"));
    }

    /**
     * Return a list note for this type of object
     *
     * @return string The note
     */
    function getNote()
    {
        return PEAR::raiseError(_("Not implemented!"));
    }

    /**
     * Return a list header for this type of object
     *
     * @return string The HTML header line
     */
    function getHeader()
    {
        $header = '';
        foreach ($this->_getHeader() as $column) {
            $header .= '<th>' . $column . '</th>';
        }
        $header .= '<th colspan="2">' . _("Action") . '</th>';
        return $header;
    }

    /**
     * Return an array of table header entries for this type of object
     *
     * @return array The header entries
     */
    function _getHeader()
    {
        return PEAR::raiseError(_("Not implemented!"));
    }
    
    /**
     * Return a list row for this object
     *
     * @return string The HTML line
     */
    function getRow()
    {
        $row = '';
        foreach ($this->_getRow() as $column) {
            $row .= '<td class="contentcell">' . htmlspecialchars($column) . '</td>';
        }
        $row .= $this->getAction();
        return $row;
    }
    
    /**
     * Return the action cell(s)
     *
     * @return string The HTML cell(s)
     */
    function getAction()
    {
        if ($this->_object->getDeleted()) {
            return '<td class="actioncell" colspan="2">' 
                . _(sprintf("%s deleted, awaiting cleanup...", 
                            $this->_object->type)) . '</td>';
        } else {
            return '<td class="actioncell" align="center"><a href="edit.php?type='
                . $this->_object->type . '&action=modify&dn=' 
                . urlencode($this->_object->getDn()) . '">' . _("Modify") . '</a></td>' .
                '<td class="actioncell" align="center"><a href="delete.php?type=' 
                . $this->_object->type . '&action=delete&dn=' 
                . urlencode($this->_object->getDn()) . '">' . _("Delete") . '</a></td>';
        }
    }

    /**
     * Return an array of table row entries for this type of object
     *
     * @return array The row entries
     */
    function _getRow()
    {
        return PEAR::raiseError(_("Not implemented!"));
    }
    

}

/*
  Local variables:
  mode: php
  indent-tabs-mode: f
  tab-width: 4
  buffer-file-coding-system: utf-8
  End:
  vim:encoding=utf-8:
 */
?>
