<?php
/*
 *  Copyright (c) 2004 Klarälvdalens Datakonsult AB
 *
 *    Written by Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 *
 *  This  program is free  software; you can redistribute  it and/or
 *  modify it  under the terms of the GNU  General Public License as
 *  published by the  Free Software Foundation; either version 2, or
 *  (at your option) any later version.
 *
 *  This program is  distributed in the hope that it will be useful,
 *  but WITHOUT  ANY WARRANTY; without even the  implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You can view the  GNU General Public License, online, at the GNU
 *  Project's homepage; see <http://www.gnu.org/licenses/gpl.html>.
 */

/** We need sieve access. */
require_once('Net/Sieve.php');

/** The name of the forward sieve script */
define('KOLAB_SIEVE_FORWARD', 'kolab-forward.siv');

/** The name of the vacation sieve script */
define('KOLAB_SIEVE_VACATION', 'kolab-vacation.siv');

/** The name of the deliver sieve script */
define('KOLAB_SIEVE_DELIVER', 'kolab-deliver.siv');

class KolabSieve {

    /**
     * A sieve connection.
     *
     * @var NetSieve
     */
    var $_sieve;

    // Funny multiline string escaping in Sieve
    function dotstuff( $str ) {
        return str_replace( "\n.", "\n..", $str );
    }

    function undotstuff( $str ) {
        return str_replace( "\n..", "\n.", $str );
    }
    
    function getDeliverFolder( $script ) {
        $inbox = false;      
        if( preg_match("/fileinto \"INBOX\/(.*)\";/", $script, $regs ) ) {
            $inbox = $regs[1];
        }
        return $inbox;
    }

    function getVacationAddresses( $script ) {
        $addresses = false;
        if( preg_match("/:addresses \\[([^\\]]*)\\]/s", $script, $regs ) ) {
            $tmp = split(',', $regs[1] );
            $addresses = array();
            foreach( $tmp as $a ) {
                if( ereg('^ *"(.*)" *$', $a, $regs ) ) {
                    $addresses[] = $regs[1];
                } else {
                    $addresses[] = $a;
                }
            }
        }
        return $addresses;
    }

    function getMailDomain( $script ) {
        $maildomain = false;
        if( preg_match( '/if not address :domain :contains "From" "(.*)" { keep; stop; }/i', $script, $regs ) ) {
            $maildomain = $regs[1];
        }
        return $maildomain;
    }

    function getReactToSpam( $script ) {
        $spam = false;
        if( preg_match('/if header :contains "X-Spam-Flag" "YES" { keep; stop; }/i', $script ) ) {
            $spam = true;
        }
        return $spam;
    }

    function getVacationDays( $script ) {
        $days = false;
        if( preg_match("/:days ([0-9]+)/s", $script, $regs ) ) {
            $days = $regs[1];
        }
        return $days;
    }

    function getVacationText( $script ) {
        $text = false;
        if( preg_match("/text:(.*\r\n)\\.\r\n/s", $script, $regs ) ) {
            $text = $regs[1];
            $text = str_replace( '\n', "\r\n", $text );
            $text = KolabSieve::undotstuff($text);
        }
        return $text;
    }

    function getForwardAddress( $script ) {
        $address = false;
        if( preg_match("/redirect \"([^\"]*)\"/s", $script, $regs ) ) {
            $address = $regs[1];
        }
        return $address;
    }

    function getKeepOnServer( $script ) {
        return ereg('"; keep;', $script, $regs );    
    }

    function createScript( $scriptname ) {
        // TODO
    }

    
    /**
     * Initialize a sieve connection.
     *
     * @param string  $uid    The user id.
     * @param string  $pass   The user pass.
     * @param string  $server The server to connect to.
     */
    function KolabSieve($uid, $pass, $server)
    {
        $this->_sieve =& new Net_Sieve($uid, $pass, $server);
    }

    /**
     * Store the forward sieve script.
     *
     * @param string  $address The address to forward to.
     * @param boolean $keep    Keep the forwarded messages on the server?
     * @param boolean $active  Set the forward script active?
     *
     * @return mixed True on success, a PEAR error otherwise
     */
    function storeForwardSieve($address, $keep, $active)
    {
        $script = "require \"fileinto\";\r\nredirect \"" . addslashes($address)
            . "\";" . ($keep?" keep;":"");
        return $this->storeSieve(KOLAB_SIEVE_FORWARD, $script, $active);

    }
    
    /**
     * Store the vacation sieve script.
     *
     * @param string  $address The address to forward to.
     * @param boolean $keep    Keep the forwarded messages on the server?
     * @param boolean $active  Set the forward script active?
     *
     * @return mixed True on success, a PEAR error otherwise
     */
    function storeVacationSieve($addresses, $maildomain, $reacttospam, $text, $days, $active)
    {
        $script = 
            "require \"vacation\";\r\n\r\n".
            (!empty($maildomain)?"if not address :domain :contains \"From\" \"".$maildomain."\" { keep; stop; }\r\n":"").
            ($reacttospam?"if header :contains \"X-Spam-Flag\" \"YES\" { keep; stop; }\r\n":"").
            "vacation :addresses [ \"".join('", "', $addresses )."\" ] :days ".
            $days . " text:\r\n".
            KolabSieve::dotstuff(trim($_REQUEST['text']))."\r\n.\r\n;\r\n\r\n";
        return $this->storeSieve(KOLAB_SIEVE_FORWARD, $script, $active);
    }
    
    /**
     * Store the deliver sieve script.
     *
     * @param string  $inbox   Folder to deliver to
     * @param boolean $active  Set the forward script active?
     *
     * @return mixed True on success, a PEAR error otherwise
     */
    function storeDeliverSieve($inbox, $active)
    {
        $script = 
            "require \"fileinto\";\r\nif header :contains [\"X-Kolab-Scheduling-Message\"] [\"FALSE\"] {\r\nfileinto \"INBOX/$inbox\";\r\n}\r\n";
        return $this->storeSieve(KOLAB_SIEVE_DELIVER, $script, $active);
    }
    
    /**
     * Store a new sieve script.
     *
     * @param string  $scriptname The name of the script
     * @param string  $script     Script content
     * @param boolean $active     Set the script active?
     *
     * @return mixed True on success, a PEAR error otherwise
     */
    function storeSieve($scriptname, $script, $active)
    {
        $result = $this->_sieve->installScript( $scriptname, $script, $active );
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (!$active && $this->_sieve->getActive() === $scriptname ) {
            $result = $this->_sieve->setActive( '' );
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
        }
        return true;
    }

    /**
     * Get forwarding information.
     *
     * @return array A list of information extracted from the forward
     * script.
     */
    function getForwardSieve()
    {
        $scripts = $this->_sieve->listScripts();

        if( in_array(KOLAB_SIEVE_FORWARD, $scripts ) ) {
            $script = $this->_sieve->getScript(KOLAB_SIEVE_FORWARD);
            $address = KolabSieve::getForwardAddress( $script );
            $keep = KolabSieve::getKeepOnServer( $script );
        } else {
            $address = '';
            $keep = true;
        }
        $active = ( $this->_sieve->getActive() === $scriptname );
        return array($address, $keep, $active);
    }

    /**
     * Get vacation information.
     *
     * @return array A list of information extracted from the vacation
     * script.
     */
    function getVacationSieve()
    {
        $addresses = $days = $text = false;
        $scripts = $this->_sieve->listScripts();
        if( in_array( KOLAB_SIEVE_VACATION, $scripts ) ) {
            $script = $this->_sieve->getScript(KOLAB_SIEVE_VACATION);
            $maildomain = KolabSieve::getMailDomain( $script );
            $reacttospam = KolabSieve::getReactToSpam( $script );
            debug("reacttospam=".($reacttospam?"true":"false"));
            $addresses = KolabSieve::getVacationAddresses( $script );
            $days = KolabSieve::getVacationDays( $script );
            $text = KolabSieve::getVacationText( $script );
        } else $reacttospam = true;
        if( $addresses === false ) {
            $object = $ldap->read( $auth->dn() );
            $addresses = array_merge( (array)$object['mail'], (array)$object['alias'] );
        }
        if( $days === false || $days < 1 ) $days = 7;
        if( $text === false ) {
            $date = strftime(_('%x'));
            $text = sprintf(
                _("I am out of office until %s.\r\n").
                _("In urgent cases, please contact Mrs. <vacation replacement>\r\n\r\n").
                _("email: <email address of vacation replacement>\r\n").
                _("phone: +49 711 1111 11\r\n").
                _("fax.:  +49 711 1111 12\r\n\r\n").
                _("Yours sincerely,\r\n").
                _("-- \r\n").
                _("<enter your name and email address here>"),
                $date);
        }
        $active = ( $this->_sieve->getActive() === KOLAB_SIEVE_VACATION);  
        return array($addresses, $maildomain, $reacttospam, $text, $days, $active);
    }

    /**
     * Get delivery information.
     *
     * @return array A list of information extracted from the deliver
     * script.
     */
    function getDeliverSieve()
    {
        $scripts = $this->_sieve->listScripts();
        $inbox = false;
        if( in_array( $scriptname, $scripts ) ) {
            // Fetch script data from server
            $script = $this->_sieve->getScript($scriptname);
            $inbox = KolabSieve::getDeliverFolder( $script );
        }
        if( $inbox === false ) $inbox = 'Inbox';
        $active = ( $this->_sieve->getActive() === $scriptname );  
        return array($inbox, $active);
    }
}


/*
  Local variables:
  mode: php
  indent-tabs-mode: f
  tab-width: 4
  buffer-file-coding-system: utf-8
  End:
  vim:encoding=utf-8:
 */
?>