<?php
	/**
	* SAPDB database class
	* @author Kai Hofmann <khofmann@probusiness.de>
	* @copyright Copyright (C) 2003,2004 Free Software Foundation, Inc. http://www.fsf.org/
	* @license http://www.fsf.org/licenses/lgpl.html GNU Lesser General Public License
	* @package phpgwapi
	* @subpackage database
	* @version $Id: class.db_sapdb.inc.php,v 1.1.2.2 2004/02/10 13:51:18 ceb Exp $
	*/

    /**
     * SAPDB database class
     *
     * @package phpgwapi
     * @subpackage database
     */
	class db extends db_
	{
		/**
		 * @var integer $UseODBCCursor Type of cursor
		 * @access private
		 */
		var $UseODBCCursor = 0;

        /**
         * @var string $type Connection type
         */
		var $type     = 'odbc';
		/**
		 * @var string $revision Api revision 
		 */
		var $revision = '1.0';
		/**
		 * @var string $Driver Database driver
		 * @access private
		 */
		var $Driver   = 'SAP DB';


		/**
		 * Connect to database
		 * 
		 * @param string $Database Database name
		 * @param string $Host Database host
		 * @param string $User Database user
		 * @param string $Password Database users password
		 * @return resource Database connection_id
		 */
		function connect($Database = '', $Host = '', $User = '', $Password = '')
		{
			/* Handle defaults */
			if ($Database == '')
			{
				$Database = $this->Database;
			}
			if ($Host == '')
			{
				$Host     = $this->Host;
			}
			if ($User == '')
			{
				$User     = $this->User;
			}
			if ($Password == '')
			{
				$Password = $this->Password;
			}
			$Driver = $this->Driver;
			/* establish connection, select database */
			if (!$this->Link_ID)
			{
				$dsn = 'Driver={' . $Driver . '};Server=' . $Host . ';Database=' . $Database;
				if ($GLOBALS['phpgw_info']['server']['db_persistent'])
				{
					$this->Link_ID = odbc_pconnect($dsn, $User, $Password, $this->UseODBCCursor);
				}
				else
				{
					$this->Link_ID = odbc_connect($dsn, $User, $Password, $this->UseODBCCursor);
				}
				if (!$this->Link_ID)
				{
					$this->halt(($GLOBALS['phpgw_info']['server']['db_persistent'] ? 'p' : '') . "connect($Host, $User, \$Password) failed.");
					return 0;
				}
			}
			return $this->Link_ID;
		}



		/**
		 * Disconnect database connection
		 *
		 * This only affects systems not using persistant connections 
		 * @return integer 1: ok; 0: not possible/already closed
		 */
		function disconnect()
		{
			if($this->Link_ID != 0)
			{
				@odbc_close($this->Link_ID);
				$this->Link_ID = 0;
				return 1;
			}
			else
			{
				return 0;
			}
		}


        /**
         * Escape strings before sending them to the database
         *
         * @param string $str the string to be escaped
         * @return string escaped sting
         */
        function db_addslashes($str)
         {
          return str_replace("'","''",$str);
         }


        /**
         * Convert a unix timestamp to a rdms specific timestamp
         *
         * @param int unix timestamp
         * @return string rdms specific timestamp
         */
		function to_timestamp($epoch)
		{
			return date('Y-m-d H:i:s',$epoch);
		}

        /**
         * Convert a rdms specific timestamp to a unix timestamp
         *
         * @param string rdms specific timestamp
         * @return int unix timestamp
         */
		function from_timestamp($timestamp)
		{
			ereg('([0-9]{4})-([0-9]{2})-([0-9]{2}) ([0-9]{2}):([0-9]{2}):([0-9]{2})',$timestamp,$parts);

			return mktime($parts[4],$parts[5],$parts[6],$parts[2],$parts[3],$parts[1]);
		}

        /**
         * Discard the current query result
         */
		function free()
		{
			@odbc_free_result($this->Query_ID);
			$this->Query_ID = 0;
		}


        /**
         * Execute a query
         *
         * @param string $Query_String the query to be executed
         * @param mixed $line the line method was called from - use __LINE__
         * @param string $file the file method was called from - use __FILE__
         * @return integer Current result if sucesful and null if failed
         */
		function query($Query_String, $line = '', $file = '')
		{



 

            
			/* No empty queries, please, since PHP4 chokes on them. 
		  	 * The empty query string is passed on from the constructor,
			 * when calling the class without a query, e.g. in situations
			 * like these: '$db = new db_Subclass;'
			 */
			if (($Query_String == '') || (!$this->connect()))
			{
				return 0;
			}

			# New query, discard previous result.
			if ($this->Query_ID)
			{
				$this->free();
			}

			if ($this->Debug)
			{
				printf("Debug: query = %s<br>\n", $Query_String);
			}

			$this->Query_ID = @odbc_exec($this->Link_ID,$Query_String);
			$this->Row = 0;

			odbc_binmode($this->Query_ID, 1);
			odbc_longreadlen($this->Query_ID, 4096);


			if (! $this->Query_ID)
			{
				$this->Errno = odbc_error($this->Link_ID);
				$this->Error = odbc_errormsg($this->Link_ID);
				$this->halt("Invalid SQL: ".$Query_String, $line, $file);
			}
			else
			{
				$this->Errno = 0;
				$this->Error = '';
			 }

			// Will return nada if it fails. That's fine.
			return $this->Query_ID;
		}


        /**
         * Execute a query with limited result set
         *
         * @param string $Query_String the query to be executed
         * @param integer $offset row to start from
         * @param mixed $line the line method was called from - use __LINE__
         * @param string $file the file method was called from - use __FILE__
         * @param int $num_rows number of rows to return (optional), if unset will use $GLOBALS['phpgw_info']['user']['preferences']['common']['maxmatchs']
         * @return integer Current result if sucesful and null if failed
         */
		function limit_query($Query_String, $offset, $line = '', $file = '', $num_rows = '')
		{
			if (! $num_rows)
			{
				$num_rows = $GLOBALS['phpgw_info']['user']['preferences']['common']['maxmatchs'];
			}

			if ($offset == 0)
			{
				$Query_String .= ' LIMIT ' . $num_rows;
			}
			else
			{
				$Query_String .= ' LIMIT ' . $offset . ',' . $num_rows;
			}

			if ($this->Debug)
			{
				printf("Debug: limit_query = %s<br>offset=%d, num_rows=%d<br>\n", $Query_String, $offset, $num_rows);
			}

			return $this->query($Query_String, $line, $file);
		}


        /**
         * Move to the next row in the results set
         *
         * @return boolean was another row found?
         */
		function next_record()
		{
			if (!$this->Query_ID)
			{
				$this->halt('next_record called with no query pending.');
				return 0;
			}

			// $this->Record = array();
			// $stat = odbc_fetch_into($this->Query_ID, ++$this->Row, &$this->Record);

			$this->Record = @odbc_fetch_array($this->Query_ID);
			$this->Row   += 1;
			$this->Errno  = odbc_error();
			$this->Error  = odbc_errormsg();

			$stat = is_array($this->Record) ? (count($this->Record) > 0) : 0;
			if (!$stat && $this->Auto_Free)
			{
				$this->free();
			}
			return $stat;
		}


        /**
         * Move to position in result set
         *
         * @param integer $pos Required row (optional), default first row
         * @return integer 1 if sucessful or 0 if not found
         */
		function seek($pos = 0)
		{
			$this->Row = $pos;
			return 1;
		}


        /**
         * Begin Transaction
         *
         * @return integer current transaction id
         */
		function transaction_begin()
		{
			return $this->query('COMMIT',__LINE__,__FILE__);
		}

        /**
         * Complete the transaction
         *
         * @return boolean True if sucessful, False if failed
         */
		function transaction_commit()
		{
			if ((!$this->Errno) && ($this->Link_ID != 0))
			{
				return odbc_exec($this->Link_ID,'COMMIT');
			}
			else
			{
				return False;
			}
		}

        /**
         * Rollback the current transaction
         *
         * @return boolean True if sucessful, False if failed
         */
		function transaction_abort()
		{
		  if ($this->Link_ID != 0)
		  {
			return @odbc_exec($this->Link_ID,'ROLLBACK');
		  }
		}


        /**
         * Find the primary key of the last insertion on the current db connection
         *
         * @param string $table name of table the insert was performed on
         * @param string $field the autoincrement primary key of the table
         * @return integer The id, -1 if failed
         */
		function get_last_insert_id($table, $field)
		{
			/* This will get the last insert ID created on the current connection.  Should only be called
			 * after an insert query is run on a table that has an auto incrementing field.  $table and
			 * $field are required, but unused here since it's unnecessary for mysql.  For compatibility
			 * with pgsql, the params must be supplied.
			 */

			if (!isset($table) || $table == '' || !isset($field) || $field == '')
			{
				return -1;
			}


			$result = @odbc_exec($this->Link_ID, "select max($field) from $table");
			if (!$result)
			{
				return -1;
			}
			$Record = @odbc_result($result,1);
			@odbc_free_result($result);
			if (is_array($Record))
			{
				return -1;
			}

			return $Record;
		}


        /**
         * Lock a table
         *
         * @param string $table name of table to lock
         * @param string $mode type of lock required (optional), default write
         * @return boolean True if sucessful, False if failed
         */
		function lock($table, $mode = 'write')
		{
			$result = $this->transaction_begin();

			if ($mode == 'write')
			{
				if (is_array($table))
				{
					while ($t = each($table))
					{
						$result = odbc_exec($this->Link_ID,'lock table ' . $t[1] . ' in share mode');
					}
				}
				else
				{
					$result = odbc_exec($this->Link_ID, 'lock table ' . $table . ' in share mode');
				}
			}
			else
			{
				$result = 1;
			}

			return $result;
		}

        /**
         * Unlock a table
         *
         * @return boolean True if sucessful, False if failed
         */
		function unlock()
		{
			return $this->transaction_commit();
		}


        /**
         * Get the number of rows affected by last update
         *
         * @return integer number of affected rows
         */
		function affected_rows()
		{
			return odbc_num_rows($this->Query_ID);
		}

        /**
         * Number of rows in current result set
         *
         * @return integer number of rows
         */
		function num_rows()
		{
			// Many ODBC drivers don't support odbc_num_rows() on SELECT statements.
			$num_rows = odbc_num_rows($this->Query_ID);

			// This is a workaround. It is intended to be ugly.
			if ($num_rows < 0)
			{
				$i=10;
				while (odbc_fetch_row($this->Query_ID, $i))
				{
					$i*=10;
				}

				$j=0;
				while ($i!=$j)
				{
					$k= $j+intval(($i-$j)/2);
					if (odbc_fetch_row($this->Query_ID, $k))
					{
						$j=$k;
					}
					else
					{
						$i=$k;
					}
					if (($i-$j)==1)
					{
						if (odbc_fetch_row($this->Query_ID, $i))
						{
							$j=$i;
						}
						else
						{
							$i=$j;
						}
					}
					//printf("$i $j $k <br>");
				}
				$num_rows=$i;
			}
			return $num_rows;
		}

        /**
         * Number of fields in current row
         *
         * @return integer number of fields
         */
		function num_fields()
		{
			return count($this->Record)/2;
			return @odbc_num_fields($this->Query_ID);
		}


        /**
         * Get the id for the next sequence
         *
         * @param string $seq_name Name of the sequence
         * @return integer sequence id
         */
		function nextid($seq_name)
		{
			$this->connect();

			if ($this->lock($this->Seq_Table))
			{
				/* get sequence number (locked) and increment */
				$q  = sprintf("select nextid from %s where seq_name = '%s'",
					$this->Seq_Table,
					$seq_name);
				$id  = @odbc_exec($this->Link_ID,$q);
				$res = @odbc_fetch_array($id);

				/* No current value, make one */
				if (!is_array($res))
				{
					$currentid = 0;
					$q = sprintf("insert into %s values('%s', %s)",
						$this->Seq_Table,
						$seq_name,
						$currentid);
					$id = @odbc_exec($this->Link_ID,$q);
				}
				else
				{
					$currentid = $res["nextid"];
				}
				$nextid = $currentid + 1;
				$q = sprintf("update %s set nextid = '%s' where seq_name = '%s'",
					$this->Seq_Table,
					$nextid,
					$seq_name);
				$id = @odbc_exec($this->Link_ID,$q);
				$this->unlock();
			}
			else
			{
				$this->halt("cannot lock ".$this->Seq_Table." - has it been created?");
				return 0;
			}
			return $nextid;
		}


        /**
         * Get description of a table
         *
         * @param string $table name of table to describe
         * @param boolean $full optional, default False summary information, True full information
         * @return array Table meta data
         */
		function metadata($table,$full=false)
		{
			$count = 0;
			$id    = 0;
			$res   = array();

			$this->connect();
			$id = odbc_exec($this->Link_ID, "select * from $table");
			if (!$id)
			{
				$this->Errno = odbc_error($this->Link_ID);
				$this->Error = odbc_errormsg($this->Link_ID);
				$this->halt('Metadata query failed.');
			}
			$count = odbc_num_fields($id);

			for ($i=1; $i <= $count; ++$i)
			{
				$res[$i]['table'] = $table;
				$name             = odbc_field_name ($id, $i);
				$res[$i]['name']  = $name;
				$res[$i]['type']  = odbc_field_type ($id, $name);
				$res[$i]['len']   = 0;  // can we determine the width of this column?
				$res[$i]['flags'] = ''; // any optional flags to report?
			}
			odbc_free_result($id);
			return $res;
		}



        /**
         * Error handler
         *
         * @param string $msg error message
         * @param integer $line line of calling method/function (optional)
         * @param string $file file of calling method/function (optional)
         */
		function halt($msg, $line = '', $file = '')
		{
			$this->Error = @odbc_errormsg($this->Link_ID);	// need to be BEFORE unlock,
			$this->Errno = @odbc_error($this->Link_ID);	// else we get its error or none

			if ($this->Link_ID)		// only if we have a link, else infinite loop
			{
				$this->unlock();	/* Just in case there is a table currently locked */
			}
			if ($this->Halt_On_Error == "no")
			{
				return;
			}
			/* Just in case there is a table currently locked */
			$this->transaction_abort();

			$this->haltmsg($msg);

			if ($file)
			{
				printf("<br><b>File:</b> %s",$file);
			}
			if ($line)
			{
				printf("<br><b>Line:</b> %s",$line);
			}

			if ($this->Halt_On_Error != "report")
			{
				echo "<p><b>Session halted.</b>";
				$GLOBALS['phpgw']->common->phpgw_exit(True);
			}
		}


        /** 
         * Display database error
         *
         * @param string $msg Error message
         */
		function haltmsg($msg)
		{
			printf("<b>Database error:</b> %s<br>\n", $msg);
			if ($this->Errno != "0" && $this->Error != "()")
			{
				printf("<b>ODBC Error</b>: %s (%s)<br>\n",$this->Errno,$this->Error);
			}
		}


        /**
         * Get a list of table names in the current database
         *
         * @return array List of the tables
         */
		function table_names()
		{
			if (!$this->Link_ID)
			{
				$this->connect();
			}
			if (!$this->Link_ID)
			{
				return array();
			}
			$return = array();
			$this->query("SELECT TABLENAME FROM DOMAIN.TABLES where owner like '" . strtoupper($this->User) . "'");
			$i=0;
			while ($this->next_record())
			{
				$return[$i]['table_name'] = strtolower($this->Record['TABLENAME']);
				$return[$i]['tablespace_name'] = $this->Database;
				$return[$i]['database'] = $this->Database;
				++$i;
			}
			return $return;
		}


        /**
         * Create a new database
         *
         * @param string $adminname Name of database administrator user (optional)
         * @param string $adminpasswd Password for the database administrator user (optional)
         */
		function create_database($adminname = '', $adminpasswd = '')
		{
			$currentUser = $this->User;
			$currentPassword = $this->Password;
			$currentDatabase = $this->Database;

			if ($adminname != '')
			{
				$this->User = $adminname;
				$this->Password = $adminpasswd;
				$this->Database = 'sapdb';
			}
			$this->disconnect();
			$this->query('create user ' . $currentUser . ' password ' . $currentPassword . ' resource not exclusive',__LINE__,__FILE__);
			$this->disconnect();

			$this->User = $currentUser;
			$this->Password = $currentPassword;
			$this->Database = $currentDatabase;
			$this->connect();
		}


        /**
         * Return the value of a filed
         *
         * @param string $Field_Name name of field
         * @return string The field value
         */
		function f($Field_Name)
		{
			return $this->Record[strtoupper($Field_Name)];
		}
		
		
		
		/**
		 * Prepare SQL statement
		 *
		 * @param string $query
		 * @return Result identifier for query_prepared_statement() or FALSE
		 * @see query_prepared_statement()
		 */
		function prepare_sql_statement($query)
		{
		  if (($query == '') || (!$this->connect()))
	  	   {
			return(FALSE);
		   }
		  return(odbc_prepare($this->connect(),$query));
		}


        /**
         * Execute prepared SQL statement
         *
         * @param resource $result_id Result identifier from prepare_sql_statement()
         * @param array $parameters_array Parameters for the prepared SQL statement
         * @return boolean TRUE on success or FALSE on failure
         * @see prepare_sql_statement()
         */
        function query_prepared_statement($result_id, $parameters_array)
         {
		  if ((!$this->connect()) || (!$result_id))
	  	   {
			return(FALSE);
		   }
		  return(odbc_execute($result_id,$parameters_array));
         }
         

	}
?>
