/* 11mar12abu
 * (c) Software Lab. Alexander Burger
 */

#include "pico.h"

/* Internal/transient hash */
unsigned long ihash(any x) {
   unsigned long g, h;
   word n;

   for (h = 0; isNum(x); x = cdr(numCell(x)))
      for (n = unDig(x); n; n >>= 8)
         g = (h = (h<<4) + (n&0xFF)) & 0xF0000000,  h = (h ^ g>>24) & ~g;
   return h % IHASH;
}

/* External hash */
unsigned long ehash(any x) {
   unsigned long h;
   word n;

   for (h = 0; isNum(x); x = cdr(numCell(x)))
      for (n = unDig(x); n; n >>= 11)
         h += n;
   return h % EHASH;
}

bool hashed(any s, any x) {
   while (isCell(x)) {
      if (s == car(x))
         return YES;
      x = cdr(x);
   }
   return NO;
}

any findHash(any s, any *p) {
   any x, y, *q, h;

   if (isCell(h = *p)) {
      x = s,  y = name(car(h));
      while (unDig(x) == unDig(y)) {
         x = cdr(numCell(x));
         y = cdr(numCell(y));
         if (!isNum(x) && !isNum(y))
            return car(h);
      }
      while (isCell(h = *(q = &cdr(h)))) {
         x = s,  y = name(car(h));
         while (unDig(x) == unDig(y)) {
            x = cdr(numCell(x));
            y = cdr(numCell(y));
            if (!isNum(x) && !isNum(y)) {
               *q = cdr(h),  cdr(h) = *p,  *p = h;
               return car(h);
            }
         }
      }
   }
   return NULL;
}

void unintern(any s, any *p) {
   any x;

   while (isCell(x = *p)) {
      if (s == car(x)) {
         *p = cdr(x);
         return;
      }
      p = &x->cdr;
   }
}

/* Get symbol name */
any name(any s) {
   for (s = tail1(s); isCell(s); s = cdr(s));
   return s;
}

// (name 'sym ['sym2]) -> sym
any doName(any ex) {
   any x, y, *p;
   unsigned long n;
   cell c1;

   x = cdr(ex),  data(c1) = EVAL(car(x));
   NeedSym(ex,data(c1));
   y = name(data(c1));
   if (!isCell(x = cdr(x)))
      return isNum(y)? consStr(y) : Nil;
   n = ihash(y);
   if (isNil(data(c1)) || isExt(data(c1)) || hashed(data(c1), Intern[n]))
      err(ex, data(c1), "Can't rename");
   Save(c1);
   x = EVAL(car(x));
   NeedSym(ex,x);
   unintern(data(c1), Transient + n);
   for (p = &tail(data(c1)); isCell(*p); p = &cdr(*p));
   *p = name(x);
   return Pop(c1);
}

/* Find or create single-char symbol */
any mkChar(int c) {
   if (c >= 0x80) {
      if (c < 0x800)
         c = 0xC0 | c>>6 & 0x1F  |  (0x80 | c & 0x3F) << 8;
      else if (c == TOP)
         c = 0xFF;
      else
         c = 0xE0 | c>>12 & 0x0F  |  (0x80 | c>>6 & 0x3F) << 8  |  (0x80 | c & 0x3F) << 16;
   }
   return consStr(box(c));
}

/* Make name */
any mkName(char *s) {
   int i;
   any nm;
   cell c1;

   i = 0,  Push(c1, nm = box(*(byte*)s++));
   while (*s)
      byteSym(*(byte*)s++, &i, &nm);
   return Pop(c1);
}

any intern(char *s) {
   any nm, x, *h;

   if (!*s)
      return Nil;
   nm = mkName(s);
   if (x = findHash(nm, h = Intern + ihash(nm)))
      return x;
   *h = cons(x = consStr(nm), *h);
   return x;
}

/* Make string */
any mkStr(char *s) {return s && *s? consStr(mkName(s)) : Nil;}

/* Get first byte of symbol name */
int firstByte(any s) {
   return !isNum(s = name(s))? 0 : unDig(s) & 0xFF;
}

int secondByte(any s) {
   return !isNum(s = name(s))? 0 : unDig(s) >> 8 & 0xFF;
}

bool isBlank(any x) {
   int c;

   if (!isSym(x))
      return NO;
   for (c = symChar(name(x)); c; c = symChar(NULL))
      if (c > ' ')
         return NO;
   return YES;
}

// (sp? 'any) -> flg
any doSpQ(any x) {
   x = cdr(x);
   return isBlank(EVAL(car(x)))? T : Nil;
}

// (pat? 'any) -> sym | NIL
any doPatQ(any x) {
   x = cdr(x);
   return isSym(x = EVAL(car(x))) && firstByte(x) == '@'? x : Nil;
}

// (fun? 'any) -> any
any doFunQ(any x) {
   x = cdr(x);
   return funq(EVAL(car(x)));
}

// (getd 'any) -> fun | NIL
any doGetd(any x) {
   x = cdr(x);
   if (!isSym(x = EVAL(car(x))))
      return Nil;
   return !isNil(funq(val(x))) || isNil(val(x)) && sharedLib(x)?
      val(x) : Nil;
}

// (all ['T | '0]) -> lst
any doAll(any x) {
   any *p;
   int mod, i;
   cell c1;

   x = cdr(x),  x = EVAL(car(x));
   if isNil(x)
      p = Intern,  mod = IHASH;
   else if (x == T)
      p = Transient,  mod = IHASH;
   else
      p = Extern,  mod = EHASH;
   Push(c1, Nil);
   for (i = 0; i < mod; ++i)
      for (x = p[i]; isCell(x); x = cdr(x))
         data(c1) = cons(car(x), data(c1));
   return Pop(c1);
}

// (intern 'sym) -> sym
any doIntern(any ex) {
   any x, y, z, *h;

   x = cdr(ex),  x = EVAL(car(x));
   NeedSym(ex,x);
   if (!isNum(y = name(x)))
      return Nil;
   if (unDig(y) == ('L'<<16 | 'I'<<8 | 'N'))
      return Nil;
   if (z = findHash(y, h = Intern + ihash(y)))
      return z;
   *h = cons(x,*h);
   return x;
}

// (extern 'sym) -> sym | NIL
any doExtern(any ex) {
   int c, i;
   any x, y, *h, nm;
   cell c1, c2;

   x = cdr(ex),  x = EVAL(car(x));
   NeedSym(ex,x);
   if (!isNum(x = name(x)))
      return Nil;
   if (!(y = findHash(x, Extern + ehash(x)))) {
      Push(c1, x);
      if ((c = symChar(x)) == '{')
         c = symChar(NULL);
      Push(c2, boxChar(c, &i, &nm));
      while ((c = symChar(NULL)) && c != '}')
         charSym(c, &i, &nm);
      if (!(y = findHash(data(c2), h = Extern + ehash(data(c2))))) {
         mkExt(y = consSym(Nil,data(c2)));
         *h = cons(y,*h);
      }
      drop(c1);
   }
   return isLife(y)? y : Nil;
}

// (==== ['sym ..]) -> NIL
any doHide(any ex) {
   any x, y, z, *h;
   int i;

   for (i = 0; i < IHASH; ++i)
      Transient[i] = Nil;
   for (x = cdr(ex); isCell(x); x = cdr(x)) {
      y = EVAL(car(x));
      NeedSym(ex,y);
      if (isNum(z = name(y)) && !findHash(z, h = Transient + ihash(z)))
         *h = cons(y,*h);
   }
   return Nil;
}

// (box? 'any) -> sym | NIL
any doBoxQ(any x) {
   x = cdr(x);
   return isSym(x = EVAL(car(x))) && !isNum(name(x))? x : Nil;
}

// (str? 'any) -> sym | NIL
any doStrQ(any x) {
   x = cdr(x);
   return isSym(x = EVAL(car(x))) &&
         !isExt(x) && !hashed(x, Intern[ihash(name(x))])? x : Nil;
}

// (ext? 'any) -> sym | NIL
any doExtQ(any x) {
   x = cdr(x);
   return isSym(x = EVAL(car(x))) && isExt(x) && isLife(x) ? x : Nil;
}

// (touch 'sym) -> sym
any doTouch(any ex) {
   any x = cdr(ex);
   x = EVAL(car(x));
   NeedSym(ex,x);
   Touch(ex,x);
   return x;
}

// (zap 'sym) -> sym
any doZap(any ex) {
   any x;

   x = cdr(ex),  x = EVAL(car(x));
   NeedSym(ex,x);
   if (isExt(x))
      db(ex,x,3);
   else {
      if (x >= Nil  &&  x <= Bye)
         protError(ex,x);
      unintern(x, Intern + ihash(name(x)));
   }
   return x;
}

// (chop 'any) -> lst
any doChop(any x) {
   int c;
   cell c1, c2;

   if (isCell(x = EVAL(cadr(x))))
      return x;
   if (!(c = symChar(name(x = xSym(x)))))
      return Nil;
   Push(c1, x);
   Push(c2, x = cons(mkChar(c), Nil));
   while (c = symChar(NULL))
      x = cdr(x) = cons(mkChar(c), Nil);
   drop(c1);
   return data(c2);
}

void pack(any x, int *i, any *nm, cell *p) {
   int c;
   cell c1;

   if (isCell(x))
      do
         pack(car(x), i, nm, p);
      while (isCell(x = cdr(x)));
   if (!isNil(x)) {
      if (isNum(x)) {
         Push(c1, x = numToSym(x, 0, 0, 0));
         c = symChar(name(x));
         if (*nm)
            charSym(c, i, nm);
         else
            Tuck(*p, c1, boxChar(c, i, nm));
         while (c = symChar(NULL))
            charSym(c, i, nm);
         drop(c1);
      }
      else if (c = symChar(name(x))) {
         if (*nm) {
            if (isExt(x))
               charSym('{', i, nm);
            charSym(c, i, nm);
         }
         else if (!isExt(x))
            Push(*p, boxChar(c, i, nm));
         else {
            Push(*p, boxChar('{', i, nm));
            charSym(c, i, nm);
         }
         while (c = symChar(NULL))
            charSym(c, i, nm);
         if (isExt(x))
            charSym('}', i, nm);
      }
   }
}

// (pack 'any ..) -> sym
any doPack(any x) {
   int i;
   any nm;
   cell c1, c2;

   x = cdr(x),  Push(c1, EVAL(car(x)));
   nm = NULL,  pack(data(c1), &i, &nm, &c2);
   while (isCell(x = cdr(x)))
      pack(data(c1) = EVAL(car(x)), &i, &nm, &c2);
   drop(c1);
   return nm? consStr(data(c2)) : Nil;
}

// (glue 'any 'lst) -> sym
any doGlue(any x) {
   int i;
   any nm;
   cell c1, c2, c3;

   x = cdr(x),  Push(c1, EVAL(car(x)));
   x = cdr(x),  Push(c2, x = EVAL(car(x)));
   if (!isCell(x)) {
      drop(c1);
      return x;
   }
   nm = NULL,  pack(car(x), &i, &nm, &c3);
   while (isCell(x = cdr(x))) {
      pack(data(c1), &i, &nm, &c3);
      pack(car(x), &i, &nm, &c3);
   }
   drop(c1);
   return nm? consStr(data(c3)) : Nil;
}

// (text 'any1 'any ..) -> sym
any doText(any x) {
   int c, n, i;
   any nm;
   any y = evSym(x = cdr(x));
   char *p, buf[bufSize(y)];
   cell c1;

   bufString(y, buf);
   if (!*(p = buf))
      return Nil;
   {
      cell arg[length(x = cdr(x))];

      for (n = 0;  isCell(x);  ++n, x = cdr(x))
         Push(arg[n], EVAL(car(x)));

      nm = NULL;
      do {
         if ((c = *p++) != '@') {
            if (nm)
               byteSym(c, &i, &nm);
            else
               i = 0,  Push(c1, nm = box(c & 0xFF));
         }
         else if (!(c = *p++))
            break;
         else if (c == '@') {
            if (nm)
               byteSym('@', &i, &nm);
            else
               i = 0,  Push(c1, nm = box('@'));
         }
         else if (c >= '1') {
            if ((c -= '1') > 8)
               c -= 7;
            if (n > c)
               pack(data(arg[c]), &i, &nm, &c1);
         }
      } while (*p);
      if (n)
         drop(arg[0]);
      else if (nm)
         drop(c1);
      return nm? consStr(data(c1)) : Nil;
   }
}

static bool pre(word n1, any y, word n2, any x) {
   for (;;) {
      if ((n1 & 0xFF) != (n2 & 0xFF))
         return NO;
      if ((n1 >>= 8) == 0) {
         if (!isNum(y = cdr(numCell(y))))
            return YES;
         n1 = unDig(y);
      }
      if ((n2 >>= 8) == 0) {
         if (!isNum(x = cdr(numCell(x))))
            return NO;
         n2 = unDig(x);
      }
   }
}

bool subStr(any y, any x) {
   word n;

   if (!isNum(y = name(y)))
      return YES;
   if (!isNum(x = name(x)))
      return NO;
   n = unDig(x);
   for (;;) {
      if (pre(unDig(y), y, n, x))
         return YES;
      if ((n >>= 8) == 0) {
         if (!isNum(x = cdr(numCell(x))))
            return NO;
         n = unDig(x);
      }
   }
}

// (pre? 'any1 'any2) -> any2 | NIL
any doPreQ(any x) {
   any y, z;
   cell c1;

   x = cdr(x),  Push(c1, evSym(x));
   x = cdr(x),  x = evSym(x);
   drop(c1);
   if (!isNum(y = name(data(c1))))
      return x;
   if (!isNum(z = name(x)))
      return Nil;
   return pre(unDig(y), y, unDig(z), z)? x : Nil;
}

// (sub? 'any1 'any2) -> any2 | NIL
any doSubQ(any x) {
   cell c1;

   x = cdr(x),  Push(c1, evSym(x));
   x = cdr(x),  x = evSym(x);
   drop(c1);
   return subStr(data(c1), x)? x : Nil;
}

// (val 'var) -> any
any doVal(any ex) {
   any x;

   x = cdr(ex),  x = EVAL(car(x));
   NeedVar(ex,x);
   if (isSym(x))
      Fetch(ex,x);
   return val(x);
}

// (set 'var 'any ..) -> any
any doSet(any ex) {
   any x;
   cell c1, c2;

   x = cdr(ex);
   do {
      Push(c1, EVAL(car(x))),  x = cdr(x);
      NeedVar(ex,data(c1));
      CheckVar(ex,data(c1));
      if (isSym(data(c1)))
         Touch(ex,data(c1));
      Push(c2, EVAL(car(x))),  x = cdr(x);
      val(data(c1)) = data(c2);
      drop(c1);
   } while (isCell(x));
   return val(data(c1));
}

// (setq var 'any ..) -> any
any doSetq(any ex) {
   any x, y;

   x = cdr(ex);
   do {
      y = car(x),  x = cdr(x);
      NeedVar(ex,y);
      CheckVar(ex,y);
      val(y) = EVAL(car(x));
   } while (isCell(x = cdr(x)));
   return val(y);
}

// (xchg 'var 'var ..) -> any
any doXchg(any ex) {
   any x, y;
   cell c1, c2;

   x = cdr(ex);
   do {
      Push(c1, EVAL(car(x))),  x = cdr(x);
      NeedVar(ex,data(c1));
      CheckVar(ex,data(c1));
      if (isSym(data(c1)))
         Touch(ex,data(c1));
      Push(c2, EVAL(car(x))),  x = cdr(x);
      NeedVar(ex,data(c2));
      CheckVar(ex,data(c2));
      if (isSym(data(c2)))
         Touch(ex,data(c2));
      y = val(data(c1)),  val(data(c1)) = val(data(c2)),  val(data(c2)) = y;
      drop(c1);
   } while (isCell(x));
   return y;
}

// (on var ..) -> T
any doOn(any ex) {
   any x = cdr(ex);
   do {
      NeedVar(ex,car(x));
      CheckVar(ex,car(x));
      val(car(x)) = T;
   } while (isCell(x = cdr(x)));
   return T;
}

// (off var ..) -> NIL
any doOff(any ex) {
   any x = cdr(ex);
   do {
      NeedVar(ex,car(x));
      CheckVar(ex,car(x));
      val(car(x)) = Nil;
   } while (isCell(x = cdr(x)));
   return Nil;
}

// (onOff var ..) -> flg
any doOnOff(any ex) {
   any x = cdr(ex);
   any y;

   do {
      NeedVar(ex,car(x));
      CheckVar(ex,car(x));
      y = val(car(x)) = isNil(val(car(x)))? T : Nil;
   } while (isCell(x = cdr(x)));
   return y;
}

// (zero var ..) -> 0
any doZero(any ex) {
   any x = cdr(ex);
   do {
      NeedVar(ex,car(x));
      CheckVar(ex,car(x));
      val(car(x)) = Zero;
   } while (isCell(x = cdr(x)));
   return Zero;
}

// (one var ..) -> 1
any doOne(any ex) {
   any x = cdr(ex);
   do {
      NeedVar(ex,car(x));
      CheckVar(ex,car(x));
      val(car(x)) = One;
   } while (isCell(x = cdr(x)));
   return One;
}

// (default var 'any ..) -> any
any doDefault(any ex) {
   any x, y;

   x = cdr(ex);
   do {
      y = car(x),  x = cdr(x);
      NeedVar(ex,y);
      CheckVar(ex,y);
      if (isNil(val(y)))
         val(y) = EVAL(car(x));
   } while (isCell(x = cdr(x)));
   return val(y);
}

// (push 'var 'any ..) -> any
any doPush(any ex) {
   any x;
   cell c1, c2;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   NeedVar(ex,data(c1));
   CheckVar(ex,data(c1));
   if (isSym(data(c1)))
      Touch(ex,data(c1));
   x = cdr(x),  Push(c2, EVAL(car(x)));
   val(data(c1)) = cons(data(c2), val(data(c1)));
   while (isCell(x = cdr(x))) {
      data(c2) = EVAL(car(x));
      val(data(c1)) = cons(data(c2), val(data(c1)));
   }
   drop(c1);
   return data(c2);
}

// (push1 'var 'any ..) -> any
any doPush1(any ex) {
   any x;
   cell c1, c2;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   NeedVar(ex,data(c1));
   CheckVar(ex,data(c1));
   if (isSym(data(c1)))
      Touch(ex,data(c1));
   x = cdr(x),  Push(c2, EVAL(car(x)));
   if (!member(data(c2), val(data(c1))))
      val(data(c1)) = cons(data(c2), val(data(c1)));
   while (isCell(x = cdr(x)))
      if (!member(data(c2) = EVAL(car(x)), val(data(c1))))
         val(data(c1)) = cons(data(c2), val(data(c1)));
   drop(c1);
   return data(c2);
}

// (pop 'var) -> any
any doPop(any ex) {
   any x, y;

   x = cdr(ex),  x = EVAL(car(x));
   NeedVar(ex,x);
   CheckVar(ex,x);
   if (isSym(x))
      Touch(ex,x);
   if (!isCell(y = val(x)))
      return y;
   val(x) = cdr(y);
   return car(y);
}

// (cut 'cnt 'var) -> lst
any doCut(any ex) {
   long n;
   any x, y;
   cell c1, c2;

   if ((n = evCnt(ex, x = cdr(ex))) <= 0)
      return Nil;
   x = cdr(x),  Push(c1, EVAL(car(x)));
   NeedVar(ex,data(c1));
   CheckVar(ex,data(c1));
   if (isSym(data(c1)))
      Touch(ex,data(c1));
   if (isCell(val(data(c1)))) {
      Push(c2, y = cons(car(val(data(c1))), Nil));
      while (isCell(val(data(c1)) = cdr(val(data(c1)))) && --n)
         y = cdr(y) = cons(car(val(data(c1))), Nil);
      drop(c1);
      return data(c2);
   }
   return val(Pop(c1));
}

// (del 'any 'var) -> lst
any doDel(any ex) {
   any x, y;
   cell c1, c2, c3;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   x = cdr(x),  Push(c2, EVAL(car(x)));
   NeedVar(ex,data(c2));
   CheckVar(ex,data(c2));
   if (isSym(data(c2)))
      Touch(ex,data(c2));
   if (isCell(x = val(data(c2)))) {
      if (equal(data(c1), car(x))) {
         drop(c1);
         return val(data(c2)) = cdr(x);
      }
      Push(c3, y = cons(car(x), Nil));
      while (isCell(x = cdr(x))) {
         if (equal(data(c1), car(x))) {
            cdr(y) = cdr(x);
            drop(c1);
            return val(data(c2)) = data(c3);
         }
         y = cdr(y) = cons(car(x), Nil);
      }
   }
   drop(c1);
   return val(data(c2));
}

// (queue 'var 'any) -> any
any doQueue(any ex) {
   any x, y;
   cell c1;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   NeedVar(ex,data(c1));
   CheckVar(ex,data(c1));
   if (isSym(data(c1)))
      Touch(ex,data(c1));
   x = cdr(x),  x = EVAL(car(x));
   if (!isCell(y = val(data(c1))))
      val(data(c1)) = cons(x,Nil);
   else {
      while (isCell(cdr(y)))
         y = cdr(y);
      cdr(y) = cons(x,Nil);
   }
   drop(c1);
   return x;
}

// (fifo 'var ['any ..]) -> any
any doFifo(any ex) {
   any x, y, z;
   cell c1;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   NeedVar(ex,data(c1));
   CheckVar(ex,data(c1));
   if (isSym(data(c1)))
      Touch(ex,data(c1));
   if (isCell(x = cdr(x))) {
      y = EVAL(car(x));
      if (isCell(z = val(data(c1))))
         val(data(c1)) = z = cdr(z) = cons(y,cdr(z));
      else {
         z = val(data(c1)) = cons(y,Nil);
         cdr(z) = z;
      }
      while (isCell(x = cdr(x)))
         val(data(c1)) = z = cdr(z) = cons(y = EVAL(car(x)), cdr(z));
   }
   else if (!isCell(z = val(data(c1))))
      y = Nil;
   else {
      if (z == cdr(z)) {
         y = car(z);
         val(data(c1)) = Nil;
      }
      else {
         y = cadr(z);
         cdr(z) = cddr(z);
      }
   }
   drop(c1);
   return y;
}

any idx(any var, any key, int flg) {
   any x, y, z, *p;
   int n;

   if (!key) {
      cell c1, c2;

      if (!isCell(x = val(var)))
         return Nil;
      y = Nil;  // Result
      Push(c1, x);  // Tree
      Push(c2, Nil);  // TOS
      for (;;) {
         while (isCell(cddr(data(c1))))
            z = data(c1),  data(c1) = cddr(z),  cddr(z) = data(c2),  data(c2) = z;
         for (;;) {
            y = cons(car(data(c1)), y);
            if (isCell(cadr(data(c1)))) {
               z = data(c1),  data(c1) = cadr(z),  cadr(z) = data(c2),  data(c2) = symPtr(z);
               break;
            }
            for (;;) {
               if (isNil(data(c2))) {
                  drop(c1);
                  return y;
               }
               if (isCell(data(c2))) {
                  z = data(c2),  data(c2) = cddr(z),  cddr(z) = data(c1),  data(c1) = z;
                  break;
               }
               z = cellPtr(data(c2)),  data(c2) = cadr(z),  cadr(z) = data(c1),  data(c1) = z;
            }
         }
      }
   }
   if (!isCell(x = val(var))) {
      if (flg > 0)
         val(var) = cons(key,Nil);
      return Nil;
   }
   p = (any*)var;
   for (;;) {
      if ((n = compare(key, car(x))) == 0) {
         if (flg < 0) {
            if (!isCell(cadr(x)))
               *p = cddr(x);
            else if (!isCell(y = cddr(x)))
               *p = cadr(x);
            else if (!isCell(z = cadr(y)))
               car(x) = car(y),  cddr(x) = cddr(y);
            else {
               while (isCell(cadr(z)))
                  z = cadr(y = z);
               car(x) = car(z),  cadr(y) = cddr(z);
            }
         }
         return x;
      }
      if (!isCell(cdr(x))) {
         if (flg > 0)
            cdr(x) = n < 0? cons(cons(key,Nil), Nil) : cons(Nil, cons(key,Nil));
         return Nil;
      }
      if (n < 0) {
         if (!isCell(cadr(x))) {
            if (flg > 0)
               cadr(x) = cons(key,Nil);
            return Nil;
         }
         x = *(p = &cadr(x));
      }
      else {
         if (!isCell(cddr(x))) {
            if (flg > 0)
               cddr(x) = cons(key,Nil);
            return Nil;
         }
         x = *(p = &cddr(x));
      }
   }
}

// (idx 'var 'any 'flg) -> lst
// (idx 'var 'any) -> lst
// (idx 'var) -> lst
any doIdx(any ex) {
   any x;
   int flg;
   cell c1, c2;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   NeedVar(ex,data(c1));
   CheckVar(ex,data(c1));
   if (!isCell(x = cdr(x)))
      x = idx(data(c1), NULL, 0);
   else {
      Push(c2, EVAL(car(x)));
      if (!isCell(cdr(x)))
         flg = 0;
      else
         flg = isNil(EVAL(cadr(x)))? -1 : +1;
      x = idx(data(c1), data(c2), flg);
   }
   drop(c1);
   return x;
}

// (lup 'lst 'any) -> lst
// (lup 'lst 'any 'any2) -> lst
any doLup(any x) {
   int n;
   cell c1, c2, c3, c4, c5;

   x = cdr(x),  data(c1) = EVAL(car(x));
   if (!isCell(data(c1)))
      return data(c1);
   Save(c1);
   x = cdr(x),  Push(c2, EVAL(car(x)));  // from
   if (isCell(x = cdr(x))) {
      Push(c3, EVAL(car(x)));  // to
      Push(c4, Nil);  // tos
      Push(c5, Nil);  // result
      for (;;) {
         while (isCell(cddr(data(c1))) && car(data(c1)) != T && (!isCell(car(data(c1))) || compare(data(c3), caar(data(c1))) >= 0))
            x = data(c1),  data(c1) = cddr(x),  cddr(x) = data(c4),  data(c4) = x;
         for (;;) {
            if (isCell(car(data(c1))) && compare(data(c2), caar(data(c1))) <= 0) {
               if (compare(data(c3), caar(data(c1))) >= 0)
                  data(c5) = cons(car(data(c1)), data(c5));
               if (isCell(cadr(data(c1)))) {
                  x = data(c1),  data(c1) = cadr(x),  cadr(x) = data(c4),  data(c4) = symPtr(x);
                  break;
               }
            }
            for (;;) {
               if (isNil(data(c4))) {
                  drop(c1);
                  return data(c5);
               }
               if (isCell(data(c4))) {
                  x = data(c4),  data(c4) = cddr(x),  cddr(x) = data(c1),  data(c1) = x;
                  break;
               }
               else
                  x = cellPtr(data(c4)),  data(c4) = cadr(x),  cadr(x) = data(c1),  data(c1) = x;
            }
         }
      }
   }
   do {
      if (car(data(c1)) == T)
         data(c1) = cadr(data(c1));
      else if (!isCell(car(data(c1))))
         data(c1) = cddr(data(c1));
      else if (n = compare(data(c2), caar(data(c1))))
         data(c1) = n < 0? cadr(data(c1)) : cddr(data(c1));
      else {
         drop(c1);
         return car(data(c1));
      }
   } while (isCell(data(c1)));
   drop(c1);
   return Nil;
}

void put(any x, any key, any val) {
   any y, z;

   if (isCell(y = tail1(x))) {
      if (isCell(car(y))) {
         if (key == cdar(y)) {
            if (isNil(val))
               Tail(x, cdr(y));
            else if (val == T)
               car(y) = key;
            else
               caar(y) = val;
            return;
         }
      }
      else if (key == car(y)) {
         if (isNil(val))
            Tail(x, cdr(y));
         else if (val != T)
            car(y) = cons(val,key);
         return;
      }
      while (isCell(z = cdr(y))) {
         if (isCell(car(z))) {
            if (key == cdar(z)) {
               if (isNil(val))
                  cdr(y) = cdr(z);
               else {
                  if (val == T)
                     car(z) = key;
                  else
                     caar(z) = val;
                  cdr(y) = cdr(z),  cdr(z) = tail1(x),  Tail(x, z);
               }
               return;
            }
         }
         else if (key == car(z)) {
            if (isNil(val))
               cdr(y) = cdr(z);
            else {
               if (val != T)
                  car(z) = cons(val,key);
               cdr(y) = cdr(z),  cdr(z) = tail1(x),  Tail(x, z);
            }
            return;
         }
         y = z;
      }
   }
   if (!isNil(val))
      Tail(x, cons(val==T? key : cons(val,key), tail1(x)));
}

any get(any x, any key) {
   any y, z;

   if (!isCell(y = tail1(x)))
      return Nil;
   if (!isCell(car(y))) {
      if (key == car(y))
         return T;
   }
   else if (key == cdar(y))
      return caar(y);
   while (isCell(z = cdr(y))) {
      if (!isCell(car(z))) {
         if (key == car(z)) {
            cdr(y) = cdr(z),  cdr(z) = tail1(x),  Tail(x, z);
            return T;
         }
      }
      else if (key == cdar(z)) {
         cdr(y) = cdr(z),  cdr(z) = tail1(x),  Tail(x, z);
         return caar(z);
      }
      y = z;
   }
   return Nil;
}

any prop(any x, any key) {
   any y, z;

   if (isCell(y = tail1(x))) {
      if (!isCell(car(y))) {
         if (key == car(y))
            return key;
      }
      else if (key == cdar(y))
         return car(y);
      while (isCell(z = cdr(y))) {
         if (!isCell(car(z))) {
            if (key == car(z)) {
               cdr(y) = cdr(z),  cdr(z) = tail1(x),  Tail(x, z);
               return key;
            }
         }
         else if (key == cdar(z)) {
            cdr(y) = cdr(z),  cdr(z) = tail1(x),  Tail(x, z);
            return car(z);
         }
         y = z;
      }
   }
   Tail(x, cons(y = cons(Nil,key), tail1(x)));
   return y;
}

// (put 'sym1|lst ['sym2|cnt ..] 'sym|0 'any) -> any
any doPut(any ex) {
   any x;
   cell c1, c2, c3;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   x = cdr(x),  Push(c2, EVAL(car(x)));
   while (isCell(cdr(x = cdr(x)))) {
      if (isCell(data(c1)))
         data(c1) = getn(data(c2), data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(data(c2)) && !unDig(data(c2))? val(data(c1)) : get(data(c1), data(c2));
      }
      data(c2) = EVAL(car(x));
   }
   NeedSym(ex,data(c1));
   Push(c3, EVAL(car(x)));
   if (isNum(data(c2)) && IsZero(data(c2))) {
      Touch(ex,data(c1));
      CheckVar(ex,data(c1));
      val(data(c1)) = x = data(c3);
   }
   else {
      if (!isNil(data(c2)))
         Touch(ex,data(c1));
      put(data(c1), data(c2), x = data(c3));
   }
   drop(c1);
   return x;
}

// (get 'sym1|lst ['sym2|cnt ..]) -> any
any doGet(any ex) {
   any x, y;
   cell c1;

   x = cdr(ex),  data(c1) = EVAL(car(x));
   if (!isCell(x = cdr(x)))
      return data(c1);
   Save(c1);
   do {
      y = EVAL(car(x));
      if (isCell(data(c1)))
         data(c1) = getn(y, data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(y) && !unDig(y)? val(data(c1)) : get(data(c1), y);
      }
   } while (isCell(x = cdr(x)));
   return Pop(c1);
}

// (prop 'sym1|lst ['sym2|cnt ..] 'sym) -> var
any doProp(any ex) {
   any x;
   cell c1, c2;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   x = cdr(x),  Push(c2, EVAL(car(x)));
   while (isCell(x = cdr(x))) {
      if (isCell(data(c1)))
         data(c1) = getn(data(c2), data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(data(c2)) && !unDig(data(c2))? val(data(c1)) : get(data(c1), data(c2));
      }
      data(c2) = EVAL(car(x));
   }
   NeedSym(ex,data(c1));
   CheckNil(ex,data(c1));
   Touch(ex,data(c1));
   return prop(Pop(c1), data(c2));
}

// (; 'sym1|lst [sym2|cnt ..]) -> any
any doSemicol(any ex) {
   any x;
   cell c1;

   x = cdr(ex),  data(c1) = EVAL(car(x));
   if (!isCell(x = cdr(x)))
      return data(c1);
   Save(c1);
   do {
      if (isCell(data(c1)))
         data(c1) = getn(car(x), data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(car(x)) && !unDig(car(x))? val(data(c1)) : get(data(c1), car(x));
      }
   } while (isCell(x = cdr(x)));
   return Pop(c1);
}

// (=: sym|0 [sym1|cnt .. sym2|0] 'any) -> any
any doSetCol(any ex) {
   any x, y, z;
   cell c1;

   x = cdr(ex),  y = val(This);
   Fetch(ex,y);
   if (z = car(x),  isCell(cdr(x = cdr(x)))) {
      y = isNum(z) && !unDig(z)? val(y) : get(y,z);
      while (z = car(x),  isCell(cdr(x = cdr(x)))) {
         if (isCell(y))
            y = getn(z,y);
         else {
            NeedSym(ex,y);
            Fetch(ex,y);
            y = isNum(z) && !unDig(z)? val(y) : get(y,z);
         }
      }
   }
   NeedSym(ex,y);
   Push(c1, EVAL(car(x)));
   if (isNum(z) && IsZero(z)) {
      Touch(ex,y);
      CheckVar(ex,y);
      val(y) = x = data(c1);
   }
   else {
      if (!isNil(z))
         Touch(ex,y);
      put(y, z, x = data(c1));
   }
   drop(c1);
   return x;
}

// (: sym|0 [sym1|cnt ..]) -> any
any doCol(any ex) {
   any x, y;

   x = cdr(ex),  y = val(This);
   Fetch(ex,y);
   y = isNum(car(x)) && !unDig(car(x))? val(y) : get(y, car(x));
   while (isCell(x = cdr(x))) {
      if (isCell(y))
         y = getn(car(x), y);
      else {
         NeedSym(ex,y);
         Fetch(ex,y);
         y = isNum(car(x)) && !unDig(car(x))? val(y) : get(y, car(x));
      }
   }
   return y;
}

// (:: sym|0 [sym1|cnt .. sym2]) -> var
any doPropCol(any ex) {
   any x, y;

   x = cdr(ex),  y = val(This);
   Fetch(ex,y);
   if (isCell(cdr(x))) {
      y = isNum(car(x)) && !unDig(car(x))? val(y) : get(y, car(x));
      while (isCell(cdr(x = cdr(x)))) {
         if (isCell(y))
            y = getn(car(x), y);
         else {
            NeedSym(ex,y);
            Fetch(ex,y);
            y = isNum(car(x)) && !unDig(car(x))? val(y) : get(y, car(x));
         }
      }
   }
   NeedSym(ex,y);
   CheckNil(ex,y);
   Touch(ex,y);
   return prop(y, car(x));
}

// (putl 'sym1|lst1 ['sym2|cnt ..] 'lst) -> lst
any doPutl(any ex) {
   any x;
   cell c1, c2;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   x = cdr(x),  Push(c2, EVAL(car(x)));
   while (isCell(x = cdr(x))) {
      if (isCell(data(c1)))
         data(c1) = getn(data(c2), data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(data(c2)) && !unDig(data(c2))? val(data(c1)) : get(data(c1), data(c2));
      }
      data(c2) = EVAL(car(x));
   }
   NeedSym(ex,data(c1));
   CheckNil(ex,data(c1));
   Touch(ex,data(c1));
   while (isCell(tail(data(c1))))
      Tail(data(c1), cdr(tail1(data(c1))));
   for (x = data(c2);  isCell(x);  x = cdr(x)) {
      if (!isCell(car(x)))
         Tail(data(c1), cons(car(x), tail1(data(c1))));
      else if (!isNil(caar(x)))
         Tail(data(c1), cons(caar(x)==T? cdar(x) : car(x), tail1(data(c1))));
   }
   drop(c1);
   return data(c2);
}

// (getl 'sym1|lst1 ['sym2|cnt ..]) -> lst
any doGetl(any ex) {
   any x, y;
   cell c1, c2;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   while (isCell(x = cdr(x))) {
      y = EVAL(car(x));
      if (isCell(data(c1)))
         data(c1) = getn(y, data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(y) && !unDig(y)? val(data(c1)) : get(data(c1), y);
      }
   }
   NeedSym(ex,data(c1));
   Fetch(ex,data(c1));
   if (!isCell(x = tail1(data(c1))))
      data(c2) = Nil;
   else {
      Push(c2, y = cons(car(x),Nil));
      while (isCell(x = cdr(x)))
         y = cdr(y) = cons(car(x),Nil);
   }
   drop(c1);
   return data(c2);
}

static void wipe(any x) {
   any y, z;

   for (y = tail1(x); isCell(y); y = cdr(y));
   if (!isNum(y)) {
      val(x) = Nil;
      tail(x) = y;
   }
   else {
      z = numCell(y);
      while (isNum(cdr(z)))
         z = numCell(cdr(z));
      if (isNil(cdr(z)) || cdr(z) == At) {
         val(x) = Nil;
         Tail(x, y);
         cdr(z) = Nil;
      }
   }
}

// (wipe 'sym|lst) -> sym
any doWipe(any x) {
   any y;

   x = cdr(x);
   if (!isNil(x = EVAL(car(x))))
      if (!isCell(x))
         wipe(x);
      else {
         y = x; do
            wipe(car(y));
         while (isCell(y = cdr(y)));
      }
   return x;
}

static any meta(any x, any y) {
   any z;

   while (isCell(x)) {
      if (isSym(car(x)))
         if (!isNil(z = get(car(x),y)) || !isNil(z = meta(val(car(x)), y)))
            return z;
      x = cdr(x);
   }
   return Nil;
}

// (meta 'obj|typ 'sym ['sym2|cnt ..]) -> any
any doMeta(any ex) {
   any x, y;
   cell c1;

   x = cdr(ex),  Push(c1, EVAL(car(x)));
   if (isSym(data(c1))) {
      Fetch(ex,data(c1));
      data(c1) = val(data(c1));
   }
   x = cdr(x),  y = EVAL(car(x));
   data(c1) = meta(data(c1), y);
   while (isCell(x = cdr(x))) {
      y = EVAL(car(x));
      if (isCell(data(c1)))
         data(c1) = getn(y, data(c1));
      else {
         NeedSym(ex,data(c1));
         Fetch(ex,data(c1));
         data(c1) = isNum(y) && !unDig(y)? val(data(c1)) : get(data(c1), y);
      }
   }
   return Pop(c1);
}

/*** Case mappings from the GNU Kaffe Project ***/
#define CHAR_UPPERCASE  1
#define CHAR_LOWERCASE  2
#define CHAR_LETTER     62
#define CHAR_DIGIT      512

static uint16_t Blocks[] = {
   0x1C2, 0x1C2, 0x1C1, 0x12C, 0x12B, 0x1A0, 0x1F8, 0x2DC, 0x25F, 0x2EE, 0x215, 0x346, 0x2DC, 0x326, 0x2BC, 0x216,
   0x15F, 0x2D4, 0x376, 0x376, 0x376, 0x369, 0xFE8F, 0x344, 0xFF85, 0xFF65, 0xFDB5, 0xFDA1, 0x1B, 0x2C4, 0x1C, 0x47,
   0xFEA8, 0xFF8C, 0x235, 0xFEFF, 0x1A, 0xFEBF, 0x26, 0xFB20, 0xFE28, 0x113, 0x104, 0xFB61, 0xFB5A, 0x10B, 0x109, 0xFE,
   0xFF08, 0x229, 0x25E, 0x1C7, 0x1FC, 0x1DC, 0xFC46, 0x229, 0xFE27, 0xFB55, 0x169, 0xFBC8, 0xFC, 0x103, 0xFB68, 0xFB48,
   0xFB28, 0xFB08, 0xFAE8, 0xFAC8, 0xFAA8, 0xFA88, 0xFA68, 0xFA48, 0x65, 0x50, 0xAB, 0x139, 0xFE0E, 0x63, 0x155, 0x1A8,
   0xF669, 0x129, 0x128, 0xF91F, 0xFE56, 0x108, 0x107, 0xFAC0, 0xFC8E, 0xFEAD, 0xC6, 0xFCA7, 0xFB95, 0xF47D, 0x9F, 0xFB17,
   0xFE20, 0xFD28, 0xFB2F, 0x3B, 0xF3B9, 0xFE57, 0xFCCE, 0xFFBB, 0xF339, 0xFA98, 0xFF8B, 0xFF3B, 0xFA54, 0xF7E3, 0xFF2B, 0xFAD7,
   0xFB69, 0xFC3A, 0xFEE5, 0xF4C8, 0xFCB0, 0xFA88, 0xFDBF, 0xF448, 0xFE45, 0xFCC7, 0xFE4F, 0xF7F1, 0xF715, 0xF2E8, 0xFD9F, 0xF348,
   0xF96A, 0xFC02, 0xFD97, 0xF2C8, 0xF2A8, 0xF4B9, 0xF4B3, 0xEF6B, 0xF86A, 0xF84A, 0xFC58, 0xF80A, 0xF7EA, 0xFC0F, 0xF7AA, 0xEE9C,
   0xFB90, 0xF74A, 0xF7FA, 0xF70A, 0xF7CA, 0xF792, 0xF471, 0xF4D2, 0xF732, 0xF64A, 0xF401, 0xF64D, 0xEFA8, 0xF5CA, 0xF5AA, 0xECA1,
   0xF569, 0xF54A, 0xF52A, 0xF50A, 0xF4EA, 0xF4CA, 0xF4AA, 0xF48A, 0xF46A, 0xF44A, 0xF42A, 0xF40A, 0xF3EA, 0xF3CA, 0xF3AA, 0xF38A,
   0xF36A, 0xF34A, 0xF32A, 0xF289, 0xF777, 0xF2CA, 0xF2AA, 0xF737, 0xEC28, 0xEC08, 0xEBE8, 0xEBC8, 0xF1EA, 0xF4A2, 0xF545, 0xEDC6,
   0xF2D7, 0xF14A, 0xE8ED, 0xE81E, 0xF0EA, 0xF597, 0xEA68, 0xEA48, 0xEA28, 0xEA08, 0xE9E8, 0xE9C8, 0xE9A8, 0xE988, 0xE968, 0xE948,
   0xE928, 0xE908, 0xE8E8, 0xE8C8, 0xE8A8, 0xE888, 0xE868, 0xE848, 0xE828, 0xE808, 0xE7E8, 0xE7C8, 0xE7A8, 0xE788, 0xE768, 0xE748,
   0xE728, 0xE708, 0xE6E8, 0xE6C8, 0xE6A8, 0xE688, 0xE668, 0xE648, 0xE628, 0xE608, 0xE5E8, 0xE5C8, 0xE5A8, 0xE588, 0xE568, 0xE548,
   0xE55F, 0xE53F, 0xE51F, 0xE4FF, 0xEFD7, 0xE4BF, 0xE49F, 0xE485, 0xEF87, 0xEF57, 0xEF57, 0xEF57, 0xEF57, 0xEF47, 0xE1AD, 0xEF46,
   0xEF46, 0xEF46, 0xE1E0, 0xE3DD, 0xEF06, 0xE9D9, 0xEBEB, 0xE244, 0xEED4, 0xEF65, 0xE1F5, 0xEF45, 0xEEE9, 0xEF7C, 0xEE74, 0xEF70,
   0xEF7D, 0xEF78, 0xEE91, 0xEFD3, 0xEE7D, 0xEE25, 0xEE27, 0xEF65, 0xEFDD, 0xEE96, 0xEFD3, 0xEFE1, 0xEF69, 0xDF88, 0xDF68, 0xDF48,
   0xED2B, 0xED3D, 0xED19, 0xEF1C, 0xEF08, 0xED47, 0xED3D, 0xED33, 0xEC2B, 0xEC0B, 0xEBEB, 0xEBCB, 0xEBCE, 0xEA7C, 0xEB69, 0xEB6C,
   0xE9B6, 0xEB0B, 0xEAEB, 0xE9E9, 0xDCA8, 0xDC88, 0xDC68, 0xDC48, 0xE910, 0xEA23, 0xEB58, 0xEB4F, 0xEB45, 0xEAE5, 0xDB68, 0xDB48,
   0xE92B, 0xE90B, 0xE8EB, 0xE8CB, 0xE8AB, 0xE88B, 0xE86B, 0xE84B, 0xDA28, 0xDA08, 0xD9E8, 0xD9C8, 0xD9A8, 0xD988, 0xD968, 0xD948,
   0xD928, 0xD908, 0xD8E8, 0xD8C8, 0xD8A8, 0xD888, 0xD868, 0xD848, 0xD828, 0xD808, 0xD7E8, 0xD7C8, 0xD7A8, 0xD788, 0xD768, 0xD748,
   0xD728, 0xD708, 0xD6E8, 0xD6C8, 0xD6A8, 0xD688, 0xD668, 0xD648, 0xD628, 0xD608, 0xD5E8, 0xD5C8, 0xD5A8, 0xD588, 0xD568, 0xD548,
   0xD528, 0xD508, 0xD4E8, 0xD4C8, 0xE2B1, 0xE28B, 0xE26B, 0xE270, 0xE22B, 0xE20B, 0xE1EB, 0xE1CB, 0xE1AB, 0xE18B, 0xE18E, 0xDD8F,
   0xE3A8, 0xDFD3, 0xD929, 0xD90A, 0xE348, 0xD8C9, 0xD8AA, 0xDCD7, 0xDCB2, 0xD681, 0xD82A, 0xD80A, 0xE268, 0xCEDE, 0xD168, 0xD148,
   0xE116, 0xE0E9, 0xE1CB, 0xE0B7, 0xE0B7, 0xE15E, 0xDF17, 0xE034, 0xE013, 0xDFF3, 0xDFD3, 0xDE6C, 0xDF93, 0xDF73, 0xDF55, 0xDF34,
   0xD56A, 0xD54A, 0xD52A, 0xD50A, 0xD4EA, 0xD4CA, 0xD4AA, 0xD48A, 0xD46A, 0xD44A, 0xD42A, 0xD40A, 0xD3EA, 0xD3CA, 0xD3AA, 0xD38A,
   0xD36A, 0xD34A, 0xD32A, 0xD30A, 0xD2EA, 0xD2CA, 0xD2AA, 0xD28A, 0xD26A, 0xD24A, 0xD22A, 0xD20A, 0xD1EA, 0xD1CA, 0xD1AA, 0xD18A,
   0xD16A, 0xD14A, 0xD12A, 0xD10A, 0xD0EA, 0xD0CA, 0xD0AA, 0xD08A, 0xD06A, 0xD04A, 0xD02A, 0xD00A, 0xCFEA, 0xCFCA, 0xCFAA, 0xCF8A,
   0xCF6A, 0xCF4A, 0xCF2A, 0xCF0A, 0xCEEA, 0xCECA, 0xCEAA, 0xCE8A, 0xCE6A, 0xCE4A, 0xCE2A, 0xCE0A, 0xCDEA, 0xCDCA, 0xCDAA, 0xCD8A,
   0xCD6A, 0xCD4A, 0xCD2A, 0xCD0A, 0xCCEA, 0xCCCA, 0xCCAA, 0xCC8A, 0xCC6A, 0xCC4A, 0xCC2A, 0xCC0A, 0xCBEA, 0xCBCA, 0xCBAA, 0xCB8A,
   0xCB6A, 0xCB4A, 0xCB2A, 0xCB0A, 0xCAEA, 0xCACA, 0xCAAA, 0xCA8A, 0xCA6A, 0xCA4A, 0xCA2A, 0xCA0A, 0xC9EA, 0xC9CA, 0xC9AA, 0xC98A,
   0xC96A, 0xC94A, 0xC92A, 0xC90A, 0xC8EA, 0xC8CA, 0xC8AA, 0xC88A, 0xC86A, 0xC84A, 0xC82A, 0xC80A, 0xC7EA, 0xC7CA, 0xC7AA, 0xC78A,
   0xC76A, 0xC74A, 0xC72A, 0xC70A, 0xC6EA, 0xC6CA, 0xC6AA, 0xC68A, 0xC66A, 0xC64A, 0xC62A, 0xC60A, 0xC5EA, 0xC5CA, 0xC5AA, 0xC58A,
   0xC56A, 0xC54A, 0xC52A, 0xC50A, 0xC4EA, 0xC4CA, 0xC4AA, 0xC48A, 0xC46A, 0xC44A, 0xC42A, 0xC40A, 0xC3EA, 0xC3CA, 0xC3AA, 0xC38A,
   0xC36A, 0xC34A, 0xC32A, 0xC30A, 0xC2EA, 0xC2CA, 0xC2AA, 0xC28A, 0xC26A, 0xC24A, 0xC22A, 0xC20A, 0xC1EA, 0xC1CA, 0xC1AA, 0xC18A,
   0xC16A, 0xC14A, 0xC12A, 0xC10A, 0xC0EA, 0xC0CA, 0xC0AA, 0xC08A, 0xC06A, 0xC04A, 0xC02A, 0xC00A, 0xBFEA, 0xBFCA, 0xBFAA, 0xBF8A,
   0xBF6A, 0xBF4A, 0xBF2A, 0xBF0A, 0xBEEA, 0xBECA, 0xBEAA, 0xBE8A, 0xBE6A, 0xBE4A, 0xBE2A, 0xBE0A, 0xBDEA, 0xBDCA, 0xBDAA, 0xBD8A,
   0xBD6A, 0xBD4A, 0xBD2A, 0xBD0A, 0xBCEA, 0xBCCA, 0xBCAA, 0xBC8A, 0xBC6A, 0xBC4A, 0xBC2A, 0xBC0A, 0xBBEA, 0xB2E0, 0xB568, 0xB548,
   0xBB6A, 0xBB4A, 0xBB2A, 0xBB0A, 0xBAEA, 0xBACA, 0xBAAA, 0xBA8A, 0xBA6A, 0xBA4A, 0xBA2A, 0xBA0A, 0xB9EA, 0xB9CA, 0xB9AA, 0xB98A,
   0xB96A, 0xB94A, 0xB92A, 0xB90A, 0xB8EA, 0xB8CA, 0xB8AA, 0xB88A, 0xB86A, 0xB84A, 0xB82A, 0xB80A, 0xB7EA, 0xB7CA, 0xB7AA, 0xB78A,
   0xB76A, 0xB74A, 0xB72A, 0xB70A, 0xB6EA, 0xB6CA, 0xB6AA, 0xB68A, 0xB66A, 0xB64A, 0xB62A, 0xB60A, 0xB5EA, 0xB5CA, 0xB5AA, 0xB58A,
   0xB56A, 0xB54A, 0xB52A, 0xB50A, 0xB4EA, 0xB4CA, 0xB4AA, 0xB48A, 0xB46A, 0xB44A, 0xB42A, 0xB40A, 0xB3EA, 0xB3CA, 0xB3AA, 0xB38A,
   0xB36A, 0xB34A, 0xB32A, 0xB30A, 0xB2EA, 0xB2CA, 0xB2AA, 0xB28A, 0xB26A, 0xB24A, 0xB22A, 0xB20A, 0xB1EA, 0xB1CA, 0xB1AA, 0xB18A,
   0xB16A, 0xB14A, 0xB12A, 0xB10A, 0xB0EA, 0xB0CA, 0xB0AA, 0xB08A, 0xB06A, 0xB04A, 0xB02A, 0xB00A, 0xAFEA, 0xAFCA, 0xAFAA, 0xAF8A,
   0xAF6A, 0xAF4A, 0xAF2A, 0xAF0A, 0xAEEA, 0xAECA, 0xAEAA, 0xAE8A, 0xAE6A, 0xAE4A, 0xAE2A, 0xAE0A, 0xADEA, 0xADCA, 0xADAA, 0xAD8A,
   0xAD6A, 0xAD4A, 0xAD2A, 0xAD0A, 0xACEA, 0xACCA, 0xACAA, 0xAC8A, 0xAC6A, 0xAC4A, 0xAC2A, 0xAC0A, 0xABEA, 0xABCA, 0xABAA, 0xAB8A,
   0xAB6A, 0xAB4A, 0xAB2A, 0xAB0A, 0xAAEA, 0xAACA, 0xAAAA, 0xAA8A, 0xAA6A, 0xAA4A, 0xAA2A, 0xAA0A, 0xA9EA, 0xA9CA, 0xA9AA, 0xA98A,
   0xA96A, 0xA94A, 0xA92A, 0xA90A, 0xA8EA, 0xA8CA, 0xA8AA, 0xA88A, 0xA86A, 0xA84A, 0xA82A, 0xA80A, 0xA7EA, 0xA7CA, 0xA7AA, 0xA78A,
   0xA76A, 0xA74A, 0xA72A, 0xA70A, 0xA6EA, 0xA6CA, 0xA6AA, 0xA68A, 0xA66A, 0xA64A, 0xA62A, 0xA60A, 0xA5EA, 0xA5CA, 0xA5AA, 0xA58A,
   0xA56A, 0xA54A, 0xA52A, 0xA50A, 0xA4EA, 0xA4CA, 0xA4AA, 0xA48A, 0xA46A, 0xA44A, 0xA42A, 0xA40A, 0xA3EA, 0xA3CA, 0xA3AA, 0xA38A,
   0xA36A, 0xA34A, 0xA32A, 0xA30A, 0xA2EA, 0xA2CA, 0xA2AA, 0xA28A, 0xA26A, 0xA24A, 0xA22A, 0xA20A, 0xA1EA, 0xA1CA, 0xA1AA, 0xA18A,
   0xA16A, 0xA14A, 0xA12A, 0xA10A, 0xA0EA, 0xA0CA, 0xA0AA, 0xA08A, 0xA06A, 0xA04A, 0xA02A, 0xA00A, 0x9FEA, 0x9FCA, 0x9FAA, 0x9F8A,
   0x9F6A, 0x9F4A, 0x9F2A, 0x9F0A, 0x9EEA, 0x9ECA, 0x9EAA, 0x9E8A, 0x9E6A, 0x9E4A, 0x9E2A, 0x9E0A, 0x9DEA, 0x9DCA, 0x9DAA, 0x9D8A,
   0x9D6A, 0x9D4A, 0x9D2A, 0x9D0A, 0x9CEA, 0x9CCA, 0x9CAA, 0x9C8A, 0x9C6A, 0x9C4A, 0x9C2A, 0x9C0A, 0x9BEA, 0x9BCA, 0x9BAA, 0x9B8A,
   0x9B6A, 0x9B4A, 0x9B2A, 0x9B0A, 0x9AEA, 0x9ACA, 0x9AAA, 0x9A8A, 0x9A6A, 0x9A4A, 0x9A2A, 0x9A0A, 0x99EA, 0x99CA, 0x99AA, 0x998A,
   0x996A, 0x994A, 0x992A, 0x990A, 0x98EA, 0x98CA, 0x98AA, 0x988A, 0x986A, 0x984A, 0x982A, 0x980A, 0x97EA, 0x97CA, 0x97AA, 0x978A,
   0x976A, 0x974A, 0x972A, 0x970A, 0x96EA, 0x96CA, 0x96AA, 0x968A, 0x966A, 0x964A, 0x962A, 0x960A, 0x95EA, 0x95CA, 0x95AA, 0x958A,
   0x956A, 0x954A, 0x952A, 0x950A, 0x94EA, 0x94CA, 0x94AA, 0x948A, 0x946A, 0x944A, 0x942A, 0x940A, 0x93EA, 0x93CA, 0x93AA, 0x938A,
   0x936A, 0x934A, 0x932A, 0x930A, 0x92EA, 0x92CA, 0x92AA, 0x928A, 0x926A, 0x924A, 0x922A, 0x920A, 0x91EA, 0x91CA, 0x91AA, 0x918A,
   0x916A, 0x914A, 0x912A, 0x910A, 0x90EA, 0x90CA, 0x90AA, 0x908A, 0x906A, 0x904A, 0x902A, 0x900A, 0x8FEA, 0x8FCA, 0x8FAA, 0x8F8A,
   0x8F6A, 0x8F4A, 0x8F2A, 0x8F0A, 0x8EEA, 0x8ECA, 0x8EAA, 0x8E8A, 0x8E6A, 0x8E4A, 0x8E2A, 0x8E0A, 0x8DEA, 0x8DCA, 0x8DAA, 0x8D8A,
   0x8D6A, 0x8D4A, 0x8D2A, 0x8D0A, 0x8CEA, 0x8CCA, 0x8CAA, 0x8C8A, 0x8C6A, 0x8C4A, 0x8C2A, 0x8C0A, 0x8BEA, 0x8BCA, 0x8BAA, 0x8B8A,
   0x8B6A, 0x8B4A, 0x8B2A, 0x8B0A, 0x8AEA, 0x8ACA, 0x8AAA, 0x8A8A, 0x8A6A, 0x8A4A, 0x8A2A, 0x8A0A, 0x89EA, 0x89CA, 0x89AA, 0x898A,
   0x896A, 0x894A, 0x892A, 0x890A, 0x88EA, 0x88CA, 0x88AA, 0x888A, 0x886A, 0x884A, 0x882A, 0x880A, 0x87EA, 0x87CA, 0x87AA, 0x878A,
   0x876A, 0x874A, 0x872A, 0x870A, 0x86EA, 0x86CA, 0x86AA, 0x868A, 0x866A, 0x864A, 0x862A, 0x860A, 0x85EA, 0x85CA, 0x85AA, 0x858A,
   0x856A, 0x854A, 0x852A, 0x850A, 0x84EA, 0x84CA, 0x84AA, 0x848A, 0x846A, 0x844A, 0x842A, 0x840A, 0x83EA, 0x83CA, 0x83AA, 0x838A,
   0x836A, 0x834A, 0x832A, 0x830A, 0x82EA, 0x82CA, 0x82AA, 0x828A, 0x826A, 0x824A, 0x822A, 0x820A, 0x81EA, 0x81CA, 0x81AA, 0x818A,
   0x816A, 0x814A, 0x812A, 0x810A, 0x80EA, 0x80CA, 0x80AA, 0x808A, 0x806A, 0x804A, 0x802A, 0x800A, 0x7FEA, 0x7FCA, 0x7FAA, 0x7F8A,
   0x7F6A, 0x7F4A, 0x7F2A, 0x7F0A, 0x7EEA, 0x7ECA, 0x7EAA, 0x7E8A, 0x7E6A, 0x7E4A, 0x7E2A, 0x7E0A, 0x7DEA, 0x7DCA, 0x7DAA, 0x7D8A,
   0x7D6A, 0x7D4A, 0x7D2A, 0x7D0A, 0x7CEA, 0x7CCA, 0x7CAA, 0x7C8A, 0x7C6A, 0x7C4A, 0x7C2A, 0x7C0A, 0x7BEA, 0x7BCA, 0x7BAA, 0x7B8A,
   0x7B6A, 0x7B4A, 0x7B2A, 0x7B0A, 0x7AEA, 0x7ACA, 0x7AAA, 0x7A8A, 0x7A6A, 0x7A4A, 0x7A2A, 0x7A0A, 0x79EA, 0x79CA, 0x79AA, 0x798A,
   0x796A, 0x794A, 0x792A, 0x790A, 0x78EA, 0x78CA, 0x78AA, 0x788A, 0x786A, 0x784A, 0x782A, 0x780A, 0x77EA, 0x77CA, 0x77AA, 0x778A,
   0x776A, 0x774A, 0x772A, 0x770A, 0x76EA, 0x76CA, 0x76AA, 0x768A, 0x766A, 0x764A, 0x762A, 0x760A, 0x75EA, 0x75CA, 0x75AA, 0x758A,
   0x756A, 0x754A, 0x752A, 0x750A, 0x74EA, 0x74CA, 0x74AA, 0x748A, 0x746A, 0x744A, 0x742A, 0x740A, 0x73EA, 0x73CA, 0x73AA, 0x738A,
   0x736A, 0x734A, 0x732A, 0x730A, 0x72EA, 0x72CA, 0x72AA, 0x728A, 0x726A, 0x724A, 0x722A, 0x720A, 0x71EA, 0x71CA, 0x71AA, 0x718A,
   0x716A, 0x714A, 0x712A, 0x710A, 0x70EA, 0x70CA, 0x70AA, 0x708A, 0x706A, 0x704A, 0x702A, 0x700A, 0x6FEA, 0x6FCA, 0x6FAA, 0x6F8A,
   0x6F6A, 0x6F4A, 0x6F2A, 0x6F0A, 0x6EEA, 0x6ECA, 0x6EAA, 0x6E8A, 0x6E6A, 0x6E4A, 0x6E2A, 0x6E0A, 0x6DEA, 0x6DCA, 0x6DAA, 0x6D8A,
   0x6D6A, 0x6D4A, 0x6D2A, 0x6D0A, 0x6CEA, 0x6CCA, 0x6CAA, 0x6C8A, 0x6C6A, 0x6C4A, 0x6C2A, 0x6C0A, 0x6BEA, 0x6BCA, 0x6BAA, 0x6B8A,
   0x6B6A, 0x6B4A, 0x6B2A, 0x6B0A, 0x6AEA, 0x6ACA, 0x6AAA, 0x6A8A, 0x6A6A, 0x6A4A, 0x6A2A, 0x6A0A, 0x69EA, 0x60F0, 0x6368, 0x6348,
   0x696A, 0x694A, 0x692A, 0x690A, 0x68EA, 0x68CA, 0x68AA, 0x688A, 0x686A, 0x684A, 0x682A, 0x680A, 0x67EA, 0x67CA, 0x67AA, 0x678A,
   0x676A, 0x674A, 0x672A, 0x670A, 0x66EA, 0x66CA, 0x66AA, 0x668A, 0x666A, 0x664A, 0x662A, 0x660A, 0x65EA, 0x65CA, 0x65AA, 0x658A,
   0x656A, 0x654A, 0x652A, 0x650A, 0x6B26, 0x6DE1, 0x6E9C, 0x5E48, 0x5E28, 0x5E08, 0x5DE8, 0x5DC8, 0x5DA8, 0x5D88, 0x5D68, 0x5D48,
   0x5D28, 0x5D08, 0x5CE8, 0x5CC8, 0x5CA8, 0x5C88, 0x5C68, 0x5C48, 0x5C28, 0x5C08, 0x5BE8, 0x5BC8, 0x5BA8, 0x5B88, 0x5B68, 0x5B48,
   0x5B28, 0x5B08, 0x5AE8, 0x5AC8, 0x5AA8, 0x5A88, 0x5A68, 0x5A48, 0x5A28, 0x5A08, 0x59E8, 0x59C8, 0x59A8, 0x5988, 0x5968, 0x5948,
   0x5928, 0x5908, 0x58E8, 0x58C8, 0x58A8, 0x5888, 0x5868, 0x5848, 0x5828, 0x5808, 0x57E8, 0x57C8, 0x57A8, 0x5788, 0x5768, 0x5748,
   0x5D6A, 0x5D4A, 0x5D2A, 0x5D0A, 0x5CEA, 0x5CCA, 0x5CAA, 0x5C8A, 0x5C6A, 0x5C4A, 0x5C2A, 0x5C0A, 0x5BEA, 0x5BCA, 0x5BAA, 0x5B8A,
   0x5B6A, 0x5B4A, 0x5B2A, 0x5B0A, 0x5AEA, 0x5ACA, 0x5AAA, 0x5A8A, 0x5A6A, 0x5A4A, 0x5A2A, 0x5A0A, 0x59EA, 0x59CA, 0x59AA, 0x598A,
   0x596A, 0x594A, 0x592A, 0x590A, 0x58EA, 0x58CA, 0x58AA, 0x588A, 0x586A, 0x584A, 0x582A, 0x580A, 0x57EA, 0x57CA, 0x57AA, 0x578A,
   0x576A, 0x574A, 0x572A, 0x570A, 0x56EA, 0x56CA, 0x56AA, 0x568A, 0x566A, 0x564A, 0x562A, 0x560A, 0x55EA, 0x55CA, 0x55AA, 0x558A,
   0x556A, 0x554A, 0x552A, 0x550A, 0x54EA, 0x54CA, 0x54AA, 0x548A, 0x546A, 0x544A, 0x542A, 0x540A, 0x53EA, 0x53CA, 0x53AA, 0x538A,
   0x536A, 0x534A, 0x532A, 0x530A, 0x52EA, 0x52CA, 0x52AA, 0x528A, 0x526A, 0x524A, 0x522A, 0x520A, 0x51EA, 0x51CA, 0x51AA, 0x518A,
   0x516A, 0x514A, 0x512A, 0x510A, 0x50EA, 0x50CA, 0x50AA, 0x508A, 0x506A, 0x504A, 0x502A, 0x500A, 0x4FEA, 0x4FCA, 0x4FAA, 0x4F8A,
   0x4F6A, 0x4F4A, 0x4F2A, 0x4F0A, 0x4EEA, 0x4ECA, 0x4EAA, 0x4E8A, 0x4E6A, 0x4E4A, 0x4E2A, 0x4E0A, 0x4DEA, 0x4DCA, 0x4DAA, 0x4D8A,
   0x4D6A, 0x4D4A, 0x4D2A, 0x4D0A, 0x4CEA, 0x4CCA, 0x4CAA, 0x4C8A, 0x4C6A, 0x4C4A, 0x4C2A, 0x4C0A, 0x4BEA, 0x4BCA, 0x4BAA, 0x4B8A,
   0x4B6A, 0x4B4A, 0x4B2A, 0x4B0A, 0x4AEA, 0x4ACA, 0x4AAA, 0x4A8A, 0x4A6A, 0x4A4A, 0x4A2A, 0x4A0A, 0x49EA, 0x49CA, 0x49AA, 0x498A,
   0x496A, 0x494A, 0x492A, 0x490A, 0x48EA, 0x48CA, 0x48AA, 0x488A, 0x486A, 0x484A, 0x482A, 0x480A, 0x47EA, 0x47CA, 0x47AA, 0x478A,
   0x476A, 0x474A, 0x472A, 0x470A, 0x46EA, 0x46CA, 0x46AA, 0x468A, 0x466A, 0x464A, 0x462A, 0x460A, 0x45EA, 0x45CA, 0x45AA, 0x458A,
   0x456A, 0x454A, 0x452A, 0x450A, 0x44EA, 0x44CA, 0x44AA, 0x448A, 0x446A, 0x444A, 0x442A, 0x440A, 0x43EA, 0x43CA, 0x43AA, 0x438A,
   0x436A, 0x434A, 0x432A, 0x430A, 0x42EA, 0x42CA, 0x42AA, 0x428A, 0x426A, 0x424A, 0x422A, 0x420A, 0x41EA, 0x41CA, 0x41AA, 0x418A,
   0x416A, 0x414A, 0x412A, 0x410A, 0x40EA, 0x40CA, 0x40AA, 0x408A, 0x406A, 0x404A, 0x402A, 0x400A, 0x3FEA, 0x3FCA, 0x3FAA, 0x3F8A,
   0x3F6A, 0x3F4A, 0x3F2A, 0x3F0A, 0x3EEA, 0x3ECA, 0x3EAA, 0x3E8A, 0x3E6A, 0x3E4A, 0x3E2A, 0x3E0A, 0x3DEA, 0x3DCA, 0x3DAA, 0x3D8A,
   0x3D6A, 0x3D4A, 0x3D2A, 0x3D0A, 0x3CEA, 0x3CCA, 0x3CAA, 0x3C8A, 0x3C6A, 0x3C4A, 0x3C2A, 0x3C0A, 0x3BEA, 0x3BCA, 0x3BAA, 0x3B8A,
   0x3B6A, 0x3B4A, 0x3B2A, 0x3B0A, 0x3AEA, 0x3ACA, 0x3AAA, 0x3A8A, 0x3A6A, 0x3A4A, 0x3A2A, 0x3A0A, 0x39EA, 0x39CA, 0x39AA, 0x398A,
   0x396A, 0x394A, 0x392A, 0x390A, 0x38EA, 0x38CA, 0x38AA, 0x388A, 0x386A, 0x384A, 0x382A, 0x380A, 0x37EA, 0x37CA, 0x37AA, 0x378A,
   0x376A, 0x374A, 0x372A, 0x370A, 0x36EA, 0x36CA, 0x36AA, 0x368A, 0x366A, 0x364A, 0x362A, 0x360A, 0x35EA, 0x35CA, 0x35AA, 0x358A,
   0x356A, 0x354A, 0x352A, 0x350A, 0x34EA, 0x34CA, 0x34AA, 0x348A, 0x346A, 0x344A, 0x342A, 0x340A, 0x33EA, 0x33CA, 0x33AA, 0x338A,
   0x336A, 0x334A, 0x332A, 0x330A, 0x32EA, 0x32CA, 0x32AA, 0x328A, 0x326A, 0x324A, 0x322A, 0x320A, 0x31EA, 0x28F2, 0x2B68, 0x2B48,
   0x3C2B, 0x3C0B, 0x3BEB, 0x3BCB, 0x3BAB, 0x3B8B, 0x3B6B, 0x3B4B, 0x3B2B, 0x3B0B, 0x3AEB, 0x3ACB, 0x3AAB, 0x3A8B, 0x3A6B, 0x3A4B,
   0x3A2B, 0x3A0B, 0x39EB, 0x39CB, 0x39AB, 0x398B, 0x396B, 0x394B, 0x392B, 0x390B, 0x38EB, 0x38CB, 0x38AB, 0x388B, 0x386B, 0x384B,
   0x382B, 0x380B, 0x37EB, 0x37CB, 0x37AB, 0x378B, 0x376B, 0x374B, 0x372B, 0x370B, 0x36EB, 0x36CB, 0x36AB, 0x368B, 0x366B, 0x364B,
   0x362B, 0x360B, 0x35EB, 0x35CB, 0x35AB, 0x358B, 0x356B, 0x354B, 0x352B, 0x350B, 0x34EB, 0x34CB, 0x34AB, 0x348B, 0x346B, 0x344B,
   0x344B, 0x342B, 0x340B, 0x33EB, 0x33CB, 0x33AB, 0x338B, 0x336B, 0x334B, 0x332B, 0x330B, 0x32EB, 0x32CB, 0x32AB, 0x328B, 0x326B,
   0x324B, 0x322B, 0x320B, 0x31EB, 0x31CB, 0x31AB, 0x318B, 0x316B, 0x314B, 0x312B, 0x310B, 0x30EB, 0x30CB, 0x30AB, 0x308B, 0x306B,
   0x304B, 0x302B, 0x300B, 0x2FEB, 0x2FCB, 0x2FAB, 0x2F8B, 0x2F6B, 0x2F4B, 0x2F2B, 0x2F0B, 0x2EEB, 0x2ECB, 0x2EAB, 0x2E8B, 0x2E6B,
   0x2E4B, 0x2E2B, 0x2E0B, 0x2DEB, 0x2DCB, 0x2DAB, 0x2D8B, 0x2D6B, 0x2D4B, 0x2D2B, 0x2D0B, 0x2CEB, 0x2CCB, 0x2CAB, 0x2C8B, 0x2C6B,
   0x2C4B, 0x2C2B, 0x2C0B, 0x2BEB, 0x2BCB, 0x2BAB, 0x2B8B, 0x2B6B, 0x2B4B, 0x2B2B, 0x2B0B, 0x2AEB, 0x2ACB, 0x2AAB, 0x2A8B, 0x2A6B,
   0x2A4B, 0x2A2B, 0x2A0B, 0x29EB, 0x29CB, 0x29AB, 0x298B, 0x296B, 0x294B, 0x292B, 0x290B, 0x28EB, 0x28CB, 0x28AB, 0x288B, 0x286B,
   0x284B, 0x282B, 0x280B, 0x27EB, 0x27CB, 0x27AB, 0x278B, 0x276B, 0x274B, 0x272B, 0x270B, 0x26EB, 0x26CB, 0x26AB, 0x268B, 0x266B,
   0x264B, 0x262B, 0x260B, 0x25EB, 0x25CB, 0x25AB, 0x258B, 0x256B, 0x254B, 0x252B, 0x250B, 0x24EB, 0x24CB, 0x24AB, 0x248B, 0x246B,
   0x244B, 0x242B, 0x240B, 0x23EB, 0x23CB, 0x23AB, 0x238B, 0x236B, 0x234B, 0x232B, 0x230B, 0x22EB, 0x22CB, 0x22AB, 0x228B, 0x226B,
   0x224B, 0x222B, 0x220B, 0x21EB, 0x21CB, 0x21AB, 0x218B, 0x216B, 0x214B, 0x212B, 0x210B, 0x20EB, 0x20CB, 0x20AB, 0x208B, 0x206B,
   0x204B, 0x202B, 0x200B, 0x1FEB, 0x1FCB, 0x1FAB, 0x1F8B, 0x1F6B, 0x1F4B, 0x1F2B, 0x1F0B, 0x1EEB, 0x1ECB, 0x1EAB, 0x1E8B, 0x1E6B,
   0x1E4B, 0x1E2B, 0x1E0B, 0x1DEB, 0x1DCB, 0x1DAB, 0x1D8B, 0x1D6B, 0x1D4B, 0x1D2B, 0x1D0B, 0x1CEB, 0x1CCB, 0x1CAB, 0x1C8B, 0x1C6B,
   0x1C4B, 0x1C2B, 0x1C0B, 0x1BEB, 0x1BCB, 0x1BAB, 0x1B8B, 0x1B6B, 0x106A, 0x104A, 0x102A, 0x100A, 0xFEA, 0xFCA, 0xFAA, 0xF8A,
   0xF6A, 0x668, 0x8E8, 0x8C8, 0x8A8, 0x888, 0x868, 0x848, 0x7D7, 0x194B, 0x7B6, 0xD1C, 0xCFC, 0xCB2, 0xCA9, 0xC9C,
   0xC7C, 0xC5C, 0xC3C, 0xC1C, 0xBFC, 0xBDC, 0xBBC, 0xB9C, 0xB7C, 0xB5E, 0xB2C, 0xB1C, 0xAB8, 0xADC, 0xA9C, 0x2C2,
   0x528, 0x166B, 0x1667, 0x3FF, 0x9FC, 0x9DC, 0x9BC, 0x659, 0xBB8, 0x15A7, 0xFC6, 0x1C0, 0x1B1, 0x9CB, 0x82C, 0x1285,
};

static uint16_t Data[] = {
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082,
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3E80, 0x3E80, 0x3001, 0x3082, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x3E80, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5198, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x3E80, 0x3E80, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202,
   0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202, 0x5202,
   0x5202, 0x2E82, 0x3E80, 0x5198, 0x2A14, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4686, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x1A1B, 0x1A1B, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4584, 0x3E80, 0x3E80, 0x3E80, 0x298,
   0x3E80, 0x298, 0x6615, 0x6696, 0x298, 0x1A97, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x4584, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x4584,
   0x4584, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x4584,
   0x4584, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x2E82,
   0x7282, 0x2E82, 0x3E80, 0x2E82, 0x4902, 0x7481, 0x7481, 0x7481, 0x7481, 0x7383, 0x1A1B, 0x1A1B, 0x1A1B, 0x6D82, 0x6D82, 0x4902,
   0x4902, 0x3E80, 0x3E80, 0x2E82, 0x4902, 0x6E01, 0x6E01, 0x7501, 0x7501, 0x3E80, 0x1A1B, 0x1A1B, 0x1A1B, 0x1B02, 0x1B82, 0x1C02,
   0x1C82, 0x1D02, 0x1D82, 0x1E02, 0x1E82, 0x1F02, 0x1F82, 0x2002, 0x2082, 0x2102, 0x2182, 0x2202, 0x2282, 0x2302, 0x2382, 0x2402,
   0x2482, 0x2502, 0x2582, 0x2602, 0x2682, 0x2702, 0x2782, 0x455, 0xC99, 0x4D6, 0xC99, 0xF, 0xF, 0xF, 0xF, 0xF,
   0x10F, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF,
   0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0x8F, 0x10F, 0x8F, 0x18F, 0x10F,
   0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0x10F, 0x10F,
   0x10F, 0x8F, 0x20C, 0x298, 0x298, 0x318, 0x39A, 0x318, 0x298, 0x298, 0x455, 0x4D6, 0x298, 0x519, 0x598, 0x614,
   0x598, 0x698, 0x709, 0x789, 0x809, 0x889, 0x909, 0x989, 0xA09, 0xA89, 0xB09, 0xB89, 0x598, 0x298, 0xC59, 0xC99,
   0xC59, 0x298, 0xD01, 0xD81, 0xE01, 0xE81, 0xF01, 0xF81, 0x1001, 0x1081, 0x1101, 0x1181, 0x1201, 0x1281, 0x1301, 0x1381,
   0x1401, 0x1481, 0x1501, 0x1581, 0x1601, 0x1681, 0x1701, 0x1781, 0x1801, 0x1881, 0x1901, 0x1981, 0x455, 0x298, 0x4D6, 0x1A1B,
   0x1A97, 0x298, 0x298, 0x298, 0xC99, 0x455, 0x4D6, 0x3E80, 0x298, 0x298, 0x298, 0x298, 0x298, 0x298, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x282C, 0x298, 0x39A, 0x39A, 0x39A, 0x39A, 0x289C, 0x289C, 0x1A1B, 0x289C, 0x2902, 0x29DD, 0xC99, 0x2A14, 0x289C, 0x1A1B,
   0x2A9C, 0x519, 0x2B0B, 0x2B8B, 0x1A1B, 0x2C02, 0x289C, 0x298, 0x1A1B, 0x2C8B, 0x2902, 0x2D5E, 0x2D8B, 0x2D8B, 0x2D8B, 0x298,
   0x298, 0x519, 0x614, 0xC99, 0xC99, 0xC99, 0x3E80, 0x298, 0x39A, 0x318, 0x298, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5405,
   0x5405, 0x5405, 0x3E80, 0x5405, 0x3E80, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x501C, 0x501C, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81,
   0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x4F81, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01,
   0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0xC99,
   0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E82, 0x2E82, 0x2E82, 0x4902, 0x4902, 0x2E82, 0x2E82, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x2E82, 0x2E82, 0x2E82, 0x2E82, 0x2E82, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5305, 0x4606, 0x5305, 0x5305, 0x3E80, 0x5305, 0x5305, 0x3E80, 0x5305, 0x5305, 0x5305, 0x5305,
   0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5398, 0x5405, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x5087, 0x5087, 0x4606, 0x5087, 0x5087, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x2D8B, 0x2D8B, 0x2D8B, 0x2D8B, 0x2D8B, 0x2D8B, 0x2D8B, 0x2D8B,
   0x2D8B, 0x2D8B, 0x2D8B, 0x2D8B, 0x840B, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x2E82, 0x3001,
   0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001,
   0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x4606,
   0x4606, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x1A1B,
   0x1A1B, 0x4701, 0x298, 0x4781, 0x4781, 0x4781, 0x3E80, 0x4801, 0x3E80, 0x4881, 0x4881, 0x4902, 0x2E01, 0x2E01, 0x2E01, 0x2E01,
   0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2F02, 0x2F02, 0x2F02, 0x2F02,
   0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02,
   0x2F02, 0x2F02, 0x2F02, 0xC99, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F82, 0x2F02, 0x2F02, 0x4A82, 0x2F02,
   0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x4B02, 0x4B82, 0x4B82, 0x3E80, 0x4C02, 0x4C82, 0x4D01, 0x4D01,
   0x4D01, 0x4D82, 0x4E02, 0x2902, 0x3E80, 0x3E80, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082,
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x2E82, 0x3B81, 0x3C03, 0x3C82, 0x3001, 0x3082, 0x3D81, 0x3E01, 0x3001, 0x3082,
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3101, 0x3182,
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x2902, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001,
   0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x4E82, 0x4F02, 0x3D02, 0x2902, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B10, 0x5B10, 0x5B10, 0x5B10, 0x5B10, 0x5B10, 0x7F0B, 0x3E80, 0x3E80,
   0x3E80, 0x7F8B, 0x800B, 0x808B, 0x810B, 0x818B, 0x820B, 0x519, 0x519, 0xC99, 0x455, 0x4D6, 0x2902, 0x3301, 0x3001, 0x3082,
   0x3001, 0x3082, 0x3381, 0x3001, 0x3082, 0x3401, 0x3401, 0x3001, 0x3082, 0x2902, 0x3481, 0x3501, 0x3581, 0x3001, 0x3082, 0x3401,
   0x3601, 0x3682, 0x3701, 0x3781, 0x3001, 0x3082, 0x2902, 0x2902, 0x3701, 0x3801, 0x2902, 0x3881, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3B81, 0x3C03, 0x3C82, 0x3B81, 0x3C03, 0x3C82, 0x3B81, 0x3C03, 0x3C82, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001,
   0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3D02, 0x3001, 0x3082, 0x501C, 0x4606, 0x4606, 0x4606,
   0x4606, 0x3E80, 0x5087, 0x5087, 0x3E80, 0x3E80, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082,
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3201, 0x3001,
   0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3282, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3901, 0x3001, 0x3082, 0x3901,
   0x2902, 0x2902, 0x3001, 0x3082, 0x3901, 0x3001, 0x3082, 0x3981, 0x3981, 0x3001, 0x3082, 0x3001, 0x3082, 0x3A01, 0x3001, 0x3082,
   0x2902, 0x3A85, 0x3001, 0x3082, 0x2902, 0x3B02, 0x4D01, 0x3001, 0x3082, 0x3001, 0x3082, 0x3E80, 0x3E80, 0x3001, 0x3082, 0x3E80,
   0x3E80, 0x3001, 0x3082, 0x3E80, 0x3E80, 0x3E80, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082,
   0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x598, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x5398, 0x3E80, 0x3E80, 0x3E80, 0x5398, 0x5398, 0x5398, 0x5398, 0x5398, 0x5398, 0x5398, 0x5398, 0x5398,
   0x5398, 0x5398, 0x5398, 0x5398, 0x5398, 0x3E80, 0x5B10, 0x5405, 0x4606, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x3E80, 0x3E80, 0x5B10, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01,
   0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01,
   0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x4D01, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80,
   0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x2902, 0x2902, 0x2902, 0x3F02, 0x3F82, 0x2902, 0x4002, 0x4002, 0x2902, 0x4082,
   0x2902, 0x4102, 0x2902, 0x2902, 0x2902, 0x2902, 0x4002, 0x2902, 0x2902, 0x4182, 0x2902, 0x2902, 0x2902, 0x2902, 0x4202, 0x4282,
   0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x4282, 0x2902, 0x2902, 0x4302, 0x2902, 0x2902, 0x4382, 0x2902, 0x2902, 0x2902, 0x2902,
   0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x4402, 0x2902, 0x2902, 0x4402, 0x2902, 0x2902, 0x2902, 0x2902, 0x4402, 0x2902,
   0x4482, 0x4482, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x4502, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902,
   0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x2902, 0x3E80, 0x3E80, 0x4584, 0x4584, 0x4584, 0x4584, 0x4584, 0x4584, 0x4584,
   0x4584, 0x4584, 0x1A1B, 0x1A1B, 0x4584, 0x4584, 0x4584, 0x4584, 0x4584, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B, 0x1A1B,
   0x1A1B, 0x1A1B, 0x4584, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101,
   0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x5101, 0x3E80, 0x3E80, 0x4584, 0x5198, 0x5198,
   0x5198, 0x5198, 0x5198, 0x5198, 0x2E01, 0x2E01, 0x3E80, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01, 0x2E01,
   0x4982, 0x4A02, 0x4A02, 0x4A02, 0x4902, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02,
   0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x2F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02,
   0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4F02, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x5198, 0x4606, 0x4606, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x4606, 0x4606, 0x4606, 0x5298, 0x4606, 0x4606, 0x5298, 0x4606, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305,
   0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5305, 0x5305,
   0x5305, 0x5298, 0x5298, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5C89, 0x5D09,
   0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x640B, 0x648B, 0x650B, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x5B88, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85,
   0x3E80, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x3E80, 0x4606, 0x4606, 0x4606, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606,
   0x5B88, 0x5B88, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3E80,
   0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5C09, 0x5C89, 0x5D09,
   0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x501C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5509, 0x5589, 0x5609, 0x5689, 0x5709, 0x5789, 0x5809, 0x5889, 0x5909,
   0x5989, 0x318, 0x5A18, 0x5A18, 0x5398, 0x3E80, 0x3E80, 0x4606, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x3E80, 0x3E80, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x6615, 0x6696, 0x5484, 0x5405,
   0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x5405, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x5198, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x5B88, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x5198, 0x5198, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x5484, 0x5484,
   0x4606, 0x4606, 0x289C, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x709, 0x789, 0x809, 0x889, 0x909, 0x989, 0xA09,
   0xA89, 0xB09, 0xB89, 0x5405, 0x5405, 0x5405, 0x5A9C, 0x5A9C, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3A85,
   0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x4606, 0x3A85, 0x3A85, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x3E80, 0x4606, 0x4606, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x5B88, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x4606, 0x3A85, 0x5B88, 0x5B88,
   0x5B88, 0x5B88, 0x5B88, 0x3E80, 0x4606, 0x5B88, 0x5B88, 0x3E80, 0x5B88, 0x5B88, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3E80, 0x5198, 0x5198,
   0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x640B,
   0x670B, 0x678B, 0x680B, 0x688B, 0x690B, 0x698B, 0x6A0B, 0x6A8B, 0x648B, 0x6B0B, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85,
   0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x4606, 0x3A85, 0x5B88, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3A85, 0x4606, 0x4606, 0x4606, 0x4606,
   0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x4606,
   0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x4606, 0x3A85, 0x3A85, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A,
   0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x39A, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x4606, 0x4606, 0x5198, 0x5198, 0x5C09,
   0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x5198, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x298, 0x298, 0x318, 0x39A, 0x318, 0x298, 0x298,
   0x6615, 0x6696, 0x298, 0x519, 0x598, 0x614, 0x598, 0x698, 0x709, 0x789, 0x809, 0x889, 0x909, 0x989, 0xA09, 0xA89,
   0xB09, 0xB89, 0x598, 0x298, 0xC99, 0xC99, 0xC99, 0x298, 0x298, 0x298, 0x298, 0x298, 0x298, 0x2A14, 0x298, 0x298,
   0x298, 0x298, 0x5B10, 0x5B10, 0x5B10, 0x5B10, 0x3E80, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009,
   0x6089, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x3E80, 0x3E80,
   0x5B88, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3E80, 0x3A85, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85,
   0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x4606, 0x3E80, 0x5B88, 0x5B88, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606,
   0x4606, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x5C09, 0x5C89,
   0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x3A85, 0x3A85, 0x39A, 0x39A, 0x610B, 0x618B, 0x620B, 0x628B,
   0x630B, 0x638B, 0x501C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x4606, 0x3A85, 0x5B88, 0x5B88, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x4606, 0x4606,
   0x5B88, 0x3E80, 0x5B88, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009,
   0x6089, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x501C, 0x4606, 0x501C, 0x4606, 0x501C,
   0x4606, 0x6615, 0x6696, 0x6615, 0x6696, 0x5B88, 0x5B88, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x3E80,
   0x3E80, 0x5B88, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x5B88, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x5B88, 0x4606,
   0x4606, 0x4606, 0x4606, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x5B88, 0x5B88, 0x5B88, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x5B88, 0x5B88, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x4584, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80, 0x3E80, 0x5C09,
   0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x5087, 0x5087, 0x5087, 0x5B88, 0x4606, 0x4606, 0x4606, 0x3E80,
   0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x5B88, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x4606, 0x4606, 0x4606,
   0x3E80, 0x4606, 0x3E80, 0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x5B88, 0x4606, 0x5B88, 0x5B88, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x4606, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198,
   0x39A, 0x5198, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x4584, 0x4606, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x4606, 0x5198, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009, 0x6089, 0x5198,
   0x5198, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x501C, 0x501C, 0x501C, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198,
   0x5198, 0x65B8, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x4606, 0x4606, 0x501C,
   0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x4606, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x3E80, 0x3E80, 0x501C, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x1A97, 0x4584, 0x4584, 0x4584, 0x3E80, 0x5C09, 0x5C89, 0x5D09, 0x5D89, 0x5E09, 0x5E89, 0x5F09, 0x5F89, 0x6009,
   0x6089, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x5198, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x5B88, 0x5B88, 0x4606,
   0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x20C, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x6615, 0x6696, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x5198, 0x5198, 0x5198, 0x6B8B, 0x6C0B, 0x6C8B, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x4606, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001,
   0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x3001, 0x3082, 0x2E82, 0x2E82, 0x2E82,
   0x2E82, 0x2E82, 0x6D02, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6E01,
   0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6E01,
   0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x3E80, 0x3E80, 0x6E01,
   0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x3E80, 0x3E80, 0x2E82, 0x6D82, 0x4902, 0x6D82, 0x4902, 0x6D82, 0x4902, 0x6D82, 0x3E80,
   0x6E01, 0x3E80, 0x6E01, 0x3E80, 0x6E01, 0x3E80, 0x6E01, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6D82, 0x6E01,
   0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E01, 0x6E82, 0x6E82, 0x6F02, 0x6F02, 0x6F02, 0x6F02, 0x6F82, 0x6F82, 0x7002,
   0x7002, 0x7082, 0x7082, 0x7102, 0x7102, 0x3E80, 0x3E80, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7203,
   0x7203, 0x7203, 0x7203, 0x7203, 0x7203, 0x7203, 0x7203, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7182, 0x7203,
   0x7203, 0x7203, 0x7203, 0x7203, 0x7203, 0x7203, 0x7203, 0x6D82, 0x6D82, 0x2E82, 0x7282, 0x2E82, 0x3E80, 0x2E82, 0x4902, 0x6E01,
   0x6E01, 0x7301, 0x7301, 0x7383, 0x1A1B, 0x7402, 0x1A1B, 0x1B02, 0x1B82, 0x1C02, 0x1C82, 0x1D02, 0x1D82, 0x1E02, 0x1E82, 0x1F02,
   0x1F82, 0x2002, 0x2082, 0x2102, 0x2182, 0x2202, 0x2282, 0x2302, 0x2382, 0x2402, 0x2482, 0x2502, 0x2582, 0x2602, 0x2682, 0x2702,
   0x2782, 0x6615, 0xC99, 0x6696, 0xC99, 0x3E80, 0x6D82, 0x6D82, 0x4902, 0x4902, 0x2E82, 0x7582, 0x2E82, 0x4902, 0x6E01, 0x6E01,
   0x7601, 0x7601, 0x7681, 0x1A1B, 0x1A1B, 0x1A1B, 0x3E80, 0x3E80, 0x2E82, 0x7282, 0x2E82, 0x3E80, 0x2E82, 0x4902, 0x7701, 0x7701,
   0x7781, 0x7781, 0x7383, 0x1A1B, 0x1A1B, 0x3E80, 0x20C, 0x20C, 0x20C, 0x20C, 0x20C, 0x20C, 0x20C, 0x782C, 0x20C, 0x20C,
   0x20C, 0x788C, 0x5B10, 0x5B10, 0x7910, 0x7990, 0x2A14, 0x7A34, 0x2A14, 0x2A14, 0x2A14, 0x2A14, 0x298, 0x298, 0x7A9D, 0x7B1E,
   0x6615, 0x7A9D, 0x7A9D, 0x7B1E, 0x6615, 0x7A9D, 0x298, 0x298, 0x298, 0x298, 0x298, 0x298, 0x298, 0x298, 0x7B8D, 0x7C0E,
   0x7C90, 0x7D10, 0x7D90, 0x7E10, 0x7E90, 0x782C, 0x318, 0x318, 0x318, 0x318, 0x318, 0x298, 0x298, 0x298, 0x298, 0x29DD,
   0x2D5E, 0x298, 0x298, 0x298, 0x298, 0x1A97, 0x7F0B, 0x2C8B, 0x2B0B, 0x2B8B, 0x7F8B, 0x800B, 0x808B, 0x810B, 0x818B, 0x820B,
   0x519, 0x519, 0xC99, 0x455, 0x4D6, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x4D01, 0x289C, 0x289C, 0x289C, 0x289C, 0x4D01, 0x289C, 0x289C, 0x2902, 0x4D01,
   0x4D01, 0x4D01, 0x2902, 0x2902, 0x4D01, 0x4D01, 0x4D01, 0x2902, 0x289C, 0x4D01, 0x289C, 0x289C, 0x289C, 0x4D01, 0x4D01, 0x4D01,
   0x4D01, 0x4D01, 0x289C, 0x289C, 0xA20A, 0xA28A, 0xA30A, 0xA38A, 0xA40A, 0xA48A, 0xA50A, 0xA58A, 0xA60A, 0x4606, 0x4606, 0x4606,
   0x4606, 0x4606, 0x4606, 0x2A14, 0x4584, 0x4584, 0x4584, 0x4584, 0x4584, 0x289C, 0x289C, 0xA68A, 0xA70A, 0xA78A, 0x3E80, 0x3E80,
   0x3E80, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0xC99, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0xC99, 0xC99, 0x289C, 0x289C, 0xC99, 0x289C, 0xC99, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0xC99, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x948A, 0x950A, 0x958A, 0x960A, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0xC99, 0xC99, 0x289C, 0x289C, 0x289C, 0x289C, 0x4D01, 0x289C, 0x8281, 0x289C, 0x4D01, 0x289C, 0x8301,
   0x8381, 0x4D01, 0x4D01, 0x2A9C, 0x2902, 0x4D01, 0x4D01, 0x289C, 0x4D01, 0x2902, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x2902, 0x289C,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x848A, 0x850A, 0x858A, 0x860A, 0x868A, 0x870A, 0x878A, 0x880A, 0x888A, 0x890A, 0x898A,
   0x8A0A, 0x8A8A, 0x8B0A, 0x8B8A, 0x8C0A, 0x8C8A, 0x8D0A, 0x8D8A, 0x8E0A, 0x8E8A, 0x8F0A, 0x8F8A, 0x900A, 0x908A, 0x910A, 0x918A,
   0x920A, 0x928A, 0x930A, 0x938A, 0x940A, 0xC99, 0xC99, 0xC59, 0xC59, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59,
   0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC99,
   0xC99, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99,
   0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC99, 0xC59, 0xC59, 0xC99, 0xC99, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59,
   0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59,
   0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC59, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0xC99, 0x289C, 0x289C, 0xC99,
   0x289C, 0x289C, 0xC99, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0xC99, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0xC99, 0xC59, 0xC59,
   0xC59, 0xC59, 0xC99, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC99, 0xC99, 0xC59, 0x519,
   0x519, 0xC99, 0xC59, 0xC59, 0xC99, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC59, 0xC99, 0xC59, 0xC99,
   0xC99, 0xC99, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99, 0xC99, 0xC99, 0xC99,
   0xC99, 0xC59, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x455,
   0x4D6, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x501C, 0x501C, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x9C1C, 0x9C1C, 0x9C1C,
   0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C1C, 0x9C9C, 0x9C9C, 0x9C9C,
   0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x9C9C, 0x7F0B, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0xC59, 0xC99, 0xC59, 0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC99,
   0xC99, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59, 0xC59,
   0xC59, 0xC59, 0xC59, 0xC99, 0xC99, 0xC59, 0xC59, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x39A, 0x39A, 0xC99, 0x1A1B, 0x289C, 0x39A, 0x39A, 0x3E80, 0x289C, 0xC99, 0xC99,
   0xC99, 0xC99, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x5B10, 0x5B10,
   0x5B10, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x289C, 0x3E80, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x289C, 0x3E80,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x840B, 0x9D0B, 0x9D8B, 0x9E0B, 0x9E8B, 0x9F0B, 0x9F8B, 0xA00B, 0xA08B, 0xA10B, 0x840B,
   0x9D0B, 0x9D8B, 0x9E0B, 0x9E8B, 0x9F0B, 0x9F8B, 0xA00B, 0xA08B, 0xA10B, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0xC59, 0xC59, 0xC59, 0xC59, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x289C, 0x501C, 0x289C,
   0x289C, 0x289C, 0x289C, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B, 0x630B,
   0x630B, 0x630B, 0x630B, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x3E80, 0x3E80, 0x3E80, 0x501C, 0x610B, 0x618B, 0x620B, 0x628B, 0xA80B, 0xA88B, 0xA90B, 0xA98B, 0xAA0B,
   0x640B, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x289C, 0x3E80, 0x289C, 0x289C,
   0x289C, 0x3E80, 0x289C, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x2C8B, 0x2B0B, 0x2B8B, 0x7F8B,
   0x800B, 0x808B, 0x810B, 0x818B, 0x820B, 0x968B, 0x970B, 0x978B, 0x980B, 0x988B, 0x990B, 0x998B, 0x9A0B, 0x9A8B, 0x9B0B, 0x9B8B,
   0x2C8B, 0x2B0B, 0x2B8B, 0x7F8B, 0x800B, 0x808B, 0x810B, 0x818B, 0x820B, 0x968B, 0x970B, 0x978B, 0x980B, 0x988B, 0x990B, 0x998B,
   0x9A0B, 0x9A8B, 0x9B0B, 0x9B8B, 0x501C, 0x501C, 0x501C, 0x501C, 0x20C, 0x298, 0x298, 0x298, 0x289C, 0x4584, 0x3A85, 0xA18A,
   0x455, 0x4D6, 0x455, 0x4D6, 0x455, 0x4D6, 0x455, 0x4D6, 0x455, 0x4D6, 0x289C, 0x289C, 0x455, 0x4D6, 0x455, 0x4D6,
   0x455, 0x4D6, 0x455, 0x4D6, 0x2A14, 0x6615, 0x6696, 0x6696, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x4606, 0x4606, 0x1A1B, 0x1A1B, 0x4584, 0x4584, 0x3E80, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85,
   0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3A85, 0x3E80, 0x501C, 0x501C, 0x630B, 0x630B, 0x630B, 0x630B, 0x501C, 0x501C,
   0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x501C, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93,
   0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93,
   0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAA93, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12,
   0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12,
   0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0xAB12, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305,
   0x5305, 0x5305, 0x5305, 0x5305, 0x519, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305,
   0x5305, 0x5305, 0x3E80, 0x5305, 0x5305, 0x5305, 0x5305, 0x5305, 0x3E80, 0x5305, 0x3E80, 0x4606, 0x4606, 0x4606, 0x4606, 0x3E80,
   0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x3E80, 0x298, 0x2A14, 0x2A14, 0x1A97, 0x1A97,
   0x6615, 0x6696, 0x6615, 0x6696, 0x6615, 0x6696, 0x6615, 0x6696, 0x6615, 0x6696, 0x6615, 0x6696, 0x3E80, 0x3E80, 0x3E80, 0x3E80,
   0x298, 0x298, 0x298, 0x298, 0x1A97, 0x1A97, 0x1A97, 0x598, 0x298, 0x598, 0x3E80, 0x298, 0x598, 0x298, 0x298, 0x2A14,
   0x6615, 0x6696, 0x6615, 0x6696, 0x6615, 0x6696, 0x318, 0x298, 0xD01, 0xD81, 0xE01, 0xE81, 0xF01, 0xF81, 0x1001, 0x1081,
   0x1101, 0x1181, 0x1201, 0x1281, 0x1301, 0x1381, 0x1401, 0x1481, 0x1501, 0x1581, 0x1601, 0x1681, 0x1701, 0x1781, 0x1801, 0x1881,
   0x1901, 0x1981, 0x6615, 0x298, 0x6696, 0x1A1B, 0x1A97,
};

static int16_t Upper[] = {
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0,
   0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0, 0xFFE0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2E7, 0x0, 0x0, 0x0, 0x0, 0x0, 0xFFE0, 0x79,
   0x0, 0xFFFF, 0x0, 0xFF18, 0x0, 0xFED4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x61, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x38, 0x0, 0xFFFF, 0xFFFE, 0xFFB1, 0x0, 0x0, 0x0, 0xFF2E, 0xFF32,
   0xFF33, 0xFF36, 0xFF35, 0xFF31, 0xFF2F, 0xFF2D, 0xFF2B, 0xFF2A, 0xFF26, 0xFF27, 0xFF25, 0x0, 0x0, 0x54, 0x0, 0x0,
   0x0, 0x0, 0x0, 0xFFDA, 0xFFDB, 0xFFE1, 0xFFC0, 0xFFC1, 0xFFC2, 0xFFC7, 0x0, 0xFFD1, 0xFFCA, 0xFFAA, 0xFFB0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0xFFD0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xFFC5, 0x8, 0x0, 0x4A, 0x56, 0x64,
   0x80, 0x70, 0x7E, 0x8, 0x0, 0x9, 0x0, 0x0, 0xE3DB, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0,
   0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0xFFF0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xFFE6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
};

static int16_t Lower[] = {
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x20, 0x0, 0x0, 0x0,
   0x1, 0x0, 0xFF39, 0x0, 0xFF87, 0x0, 0xD2, 0xCE, 0xCD, 0x4F, 0xCA, 0xCB, 0xCF, 0x0, 0xD3, 0xD1,
   0xD5, 0xD6, 0xDA, 0xD9, 0xDB, 0x0, 0x0, 0x2, 0x1, 0x0, 0x0, 0xFF9F, 0xFFC8, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x26, 0x25,
   0x40, 0x3F, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x50,
   0x0, 0x0, 0x30, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xFFF8, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0xFFF8, 0x0, 0xFFB6, 0xFFF7, 0x0, 0xFFAA, 0xFF9C, 0x0, 0xFF90, 0xFFF9, 0xFF80, 0xFF82,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0xE2A3, 0xDF41, 0xDFBA, 0x0, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
   0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1A, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
};

static inline int charType(int c) {return Data[Blocks[c>>5]+c & 0xFFFF] & 0x1F;}

static inline bool isLowc(int c) {return charType(c) == CHAR_LOWERCASE;}
static inline bool isUppc(int c) {return charType(c) == CHAR_UPPERCASE;}

static inline bool isLetterOrDigit(int c) {
   return (1 << charType(c)) & (CHAR_DIGIT | CHAR_LETTER);
}

static int toUpperCase(int c) {
   return c + Upper[Data[Blocks[c>>5]+c & 0xFFFF] >> 7];
}

static int toLowerCase(int c) {
   return c + Lower[Data[Blocks[c>>5]+c & 0xFFFF] >> 7];
}

// (low? 'any) -> sym | NIL
any doLowQ(any x) {
   x = cdr(x);
   return isSym(x = EVAL(car(x))) && isLowc(symChar(name(x)))? x : Nil;
}

// (upp? 'any) -> sym | NIL
any doUppQ(any x) {
   x = cdr(x);
   return isSym(x = EVAL(car(x))) && isUppc(symChar(name(x)))? x : Nil;
}

// (lowc 'any) -> any
any doLowc(any x) {
   int c, i;
   any nm;
   cell c1, c2;

   x = cdr(x);
   if (!isSym(x = EVAL(car(x))) || !(c = symChar(name(x))))
      return x;
   Push(c1, x);
   Push(c2, boxChar(toLowerCase(c), &i, &nm));
   while (c = symChar(NULL))
      charSym(toLowerCase(c), &i, &nm);
   drop(c1);
   return consStr(data(c2));
}

// (uppc 'any) -> any
any doUppc(any x) {
   int c, i;
   any nm;
   cell c1, c2;

   x = cdr(x);
   if (!isSym(x = EVAL(car(x))) || !(c = symChar(name(x))))
      return x;
   Push(c1, x);
   Push(c2, boxChar(toUpperCase(c), &i, &nm));
   while (c = symChar(NULL))
      charSym(toUpperCase(c), &i, &nm);
   drop(c1);
   return consStr(data(c2));
}

// (fold 'any ['cnt]) -> sym
any doFold(any ex) {
   int n, c, i;
   any x, nm;
   cell c1, c2;

   x = cdr(ex);
   if (!isSym(x = EVAL(car(x))) || !(c = symChar(name(x))))
      return Nil;
   while (!isLetterOrDigit(c))
      if (!(c = symChar(NULL)))
         return Nil;
   Push(c1, x);
   n = isCell(x = cddr(ex))? evCnt(ex,x) : 24;
   Push(c2, boxChar(toLowerCase(c), &i, &nm));
   while (c = symChar(NULL))
      if (isLetterOrDigit(c)) {
         if (!--n)
            break;
         charSym(toLowerCase(c), &i, &nm);
      }
   drop(c1);
   return consStr(data(c2));
}
