/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmviewport
 * @short_description: An abstract class doing a visual projection of a canvas.
 * @see_also: #PgmCanvas, #PgmViewportFactory, #PgmDrawable.
 *
 * <refsect2>
 * <para>
 * A #PgmViewport object is used to do a visual projection of #PgmCanvas object.
 * Each #PgmDrawable added to the canvas is rendered on the viewport converting
 * canvas coordinates to display coordinates. Viewport is a base abstract class
 * from which implementation plugins inherit. You can instantiate several
 * viewport implementations through a #PgmViewportFactory. A viewport can handle
 * 0 or 1 canvas, if there's no canvas binded, nothing is projected except
 * the background color of the viewport.
 * </para>
 * <title>Implementation capacities</title>
 * <para>
 * Each viewport implementations are based on a dedicated graphical library such
 * as DirectFB, OpenGL or Direct3D to project the binded canvas. These libraries
 * can be completely different, some of them being 2d-based generating their
 * output through blit operations, other being 3d-based generating their output
 * through a 3d pipeline. Each of them can also optimize part of their rendering
 * paths thanks to dedicated hardware. These differences can lead to viewport
 * implementations with different capacities. The result being that some
 * properties of a drawable could not be correctly projected by some
 * implementations. You can retrieve the mask capacities handled by a Viewport
 * at runtime with the pgm_viewport_get_caps_mask() method, and you can get the
 * list of handled color spaces with pgm_viewport_get_pixel_formats().
 * </para>
 * <title>Event handling</title>
 * <para>
 * A viewport is responsible of the event handling. It catches the events
 * generated by the underlying backend, and converts them building Pigment
 * events, which are then dispatched through signals. You can connect functions
 * to these signals using g_signal_connect() , in UI programming these
 * functions are often called callbacks. Each time an event is generated, the
 * corresponding signal is emited and all the connected callbacks are called.
 * </para>
 * <title>Viewport size</title>
 * <para>
 * Practically speaking, the viewport is basically either a window on your
 * screen or your complete monitor/TV. When you define the size of the viewport,
 * the implementation will try to change the window size, or the video mode if
 * you set it to be fullscreen. Some viewport implementation are not able to
 * obtain the screen size in millimeters and you have to set it, so that the
 * viewport can calculate the appropriate projections taking in account non
 * square pixels.
 * </para>
 * <title>Recommended usage of viewport size</title>
 * <para>
 * Applications are strongly encouraged to use the viewport the following way:
 * <itemizedlist>
 * <listitem>
 * <para>
 * The application initializes Pigment using pgm_init and then try to get
 * the screen size in millimeters, the screen resolution and the viewport size
 * (window size or fullscreen size). Looking at those parameters the application
 * can calculate the pixel aspect ratio of the screen and figure out the visual
 * aspect ratio of the viewport (4:3, 16:9, etc).
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * When the application has defined the visual aspect ratio of the viewport, it
 * can decide the visual aspect ratio it's going to generate and set the canvas
 * size accordingly. Let's say for example that the viewport size is a 750x400
 * pixels window and that we have square pixels, this is a bit wider than 16:9.
 * The application decides to generate a 16:9 style user interface and sets the
 * canvas to 16.0x9.0. The viewport will draw black borders and project that
 * canvas as a 711x400 pixels large rectangle in the middle of that window.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * If the viewport size changes (window resize), the application can decide to
 * use another aspect ratio or stick with the current one.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-06-10 (0.1.5)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmviewport.h"
#include "pgmimage.h"
#include "pgmmarshal.h"

#ifdef G_OS_UNIX
#include <unistd.h> /* pipe, close */
#endif /* G_OS_UNIX */

#ifdef WIN32
#include <fcntl.h>    /* _O_BINARY */
#include <io.h>       /* _pipe, _close */
#define pipe(handle)  _pipe(handle, 4096, _O_BINARY)
#define close(handle) _close(handle)
#endif /* WIN32 */

/* Z zero depth value in the [0.0, 1.0f] range */
#define DEPTH_ZERO_Z 0.68791816f

/* Multi-press event constants */
#define MULTI_PRESS_DELAY    250 /* in milliseconds */
#define MULTI_PRESS_DISTANCE   5 /* in pixels */

GST_DEBUG_CATEGORY_STATIC (pgm_viewport_debug);
#define GST_CAT_DEFAULT pgm_viewport_debug

/* Viewport signals */
enum {
  BUTTON_PRESS_EVENT,
  BUTTON_PRESSURE_EVENT,
  BUTTON_RELEASE_EVENT,
  KEY_PRESS_EVENT,
  KEY_RELEASE_EVENT,
  CONFIGURE_EVENT,
  DELETE_EVENT,
  SCROLL_EVENT,
  MOTION_NOTIFY_EVENT,
  EXPOSE_EVENT,
  DRAG_MOTION_EVENT,
  DRAG_DROP_EVENT,
  DRAG_LEAVE_EVENT,
  STATE_EVENT,
  WIN32_MESSAGE_EVENT,
  PIXELS_READ,
  UPDATE_PASS,
  LAST_SIGNAL
};

/* Pixel rectangle */
typedef struct {
  guint8 *pixels;
  guint   width;
  guint   height;
} PixelRectangle;

#ifdef WIN32
/* Win32 message */
typedef struct {
  PgmEventType type;
  PgmEvent *event;
  gint32   *message_processed;
  LRESULT  *result;
} Win32Message;
#endif /* WIN32 */

/* Mouse picking processing function as its defined in pgmdrawable.h */
typedef void (*DoPickingFunc) (PgmDrawable *drawable,
                               PgmEvent *event,
                               PgmVec3 *p1,
                               PgmVec3 *p2,
                               guint16 *emission_mask);

static GstObjectClass *parent_class = NULL;
static guint pgm_viewport_signals[LAST_SIGNAL] = { 0 };

/* Private functions */

/* Copy a layer reversing it and taking a ref on each drawable */
static GList*
copy_layer_reversed  (GList *layer)
{
  GList *new_layer = NULL;

  if (G_LIKELY (layer != NULL))
    {
      /* Create the last entry of the new list */
      new_layer = g_slice_new (GList);
      new_layer->data = g_object_ref (layer->data);
      new_layer->next = NULL;
      layer = layer->next;

      /* Create the previous entries */
      while (layer)
        {
          new_layer->prev = g_slice_new (GList);
          new_layer->prev->next = new_layer;
          new_layer = new_layer->prev;
          new_layer->data = g_object_ref (layer->data);
          layer = layer->next;
        }

      /* Set the previous pointer if the first entry of the new list to NULL */
      new_layer->prev = NULL;
    }

  return new_layer;
}

/* Call a drawable picking function for each drawables in a layer */
static void
layer_foreach_do_picking (PgmCanvas *canvas,
                          GList *layer,
                          DoPickingFunc do_picking,
                          PgmEvent *event,
                          PgmVec3 *p1,
                          PgmVec3 *p2,
                          guint16 *emission_mask)
{
  PgmDrawable *drawable;
  GList *list, *walk;

  /* Get a reversed copy of the layer so that we go from the nearest to the
   * farthest, the copy takes a ref on each drawable making sure drawables in
   * the layer are not disposed while picked */
  GST_OBJECT_LOCK (canvas);
  list = copy_layer_reversed (layer);
  GST_OBJECT_UNLOCK (canvas);

  /* Iterate picking on the drawables. Since the user can return TRUE in its
   * signal handler to prevent further propagation of the drawable picking
   * signals, the signal emission mask allows to filter signals which still need
   * to be emitted. The refs taken on each drawable during the copy are released
   * after use in the loop. */
  walk = list;
  while (walk && emission_mask)
    {
      drawable = (PgmDrawable*) walk->data;
      do_picking (drawable, event, p1, p2, emission_mask);
      g_object_unref (drawable);
      walk = walk->next;
    }

  /* And clean up */
  g_list_free (list);
  walk = NULL;
}

/* Popagate a mouse event to the canvas to process picking */
static void
propagate_picking_event (PgmViewport *viewport,
                         PgmEvent *event)
{
  PgmCanvas *canvas = viewport->canvas;
  PgmEventButton *button = (PgmEventButton*) event;
  guint16 emission_mask = PGM_DRAWABLE_PICKING_MASK;
  gfloat x1, y1, z1, x2, y2, z2;
  PgmVec3 p1, p2;

  /* Check if there's a canvas bound */
  if (G_UNLIKELY (!viewport->canvas))
    return;

  /* Create the picking line in canvas space */
  pgm_viewport_to_canvas (viewport, &x1, &y1, &z1, button->x, button->y, 0.0f);
  pgm_vec3_set_from_scalars (&p1, x1, y1, z1);
  pgm_viewport_to_canvas (viewport, &x2, &y2, &z2, button->x, button->y, 1.0f);
  pgm_vec3_set_from_scalars (&p2, x2, y2, z2);

  /* Propagate the picking to each drawables */
  switch (event->type)
    {
    case PGM_BUTTON_PRESS:
      layer_foreach_do_picking (canvas, canvas->near_layer,
                                (DoPickingFunc) _pgm_drawable_do_press_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle_layer,
                                (DoPickingFunc) _pgm_drawable_do_press_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far_layer,
                                (DoPickingFunc) _pgm_drawable_do_press_event,
                                event, &p1, &p2, &emission_mask);
      break;
    case PGM_BUTTON_PRESSURE:
      layer_foreach_do_picking (canvas, canvas->near_layer,
                                (DoPickingFunc) _pgm_drawable_do_pressure_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle_layer,
                                (DoPickingFunc) _pgm_drawable_do_pressure_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far_layer,
                                (DoPickingFunc) _pgm_drawable_do_pressure_event,
                                event, &p1, &p2, &emission_mask);
      break;
    case PGM_BUTTON_RELEASE:
      layer_foreach_do_picking (canvas, canvas->near_layer,
                                (DoPickingFunc) _pgm_drawable_do_release_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle_layer,
                                (DoPickingFunc) _pgm_drawable_do_release_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far_layer,
                                (DoPickingFunc) _pgm_drawable_do_release_event,
                                event, &p1, &p2, &emission_mask);
      break;
    case PGM_MOTION_NOTIFY:
      layer_foreach_do_picking (canvas, canvas->near_layer,
                                (DoPickingFunc) _pgm_drawable_do_motion_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle_layer,
                                (DoPickingFunc) _pgm_drawable_do_motion_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far_layer,
                                (DoPickingFunc) _pgm_drawable_do_motion_event,
                                event, &p1, &p2, &emission_mask);
      break;
    case PGM_SCROLL:
      emission_mask = PGM_DRAWABLE_PICKING_SCROLLED;
      layer_foreach_do_picking (canvas, canvas->near_layer,
                                (DoPickingFunc) _pgm_drawable_do_scroll_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle_layer,
                                (DoPickingFunc) _pgm_drawable_do_scroll_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far_layer,
                                (DoPickingFunc) _pgm_drawable_do_scroll_event,
                                event, &p1, &p2, &emission_mask);
      break;

    default:
      break;
    }
}

/* Emit double and triple button press event if needed */
static void
emit_multi_press_event (PgmViewport *viewport,
                        PgmEventButton *event)
{
  PgmEvent *new_event = NULL;
  /* Delay */
  static guint32 first_time = 0;
  static guint32 second_time = 0;
  guint32 current_time = event->time;
  /* Distance */
  static gint first_x = 0;
  static gint first_y = 0;
  gint current_x = (gint) event->x;
  gint current_y = (gint) event->y;

  /* Second click */
  if (current_time - first_time < MULTI_PRESS_DELAY
      && ABS (current_x - first_x) < MULTI_PRESS_DISTANCE
      && ABS (current_y - first_y) < MULTI_PRESS_DISTANCE)
    {
      new_event = pgm_event_copy ((PgmEvent *) event);
      new_event->type = PGM_DOUBLE_BUTTON_PRESS;
      g_signal_emit (G_OBJECT (viewport),
                     pgm_viewport_signals[BUTTON_PRESS_EVENT], 0, new_event);
      pgm_event_free (new_event);
      second_time = current_time;
    }
  /* Third click */
  else if (current_time - second_time < MULTI_PRESS_DELAY
           && ABS (current_x - first_x) < MULTI_PRESS_DISTANCE
           && ABS (current_y - first_y) < MULTI_PRESS_DISTANCE)
    {
      new_event = pgm_event_copy ((PgmEvent *) event);
      new_event->type = PGM_TRIPLE_BUTTON_PRESS;
      g_signal_emit (G_OBJECT (viewport),
                     pgm_viewport_signals[BUTTON_PRESS_EVENT], 0, new_event);
      pgm_event_free (new_event);
      first_time = 0;
      second_time = 0;
      first_x = 0;
      first_y = 0;
    }
  /* First click */
  else
    {
      first_time = current_time;
      first_x = current_x;
      first_y = current_y;
    }
}

/* Update the states of the viewport depending on the PgmEventState */
static void
update_states (PgmViewport *viewport,
               PgmViewportState state_mask)
{
  GST_OBJECT_LOCK (viewport);
  viewport->iconified = state_mask & PGM_VIEWPORT_ICONIFIED;
  GST_OBJECT_UNLOCK (viewport);
}

/* IO watch callback for pushed events. It dispatches the events from the queue,
 * emitting the correct signal for each of them. The queue is then freed. */
static gboolean
do_events (GIOChannel *source,
           GIOCondition condition,
           gpointer data)
{
  PgmViewport *viewport = PGM_VIEWPORT (data);
  GList *events, *walk;
  gchar buf;

  g_io_channel_read_chars (source, &buf, 1, NULL, NULL);

  /* Reverse the list since we prepend */
  g_mutex_lock (viewport->event_lock);
  events = g_list_reverse (viewport->event_list);
  viewport->event_list = NULL;
  g_mutex_unlock (viewport->event_lock);

  /* Dispatch all the events from the list */
  walk = events;
  while (walk)
    {
      PgmEvent *event = (PgmEvent *) walk->data;

      switch (event->type)
        {
        case PGM_MOTION_NOTIFY:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[MOTION_NOTIFY_EVENT], 0, event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_KEY_PRESS:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[KEY_PRESS_EVENT], 0, event);
          break;

        case PGM_KEY_RELEASE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[KEY_RELEASE_EVENT], 0, event);
          break;

        case PGM_SCROLL:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[SCROLL_EVENT], 0, event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_BUTTON_PRESS:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[BUTTON_PRESS_EVENT], 0, event);
          emit_multi_press_event (viewport, (PgmEventButton *) event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_BUTTON_PRESSURE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[BUTTON_PRESSURE_EVENT], 0, event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_BUTTON_RELEASE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[BUTTON_RELEASE_EVENT], 0, event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_EXPOSE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[EXPOSE_EVENT], 0, event);
          break;

        case PGM_CONFIGURE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[CONFIGURE_EVENT], 0, event);
          break;

        case PGM_DRAG_MOTION:
          {
            PgmViewportClass *klass = PGM_VIEWPORT_GET_CLASS (viewport);
            gboolean accepted = FALSE;

            g_signal_emit (G_OBJECT (viewport),
                           pgm_viewport_signals[DRAG_MOTION_EVENT], 0, event,
                           &accepted);
            if (klass->set_drag_status)
              klass->set_drag_status (viewport, accepted);
          }
          break;

        case PGM_DRAG_DROP:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[DRAG_DROP_EVENT], 0, event);
          break;

        case PGM_DRAG_LEAVE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[DRAG_LEAVE_EVENT], 0, event);
          break;

        case PGM_STATE:
          update_states (viewport, ((PgmEventState *) event)->state_mask);
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[STATE_EVENT], 0, event);
          break;

        case PGM_DELETE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[DELETE_EVENT], 0, event);
          break;

#ifdef WIN32
        case PGM_WIN32_MESSAGE:
          {
            Win32Message *message = (Win32Message*) event;
            g_signal_emit (G_OBJECT (viewport),
                           pgm_viewport_signals[WIN32_MESSAGE_EVENT], 0,
                           message->event, (gpointer) message->result);
            g_atomic_int_set (message->message_processed, 1);
            event = message->event;
            g_slice_free (Win32Message, message);
          }
          break;
#endif /* WIN32 */

        default:
          break;
        }

      pgm_event_free (event);
      walk = walk->next;
    }

  g_list_free (events);
  events = NULL;

  return TRUE;
}

/* IO watch callback for pushed pixels */
static gboolean
do_pixels (GIOChannel *source,
           GIOCondition condition,
           gpointer data)
{
  PgmViewport *viewport = PGM_VIEWPORT (data);
  GList *walk = NULL, *rectangles = NULL;
  gchar buf;

  g_io_channel_read_chars (source, &buf, 1, NULL, NULL);

  g_mutex_lock (viewport->pixel_lock);
  rectangles = viewport->pixel_list;
  viewport->pixel_list = NULL;
  g_mutex_unlock (viewport->pixel_lock);

  /* Emit signals for all the pixel areas from the list */
  walk = rectangles;
  while (walk)
    {
      PixelRectangle *rectangle = (PixelRectangle*) walk->data;

      /* Emit "pixels-read" signal with the pixel area */
      g_signal_emit (G_OBJECT (viewport), pgm_viewport_signals[PIXELS_READ], 0,
                     rectangle->width, rectangle->height, rectangle->pixels);

      /* Then free our rectangle */
      g_slice_free (PixelRectangle, rectangle);
      rectangle = NULL;

      walk = walk->next;
    }

  g_list_free (rectangles);
  rectangles = NULL;

  return TRUE;
}

/* Canvas "size-changed" handler */
static void
canvas_size_changed_cb (PgmCanvas *canvas,
                        gpointer data)
{
  PgmViewport *viewport = PGM_VIEWPORT (data);

  pgm_viewport_update_projection (viewport);
}

/* Create an IO watch */
static _PgmIOWatch*
new_io_watch (PgmViewport *viewport,
              GIOFunc watch_func)
{
  _PgmIOWatch *watch;

  watch = g_slice_new (_PgmIOWatch);

  watch->fd[0] = -1;
  watch->fd[1] = -1;
  watch->in = NULL;
  watch->out = NULL;
  watch->tag = 0;

  /* Create the pipe */
  if (G_UNLIKELY (pipe (watch->fd) == -1))
    {
      GST_ERROR_OBJECT (viewport, "pipe failed");
      return NULL;
    }

  /* Build an IO channel for reading */
  watch->out = g_io_channel_unix_new (watch->fd[0]);
  if (G_UNLIKELY (!watch->out))
    {
      GST_ERROR_OBJECT (viewport, "g_io_channel_unix_new failed");
      close (watch->fd[0]);
      close (watch->fd[1]);
      return NULL;
    }

  /* Build an IO channel for writing */
  watch->in = g_io_channel_unix_new (watch->fd[1]);
  if (G_UNLIKELY (!watch->in))
    {
      GST_ERROR_OBJECT (viewport, "g_io_channel_unix_new failed");
      g_io_channel_unref (watch->out);
      close (watch->fd[0]);
      close (watch->fd[1]);
      return NULL;
    }

  /* We don't close the fd ourselves because otherwise it deadlocks on windows */
  g_io_channel_set_close_on_unref (watch->in, TRUE);
  g_io_channel_set_close_on_unref (watch->out, TRUE);

  /* Add the watch to dispatch in the main loop using the default GMainContext */
  watch->tag = g_io_add_watch (watch->out, G_IO_IN, watch_func, viewport);

  return watch;
}

/* Free an IO watch */
static void
free_io_watch (_PgmIOWatch *watch)
{
  /* Event data cleaning */
  if (G_LIKELY (watch->tag))
    {
      g_source_remove (watch->tag);
      watch->tag = 0;
    }
  if (G_LIKELY (watch->out))
    {
      g_io_channel_unref (watch->out);
      watch->out = NULL;
    }
  if (G_LIKELY (watch->in))
    {
      g_io_channel_unref (watch->in);
      watch->in = NULL;
    }

  g_slice_free (_PgmIOWatch, watch);
  watch = NULL;
}

/* GObject stuff */

G_DEFINE_TYPE (PgmViewport, pgm_viewport, GST_TYPE_OBJECT);

static void
pgm_viewport_dispose (GObject *object)
{
  PgmViewport *viewport = PGM_VIEWPORT (object);

  /* Frees event handling data */
  free_io_watch (viewport->event_watch);
  g_mutex_free (viewport->event_lock);
  if (viewport->event_list)
    {
      GList *walk = viewport->event_list;
      while (walk)
        {
          pgm_event_free ((PgmEvent*) walk->data);
          walk = walk->next;
        }
      g_list_free (viewport->event_list);
      viewport->event_list = NULL;
    }

  /* Free pixel handling data */
  free_io_watch (viewport->pixel_watch);
  g_mutex_free (viewport->pixel_lock);
  if (viewport->pixel_list)
    {
      GList *walk = viewport->pixel_list;
      while (walk)
        {
          g_slice_free (PixelRectangle, walk->data);
          walk->data = NULL;
          walk = walk->next;
        }
      g_list_free (viewport->pixel_list);
      viewport->pixel_list = NULL;
    }

  /* Free projection matrices */
  pgm_mat4x4_free (viewport->projection);
  pgm_mat4x4_free (viewport->inv_projection);

  if (viewport->icon)
    g_object_unref (viewport->icon);

  g_free (viewport->title);
  viewport->title = NULL;

  if (viewport->message_filter)
    {
      g_list_free (viewport->message_filter);
      viewport->message_filter = NULL;
    }

  if (viewport->canvas)
    gst_object_unref (GST_OBJECT_CAST (viewport->canvas));

  gst_object_unref (GST_OBJECT_CAST (viewport->factory));

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_viewport_class_init (PgmViewportClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmViewport::button-press-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventButton
   *
   * Will be emitted on a mouse button press on the @viewport.
   */
  pgm_viewport_signals[BUTTON_PRESS_EVENT] =
    g_signal_new ("button-press-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, button_press_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::button-pressure-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventButton
   *
   * Will be emitted on a mouse button pressure change on the @viewport.
   */
  pgm_viewport_signals[BUTTON_PRESSURE_EVENT] =
    g_signal_new ("button-pressure-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, button_pressure_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::button-release-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventButton
   *
   * Will be emitted on a mouse button release on the @viewport.
   */
  pgm_viewport_signals[BUTTON_RELEASE_EVENT] =
    g_signal_new ("button-release-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, button_release_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::key-press-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventKey
   *
   * Will be emitted on a keyboard press on the @viewport.
   */
  pgm_viewport_signals[KEY_PRESS_EVENT] =
    g_signal_new ("key-press-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, key_press_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::key-release-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventKey
   *
   * Will be emitted on a keyboard release on the @viewport.
   */
  pgm_viewport_signals[KEY_RELEASE_EVENT] =
    g_signal_new ("key-release-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, key_release_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::configure-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventConfigure
   *
   * Will be emitted on a @viewport position or size change.
   */
  pgm_viewport_signals[CONFIGURE_EVENT] =
    g_signal_new ("configure-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, configure_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::delete-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventDelete
   *
   * Will be emitted on a @viewport close request (ie the user clicked the
   * close button of the window manager).
   */
  pgm_viewport_signals[DELETE_EVENT] =
    g_signal_new ("delete-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, delete_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::scroll-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventScroll
   *
   * Will be emitted on a mouse scroll on the @viewport.
   */
  pgm_viewport_signals[SCROLL_EVENT] =
    g_signal_new ("scroll-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, scroll_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::motion-notify-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventMotion
   *
   * Will be emitted on a mouse move on the @viewport.
   */
  pgm_viewport_signals[MOTION_NOTIFY_EVENT] =
    g_signal_new ("motion-notify-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, motion_notify_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::expose-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventExpose
   *
   * Will be emitted when the @viewport visibility status has changed.
   */
  pgm_viewport_signals[EXPOSE_EVENT] =
    g_signal_new ("expose-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, expose_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::drag-motion-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventDnd
   *
   * Will be emitted when the user moves the cursor over @viewport during a
   * drag. The signal handler must determine whether the cursor position is in
   * a drop zone or not. If it is not in a drop zone, it returns FALSE and the
   * "drag-drop-event" signal will not be called. Otherwise, the handler
   * returns TRUE.
   */
  pgm_viewport_signals[DRAG_MOTION_EVENT] =
    g_signal_new ("drag-motion-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, drag_motion_event),
                  NULL, NULL, pgm_marshal_BOOLEAN__BOXED, G_TYPE_BOOLEAN,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::drag-drop-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventDnd
   *
   * Will be emitted when the user drops data onto @viewport. Note that you have
   * to connect a handler to the signal "drag-motion-event" and return TRUE to
   * accept a drag. This signal is not emitted if a drag has not been accepted.
   */
  pgm_viewport_signals[DRAG_DROP_EVENT] =
    g_signal_new ("drag-drop-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, drag_drop_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::drag-leave-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventDnd
   *
   * Will be emitted when the user leaves @viewport during a drag. A typical
   * reason to connect to this signal is to undo things done in
   * "drag-motion-event".
   */
  pgm_viewport_signals[DRAG_LEAVE_EVENT] =
    g_signal_new ("drag-leave-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, drag_leave_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::state-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventState
   *
   * Will be emitted when the state of @viewport changes, being a visibility
   * change, a fullscreen change or an iconification change.
   */
  pgm_viewport_signals[STATE_EVENT] =
    g_signal_new ("state-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, state_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
#ifdef WIN32
  /**
   * PgmViewport::win32-message-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventWin32Message
   *
   * Will be emitted when a Win32 message is received by @viewport. The
   * signal is emitted only if the message type is in the filter list defined
   * by pgm_viewport_set_message_filter(). It is prior to all the other
   * @viewport signal events.
   */
  pgm_viewport_signals[WIN32_MESSAGE_EVENT] =
    g_signal_new ("win32-message-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, win32_message_event),
                  NULL, NULL, pgm_marshal_POINTER__BOXED, G_TYPE_POINTER,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
#endif /* WIN32 */
  /**
   * PgmViewport::pixels-read:
   * @viewport: the #PgmViewport
   * @width: the width of the pixels area
   * @height: the height of the pixels area
   * @pixels: the pixels area that has been filled. It has been allocated
   * by the application and must be freed by the application.
   *
   * Will be emitted when pixels have been read following a call to
   * pgm_viewport_read_pixels().
   */
  pgm_viewport_signals[PIXELS_READ] =
    g_signal_new ("pixels-read", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, pixels_read),
                  NULL, NULL, pgm_marshal_VOID__UINT_UINT_POINTER,
                  G_TYPE_NONE, 3, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_POINTER);
  /**
   * PgmViewport::update-pass:
   * @viewport: the #PgmViewport
   *
   * Will be emitted prior to any rendering pass happening in the Pigment
   * internal rendering thread. It is highly recommended to update all the
   * Pigment drawables rendered by @viewport in this signal. It ensures the
   * best performance and rendering correctness since the object update pass
   * will be completely synchronized prior to the rendering pass.
   *
   * Note that this is the only signal in Pigment which is emitted from another
   * thread than the one of the Pigment main loop.
   */
  pgm_viewport_signals[UPDATE_PASS] =
    g_signal_new ("update-pass", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, update_pass),
                  NULL, NULL, pgm_marshal_VOID__VOID, G_TYPE_NONE, 0);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_viewport_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_viewport_debug, "pgm_viewport", 0,
                           "viewport object");
}

static void
pgm_viewport_init (PgmViewport *viewport)
{
  /* Public data default values */
  viewport->canvas = NULL;
  viewport->factory = NULL;
  viewport->width = 800;
  viewport->height = 600;
  viewport->width_mm = -1;
  viewport->height_mm = -1;
  viewport->projected_width = 800;
  viewport->projected_height = 600;
  viewport->projected_x = 0;
  viewport->projected_y = 0;
  viewport->rotation = PGM_VIEWPORT_ROTATION_NONE;
  viewport->reflection = PGM_VIEWPORT_REFLECTION_NONE;
  viewport->viewport_ratio = 4.0f/3.0f;
  viewport->canvas_ratio = 4.0f/3.0f;
  viewport->pixel_aspect_ratio = 1.0f;
  viewport->title = g_strdup ("");
  viewport->cursor = PGM_VIEWPORT_LEFT_ARROW;
  viewport->icon = NULL;
  viewport->fullscreen = FALSE;
  viewport->visible = FALSE;
  viewport->opacity = 255;
  viewport->decorated = TRUE;
  viewport->iconified = FALSE;
  viewport->alpha_blending = TRUE;
  viewport->message_filter = NULL;

  /* Projection matrix */
  viewport->projection = pgm_mat4x4_new ();
  viewport->inv_projection = pgm_mat4x4_new ();
  viewport->projection_hflip = 1.0f;
  viewport->projection_vflip = 1.0f;
  viewport->projection_rotation = 0.0f;

  /* Event handling */
  viewport->event_watch = new_io_watch (viewport, (GIOFunc) do_events);
  viewport->event_watch_up = viewport->event_watch && viewport->event_watch->in;
  viewport->event_lock = g_mutex_new ();
  viewport->event_list = NULL;

  /* Pixel handling */
  viewport->pixel_watch = new_io_watch (viewport, (GIOFunc) do_pixels);
  viewport->pixel_watch_up = viewport->pixel_watch && viewport->pixel_watch->in;
  viewport->pixel_lock = g_mutex_new ();
  viewport->pixel_list = NULL;
}

/**
 * pgm_viewport_set_title:
 * @viewport: a #PgmViewport object.
 * @title: the title.
 *
 * Sets the title appearing in the title bar and in the iconified
 * window list. The string is copied and can be freed when the call returns.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_title (PgmViewport *viewport,
                        const gchar *title)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (G_UNLIKELY (viewport->title))
    g_free (viewport->title);
  viewport->title = g_strdup (title);

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_title)
    ret = klass->set_title (viewport, title);

  return ret;
}

/**
 * pgm_viewport_get_title:
 * @viewport: a #PgmViewport object.
 * @title: a pointer to a pointer to a gchar where the title string will be
 * stored. g_free() after use.
 *
 * Retrieves the title of @viewport in @title.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_title (PgmViewport *viewport,
                        gchar **title)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (title != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *title = g_strdup (viewport->title);

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_show:
 * @viewport: a #PgmViewport object.
 *
 * Makes @viewport visible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_show (PgmViewport *viewport)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->visible == TRUE)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_OK;
    }

  viewport->visible = TRUE;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->show)
    ret = klass->show (viewport);

  return ret;
}

/**
 * pgm_viewport_hide:
 * @viewport: a #PgmViewport object.
 *
 * Makes @viewport invisible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_hide (PgmViewport *viewport)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->visible == FALSE)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_OK;
    }

  viewport->visible = FALSE;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->hide)
    ret = klass->hide (viewport);

  return ret;
}

/**
 * pgm_viewport_is_visible:
 * @viewport: a #PgmViewport object.
 * @visible: a pointer to a #gboolean where the visible state of the viewport
 * is going to be stored.
 *
 * Retrieves whether @viewport is visible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_is_visible (PgmViewport *viewport,
                         gboolean *visible)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (visible != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *visible = viewport->visible;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_decorated:
 * @viewport: a #PgmViewport object.
 * @decorated: %TRUE to decorate the viewport.
 *
 * By default, viewports should be (depending on the plugin) decorated with a
 * title bar and resize controls. This function allows to disable these
 * decorations, creating a borderless viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_decorated (PgmViewport *viewport,
                            gboolean decorated)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->decorated = decorated;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_decorated)
    ret = klass->set_decorated (viewport, decorated);

  return ret;
}

/**
 * pgm_viewport_get_decorated:
 * @viewport: a #PgmViewport object.
 * @decorated: a pointer the a #gboolean where the decorated state will be
 * stored.
 *
 * Retrieves in @decorated whether @viewport is decorated.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_decorated (PgmViewport *viewport,
                            gboolean *decorated)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (decorated != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *decorated = viewport->decorated;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_cursor:
 * @viewport: a #PgmViewport object.
 * @cursor: the cursor to set.
 *
 * Sets @cursor as the current cursor of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_cursor (PgmViewport *viewport,
                         PgmViewportCursor cursor)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->cursor == cursor)
    {
      GST_OBJECT_UNLOCK (viewport);
      return ret;
    }

  viewport->cursor = cursor;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_cursor)
    ret = klass->set_cursor (viewport, cursor);

  return ret;
}

/**
 * pgm_viewport_get_cursor:
 * @viewport: a #PgmViewport object.
 * @cursor: a pointer #PgmViewportCursor where the @viewport cursor is going
 * to be stored.
 *
 * Retrieves the @cursor of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_cursor (PgmViewport *viewport,
                         PgmViewportCursor *cursor)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (cursor != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *cursor = viewport->cursor;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_icon:
 * @viewport: a #PgmViewport object.
 * @icon: a #GdkPixbuf object to set as icon for @viewport.
 *
 * Sets @icon as the current icon of @viewport (shown by the window manager
 * e.g. when the @viewport window is minimised).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_icon (PgmViewport *viewport,
                       GdkPixbuf *icon)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->icon == icon)
    {
      GST_OBJECT_UNLOCK (viewport);
      return ret;
    }

  if (viewport->icon)
    g_object_unref (viewport->icon);

  if (icon)
    viewport->icon = g_object_ref (icon);
  else
    viewport->icon = NULL;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_icon)
    ret = klass->set_icon (viewport, icon);

  return ret;
}

/**
 * pgm_viewport_get_icon:
 * @viewport: a #PgmViewport object.
 * @icon: a double pointer #GdkPixbuf where the @viewport icon is going
 * to be stored. Unref after usage.
 *
 * Retrieves the @icon of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_icon (PgmViewport *viewport,
                       GdkPixbuf **icon)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (icon != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->icon)
    *icon = g_object_ref (viewport->icon);
  else
    *icon = NULL;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_size:
 * @viewport: a #PgmViewport object.
 * @width: the viewport width.
 * @height: the viewport height.
 *
 * Sets @viewport size in pixels to (@width,@height).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_size (PgmViewport *viewport,
                       gint width,
                       gint height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_X;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->width == width && viewport->height == height)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_OK;
    }

  viewport->width = MAX (1, width);
  viewport->height = MAX (1, height);

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_size)
    ret = klass->set_size (viewport, viewport->width, viewport->height);

  pgm_viewport_update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_get_size:
 * @viewport: a #PgmViewport object.
 * @width: a pointer to a #gint where the viewport width in pixels is
 * going to be stored.
 * @height: a pointer to a #gint where the viewport height in pixels is
 * going to be stored.
 *
 * Retrieves the size (@width,@height) of @viewport in pixels.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_size (PgmViewport *viewport,
                       gint *width,
                       gint *height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *width = viewport->width;
  *height = viewport->height;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_size)
    ret = klass->get_size (viewport, width, height);

  return ret;
}

/**
 * pgm_viewport_set_alpha_blending:
 * @viewport: a #PgmViewport object.
 * @alpha_blending: the alpha blending state.
 *
 * Enable or disable alpha blending on @viewport.
 *
 * The function is useful to improve the rendering performance. For instance
 * it can make HD video playback in fullscreen smoother, when no blended
 * drawables need to be drawn over it. 
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_alpha_blending (PgmViewport *viewport,
                                 gboolean alpha_blending)
{
  PgmError ret = PGM_ERROR_OK;
  PgmViewportClass *klass;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->alpha_blending == alpha_blending)
    {
      GST_OBJECT_UNLOCK (viewport);
      return ret;
    }

  viewport->alpha_blending = alpha_blending;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_alpha_blending)
    ret = klass->set_alpha_blending (viewport, alpha_blending);

  return ret;
}

/**
 * pgm_viewport_get_alpha_blending:
 * @viewport: a #PgmViewport object.
 * @alpha_blending: a pointer to a #gboolean where the alpha blending state
 * is going to be stored.
 *
 * Retrieves in @alpha_blending the alpha blending state of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_alpha_blending (PgmViewport *viewport,
                                 gboolean *alpha_blending)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (alpha_blending != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *alpha_blending = viewport->alpha_blending;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_opacity:
 * @viewport: a #PgmViewport object.
 * @opacity: the opacity between 0 and 255.
 *
 * Set the opacity of @viewport.
 *
 * Note that this function only works if @viewport supports the
 * #PGM_VIEWPORT_OPACITY capacity. The opacity is considered by compositing
 * managers for blending with other applications.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_opacity (PgmViewport *viewport,
                          guchar opacity)
{
  PgmError ret = PGM_ERROR_OK;
  PgmViewportClass *klass;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->opacity = opacity;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_opacity)
    ret = klass->set_opacity (viewport, opacity);

  return ret;
}

/**
 * pgm_viewport_get_opacity:
 * @viewport: a #PgmViewport object.
 * @opacity: a pointer to a #guchar where the opacity is going to be stored.
 *
 * Retrieves in @opacity the opacity of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_opacity (PgmViewport *viewport,
                          guchar *opacity)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (opacity != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *opacity = viewport->opacity;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_fullscreen:
 * @viewport: a #PgmViewport object.
 * @fullscreen: the fullscreen state.
 *
 * Sets/unsets fullscreen mode of @viewport function of @fullscreen.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_fullscreen (PgmViewport *viewport,
                             gboolean fullscreen)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->fullscreen == fullscreen)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_OK;
    }

  viewport->fullscreen = fullscreen;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_fullscreen)
    ret = klass->set_fullscreen (viewport, fullscreen);

  return ret;
}

/**
 * pgm_viewport_get_fullscreen:
 * @viewport: a #PgmViewport object.
 * @fullscreen: a pointer to a #gboolean where the fullscreen state is going
 * to be stored.
 *
 * Retrieves the fullscreen state of @viewport in @fullscreen.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_fullscreen (PgmViewport *viewport,
                             gboolean *fullscreen)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (fullscreen != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *fullscreen = viewport->fullscreen;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_iconified:
 * @viewport: a #PgmViewport object.
 * @iconified: %TRUE to iconify the viewport.
 *
 * Asks to iconify (i.e. minimize) @viewport depending on the @iconified value.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_iconified (PgmViewport *viewport,
                            gboolean iconified)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->iconified == iconified)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_OK;
    }

  viewport->iconified = iconified;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_iconified)
    ret = klass->set_iconified (viewport, iconified);

  return ret;
}

/**
 * pgm_viewport_get_iconified:
 * @viewport: a #PgmViewport object.
 * @iconified: a pointer the a #gboolean where the iconified state will be
 * stored.
 *
 * Retrieves in @iconified whether @viewport is iconified.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_iconified (PgmViewport *viewport,
                            gboolean *iconified)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (iconified != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *iconified = viewport->iconified;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_focus:
 * @viewport: a #PgmViewport object.
 *
 * Presents @viewport to the user. This may mean raising the window in the
 * stacking order, deiconifying it and/or giving it the keyboard focus, possibly
 * dependent on the user's platform, window manager, and preferences.
 *
 * Note that if @viewport is hidden, this function calls pgm_viewport_show() as
 * well.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_focus (PgmViewport *viewport)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->focus)
    ret = klass->focus (viewport);

  return ret;
}

/**
 * pgm_viewport_set_screen_resolution:
 * @viewport: a #PgmViewport object.
 * @width: the screen width in pixels.
 * @height: the screen height in pixels.
 *
 * Sets a new resolution (@width,@height) of @viewport in pixels. This is
 * changing the video mode or the display resolution so you are strongly
 * encouraged to restore it to original value when exiting.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_screen_resolution (PgmViewport *viewport,
                                    gint width,
                                    gint height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_screen_resolution)
    ret = klass->set_screen_resolution (viewport, width, height);

  pgm_viewport_update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_get_screen_resolution:
 * @viewport: a #PgmViewport object.
 * @width: a pointer to a #gint where the screen width in pixels is going to be
 * stored.
 * @height: a pointer to a #gint where the screen height in pixels is going to
 * be stored.
 *
 * Retrieves the resolution (@width,@height) of @viewport in pixels.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_screen_resolution (PgmViewport *viewport,
                                    gint *width,
                                    gint *height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_screen_resolution)
    ret = klass->get_screen_resolution (viewport, width, height);

  return ret;
}

/**
 * pgm_viewport_set_screen_size_mm:
 * @viewport: a #PgmViewport object.
 * @width: the screen width in millimeters.
 * @height: the screen height in millimeters.
 *
 * Sets the physical screen size in millimeters. This is used if the @viewport
 * implementation cannot retrieve that information by itself. You have to
 * define the physical ratio of the screen so that @viewport can calculate the
 * pixel aspect ratio.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_screen_size_mm (PgmViewport *viewport,
                                 gint width,
                                 gint height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->width_mm = width;
  viewport->height_mm = height;

  GST_OBJECT_UNLOCK (viewport);

  GST_DEBUG_OBJECT (viewport, "forcing screen size to %d mm x %d mm", width,
                    height);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_screen_size_mm)
    ret = klass->set_screen_size_mm (viewport, width, height);

  pgm_viewport_update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_get_screen_size_mm:
 * @viewport: a #PgmViewport object.
 * @width: a pointer to a #gint where the screen width in millimeters is going
 * to be stored.
 * @height: a pointer to a #gint where the screen height in millimeters is going
 * to be stored.
 *
 * Retrieves the physical (@width,@height) size of @viewport in millimeters.
 * If the plugin can not provide that information it will return -1 as width
 * and height. In that case the application should provide the screen size
 * through pgm_viewport_set_screen_size_mm().
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_screen_size_mm (PgmViewport *viewport,
                                 gint *width,
                                 gint *height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  /* If we have a screen size defined already (cached or forced), we use it */
  if (viewport->width_mm != -1 && viewport->height_mm != -1)
    {
      *width = viewport->width_mm;
      *height = viewport->height_mm;
      GST_OBJECT_UNLOCK (viewport);
      GST_LOG_OBJECT (viewport, "screen size %dx%d mm (from cache)",
                        *width, *height);
    }
  else
    {
      GST_OBJECT_UNLOCK (viewport);

      klass = PGM_VIEWPORT_GET_CLASS (viewport);

      /* Otherwise we get it from plugin */
      if (klass->get_screen_size_mm)
        ret = klass->get_screen_size_mm (viewport, width, height);

      /* If it's a valid size we cache it */
      if (*width != -1 && *height != -1)
        {
          GST_OBJECT_LOCK (viewport);

          viewport->width_mm = *width;
          viewport->height_mm = *height;

          GST_OBJECT_UNLOCK (viewport);

          GST_LOG_OBJECT (viewport, "caching screen size %dx%d mm", *width,
                            *height);
        }
    }

  return ret;
}

/**
 * pgm_viewport_push_event:
 * @viewport: a #PgmViewport object.
 * @event: the #PgmEvent to push.
 *
 * Push an event in the list.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_push_event (PgmViewport *viewport,
                         PgmEvent *event)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (event != NULL, PGM_ERROR_X);

  g_mutex_lock (viewport->event_lock);
  viewport->event_list = g_list_prepend (viewport->event_list, event);
  g_mutex_unlock (viewport->event_lock);

  if (G_LIKELY (viewport->event_watch_up))
    {
      g_io_channel_write_chars (viewport->event_watch->in, "#", 1, NULL, NULL);
      g_io_channel_flush (viewport->event_watch->in, NULL);
    }

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_canvas:
 * @viewport: a #PgmViewport object.
 * @canvas: the #PgmCanvas to bind.
 *
 * Sets the @canvas to be projected by the @viewport. This function increases
 * the refcount on the canvas. Any previously set canvas on @viewport is
 * unreffed. If @canvas is NULL, the previously set canvas is unreffed.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_canvas (PgmViewport *viewport,
                         PgmCanvas *canvas)
{
  PgmViewportClass *klass;
  PgmCanvas *tmp = NULL;
  PgmError ret = PGM_ERROR_OK;
  gulong pixel_formats = 0;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  GST_OBJECT_LOCK (viewport);

  /* Is it the canvas already binded? If yes just return. */
  if (viewport->canvas == canvas)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_X;
    }

  /* Get the viewport pixel formats */
  if (klass->get_pixel_formats)
    {
      GST_OBJECT_UNLOCK (viewport);
      klass->get_pixel_formats (viewport, &pixel_formats);
      GST_OBJECT_LOCK (viewport);
    }

  /* If there was a previous canvas */
  if (viewport->canvas)
    {
      /* Backup the pointer */
      tmp = viewport->canvas;
      /* And update the previous canvas pixel formats mask */
      _pgm_canvas_remove_pixel_formats (tmp, pixel_formats);
      /* Disconnect "size-changed" handler */
      g_signal_handler_disconnect (tmp, viewport->canvas_size_handler);
    }

  /* If there's a new canvas */
  if (canvas)
    {
      /* Bind it taking a refcount */
      viewport->canvas = gst_object_ref (GST_OBJECT_CAST (canvas));
      /* Update the new canvas pixel formats mask */
      _pgm_canvas_add_pixel_formats (canvas, pixel_formats);
      /* Connect "size-changed" handler */
      viewport->canvas_size_handler =
        g_signal_connect (G_OBJECT (canvas), "size-changed",
                          G_CALLBACK (canvas_size_changed_cb),
                          (gpointer) viewport);
    }
  else
    viewport->canvas = NULL;

  GST_OBJECT_UNLOCK (viewport);

  /* Let the plugin do what it needs with the canvas */
  if (klass->set_canvas)
    ret = klass->set_canvas (viewport, canvas);

  /* And then, unref the previous canvas */
  if (tmp)
    gst_object_unref (GST_OBJECT_CAST (tmp));

  pgm_viewport_update_projection (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_get_canvas:
 * @viewport: a #PgmViewport object.
 * @canvas: a pointer to #PgmCanvas pointer in which the canvas bound to
 * @viewport is going to be stored. The refcount of canvas is increased.
 *
 * Retrieves the current canvas or NULL if there's no canvas bound.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_canvas (PgmViewport *viewport,
                         PgmCanvas **canvas)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (canvas != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (!viewport->canvas)
    {
      GST_LOG_OBJECT (viewport, "no canvas bound");
      *canvas = NULL;
    }
  else
    *canvas = gst_object_ref (GST_OBJECT_CAST (viewport->canvas));

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_canvas_rotation:
 * @viewport: a #PgmViewport object.
 * @rotation: the #PgmViewportRotation rotation type.
 *
 * Affects the way @viewport projects its canvas applying the specified @rotation.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_canvas_rotation (PgmViewport *viewport,
                                  PgmViewportRotation rotation)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  switch (rotation)
    {
    case PGM_VIEWPORT_ROTATION_NONE:
      viewport->projection_rotation = 0.0f;
      break;

    case PGM_VIEWPORT_ROTATION_90:
      viewport->projection_rotation = (gfloat) -G_PI / 2.0f;
      break;

    case PGM_VIEWPORT_ROTATION_180:
      viewport->projection_rotation = (gfloat) G_PI;
      break;

    case PGM_VIEWPORT_ROTATION_270:
      viewport->projection_rotation = (gfloat) G_PI / 2.0;
      break;

    default:
      GST_ERROR_OBJECT (viewport, "unknown viewport rotation type");
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_X;
    }

  viewport->rotation = rotation;

  GST_OBJECT_UNLOCK (viewport);

  pgm_viewport_update_projection (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_get_canvas_rotation:
 * @viewport: a #PgmViewport object.
 * @rotation: a pointer to #PgmViewportRotation where the current rotation
 * will be stored.
 *
 * Retrieves the current @rotation applied by @viewport on its canvas.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_canvas_rotation (PgmViewport *viewport,
                                  PgmViewportRotation *rotation)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (rotation != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *rotation = viewport->rotation;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_canvas_reflection:
 * @viewport: a #PgmViewport object.
 * @reflection: the #PgmViewportReflection reflection type.
 *
 * Affects the way @viewport projects its canvas applying the specified
 * @reflection.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_canvas_reflection (PgmViewport *viewport,
                                    PgmViewportReflection reflection)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  switch (reflection)
    {
    case PGM_VIEWPORT_REFLECTION_NONE:
      viewport->projection_hflip = 1.0f;
      viewport->projection_vflip = 1.0f;
      break;

    case PGM_VIEWPORT_REFLECTION_HORIZONTAL_FLIP:
      viewport->projection_hflip = -1.0f;
      viewport->projection_vflip = 1.0f;
      break;

    case PGM_VIEWPORT_REFLECTION_VERTICAL_FLIP:
      viewport->projection_hflip = 1.0f;
      viewport->projection_vflip = -1.0f;
      break;

    default:
      GST_ERROR_OBJECT (viewport, "unknown viewport reflection type");
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_X;
    }

  viewport->reflection = reflection;

  GST_OBJECT_UNLOCK (viewport);

  pgm_viewport_update_projection (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_get_canvas_reflection:
 * @viewport: a #PgmViewport object.
 * @reflection: a pointer to #PgmViewportReflection where the current reflection
 * will be stored.
 *
 * Retrieves the current @reflection applied by @viewport on its canvas.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_canvas_reflection (PgmViewport *viewport,
                                    PgmViewportReflection *reflection)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (reflection != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *reflection = viewport->reflection;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_message_filter:
 * @viewport: a #PgmViewport object.
 * @filter: the #GList containing the message types to filter.
 *
 * Sets the list of message types filtered by @viewport for the
 * "win32-message-events" signal. The previously set filter is discarded. A
 * NULL @filter is equivalent to a void list. The list is copied internally and
 * have to be freed by the caller.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_message_filter (PgmViewport *viewport,
                                 GList *filter)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->message_filter)
    g_list_free (viewport->message_filter);

  viewport->message_filter = g_list_copy (filter);

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_message_filter)
    ret = klass->set_message_filter (viewport, filter);

  return ret;
}

/**
 * pgm_viewport_get_message_filter:
 * @viewport: a #PgmViewport object.
 * @filter: a pointer to #GList pointer in which the filter list is going to
 * be stored. g_list_free() after use.
 *
 * Retrieves the current list of message types filtered by @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_message_filter (PgmViewport *viewport,
                                 GList **filter)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (filter != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *filter = g_list_copy (viewport->message_filter);

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_update_projection:
 * @viewport: a #PgmViewport object.
 *
 * Update the projection. The projection update is done automatically by
 * Pigment, this function is only useful for plugin that have changed for
 * instance the size of the viewport, and want to adapt the projection
 * accordingly. 
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_update_projection (PgmViewport *viewport)
{
  PgmCanvas *canvas = viewport->canvas;
  PgmError ret = PGM_ERROR_X;
  PgmViewportClass *klass;
  gint resolution_width;
  gint resolution_height;
  gint screen_width;
  gint screen_height;
  gfloat tmp;

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  /* Get resolution and screen size */
  pgm_viewport_get_screen_resolution (viewport, &resolution_width,
                                      &resolution_height);
  pgm_viewport_get_screen_size_mm (viewport, &screen_width, &screen_height);

  GST_OBJECT_LOCK (viewport);

  /* Get pixel-aspect-ratio function of the resolution and screen size */
  viewport->pixel_aspect_ratio = ((gfloat) resolution_width / resolution_height)
    * ((gfloat) screen_height / screen_width);

  /* Get the viewport ratio */
  viewport->viewport_ratio = (gfloat) viewport->width / viewport->height;

  /* Get the canvas ratio depending on the requested rotation */
  if (viewport->rotation == PGM_VIEWPORT_ROTATION_NONE
      || viewport->rotation == PGM_VIEWPORT_ROTATION_180)
    {
      if (canvas)
        {
          tmp = canvas->width * viewport->pixel_aspect_ratio;
          viewport->canvas_ratio = tmp / canvas->height;
        }
      else
        viewport->canvas_ratio = 4.0f / 3.0f * viewport->pixel_aspect_ratio;
    }
  else
    {
      if (canvas)
        {
          tmp = canvas->height * viewport->pixel_aspect_ratio;
          viewport->canvas_ratio = tmp / canvas->width;
        }
      else
        viewport->canvas_ratio = 3.0f / 4.0f * viewport->pixel_aspect_ratio;
    }

  /* Get the projected position and size */
  if (viewport->viewport_ratio >= viewport->canvas_ratio)
    {
      tmp = viewport->width - viewport->height * viewport->canvas_ratio;
      viewport->projected_x = (gint) (tmp * 0.5f);
      viewport->projected_y = 0;
      viewport->projected_width = (gint) (viewport->width - tmp);
      viewport->projected_height = viewport->height;
    }
  else
    {
      tmp = viewport->height - viewport->width / viewport->canvas_ratio;
      viewport->projected_x = 0;
      viewport->projected_y = (gint) (tmp * 0.5f);
      viewport->projected_width = viewport->width;
      viewport->projected_height = (gint) (viewport->height - tmp);
    }

  /* Update projection matrices */
  if (canvas)
    {
      gfloat x_scale, y_scale;

      /* FIXME: Explanations really needed here... */
      pgm_mat4x4_set_from_scalars (viewport->projection,
                                   763.94f,   0.0f,   0.0f,    0.0f,
                                     0.0f,  763.94f,  0.0f,    0.0f,
                                     0.0f,    0.0f,  -1.1f, -553.22f,
                                     0.0f,    0.0f,  -1.0f,    0.0f);

      /* Adapt the scaling variables to the requested rotation */
      if (viewport->rotation == PGM_VIEWPORT_ROTATION_NONE
          || viewport->rotation == PGM_VIEWPORT_ROTATION_180)
        {
          x_scale = 2.0f / canvas->width;
          y_scale = 2.0f / canvas->height;
        }
      else
        {
          x_scale = 2.0f / canvas->height;
          y_scale = 2.0f / canvas->width;
        }

      /* Adapt the scaling variables to the requested reflection */
      x_scale *= viewport->projection_vflip;
      y_scale *= viewport->projection_hflip;

      /* Scale the projection to fit the viewport. The projection is flipped
       * horizontally because the origin is at the upper-left in Pigment. */
      pgm_mat4x4_scale_from_scalars (viewport->projection,
                                     x_scale, -y_scale, 1.0f);

      /* Rotate the canvas with the requested angle */
      pgm_mat4x4_rotate_z (viewport->projection, viewport->projection_rotation);

      /* Then finally translate to the upper-left origin */
      pgm_mat4x4_translate_from_scalars (viewport->projection,
                                         -canvas->width / 2.0f,
                                         -canvas->height / 2.0f,
                                         -763.9432905087f);

      /* Store inverse matrix */
      pgm_mat4x4_free (viewport->inv_projection);
      viewport->inv_projection = pgm_mat4x4_inverse (viewport->projection);
    }

  GST_OBJECT_UNLOCK (viewport);

  /* Let the plugin apply the parameters */
  if (klass->update_projection)
    ret = klass->update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_from_canvas:
 * @viewport: a #PgmViewport object.
 * @viewport_x: a gfloat address to store the projected viewport x component.
 * @viewport_y: a gfloat address to store the projected viewport y component.
 * @viewport_z: a gfloat address to store the projected viewport z component.
 * @canvas_x: the canvas x component to project.
 * @canvas_y: the canvas y component to project.
 * @canvas_z: the canvas z component to project.
 *
 * Retrieves the projection of a 3-components vector in canvas coordinates on
 * the viewport in a 3-components vector in viewport coordinates.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_from_canvas (PgmViewport *viewport,
                          gfloat *viewport_x,
                          gfloat *viewport_y,
                          gfloat *viewport_z,
                          gfloat canvas_x,
                          gfloat canvas_y,
                          gfloat canvas_z)
{
  PgmVec4 *viewport_vec, canvas_vec;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (viewport->canvas, PGM_ERROR_X);
  g_return_val_if_fail (viewport_x && viewport_y && viewport_z, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  /* Convert from world (canvas) coordinates to clipping coordinates */
  pgm_vec4_set_from_scalars (&canvas_vec, canvas_x, canvas_y, canvas_z, 1.0f);
  viewport_vec = pgm_mat4x4_multiply_vec4 (viewport->projection, &canvas_vec);

  /* Perspective divide, convert from clipping coordinates to NDC */
  viewport_vec->v[0] /= viewport_vec->v[3];
  viewport_vec->v[1] /= viewport_vec->v[3];
  viewport_vec->v[2] /= viewport_vec->v[3];

  /* Convert from NDC to viewport coordinates */
  *viewport_x = (viewport_vec->v[0] + 1.0f) / 2.0f * viewport->projected_width;
  *viewport_y = (1.0f - viewport_vec->v[1]) / 2.0f * viewport->projected_height;
  *viewport_z = (viewport_vec->v[2] + 1.0f) / 2.0f;

  /* And add the position offset */
  *viewport_x += viewport->projected_x;
  *viewport_y += viewport->projected_y;

  GST_OBJECT_UNLOCK (viewport);

  pgm_vec4_free (viewport_vec);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_to_canvas:
 * @viewport: a #PgmViewport object.
 * @canvas_x: a gfloat address to store the unprojected canvas x component.
 * @canvas_y: a gfloat address to store the unprojected canvas y component.
 * @canvas_z: a gfloat address to store the unprojected canvas z component.
 * @viewport_x: the viewport x component to project.
 * @viewport_y: the viewport y component to project.
 * @viewport_z: the viewport z component to project.
 *
 * Retrieves the projection of a 3-components vector in viewport coordinates on
 * the viewport in a 3-components vector in canvas coordinates. The z component
 * of the viewport being clamped in the range [0.0, 1.0].
 *
 * Note that if you want to get a canvas vector with a z coordinates of 0.0f,
 * you can give -1.0f to @viewport_z. It can be useful when you want to convert
 * 2d position in viewport coordinates to canvas coordinates. 
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_to_canvas (PgmViewport *viewport,
                        gfloat *canvas_x,
                        gfloat *canvas_y,
                        gfloat *canvas_z,
                        gfloat viewport_x,
                        gfloat viewport_y,
                        gfloat viewport_z)
{
  PgmVec4 *canvas_vec, viewport_vec;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (viewport->canvas, PGM_ERROR_X);
  g_return_val_if_fail (canvas_x && canvas_y && canvas_z, PGM_ERROR_X);

  /* Get canvas coordinates at z zero */
  if (viewport_z == -1.0f)
    viewport_z = DEPTH_ZERO_Z;
  else
    viewport_z = CLAMP (viewport_z, 0.0f, 1.0f);

  GST_OBJECT_LOCK (viewport);

  /* Remove the position offset */
  pgm_vec4_set_from_scalars (&viewport_vec, viewport_x, viewport_y,
                             viewport_z, 1.0f);
  viewport_vec.v[0] -= viewport->projected_x;
  viewport_vec.v[1] -= viewport->projected_y;

  /* Convert from viewport coordinates to clipping coordinates */
  viewport_vec.v[0] =
    ((2.0f * viewport_vec.v[0]) / viewport->projected_width) - 1.0f;
  viewport_vec.v[1] =
    1.0f - ((2.0f * viewport_vec.v[1]) / viewport->projected_height);
  viewport_vec.v[2] = (2.0f * viewport_vec.v[2]) - 1.0f;

  /* Convert from clipping coordinates to world (canvas) coordinates */
  canvas_vec = pgm_mat4x4_multiply_vec4 (viewport->inv_projection,
                                         &viewport_vec);

  GST_OBJECT_UNLOCK (viewport);

  /* No guarantee that w is 1, so rescale appropriately */
  *canvas_x = canvas_vec->v[0] / canvas_vec->v[3];
  *canvas_y = canvas_vec->v[1] / canvas_vec->v[3];
  *canvas_z = canvas_vec->v[2] / canvas_vec->v[3];

  pgm_vec4_free (canvas_vec);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_get_embedding_id:
 * @viewport: a #PgmViewport object.
 * @embedding_id: a pointer to a #gulong where the embedding ID is going to
 * be stored.
 *
 * Gets the embedding ID of @viewport. It can be used by other applications
 * to embed it.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_embedding_id (PgmViewport *viewport,
                               gulong *embedding_id)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (embedding_id != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_embedding_id)
    ret = klass->get_embedding_id (viewport, embedding_id);

  return ret;
}

/**
 * pgm_viewport_get_pixel_formats:
 * @viewport: a #PgmViewport object.
 * @formats_mask: a pointer to a #gulong where the mask of supported
 * #PgmImagePixelFormat is going to be stored.
 *
 * Gets the list of supported pixel formats of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_pixel_formats (PgmViewport *viewport,
                                gulong *formats_mask)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (formats_mask != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_pixel_formats)
    ret = klass->get_pixel_formats (viewport, formats_mask);

  return ret;
}

/**
 * pgm_viewport_get_caps_mask:
 * @viewport: a #PgmViewport object.
 * @caps_mask: a pointer to a #gulong where the mask of #PgmViewportCapacity is
 * going to be stored.
 *
 * Retrieves the mask of supported #PgmViewportCapacity.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_caps_mask (PgmViewport *viewport,
                            gulong *caps_mask)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (caps_mask != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_caps_mask)
    ret = klass->get_caps_mask (viewport, caps_mask);

  return ret;
}

/**
 * pgm_viewport_get_frame_rate:
 * @viewport: a #PgmViewport object.
 * @frame_rate: a pointer to a #guint where the frame rate is going to be
 * stored.
 *
 * Retrieves the frame rate in frames per second of the rendering of the
 * canvas on @viewport. The value retrieved changes each second, and
 * represents the number of frames that have been rendered in the previous
 * second.
 *
 * That function could be useful to check the rendering performance on a
 * system, and for instance tell the user if too many frames are dropped.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_frame_rate (PgmViewport *viewport,
                             guint *frame_rate)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (frame_rate != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_frame_rate)
    ret = klass->get_frame_rate (viewport, frame_rate);

  return ret;
}

/**
 * pgm_viewport_read_pixels:
 * @viewport: a #PgmViewport object.
 * @x: the x coordinate of the first pixel to read from the left of @viewport.
 * @y: the y coordinate of the first pixel to read from the top of @viewport.
 * @width: the width of the pixel area to read, corresponds to a single pixel.
 * @height: the height of the pixel area to read, corresponds to a single pixel.
 * @pixels: a pre-allocated memory area where the pixels read will be written.
 * The buffer must be freed by the application, in the "pixels-read" signal
 * callback for instance.
 *
 * Requests the reading of a pixel area in @viewport, starting with the pixel
 * whose top-left corner is at location (@x, @y) and dimension is
 * (@width, @height). Once the pixel area is obtained, the "pixels-read" signal
 * is emitted with the requested pixels stored at location @pixels. The
 * retrieved pixel area is in the RGBA color space.
 *
 * @pixels should be allocated by the application with the necessary size
 * (width * height * 4) before any call to that function. A callback must be
 * connected to the "pixels-read" signal before any call to that function.
 * @x, @y, @width and @height are clamped if their values lie outside @viewport
 * size.
 *
 * MT Safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_read_pixels (PgmViewport *viewport,
                          guint x,
                          guint y,
                          guint width,
                          guint height,
                          guint8 *pixels)
{
  PgmError ret = PGM_ERROR_OK;
  PgmViewportClass *klass;
  gulong handler_id;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (pixels != NULL, PGM_ERROR_X);

  /* A callback must be connected to the "pixels-read" signal */
  handler_id = g_signal_handler_find (viewport, G_SIGNAL_MATCH_ID
                                      | G_SIGNAL_MATCH_UNBLOCKED,
                                      pgm_viewport_signals[PIXELS_READ], 0,
                                      NULL, NULL, NULL);
  g_return_val_if_fail (handler_id, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  /* x and y are clamped to the viewport size */
  x = MIN (x, (guint) viewport->width);
  y = MIN (y, (guint) viewport->height);

  /* width and height are clamped to the remaining size */
  width = MIN (width, viewport->width - x);
  height = MIN (height, viewport->height - y);

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->read_pixels)
    ret = klass->read_pixels (viewport, x, y, width, height, pixels);

  return ret;
}

/**
 * pgm_viewport_push_pixels:
 * @viewport: a #PgmViewport object.
 * @width: the width of the pixel area.
 * @height: the height of the pixel area.
 * @pixels: the pixels to push.
 *
 * Push a pixel area to @viewport. The function will emit the "pixels-read"
 * signal from the Pigment main loop with the given pixel area parameters.
 *
 * A plugin must call this function for each read_pixels() call that does not
 * return an error, because the user may free the pixel array in the callback
 * called from there.
 *
 * This function should only be used by plugins.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_push_pixels (PgmViewport *viewport,
                          guint width,
                          guint height,
                          guint8 *pixels)
{
  PixelRectangle *rectangle;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (pixels, PGM_ERROR_X);

  rectangle = g_slice_new (PixelRectangle);
  if (!rectangle)
    return PGM_ERROR_X;

  rectangle->width = width;
  rectangle->height = height;
  rectangle->pixels = pixels;

  g_mutex_lock (viewport->pixel_lock);
  viewport->pixel_list = g_list_prepend (viewport->pixel_list, rectangle);
  g_mutex_unlock (viewport->pixel_lock);

  if (G_LIKELY (viewport->pixel_watch_up))
    {
      g_io_channel_write_chars (viewport->pixel_watch->in, "#", 1, NULL, NULL);
      g_io_channel_flush (viewport->pixel_watch->in, NULL);
    }

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_emit_update_pass:
 * @viewport: the #PgmViewport object.
 *
 * Emits the 'update-pass' signal on @viewport.
 *
 * This function should only be used by plugins.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_emit_update_pass (PgmViewport *viewport)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  g_signal_emit (G_OBJECT (viewport), pgm_viewport_signals[UPDATE_PASS], 0);

  return PGM_ERROR_OK;
}

/* Protected methods */

#ifdef WIN32
/**
 * pgm_viewport_push_win32_message:
 * @viewport: [self] the #PgmViewport object.
 * @event: [in] the Win32 message event.
 * @message_processed: [out] the atomically set condition.
 * @result: [out] the WinProc result.
 *
 * Push a Win32 message event in the list, the message_processed condition has
 * to be atomiocally set once the result value is set so that the caller can
 * return from the nested WinProc.
 *
 * This function should only be used by plugins.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
_pgm_viewport_push_win32_message (PgmViewport *viewport,
                                  PgmEvent *event,
                                  gint32 *message_processed,
                                  LRESULT *result)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (event != NULL, PGM_ERROR_X);
  g_return_val_if_fail (message_processed != NULL, PGM_ERROR_X);
  g_return_val_if_fail (result != NULL, PGM_ERROR_X);

  if (G_LIKELY (viewport->event_watch_up))
    {
      Win32Message *message;

      message = g_slice_new (Win32Message);
      if (!message)
        return PGM_ERROR_X;

      message->type = PGM_WIN32_MESSAGE;
      message->event = event;
      message->message_processed = message_processed;
      message->result = result;

      g_mutex_lock (viewport->event_lock);
      viewport->event_list = g_list_prepend (viewport->event_list, message);
      g_mutex_unlock (viewport->event_lock);

      g_io_channel_write_chars (viewport->event_watch->in, "#", 1, NULL, NULL);
      g_io_channel_flush (viewport->event_watch->in, NULL);
    }
  else
    {
      *result = 0;
      g_atomic_int_set (message_processed, 1);
    }

  return PGM_ERROR_OK;
}
#endif /* WIN32 */
