/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment sphere example
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Guillaume Emont <guillaume@fluendo.com>
 */

#include <pgm/pgm.h>
#include <cairo.h>
#include <math.h>

#define offset 5

static void
key_press_event_cb (PgmViewport *viewport,
                    PgmEventKey *event,
                    PgmDrawable *drawable)
{
  gfloat x, y, z;

  pgm_drawable_get_position (drawable, &x, &y, &z);
  switch (event->keyval)
    {
    case PGM_Up:
      y -= offset;
      break;
    case PGM_Down:
      y += offset;
      break;
    case PGM_Left:
      x -= offset;
      break;
    case PGM_Right:
      x += offset;
      break;
    case PGM_s:
    case PGM_S:
      z -= offset;
      break;
    case PGM_x:
    case PGM_X:
      z += offset;
      break;

    case PGM_q:
    case PGM_Q:
      pgm_main_quit ();
      return;

    default:
      return;
    }

  pgm_drawable_set_position (drawable, x, y, z);
  pgm_viewport_update (viewport);
}

static void
configure_event_cb (PgmViewport *viewport,
                    PgmEventConfigure *event,
                    gpointer data)
{
  pgm_viewport_update (viewport);
}

static void
expose_event_cb (PgmViewport *viewport,
                 PgmEventExpose *event,
                 gpointer data)
{
  pgm_viewport_update (viewport);
}

/* allocates a memory buffer in data and draws a gradient sphere in it */
PgmError
cairo_drawing (guchar **data,
               gint width,
               gint height)
{
  PgmError err = PGM_ERROR_OK;
  cairo_t *ctx;
  cairo_surface_t *surface;
  cairo_pattern_t *pat;

  if (!data)
    return PGM_ERROR_X;

  *data = calloc (width * height * 4, sizeof (guchar));
  surface = cairo_image_surface_create_for_data (*data, CAIRO_FORMAT_ARGB32,
                                                 width, height, width * 4);
  ctx = cairo_create (surface);
  if (!(surface && ctx))
    {
      err = PGM_ERROR_X;
      goto tidy;
    }
  cairo_scale (ctx, width, height);

  pat = cairo_pattern_create_radial (0.45, 0.4, 0.1, 0.4, 0.4, 0.5);
  if (!pat)
    {
      err = PGM_ERROR_X;
      goto tidy;
    }
  cairo_pattern_add_color_stop_rgba (pat, 0, 0.8, 0.8, 1, 1);
  cairo_pattern_add_color_stop_rgba (pat, 1, 0, 0, 1, 1);

  cairo_set_source (ctx, pat);
  cairo_arc (ctx, 0.5, 0.5, 0.3, 0, 2 * M_PI);
  cairo_fill (ctx);

tidy:
  cairo_pattern_destroy (pat);
  cairo_surface_destroy (surface);
  cairo_destroy (ctx);

  return err;
}

int
main (int argc,
      char *argv[])
{
  PgmError ret = PGM_ERROR_OK;
  PgmViewportFactory *factory;
  PgmViewport *gl;
  PgmCanvas *canvas;
  PgmDrawable *txt;
  PgmDrawable *img;
  guchar *data;

  g_print ("Use the arrows, 's' and 'x' to move the sphere around\n");

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  factory = pgm_viewport_factory_new ("opengl");
  pgm_viewport_factory_create (factory, &gl);
  if (!gl)
    {
      g_print ("No \"opengl\" viewport found\n");
      return -1;
    }
  pgm_viewport_set_title (gl, "Sphere");

  /* sphere creation */
  ret = cairo_drawing (&data, 400, 400);
  g_assert (PGM_ERROR_OK == ret);

  img = pgm_image_new_from_buffer (PGM_IMAGE_BGRA, 400, 400, 400 * 4,
                                   400 * 400 * 4, data);
  pgm_drawable_set_bg_color (img, 255, 255, 255, 0);
  pgm_drawable_set_position (img, 200, 100, 0);
  pgm_drawable_set_size (img, 400, 400);
  pgm_drawable_show (img);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 800, 600);
  pgm_viewport_set_canvas (gl, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, PGM_DRAWABLE (img));

  /* Main loop */
  g_signal_connect (G_OBJECT (gl), "key-press-event",
                    G_CALLBACK (key_press_event_cb), img);
  g_signal_connect (G_OBJECT (gl), "configure-event",
                    G_CALLBACK (configure_event_cb), NULL);
  g_signal_connect (G_OBJECT (gl), "expose-event",
                    G_CALLBACK (expose_event_cb), NULL);
  g_signal_connect (G_OBJECT (gl), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  pgm_viewport_show (gl);
  pgm_viewport_update (gl);
  pgm_main ();

  /* Deinit */
  g_object_unref (canvas);
  g_object_unref (gl);
  g_object_unref (factory);
  pgm_deinit ();
  free (data);

  return 0;
}
