# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import pgm
from pypgmtools.graph.group import Group
from pypgmtools.timing import implicit

from math import *

class RingList(Group):

    def __init__(self,
                 canvas,
                 layer,
                 selected_item = 0):

        Group.__init__(self, canvas, layer)

        self._selected_item    = selected_item
        self._canvas           = canvas

        self._attributes       = []
        self.widgets   = []
        self.animated_widgets = []

        self._layout()


    def _compute_position(self, index):
        if len(self.widgets) > 0:
            c = float(index) / len(self.widgets) * pi * 2.0
            x = sin(c)
            y = cos(c) / 1.5
            z = cos(c) * 500.0 - 200.0
        else:
            x = 0.0
            y = 0.0
            z = 0.0

        position = (x, y, z)
        return position

    def _compute_opacity(self, index):
        return 255

    def _compute_zoom(self, index):
        return 1.0

    def _compute_attributes(self, index):
        return (self._compute_position(index),
                self._compute_opacity(index),
                self._compute_zoom(index))

    def _layout(self):
        self._attributes = []
        for i in xrange(len(self.widgets)):
            self._attributes.append(self._compute_attributes(i))

    def _update(self):
        for i in xrange(len(self._attributes)):
            j = (i + self._selected_item) % len(self.animated_widgets)
            widget = self.animated_widgets[j]
            #widget.opacity  = self._attributes[i][1]
            widget.position = self._attributes[i][0]
            zoom = self._attributes[i][2]
            #widget.size = (widget.width * zoom, widget.height * zoom)

    def selected_item__get(self):
        return self._selected_item

    def selected_item__set(self, index):
        if index >= 0 and index < len(self.widgets):
            # FIXME
            """
            if index > self._selected_item > 0:
                increment = 1
            else:
                increment = -1

            for i in xrange(self._selected_item+increment, index+increment, increment):
                self._selected_item = i % len(self.widgets)
                self._update()
            """

            self._selected_item = index
            self._update()

    def __len__(self):
        return len(self.widgets)

    def insert(self, index, widget):
        # text insertion
        if len(self.widgets) == 0:
            widget.position = self._compute_position(0)
        elif abs(index) < len(self.widgets):
            # insert it at the position of self.animated_widgets[index]
            widget.position = self.animated_widgets[index].position
        else:
            widget.position = self._compute_position(len(self.widgets))

        # FIXME: should take the zoom into account
        #widget.size = (self._width, self._widget_height)

        widget.opacity = 255

        widget.visible = True

        Group.add(self, widget)
        self.widgets.insert(index, widget)

        animated_widget = implicit.AnimatedObject(widget)
        animated_widget.setup_next_animations(duration = 300)
        self.animated_widgets.insert(index, animated_widget)

        if index < self._selected_item:
            self._selected_item += 1

        self._layout()
        self._update()

    def append(self, widget):
        self.insert(len(self.widgets), widget)

    def pop(self, index):
        static = self.widgets.pop(index)

        Group.remove(self, static)
        self.animated_widgets.pop(index)

        if index == self._selected_item and index == len(self.widgets):
            self.selected_item -= 1

        self._layout()
        self._update()

    def __getitem__(self, index):
        return self.widgets[index]


if __name__ == "__main__":
    import os, pgm, gobject, gst
    from pypgmtools.graph.text import Text
    from pypgmtools.graph.image import Image
    from pypgmtools.graph.group import Group

    def create_text(label):
        txt = Text()
        txt.label = label
        txt.font_family = "Bitstream DejaVu"
        txt.font_height = 0.225
        txt.fg_color = (255, 255, 255, 255)
        txt.bg_color = (255, 0, 0, 0)
        txt.ellipsize = pgm.TEXT_ELLIPSIZE_END
        txt.alignment = pgm.TEXT_ALIGN_CENTER
        return txt

    def create_img(img_file):
        img = Image()
        img.set_from_fd(os.open(img_file, os.O_RDONLY))
        img.fg_color = (255, 255, 255, 255)
        img.bg_color = (100, 200, 100, 0)
        return img

    def create_video(video_uri):
        img = Image()
        img.fg_color = (255, 255, 255, 255)
        img.bg_color = (0, 0, 0, 0)
        img.alignment = pgm.IMAGE_LEFT

        # GStreamer pipeline setup
        pipeline = gst.element_factory_make('playbin')
        sink = gst.element_factory_make('pgmimagesink')
        pipeline.set_property('uri', video_uri)
        pipeline.set_property('video-sink', sink)
        sink.set_property('image', img)
        pipeline.set_state(gst.STATE_PLAYING)

        return img


    def on_key_press(viewport, event, list_widget):
        if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
            pgm.main_quit()

        elif event.keyval == pgm.keysyms.b:
            list_widget.selected_item = 0

        elif event.keyval == pgm.keysyms.g:
            list_widget.selected_item = len(list_widget) - 1

        elif event.keyval == pgm.keysyms.Down or \
             event.keyval == pgm.keysyms.Right:
            n = list_widget.selected_item + 1
            list_widget.selected_item = n % len(list_widget)

        elif event.keyval == pgm.keysyms.Up or \
             event.keyval == pgm.keysyms.Left:
            n = list_widget.selected_item - 1
            if n < 0:
                list_widget.selected_item = len(list_widget) - 1
            else:
                list_widget.selected_item = n

        # add an example entry at the 3rd position
        elif event.keyval == pgm.keysyms.space:
            list_widget.insert(2, create_text("Test"))
            #list_widget.append(create_text("Test"))

        # remove the currently selected item
        elif event.keyval == pgm.keysyms.Return:
            list_widget.pop(list_widget.selected_item)

    def on_delete(viewport, event):
        pgm.main_quit()



    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'TextList widget-test'

    # Canvas and image drawable creation
    canvas = pgm.Canvas()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)
    gl.show()

    list_widget = RingList(canvas, pgm.DRAWABLE_MIDDLE)
    list_widget.position = (1.5, 0.5, 0.0)

    for i in xrange(5):
        #list_widget.append(create_text("%s" % i))
        list_widget.append(create_img("examples/pictures/fluendo.png"))

        """
        t = create_text("%s" % i)
        i = create_img("examples/pictures/fluendo.png")
        g = Group(canvas, pgm.DRAWABLE_MIDDLE)
        g.add(t)
        g.add(i)
        t.position = (0.0, 0.0, 0.0)
        t.size = (1.0, 0.5, 0.0)
        i.position = (0.0, 0.5, 0.0)
        i.size = (1.0, 0.5, 0.0)
        g.visible = True
        list_widget.append(g)
        """

#    list_widget.append(create_video("file:///home/kaleo/media/vrac/Xraalthraal_and_John.mov"))

    list_widget.selected_item = 3
    list_widget.visible = True

    # Let's start a mainloop
    gl.connect('key-press-event', on_key_press, list_widget)
    gl.connect('delete-event', on_delete)
    gobject.timeout_add(15, gl.update)
    pgm.main()
