# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image # FIXME: should be replaced with Rectangle
import pgm
from pypgmtools.timing import implicit
import gst


class Scrollbar(Group):

    def __init__(self, canvas, length, shown):
        """
        The Scrollbar is a vertical bar
        @param canvas : the canvas to paint on
        @type canvas: a pgm.Canvas
        @param lenght: the length of the process
        @type process: anything, with which we can do the mathematic things
        @param shown: how many are shown?
        @type shown: anything, with which we can do the mathematic things
        """
        Group.__init__(self,canvas)
        ## Making background of the bar.
        self._bg = Image()
        self._list = length
        if self._list <= 0:
            self._list = 1
        if list >= shown:
            self._shown = shown
        else:
            self._shown = list
        self._cur_pos = 1
        self._progression = 0
        self._bg.height = 0
        self._bg.width = 0
        self._bg.size = (0,0)
        self._bg.bg_color = (0,0,0,255) ##We are black
        self._bg.visible = True

        self._cursor = Image()
        self._cursor.height = 0
        self._cursor.width = 0
        self._cursor.visible = True
        self._cursor.bg_color= (255,255,255,255) ## We are white
        self._animated_cursor = implicit.AnimatedObject(self._cursor)
        self.add(self._bg)
        self.add(self._cursor)
        self.visible = True

    def size__set(self,size):
        """
        Set the size of the bar. The bar is automatically new painted
        """
        self._bg.size = size
        self._cursor.width = size[0]
        self._set_cursor_to_position()

    def length__set(self,num):
        """
        Set the length. The bar is automatically new painted
        @param length: the lenght of the process
        @type lenght: as above: something to do mathematic stuff
        """
        if num <= 0:
            num = 1
        self._list = num
        if self._list < self._shown:
            self._shown = self._list
        if self._cur_pos >  self._list:
            self._cur_pos = 1
        self._set_cursor_to_position()

    def number_shown__set(self,num):
        if num <= 0:
            num = 1
        self._shown = num
        self._set_cursor_to_position()

    def number_shown__get(self):
        return self._shown

    def length__get(self):
        """
        to get the length:
        @return: returns the length
        """
        return self._list

    def _set_cursor_to_position(self):
        """
        Here the position is calculated and the bar is painted
        """
        height = self._bg.height
        if (self._list * self._shown) < 1: ## We can't do this if self._list or self._shown == 0
            self._list = 1
            self._shown = 1
            self._cur_pos = 1
        height = (self._bg.height / self._list * self._shown)
        self._animated_cursor.position = (self._bg.position[0],
                                                                self._bg.position[1] + (
                                                                ((self._bg.height - height )
                                                                / self._list  * self._cur_pos))  ,
                                                                self._bg.position[2]+1)
        self._animated_cursor.height = height

    def cursor_position__set(self, num):
        """
        Set the position of the cursor. The bar is automatically new painted
        @param position: the position to set to
        @type position: something to do mathematic stuff
        """
        if num < 0 or num > self._list:
            pass
        else:
            self._cur_pos = num
            self._set_cursor_to_position()

    def cursor_position__get(self):
        """
        returns the position of the cursor
        @return : the position of the cursor
        """
        return self._cur_pos

    def next_item(self):
        """
        go to the next item.  The bar is automatically new painted
        """
        if not self._cur_pos >= self._list:
            self._cur_pos = self._cur_pos + 1
            self._set_cursor_to_position()

    def previous_item(self):
        """
        go to the previous item.  The bar is automatically new painted
        """
        if not self._cur_pos <= 0:
            self._cur_pos = self._cur_pos - 1
            self._set_cursor_to_position()

    def bg_color__set(self, color):
        """
        Set the background color. The bar is automatically new painted
        @param color: color of the background
        """
        self._bg.bg_color = color

    def bg_color__get(self):
        """
        returns the current background color
        """
        return self._bg.bg_color

    def bar_color__set(self, color):
        """
        Set the color of the bar. The bar is automatically new painted
        @param color: color of the bar
        """
        self._bar.bg_color = color

    def bar_color__get(self):
        """
        return the current color of the bar
        """
        return self._bar.bg_color

class LinedScrollbar(Scrollbar):

    def __init__(self, canvas, length, shown, strength):
        """
        The LinedScrollbar inheritets from the Scrollbar, but it shows
        only the borders of the bar and the cursor
        @param canvas : the canvas to paint on
        @type canvas: a pgm.Canvas
        @param lenght: the length of the process
        @type process: anything, with which we can do the mathematic things
        @param shown: how many are shown?
        @type shown: anything, with which we can do the mathematic things
        """
        Scrollbar.__init__(self, canvas, length, shown)
        self._bg.visible = False
        ## We need it, but we don't show it!
        self._left = Image()
        self._left.width = strength
        self._left.visible = True
        self.add(self._left)

        self._right = Image()
        self._right.width = strength
        self._right.visible = True
        self.add(self._right)

    def position__set(self, position):
        """
        Set the position of the bar
        """
        Scrollbar.position__set(self,position)
        self._repositioning()

    def _repositioning(self):
        """
        reposition the borders
        """
        self._left.position = (self._bg.position[0] - self._left.width,
                                            self._bg.position[1],
                                            self._bg.position[2] + 2)
        self._right.position = (self._bg.position[0] - self._left.width +
                                            self._bg.width,
                                            self._bg.position[1],
                                            self._bg.position[2] + 2)

    def bg_color__set(self, color):
        """
        Set the color of the borders. The bar is automatically new painted
        @param color: color of the background
        """
        self._left.bg_color = color
        self._right.bg_color = color

    def size__set(self,size):
        """
        Set the size of the bar. The bar is automatically new painted
        """
        Scrollbar.size__set(self,size)
        self._left.height = size[1]
        self._right.height = size[1]
        self._repositioning()

if __name__ == "__main__":
    from pypgmtools.graph.image import Image
    from pypgmtools.graph.text import Text
    import pgm
    import os, gobject, sys

    def handle_events(bar, gl, loop):
        for event in gl.get_events():
            if event.type == pgm.DELETE:
                loop.quit()

            if event.type == pgm.KEY_PRESS:
                if event.keyval == pgm.KEY_ESCAPE or event.keyval == pgm.KEY_q:
                    loop.quit()

                elif event.keyval == pgm.KEY_UP:
                    bar.previous_item()
                    lbar.previous_item()
                    #print bar.progress

                elif event.keyval == pgm.KEY_DOWN:
                    bar.next_item()
                    lbar.next_item()
                    #print bar.progress

                elif event.keyval == pgm.KEY_p:
                    bar.number_shown = bar.number_shown + 10
                    lbar.number_shown = lbar.number_shown + 10
                    #print bar.progress

                elif event.keyval == pgm.KEY_o:
                    bar.number_shown = bar.number_shown - 10
                    lbar.number_shown = lbar.number_shown - 10
                    #print bar.progress

                elif event.keyval == pgm.KEY_e:
                    bar.list_length = bar.list_length  - 10
                    lbar.list_length = lbar.list_length - 10
                    print bar.list_length, lbar.list_length
                elif event.keyval == pgm.KEY_r:
                    bar.list_length = bar.list_length  + 10
                    lbar.list_length = lbar.list_length + 10
                    print bar.list_length, lbar.list_length
                elif event.keyval == pgm.KEY_i:
                    bar.cursor_position = 400
                    lbar.cursor_position = 400
        return True


    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'bar widget'

    canvas = pgm.Canvas()

    gl.set_canvas(canvas)
    bar = Scrollbar(canvas, 100, 12)
    bar.bg_color = 38, 139, 200, 255
    bar.cursor_color = 200, 239, 250, 155
    bar.position = (canvas.width * 0.05, canvas.height * 0.1, 0)
    bar.size = (canvas.width * 0.01, canvas.height * 0.8)

    lbar = LinedScrollbar(canvas, 800, 100, canvas.width * 0.001)
    lbar.bg_color = 200, 200, 200, 255
    lbar.cursor_color = 200, 239, 250, 155
    lbar.position = (canvas.width * 0.8, canvas.height * 0.1, 0)
    lbar.size = (canvas.width * 0.01, canvas.height * 0.8)

    loop = gobject.MainLoop()
    gobject.timeout_add(5, gl.update)
    gobject.timeout_add(15, handle_events, bar, gl, loop)
    gobject.timeout_add(15, handle_events, lbar, gl, loop)
    loop.run()
